/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.polaris.service.catalog;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.maprfs.AbstractMapRFileSystem;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.authorize.UsersACLsManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URI;


public class CatalogMfsUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(CatalogMfsUtils.class);

    private static final String PUBLIC_ACE = "rf:p,wf:p,ef:p,rd:p,ac:p,dc:p,ld:p";

    /**
     * Build string with ACE to allow  users access to catalog
     *
     * @param users - list of users that should have access to catalog
     * @return ACE in string format
     */
    public static String buildACEStrForUsers(String users) throws IOException {
        UserGroupInformation ugi = UserGroupInformation.getCurrentUser();
        StringBuilder aceUserList = new StringBuilder();
        aceUserList.append(":(");
        for (String user : users.split(",")) {
            aceUserList.append("u:").append(user).append("|");
        }
        aceUserList.append("u:").append(ugi.getShortUserName()).append(")");
        StringBuilder ace = new StringBuilder();

        //read
        ace.append(UsersACLsManager.ACE.READ_FILE);
        ace.append(aceUserList);

        //write
        ace.append(",").append(UsersACLsManager.ACE.WRITE_FILE);
        ace.append(aceUserList);

        //execute
        ace.append(",").append(UsersACLsManager.ACE.EXECUTE_FILE);
        ace.append(aceUserList);

        //read dir
        ace.append(",").append(UsersACLsManager.ACE.READ_DIR);
        ace.append(aceUserList);

        //lookup dir owner and all allow users
        ace.append(",").append(UsersACLsManager.ACE.LOOKUP_DIR);
        ace.append(aceUserList);

        //add child only for owner
        ace.append(",").append(UsersACLsManager.ACE.ADD_CHILD);
        ace.append(aceUserList);

        //delete child only for owner
        ace.append(",").append(UsersACLsManager.ACE.DELETE_CHILD);
        ace.append(aceUserList);

        return ace.toString();
    }

    /**
     * Change ACE for catalog directory
     */
    public static void changeCatalogAce(AbstractMapRFileSystem fs, String location, String users) throws IOException {
        String ace = CatalogMfsUtils.buildACEStrForUsers(users);
        fs.setAces(new Path(location), ace, true, 0, 0, true);
    }

    /**
     * Set ACE for users to catalog directory at MFS or set public ACE if catalog-users property is unset
     **/
    public static void changeCatalogDirectoryPermission(String location, String users, String catalogName) {
        try {
            AbstractMapRFileSystem mfs = (AbstractMapRFileSystem) FileSystem.get(new URI(location), new Configuration());
            if (users != null && !users.isEmpty() && !users.equals("*")) {
                changeCatalogAce(mfs, location, users);
            } else {
                mfs.setAces(new Path(location), PUBLIC_ACE, true, 0, 0, true);
            }
        } catch (Exception e) {
            LOGGER.warn("Can't change ACE for catalog {}", catalogName, e);
        }
    }

}
