/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.polaris.service.quarkus.config;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.smallrye.config.ConfigSourceContext;
import io.smallrye.config.ConfigSourceFactory;
import io.smallrye.config.common.MapBackedConfigSource;
import org.apache.hadoop.conf.Configuration;
import org.eclipse.microprofile.config.spi.ConfigSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.apache.hadoop.fs.CommonConfigurationKeysPublic.HADOOP_SECURITY_CREDENTIAL_PROVIDER_PATH;

/*
 *
 * Implementation SSL configuration that uses Hadoop credentials provider
 *
 * */
public class SslConfigSourceFactory implements ConfigSourceFactory {

    private static final Logger LOGGER = LoggerFactory.getLogger(SslConfigSourceFactory.class);
    private static final String KEYSTORE_PASSWORD = "ssl.server.keystore.password";
    private static final String SERVICE_PASSWORD = "quarkus.http.ssl.certificate.key-store-password";
    private static final String MANAGEMENT_PASSWORD = "quarkus.management.ssl.certificate.key-store-password";

    @Override
    public Iterable<ConfigSource> getConfigSources(ConfigSourceContext context) {
        return getConfigSource();
    }

    protected List<ConfigSource> getConfigSource() {
        final List<ConfigSource> list = new ArrayList<>();
        String creds = System.getenv("POLARIS_SSL_CREDS_STORE");
        if (creds != null) {
            Configuration conf = new Configuration();
            conf.set(HADOOP_SECURITY_CREDENTIAL_PROVIDER_PATH, creds);
            LOGGER.debug("Getting keystore password from Hadoop credentials provider");
            Map<String, String> map = new HashMap<>();
            try {
                char[] sslKeystorePass = conf.getPassword(KEYSTORE_PASSWORD);
                if (sslKeystorePass != null) {
                    map.put(SERVICE_PASSWORD, new String(sslKeystorePass));
                    map.put(MANAGEMENT_PASSWORD, new String(sslKeystorePass));
                }
            } catch (IOException ioe) {
                LOGGER.warn("Cannot find {} option ", KEYSTORE_PASSWORD, ioe);
            }
            list.add(new InMemoryConfigSource("ssl-keystore-config", map, 400));
        }
        return list;
    }

    private static final class InMemoryConfigSource extends MapBackedConfigSource {
        public InMemoryConfigSource(String name, Map<String, String> propertyMap, int defaultOrdinal) {
            super(name, propertyMap, defaultOrdinal);
        }
    }

}
