/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.polaris.quarkus.common.config;

import java.io.IOException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.hadoop.conf.Configuration;
import org.eclipse.microprofile.config.spi.ConfigSource;

import io.smallrye.config.ConfigSourceContext;
import io.smallrye.config.ConfigSourceFactory;
import io.smallrye.config.common.MapBackedConfigSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.apache.hadoop.fs.CommonConfigurationKeysPublic.HADOOP_SECURITY_CREDENTIAL_PROVIDER_PATH;
import static org.apache.polaris.core.storage.aws.AwsCredentialsStorageIntegration.POLARIS_AWS_ACCESS_KEY;
import static org.apache.polaris.core.storage.aws.AwsCredentialsStorageIntegration.POLARIS_AWS_SECRET_KEY;
/*
 * Implementation for Credentials Config Source Factory.
 * Support uses Hadoop credentials provider to get username/password for quarkus datasource and
 * access/secret key for Object Store
 * */
public class CredentialsConfigSourceFactory implements ConfigSourceFactory {

    private static final Logger LOGGER = LoggerFactory.getLogger(CredentialsConfigSourceFactory.class);
    private static final String JDBC_USERNAME = "quarkus.datasource.username";
    private static final String JDBC_PASSWORD = "quarkus.datasource.password";
    List<String> properties = List.of(JDBC_USERNAME, JDBC_PASSWORD,
            POLARIS_AWS_ACCESS_KEY, POLARIS_AWS_SECRET_KEY);

    @Override
    public Iterable<ConfigSource> getConfigSources(ConfigSourceContext context) {
        return getConfigSource();
    }

    protected List<ConfigSource> getConfigSource() {
        final List<ConfigSource> list = new ArrayList<>();
        String creds = System.getenv("POLARIS_CREDS_STORE");
        if (creds != null) {
            Configuration conf = new Configuration();
            conf.set(HADOOP_SECURITY_CREDENTIAL_PROVIDER_PATH, creds);
            LOGGER.debug("Getting username/password from Hadoop credentials provider");

            Map<String, String> map = properties.stream().map(item -> {
                        try {
                            return new AbstractMap.SimpleEntry<>(item, conf.getPassword(item));
                        } catch (IOException ioe) {
                            LOGGER.warn("Cannot find {} option ", item, ioe);
                        }
                        return new AbstractMap.SimpleEntry<>(item, null);
                    })
                    .filter(entry -> Objects.nonNull(entry.getValue()))
                    .collect(Collectors.toMap(Map.Entry::getKey, value -> String.valueOf((char[]) value.getValue())));
            list.add(new InMemoryConfigSource("custom-jdbc-config", map, 400));
        }
        return list;
    }

    private static final class InMemoryConfigSource extends MapBackedConfigSource {
        public InMemoryConfigSource(String name, Map<String, String> propertyMap, int defaultOrdinal) {
            super(name, propertyMap, defaultOrdinal);
        }
    }

}
