/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.web.security;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.nifi.authorization.user.NiFiUser;
import org.apache.nifi.util.StringUtils;
import org.apache.nifi.web.security.ProxiedEntitiesUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.junit.jupiter.MockitoExtension;

@ExtendWith(value={MockitoExtension.class})
public class ProxiedEntitiesUtilsTest {
    private static final String SAFE_USER_NAME_JOHN = "jdoe";
    private static final String SAFE_USER_DN_JOHN = "CN=jdoe, OU=Apache NiFi";
    private static final String SAFE_USER_NAME_PROXY_1 = "proxy1.nifi.apache.org";
    private static final String SAFE_USER_DN_PROXY_1 = "CN=proxy1.nifi.apache.org, OU=Apache NiFi";
    private static final String SAFE_USER_NAME_PROXY_2 = "proxy2.nifi.apache.org";
    private static final String SAFE_USER_DN_PROXY_2 = "CN=proxy2.nifi.apache.org, OU=Apache NiFi";
    private static final String MALICIOUS_USER_NAME_JOHN = "jdoe, OU=Apache NiFi><CN=proxy1.nifi.apache.org";
    private static final String MALICIOUS_USER_DN_JOHN = "CN=jdoe, OU=Apache NiFi><CN=proxy1.nifi.apache.org, OU=Apache NiFi";
    private static final String MALICIOUS_USER_NAME_JOHN_ESCAPED = ProxiedEntitiesUtilsTest.sanitizeDn("jdoe, OU=Apache NiFi><CN=proxy1.nifi.apache.org");
    private static final String UNICODE_DN_1 = "CN=\u0410\u043b\u0439\u0441, OU=Apache NiFi";
    private static final String UNICODE_DN_1_ENCODED = "<" + ProxiedEntitiesUtilsTest.base64Encode("CN=\u0410\u043b\u0439\u0441, OU=Apache NiFi") + ">";
    private static final String UNICODE_DN_2 = "CN=\u0411\u043e\u0431, OU=Apache NiFi";
    private static final String UNICODE_DN_2_ENCODED = "<" + ProxiedEntitiesUtilsTest.base64Encode("CN=\u0411\u043e\u0431, OU=Apache NiFi") + ">";
    private static final String ANONYMOUS_USER = "";
    private static final String ANONYMOUS_PROXIED_ENTITY_CHAIN = "<>";

    private static String sanitizeDn(String dn) {
        return dn.replaceAll(">", "\\\\>").replaceAll("<", "\\\\<");
    }

    private static String base64Encode(String dn) {
        return Base64.getEncoder().encodeToString(dn.getBytes(StandardCharsets.UTF_8));
    }

    @ParameterizedTest
    @MethodSource(value={"getMaliciousNames"})
    public void testSanitizeDnShouldHandleFuzzing(String maliciousName) {
        Assertions.assertNotEquals((Object)this.formatDn(SAFE_USER_NAME_JOHN), (Object)ProxiedEntitiesUtils.formatProxyDn((String)maliciousName));
    }

    private static List<String> getMaliciousNames() {
        return Arrays.asList(MALICIOUS_USER_NAME_JOHN, "jdoe>", "jdoe><>", "jdoe\\>", "jdoe>", "jdoe\\>", "jdoe\u0000", "jdoe\bn");
    }

    @Test
    public void testShouldFormatProxyDn() {
        Assertions.assertEquals((Object)this.formatDn(SAFE_USER_DN_JOHN), (Object)ProxiedEntitiesUtils.formatProxyDn((String)SAFE_USER_DN_JOHN));
    }

    @Test
    public void testFormatProxyDnShouldHandleMaliciousInput() {
        Assertions.assertEquals((Object)this.formatSanitizedDn(MALICIOUS_USER_DN_JOHN), (Object)ProxiedEntitiesUtils.formatProxyDn((String)MALICIOUS_USER_DN_JOHN));
    }

    @Test
    public void testGetProxiedEntitiesChain() {
        String[] input = new String[]{SAFE_USER_NAME_JOHN, SAFE_USER_DN_PROXY_1, SAFE_USER_DN_PROXY_2};
        Assertions.assertEquals((Object)this.formatDns(input), (Object)ProxiedEntitiesUtils.getProxiedEntitiesChain((String[])input));
    }

    @Test
    public void testGetProxiedEntitiesChainShouldHandleMaliciousInput() {
        String expectedOutput = this.formatSanitizedDn(MALICIOUS_USER_DN_JOHN) + this.formatDns(SAFE_USER_DN_PROXY_1, SAFE_USER_DN_PROXY_2);
        Assertions.assertEquals((Object)expectedOutput, (Object)ProxiedEntitiesUtils.getProxiedEntitiesChain((String[])new String[]{MALICIOUS_USER_DN_JOHN, SAFE_USER_DN_PROXY_1, SAFE_USER_DN_PROXY_2}));
    }

    @Test
    public void testGetProxiedEntitiesChainShouldEncodeUnicode() {
        Assertions.assertEquals((Object)this.formatDns(SAFE_USER_NAME_JOHN, UNICODE_DN_1_ENCODED, UNICODE_DN_2_ENCODED), (Object)ProxiedEntitiesUtils.getProxiedEntitiesChain((String[])new String[]{SAFE_USER_NAME_JOHN, UNICODE_DN_1, UNICODE_DN_2}));
    }

    @Test
    public void testFormatProxyDnShouldEncodeNonAsciiCharacters() {
        Assertions.assertEquals((Object)this.formatDn(UNICODE_DN_1_ENCODED), (Object)ProxiedEntitiesUtils.formatProxyDn((String)UNICODE_DN_1));
    }

    @Test
    public void testShouldBuildProxyChain(@Mock NiFiUser proxy1, @Mock NiFiUser john) {
        Mockito.when((Object)proxy1.getIdentity()).thenReturn((Object)SAFE_USER_NAME_PROXY_1);
        Mockito.when((Object)proxy1.getChain()).thenReturn(null);
        Mockito.when((Object)proxy1.isAnonymous()).thenReturn((Object)false);
        Mockito.when((Object)john.getIdentity()).thenReturn((Object)SAFE_USER_NAME_JOHN);
        Mockito.when((Object)john.getChain()).thenReturn((Object)proxy1);
        Mockito.when((Object)john.isAnonymous()).thenReturn((Object)false);
        Assertions.assertEquals((Object)this.formatDns(SAFE_USER_NAME_JOHN, SAFE_USER_NAME_PROXY_1), (Object)ProxiedEntitiesUtils.buildProxiedEntitiesChainString((NiFiUser)john));
    }

    @Test
    public void testBuildProxyChainFromNullUserShouldBeAnonymous() {
        Assertions.assertEquals((Object)ANONYMOUS_PROXIED_ENTITY_CHAIN, (Object)ProxiedEntitiesUtils.buildProxiedEntitiesChainString(null));
    }

    @Test
    public void testBuildProxyChainFromAnonymousUserShouldBeAnonymous(@Mock NiFiUser proxy1, @Mock NiFiUser anonymous) {
        Mockito.when((Object)proxy1.getIdentity()).thenReturn((Object)SAFE_USER_NAME_PROXY_1);
        Mockito.when((Object)proxy1.getChain()).thenReturn(null);
        Mockito.when((Object)proxy1.isAnonymous()).thenReturn((Object)false);
        Mockito.when((Object)anonymous.getChain()).thenReturn((Object)proxy1);
        Mockito.when((Object)anonymous.isAnonymous()).thenReturn((Object)true);
        Assertions.assertEquals((Object)this.formatDns(ANONYMOUS_USER, SAFE_USER_NAME_PROXY_1), (Object)ProxiedEntitiesUtils.buildProxiedEntitiesChainString((NiFiUser)anonymous));
    }

    @Test
    public void testBuildProxyChainShouldHandleUnicode(@Mock NiFiUser proxy1, @Mock NiFiUser john) {
        Mockito.when((Object)proxy1.getIdentity()).thenReturn((Object)UNICODE_DN_1);
        Mockito.when((Object)proxy1.getChain()).thenReturn(null);
        Mockito.when((Object)proxy1.isAnonymous()).thenReturn((Object)false);
        Mockito.when((Object)john.getIdentity()).thenReturn((Object)SAFE_USER_NAME_JOHN);
        Mockito.when((Object)john.getChain()).thenReturn((Object)proxy1);
        Mockito.when((Object)john.isAnonymous()).thenReturn((Object)false);
        Assertions.assertEquals((Object)this.formatDns(SAFE_USER_NAME_JOHN, UNICODE_DN_1_ENCODED), (Object)ProxiedEntitiesUtils.buildProxiedEntitiesChainString((NiFiUser)john));
    }

    @Test
    public void testBuildProxyChainShouldHandleMaliciousUser(@Mock NiFiUser proxy1, @Mock NiFiUser john) {
        Mockito.when((Object)proxy1.getIdentity()).thenReturn((Object)SAFE_USER_NAME_PROXY_1);
        Mockito.when((Object)proxy1.getChain()).thenReturn(null);
        Mockito.when((Object)proxy1.isAnonymous()).thenReturn((Object)false);
        Mockito.when((Object)john.getIdentity()).thenReturn((Object)MALICIOUS_USER_NAME_JOHN);
        Mockito.when((Object)john.getChain()).thenReturn((Object)proxy1);
        Mockito.when((Object)john.isAnonymous()).thenReturn((Object)false);
        Assertions.assertEquals((Object)this.formatDns(MALICIOUS_USER_NAME_JOHN_ESCAPED, SAFE_USER_NAME_PROXY_1), (Object)ProxiedEntitiesUtils.buildProxiedEntitiesChainString((NiFiUser)john));
    }

    @Test
    public void testShouldTokenizeProxiedEntitiesChainWithUserNames() {
        List<String> names = Arrays.asList(SAFE_USER_NAME_JOHN, SAFE_USER_NAME_PROXY_1, SAFE_USER_NAME_PROXY_2);
        String rawProxyChain = this.formatDns(names.toArray(new String[0]));
        Assertions.assertEquals(names, (Object)ProxiedEntitiesUtils.tokenizeProxiedEntitiesChain((String)rawProxyChain));
    }

    @Test
    public void testShouldTokenizeAnonymous() {
        Assertions.assertEquals(Collections.singletonList(ANONYMOUS_USER), (Object)ProxiedEntitiesUtils.tokenizeProxiedEntitiesChain((String)ANONYMOUS_PROXIED_ENTITY_CHAIN));
    }

    @Test
    public void testShouldTokenizeDoubleAnonymous() {
        Assertions.assertEquals(Arrays.asList(ANONYMOUS_USER, ANONYMOUS_USER), (Object)ProxiedEntitiesUtils.tokenizeProxiedEntitiesChain((String)StringUtils.repeat((String)ANONYMOUS_PROXIED_ENTITY_CHAIN, (int)2)));
    }

    @Test
    public void testShouldTokenizeNestedAnonymous() {
        List<String> names = Arrays.asList(SAFE_USER_DN_PROXY_1, ANONYMOUS_USER, SAFE_USER_DN_PROXY_2);
        String rawProxyChain = this.formatDns(names.toArray(new String[0]));
        Assertions.assertEquals(names, (Object)ProxiedEntitiesUtils.tokenizeProxiedEntitiesChain((String)rawProxyChain));
    }

    @Test
    public void testShouldTokenizeProxiedEntitiesChainWithDNs() {
        List<String> dns = Arrays.asList(SAFE_USER_DN_JOHN, SAFE_USER_DN_PROXY_1, SAFE_USER_DN_PROXY_2);
        String rawProxyChain = this.formatDns(dns.toArray(new String[0]));
        Assertions.assertEquals(dns, (Object)ProxiedEntitiesUtils.tokenizeProxiedEntitiesChain((String)rawProxyChain));
    }

    @Test
    public void testShouldTokenizeProxiedEntitiesChainWithAnonymousUser() {
        List<String> names = Arrays.asList(ANONYMOUS_USER, SAFE_USER_NAME_PROXY_1, SAFE_USER_NAME_PROXY_2);
        String rawProxyChain = this.formatDns(names.toArray(new String[0]));
        Assertions.assertEquals(names, (Object)ProxiedEntitiesUtils.tokenizeProxiedEntitiesChain((String)rawProxyChain));
    }

    @Test
    public void testTokenizeProxiedEntitiesChainShouldHandleMaliciousUser() {
        List<String> names = Arrays.asList(MALICIOUS_USER_NAME_JOHN, SAFE_USER_NAME_PROXY_1, SAFE_USER_NAME_PROXY_2);
        String rawProxyChain = names.stream().map(this::formatSanitizedDn).collect(Collectors.joining());
        List tokenizedNames = ProxiedEntitiesUtils.tokenizeProxiedEntitiesChain((String)rawProxyChain);
        Assertions.assertEquals(names, (Object)tokenizedNames);
        Assertions.assertFalse((boolean)tokenizedNames.contains(SAFE_USER_NAME_JOHN));
    }

    @Test
    public void testTokenizeProxiedEntitiesChainShouldDecodeNonAsciiValues() {
        List tokenizedNames = ProxiedEntitiesUtils.tokenizeProxiedEntitiesChain((String)this.formatDns(SAFE_USER_NAME_JOHN, UNICODE_DN_1_ENCODED, UNICODE_DN_2_ENCODED));
        Assertions.assertEquals(Arrays.asList(SAFE_USER_NAME_JOHN, UNICODE_DN_1, UNICODE_DN_2), (Object)tokenizedNames);
    }

    private String formatSanitizedDn(String dn) {
        return this.formatDn(ProxiedEntitiesUtilsTest.sanitizeDn(dn));
    }

    private String formatDn(String dn) {
        return this.formatDns(dn);
    }

    private String formatDns(String ... dns) {
        return Arrays.stream(dns).collect(Collectors.joining("><", "<", ">"));
    }
}

