/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.web.security.x509.ocsp;

import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Date;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.ExtendedKeyUsage;
import org.bouncycastle.asn1.x509.KeyPurposeId;
import org.bouncycastle.asn1.x509.KeyUsage;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.asn1.x509.X509Extension;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OcspCertificateValidatorTest {
    private static final Logger logger = LoggerFactory.getLogger(OcspCertificateValidatorTest.class);
    private static final int KEY_SIZE = 2048;
    private static final long YESTERDAY = System.currentTimeMillis() - 86400000L;
    private static final long ONE_YEAR_FROM_NOW = System.currentTimeMillis() + 31536000000L;
    private static final String SIGNATURE_ALGORITHM = "SHA256withRSA";
    private static final String PROVIDER = "BC";

    @BeforeAll
    public static void setUpOnce() {
        Security.addProvider((Provider)new BouncyCastleProvider());
    }

    private static KeyPair generateKeyPair() throws NoSuchAlgorithmException {
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("RSA");
        keyPairGenerator.initialize(2048);
        return keyPairGenerator.generateKeyPair();
    }

    private static X509Certificate generateCertificate(String dn) throws IOException, NoSuchAlgorithmException, CertificateException, OperatorCreationException {
        KeyPair keyPair = OcspCertificateValidatorTest.generateKeyPair();
        return OcspCertificateValidatorTest.generateCertificate(dn, keyPair);
    }

    private static X509Certificate generateCertificate(String dn, KeyPair keyPair) throws IOException, CertificateException, OperatorCreationException {
        PrivateKey privateKey = keyPair.getPrivate();
        ContentSigner sigGen = new JcaContentSignerBuilder(SIGNATURE_ALGORITHM).setProvider(PROVIDER).build(privateKey);
        SubjectPublicKeyInfo subPubKeyInfo = SubjectPublicKeyInfo.getInstance((Object)keyPair.getPublic().getEncoded());
        Date startDate = new Date(YESTERDAY);
        Date endDate = new Date(ONE_YEAR_FROM_NOW);
        X509v3CertificateBuilder certBuilder = new X509v3CertificateBuilder(new X500Name(dn), BigInteger.valueOf(System.currentTimeMillis()), startDate, endDate, new X500Name(dn), subPubKeyInfo);
        certBuilder.addExtension(X509Extension.keyUsage, true, (ASN1Encodable)new KeyUsage(184));
        ArrayList<KeyPurposeId> ekUsages = new ArrayList<KeyPurposeId>();
        ekUsages.add(KeyPurposeId.id_kp_clientAuth);
        ekUsages.add(KeyPurposeId.id_kp_serverAuth);
        certBuilder.addExtension(X509Extension.extendedKeyUsage, false, (ASN1Encodable)new ExtendedKeyUsage(ekUsages.toArray(new KeyPurposeId[0])));
        X509CertificateHolder certificateHolder = certBuilder.build(sigGen);
        return new JcaX509CertificateConverter().setProvider(PROVIDER).getCertificate(certificateHolder);
    }

    private static X509Certificate generateIssuedCertificate(String dn, String issuerDn, PrivateKey issuerKey) throws NoSuchAlgorithmException, CertificateException, OperatorCreationException {
        KeyPair keyPair = OcspCertificateValidatorTest.generateKeyPair();
        return OcspCertificateValidatorTest.generateIssuedCertificate(dn, keyPair.getPublic(), issuerDn, issuerKey);
    }

    private static X509Certificate generateIssuedCertificate(String dn, PublicKey publicKey, String issuerDn, PrivateKey issuerKey) throws CertificateException, OperatorCreationException {
        ContentSigner sigGen = new JcaContentSignerBuilder(SIGNATURE_ALGORITHM).setProvider(PROVIDER).build(issuerKey);
        SubjectPublicKeyInfo subPubKeyInfo = SubjectPublicKeyInfo.getInstance((Object)publicKey.getEncoded());
        Date startDate = new Date(YESTERDAY);
        Date endDate = new Date(ONE_YEAR_FROM_NOW);
        X509v3CertificateBuilder v3CertGen = new X509v3CertificateBuilder(new X500Name(issuerDn), BigInteger.valueOf(System.currentTimeMillis()), startDate, endDate, new X500Name(dn), subPubKeyInfo);
        X509CertificateHolder certificateHolder = v3CertGen.build(sigGen);
        return new JcaX509CertificateConverter().setProvider(PROVIDER).getCertificate(certificateHolder);
    }

    @Test
    public void testShouldGenerateCertificate() throws Exception {
        String testDn = "CN=This is a test";
        X509Certificate certificate = OcspCertificateValidatorTest.generateCertificate("CN=This is a test");
        logger.info("Generated certificate: \n{}", (Object)certificate);
        Assertions.assertEquals((Object)"CN=This is a test", (Object)certificate.getSubjectDN().getName());
        Assertions.assertEquals((Object)"CN=This is a test", (Object)certificate.getIssuerDN().getName());
        certificate.verify(certificate.getPublicKey());
    }

    @Test
    public void testShouldGenerateCertificateFromKeyPair() throws Exception {
        String testDn = "CN=This is a test";
        KeyPair keyPair = OcspCertificateValidatorTest.generateKeyPair();
        X509Certificate certificate = OcspCertificateValidatorTest.generateCertificate("CN=This is a test", keyPair);
        logger.info("Generated certificate: \n{}", (Object)certificate);
        Assertions.assertEquals((Object)keyPair.getPublic(), (Object)certificate.getPublicKey());
        Assertions.assertEquals((Object)"CN=This is a test", (Object)certificate.getSubjectDN().getName());
        Assertions.assertEquals((Object)"CN=This is a test", (Object)certificate.getIssuerDN().getName());
        certificate.verify(certificate.getPublicKey());
    }

    @Test
    public void testShouldGenerateIssuedCertificate() throws Exception {
        String testDn = "CN=This is a signed test";
        String issuerDn = "CN=Issuer CA";
        KeyPair issuerKeyPair = OcspCertificateValidatorTest.generateKeyPair();
        PrivateKey issuerPrivateKey = issuerKeyPair.getPrivate();
        X509Certificate issuerCertificate = OcspCertificateValidatorTest.generateCertificate("CN=Issuer CA", issuerKeyPair);
        logger.info("Generated issuer certificate: \n{}", (Object)issuerCertificate);
        X509Certificate certificate = OcspCertificateValidatorTest.generateIssuedCertificate("CN=This is a signed test", "CN=Issuer CA", issuerPrivateKey);
        logger.info("Generated signed certificate: \n{}", (Object)certificate);
        Assertions.assertEquals((Object)issuerKeyPair.getPublic(), (Object)issuerCertificate.getPublicKey());
        Assertions.assertEquals((Object)"CN=This is a signed test", (Object)certificate.getSubjectX500Principal().getName());
        Assertions.assertEquals((Object)"CN=Issuer CA", (Object)certificate.getIssuerX500Principal().getName());
        certificate.verify(issuerCertificate.getPublicKey());
        SignatureException se = (SignatureException)Assertions.assertThrows(SignatureException.class, () -> certificate.verify(certificate.getPublicKey()));
        Assertions.assertTrue((boolean)se.getMessage().contains("certificate does not verify with supplied key"));
    }
}

