/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.nifi.annotation.behavior.DefaultRunDuration;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.SideEffectFree;
import org.apache.nifi.annotation.behavior.SupportsBatching;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.components.DescribedValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;

@SideEffectFree
@SupportsBatching(defaultDuration=DefaultRunDuration.TWENTY_FIVE_MILLIS)
@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@Tags(value={"attributes", "modification", "filter", "retain", "remove", "delete", "regex", "regular expression", "Attribute Expression Language"})
@CapabilityDescription(value="Filters the attributes of a FlowFile by retaining specified attributes and removing the rest or by removing specified attributes and retaining the rest.")
public class FilterAttribute
extends AbstractProcessor {
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("All successful FlowFiles are routed to this relationship").build();
    public static final PropertyDescriptor FILTER_MODE = new PropertyDescriptor.Builder().name("Filter Mode").displayName("Filter Mode").description("Specifies the strategy to apply on filtered attributes. Either 'Remove' or 'Retain' only the matching attributes.").required(true).allowableValues(FilterMode.class).expressionLanguageSupported(ExpressionLanguageScope.NONE).defaultValue(FilterMode.RETAIN.getValue()).build();
    public static final PropertyDescriptor MATCHING_STRATEGY = new PropertyDescriptor.Builder().name("Attribute Matching Strategy").displayName("Attribute Matching Strategy").description("Specifies the strategy to filter attributes by.").required(true).expressionLanguageSupported(ExpressionLanguageScope.NONE).allowableValues(MatchingStrategy.class).defaultValue(MatchingStrategy.ENUMERATION.getValue()).build();
    public static final PropertyDescriptor ATTRIBUTE_ENUMERATION = new PropertyDescriptor.Builder().name("Filtered Attributes").displayName("Filtered Attributes").description("A set of attribute names to filter from FlowFiles. Each attribute name is separated by the comma delimiter ','.").required(true).dependsOn(MATCHING_STRATEGY, MatchingStrategy.ENUMERATION.getValue(), new String[0]).addValidator(StandardValidators.NON_BLANK_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).build();
    public static final PropertyDescriptor ATTRIBUTE_PATTERN = new PropertyDescriptor.Builder().name("Filtered Attributes Pattern").displayName("Filtered Attributes Pattern").description("A regular expression to match names of attributes to filter from FlowFiles.").required(true).dependsOn(MATCHING_STRATEGY, MatchingStrategy.PATTERN.getValue(), new String[0]).addValidator(StandardValidators.REGULAR_EXPRESSION_WITH_EL_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).build();
    private static final String DELIMITER_VALUE = ",";
    private static final Set<Relationship> relationships = Collections.singleton(REL_SUCCESS);
    private static final List<PropertyDescriptor> properties = Collections.unmodifiableList(Arrays.asList(FILTER_MODE, MATCHING_STRATEGY, ATTRIBUTE_ENUMERATION, ATTRIBUTE_PATTERN));
    private volatile Predicate<String> cachedMatchingPredicate;

    public Set<Relationship> getRelationships() {
        return relationships;
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return properties;
    }

    @OnScheduled
    public void onScheduled(ProcessContext context) {
        MatchingStrategy matchingStrategy = FilterAttribute.getMatchingStrategy(context);
        this.cachedMatchingPredicate = null;
        if (matchingStrategy == MatchingStrategy.ENUMERATION && !context.getProperty(ATTRIBUTE_ENUMERATION).isExpressionLanguagePresent()) {
            this.cachedMatchingPredicate = FilterAttribute.determineMatchingPredicateBasedOnEnumeration(context, null);
        }
        if (matchingStrategy == MatchingStrategy.PATTERN && !context.getProperty(ATTRIBUTE_PATTERN).isExpressionLanguagePresent()) {
            this.cachedMatchingPredicate = FilterAttribute.determineMatchingPredicateBasedOnRegex(context, null);
        }
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        Predicate<String> matchingPredicate = this.determineMatchingPredicate(context, flowFile);
        FilterMode filterMode = FilterAttribute.getFilterMode(context);
        Predicate<String> isMatched = filterMode == FilterMode.RETAIN ? matchingPredicate : matchingPredicate.negate();
        HashSet attributesToRemove = new HashSet(flowFile.getAttributes().keySet());
        attributesToRemove.removeIf(isMatched);
        FlowFile updatedFlowFile = session.removeAllAttributes(flowFile, attributesToRemove);
        session.transfer(updatedFlowFile, REL_SUCCESS);
    }

    private Predicate<String> determineMatchingPredicate(ProcessContext context, FlowFile flowFile) {
        if (this.cachedMatchingPredicate != null) {
            return this.cachedMatchingPredicate;
        }
        MatchingStrategy matchingStrategy = FilterAttribute.getMatchingStrategy(context);
        switch (matchingStrategy) {
            case ENUMERATION: {
                return FilterAttribute.determineMatchingPredicateBasedOnEnumeration(context, flowFile);
            }
            case PATTERN: {
                return FilterAttribute.determineMatchingPredicateBasedOnRegex(context, flowFile);
            }
        }
        throw new IllegalArgumentException("Cannot determine matching predicate for unsupported strategy [" + matchingStrategy + "]");
    }

    private static Predicate<String> determineMatchingPredicateBasedOnEnumeration(ProcessContext context, FlowFile flowFile) {
        String attributeSetDeclaration = FilterAttribute.getAttributeSet(context, flowFile);
        String delimiter = FilterAttribute.getDelimiter();
        Set attributeSet = Arrays.stream(attributeSetDeclaration.split(Pattern.quote(delimiter))).map(String::trim).filter(attributeName -> !attributeName.isEmpty()).collect(Collectors.toSet());
        return attributeSet::contains;
    }

    private static Predicate<String> determineMatchingPredicateBasedOnRegex(ProcessContext context, FlowFile flowFile) {
        Pattern attributeRegex = FilterAttribute.getAttributeRegex(context, flowFile);
        return attributeName -> attributeRegex.matcher((CharSequence)attributeName).matches();
    }

    private static FilterMode getFilterMode(ProcessContext context) {
        return FilterMode.fromValue(context.getProperty(FILTER_MODE).getValue());
    }

    private static MatchingStrategy getMatchingStrategy(ProcessContext context) {
        return MatchingStrategy.fromValue(context.getProperty(MATCHING_STRATEGY).getValue());
    }

    private static String getAttributeSet(ProcessContext context, FlowFile flowFile) {
        return context.getProperty(ATTRIBUTE_ENUMERATION).evaluateAttributeExpressions(flowFile).getValue();
    }

    private static String getDelimiter() {
        return DELIMITER_VALUE;
    }

    private static Pattern getAttributeRegex(ProcessContext context, FlowFile flowFile) {
        return Pattern.compile(context.getProperty(ATTRIBUTE_PATTERN).evaluateAttributeExpressions(flowFile).getValue());
    }

    private static <E extends Enum<E>> E enumConstantFromValue(Class<E> enumClass, String value) {
        for (Enum enumConstant : (Enum[])enumClass.getEnumConstants()) {
            if (!((DescribedValue)enumConstant).getValue().equals(value)) continue;
            return (E)enumConstant;
        }
        throw new IllegalArgumentException(String.format("Unknown %s value [%s]", enumClass.getSimpleName(), value));
    }

    static enum MatchingStrategy implements DescribedValue
    {
        ENUMERATION("Enumerate attributes", "Provides a set of attribute keys to filter for, separated by a comma delimiter ','."),
        PATTERN("Use regular expression", "Provides a regular expression to match keys of attributes to filter for.");

        private final String value;
        private final String description;

        private MatchingStrategy(String value, String description) {
            this.value = value;
            this.description = description;
        }

        public String getValue() {
            return this.value;
        }

        public String getDisplayName() {
            return this.value;
        }

        public String getDescription() {
            return this.description;
        }

        public static MatchingStrategy fromValue(String value) {
            return FilterAttribute.enumConstantFromValue(MatchingStrategy.class, value);
        }
    }

    static enum FilterMode implements DescribedValue
    {
        RETAIN("Retain", "Retains only the attributes matching the filter, all other attributes are removed."),
        REMOVE("Remove", "Removes the attributes matching the filter, all other attributes are retained.");

        private final String value;
        private final String description;

        private FilterMode(String value, String description) {
            this.value = value;
            this.description = description;
        }

        public String getValue() {
            return this.value;
        }

        public String getDisplayName() {
            return this.value;
        }

        public String getDescription() {
            return this.description;
        }

        public static FilterMode fromValue(String value) {
            return FilterAttribute.enumConstantFromValue(FilterMode.class, value);
        }
    }
}

