/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.parameter;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.commons.io.IOUtils;
import org.apache.nifi.annotation.behavior.Restricted;
import org.apache.nifi.annotation.behavior.Restriction;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.ConfigVerificationResult;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.RequiredPermission;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.components.Validator;
import org.apache.nifi.controller.ConfigurationContext;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.parameter.AbstractParameterProvider;
import org.apache.nifi.parameter.Parameter;
import org.apache.nifi.parameter.ParameterDescriptor;
import org.apache.nifi.parameter.ParameterGroup;
import org.apache.nifi.parameter.ParameterProviderInitializationContext;
import org.apache.nifi.parameter.VerifiableParameterProvider;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.stream.io.LimitingInputStream;

@Tags(value={"file"})
@CapabilityDescription(value="Fetches parameters from files.  Parameter groups are indicated by a set of directories, and files within the directories map to parameter names. The content of the file becomes the parameter value.")
@Restricted(restrictions={@Restriction(requiredPermission=RequiredPermission.READ_FILESYSTEM, explanation="Provides operator the ability to read from any file that NiFi has access to.")})
public class FileParameterProvider
extends AbstractParameterProvider
implements VerifiableParameterProvider {
    private static final int MAX_SIZE_LIMIT = 8096;
    private static final Charset DEFAULT_CHARSET = StandardCharsets.UTF_8;
    private static final AllowableValue BASE64_ENCODING = new AllowableValue("base64", "Base64", "File content is Base64-encoded, and will be decoded before providing the value as a Parameter.");
    private static final AllowableValue PLAIN_TEXT = new AllowableValue("plaintext", "Plain text", "File content is not encoded, and will be provided directly as a Parameter value.");
    public static final PropertyDescriptor PARAMETER_GROUP_DIRECTORIES = new PropertyDescriptor.Builder().name("parameter-group-directories").displayName("Parameter Group Directories").description("A comma-separated list of directory absolute paths that will map to named parameter groups.  Each directory that contains files will map to a parameter group, named after the innermost directory in the path.  Files inside the directory will map to parameter names, whose values are the content of each respective file.").addValidator((Validator)new MultiDirectoryExistsValidator()).required(true).build();
    public static final PropertyDescriptor PARAMETER_VALUE_BYTE_LIMIT = new PropertyDescriptor.Builder().name("parameter-value-byte-limit").displayName("Parameter Value Byte Limit").description("The maximum byte size of a parameter value.  Since parameter values are pulled from the contents of files, this is a safeguard that can prevent memory issues if large files are included.").addValidator(StandardValidators.createDataSizeBoundsValidator((long)1L, (long)8096L)).defaultValue("256 B").required(true).build();
    public static final PropertyDescriptor PARAMETER_VALUE_ENCODING = new PropertyDescriptor.Builder().name("parameter-value-encoding").displayName("Parameter Value Encoding").description("Indicates how parameter values are encoded inside Parameter files.").allowableValues(new AllowableValue[]{BASE64_ENCODING, PLAIN_TEXT}).defaultValue(BASE64_ENCODING.getValue()).required(true).build();
    private List<PropertyDescriptor> properties;

    protected void init(ParameterProviderInitializationContext config) {
        ArrayList<PropertyDescriptor> properties = new ArrayList<PropertyDescriptor>();
        properties.add(PARAMETER_GROUP_DIRECTORIES);
        properties.add(PARAMETER_VALUE_BYTE_LIMIT);
        properties.add(PARAMETER_VALUE_ENCODING);
        this.properties = Collections.unmodifiableList(properties);
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return this.properties;
    }

    public List<ParameterGroup> fetchParameters(ConfigurationContext context) {
        ArrayList<ParameterGroup> parameterGroups = new ArrayList<ParameterGroup>();
        Collection<File> groupDirectories = this.getDirectories(context, PARAMETER_GROUP_DIRECTORIES);
        groupDirectories.forEach(directory -> parameterGroups.add(this.getParameterGroup(context, (File)directory, directory.getName())));
        AtomicInteger groupedParameterCount = new AtomicInteger(0);
        HashSet groupNames = new HashSet();
        parameterGroups.forEach(group -> {
            groupedParameterCount.addAndGet(group.getParameters().size());
            groupNames.add(group.getGroupName());
        });
        this.getLogger().info("Fetched {} parameters.  Group names: {}", new Object[]{groupedParameterCount.get(), groupNames});
        return parameterGroups;
    }

    public List<ConfigVerificationResult> verify(ConfigurationContext context, ComponentLog verificationLogger) {
        ArrayList<ConfigVerificationResult> results = new ArrayList<ConfigVerificationResult>();
        try {
            List<ParameterGroup> parameterGroups = this.fetchParameters(context);
            Set parameterGroupNames = parameterGroups.stream().map(ParameterGroup::getGroupName).collect(Collectors.toSet());
            long parameterCount = parameterGroups.stream().flatMap(group -> group.getParameters().stream()).count();
            results.add(new ConfigVerificationResult.Builder().outcome(ConfigVerificationResult.Outcome.SUCCESSFUL).verificationStepName("Fetch Parameters").explanation(String.format("Fetched %s files as parameters.", parameterCount)).build());
        }
        catch (IllegalArgumentException e) {
            verificationLogger.error("Failed to fetch parameters", (Throwable)e);
            results.add(new ConfigVerificationResult.Builder().outcome(ConfigVerificationResult.Outcome.FAILED).verificationStepName("Fetch Parameters").explanation("Failed to fetch parameters: " + e.getMessage()).build());
        }
        return results;
    }

    private String getParameterValue(String rawValue, ParameterValueEncoding encoding) {
        if (ParameterValueEncoding.BASE64 == encoding) {
            return new String(Base64.getDecoder().decode(rawValue.getBytes(DEFAULT_CHARSET)), DEFAULT_CHARSET);
        }
        return rawValue;
    }

    private Collection<File> getDirectories(ConfigurationContext context, PropertyDescriptor descriptor) {
        return context.getProperty(descriptor).isSet() ? (Collection)Arrays.stream(context.getProperty(descriptor).getValue().split(",")).map(String::trim).map(File::new).collect(Collectors.toSet()) : Collections.emptySet();
    }

    private ParameterGroup getParameterGroup(ConfigurationContext context, File directory, String groupName) {
        int parameterSizeLimit = context.getProperty(PARAMETER_VALUE_BYTE_LIMIT).asDataSize(DataUnit.B).intValue();
        ParameterValueEncoding parameterEncoding = ParameterValueEncoding.valueOf(context.getProperty(PARAMETER_VALUE_ENCODING).getValue().toUpperCase());
        File[] files = directory.listFiles();
        ArrayList<Parameter> parameters = new ArrayList<Parameter>();
        for (File file : files) {
            if (file.isDirectory() || file.isHidden()) continue;
            String parameterName = file.getName();
            try (BufferedInputStream in = new BufferedInputStream((InputStream)new LimitingInputStream((InputStream)new FileInputStream(file), (long)parameterSizeLimit));){
                String rawValue = IOUtils.toString((InputStream)in, (Charset)Charset.defaultCharset()).trim();
                String parameterValue = this.getParameterValue(rawValue, parameterEncoding);
                if (parameterValue.length() >= parameterSizeLimit) {
                    this.getLogger().warn("Parameter {} may be truncated at {} bytes", new Object[]{parameterName, parameterValue.length()});
                }
                ParameterDescriptor parameterDescriptor = new ParameterDescriptor.Builder().name(parameterName).build();
                parameters.add(new Parameter(parameterDescriptor, parameterValue, null, Boolean.valueOf(true)));
            }
            catch (IOException e) {
                throw new RuntimeException(String.format("Failed to read file [%s]", file), e);
            }
        }
        return new ParameterGroup(groupName, parameters);
    }

    public static class MultiDirectoryExistsValidator
    implements Validator {
        public ValidationResult validate(String subject, String value, ValidationContext context) {
            Object reason = null;
            if (value == null) {
                reason = "At least one directory is required";
            } else {
                Set directories = Arrays.stream(value.split(",")).map(String::trim).collect(Collectors.toSet());
                try {
                    for (String directory : directories) {
                        File file = new File(directory);
                        if (!file.exists()) {
                            reason = "Directory " + file.getName() + " does not exist";
                            continue;
                        }
                        if (file.isDirectory()) continue;
                        reason = "Path " + file.getName() + " does not point to a directory";
                    }
                }
                catch (Exception e) {
                    reason = "Value is not a valid directory name";
                }
            }
            return new ValidationResult.Builder().subject(subject).input(value).explanation((String)reason).valid(reason == null).build();
        }
    }

    private static enum ParameterValueEncoding {
        BASE64,
        PLAINTEXT;

    }
}

