/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.slack;

import com.slack.api.bolt.App;
import com.slack.api.bolt.AppConfig;
import com.slack.api.methods.MethodsClient;
import com.slack.api.methods.SlackApiException;
import com.slack.api.methods.request.chat.ChatPostMessageRequest;
import com.slack.api.methods.request.files.FilesUploadV2Request;
import com.slack.api.methods.response.chat.ChatPostMessageResponse;
import com.slack.api.methods.response.files.FilesUploadV2Response;
import com.slack.api.model.File;
import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.configuration.DefaultSettings;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.annotation.lifecycle.OnStopped;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.Validator;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.slack.util.ChannelMapper;
import org.apache.nifi.processors.slack.util.RateLimit;
import org.apache.nifi.stream.io.StreamUtils;
import org.apache.nifi.util.FormatUtils;

@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@CapabilityDescription(value="Posts a message to the specified Slack channel. The content of the message can be either a user-defined message that makes use of Expression Language or the contents of the FlowFile can be sent as the message. If sending a user-defined message, the contents of the FlowFile may also be optionally uploaded as a file attachment.")
@Tags(value={"slack", "conversation", "chat.postMessage", "social media", "team", "text", "unstructured", "write", "upload", "send", "publish"})
@WritesAttributes(value={@WritesAttribute(attribute="slack.channel.id", description="The ID of the Slack Channel from which the messages were retrieved"), @WritesAttribute(attribute="slack.ts", description="The timestamp of the slack messages that was sent; this is used by Slack as a unique identifier")})
@DefaultSettings(yieldDuration="3 sec")
public class PublishSlack
extends AbstractProcessor {
    static final AllowableValue PUBLISH_STRATEGY_CONTENT_AS_MESSAGE = new AllowableValue("Send FlowFile Content as Message", "Send FlowFile Content as Message", "The contents of the FlowFile will be sent as the message text.");
    static final AllowableValue PUBLISH_STRATEGY_USE_PROPERTY = new AllowableValue("Use 'Message Text' Property", "Use 'Message Text' Property", "The value of the Message Text Property will be sent as the message text.");
    static final PropertyDescriptor ACCESS_TOKEN = new PropertyDescriptor.Builder().name("Access Token").description("OAuth Access Token used for authenticating/authorizing the Slack request sent by NiFi. This may be either a User Token or a Bot Token. The token must be granted the chat:write scope. Additionally, in order to upload FlowFile contents as an attachment, it must be granted files:write.").addValidator(StandardValidators.NON_EMPTY_VALIDATOR).required(true).sensitive(true).build();
    static PropertyDescriptor CHANNEL = new PropertyDescriptor.Builder().name("Channel").description("The name or identifier of the channel to send the message to. If using a channel name, it must be prefixed with the # character. For example, #general. This is valid only for public channels. Otherwise, the unique identifier of the channel to publish to must be provided.").addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).required(true).build();
    static PropertyDescriptor PUBLISH_STRATEGY = new PropertyDescriptor.Builder().name("Publish Strategy").description("Specifies how the Processor will send the message or file to Slack.").required(true).allowableValues(new AllowableValue[]{PUBLISH_STRATEGY_CONTENT_AS_MESSAGE, PUBLISH_STRATEGY_USE_PROPERTY}).defaultValue(PUBLISH_STRATEGY_CONTENT_AS_MESSAGE.getValue()).build();
    static PropertyDescriptor CHARACTER_SET = new PropertyDescriptor.Builder().name("Character Set").description("Specifies the name of the Character Set used to encode the FlowFile contents.").required(true).defaultValue("UTF-8").addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).dependsOn(PUBLISH_STRATEGY, new AllowableValue[]{PUBLISH_STRATEGY_CONTENT_AS_MESSAGE}).build();
    static PropertyDescriptor MESSAGE_TEXT = new PropertyDescriptor.Builder().name("Message Text").description("The text of the message to send to Slack.").expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).required(true).addValidator(Validator.VALID).dependsOn(PUBLISH_STRATEGY, new AllowableValue[]{PUBLISH_STRATEGY_USE_PROPERTY}).build();
    static PropertyDescriptor SEND_CONTENT_AS_ATTACHMENT = new PropertyDescriptor.Builder().name("Include FlowFile Content as Attachment").description("Specifies whether or not the contents of the FlowFile should be uploaded as an attachment to the Slack message.").allowableValues(new String[]{"true", "false"}).required(true).dependsOn(PUBLISH_STRATEGY, new AllowableValue[]{PUBLISH_STRATEGY_USE_PROPERTY}).defaultValue("false").build();
    static PropertyDescriptor MAX_FILE_SIZE = new PropertyDescriptor.Builder().name("Max FlowFile Size").description("The maximum size of a FlowFile that can be sent to Slack. If any FlowFile exceeds this size, it will be routed to failure. This plays an important role because the entire contents of the file must be loaded into NiFi's heap in order to send the data to Slack.").required(true).dependsOn(SEND_CONTENT_AS_ATTACHMENT, "true", new String[0]).addValidator(StandardValidators.DATA_SIZE_VALIDATOR).defaultValue("1 MB").build();
    static PropertyDescriptor THREAD_TS = new PropertyDescriptor.Builder().name("Thread Timestamp").description("The Timestamp identifier for the thread that this message is to be a part of. If not specified, the message will be a top-level message instead of being in a thread.").addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).required(false).build();
    private static final List<PropertyDescriptor> properties = Collections.unmodifiableList(Arrays.asList(ACCESS_TOKEN, CHANNEL, PUBLISH_STRATEGY, MESSAGE_TEXT, CHARACTER_SET, SEND_CONTENT_AS_ATTACHMENT, MAX_FILE_SIZE, THREAD_TS));
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("FlowFiles are routed to success after being successfully sent to Slack").build();
    public static final Relationship REL_RATE_LIMITED = new Relationship.Builder().name("rate limited").description("FlowFiles are routed to 'rate limited' if the Rate Limit has been exceeded").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("FlowFiles are routed to 'failure' if unable to be sent to Slack for any other reason").build();
    private static final Set<Relationship> relationships = Collections.unmodifiableSet(new HashSet<Relationship>(Arrays.asList(REL_SUCCESS, REL_RATE_LIMITED, REL_FAILURE)));
    private final RateLimit rateLimit = new RateLimit(this.getLogger());
    private volatile ChannelMapper channelMapper;
    private volatile App slackApp;
    private volatile MethodsClient client;

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return properties;
    }

    public Set<Relationship> getRelationships() {
        return relationships;
    }

    @OnScheduled
    public void setup(ProcessContext context) {
        this.slackApp = this.createSlackApp(context);
        this.client = this.slackApp.client();
        this.channelMapper = new ChannelMapper(this.client);
    }

    @OnStopped
    public void shutdown() {
        if (this.slackApp != null) {
            this.slackApp.stop();
        }
    }

    private App createSlackApp(ProcessContext context) {
        String botToken = context.getProperty(ACCESS_TOKEN).getValue();
        AppConfig appConfig = AppConfig.builder().singleTeamBotToken(botToken).build();
        return new App(appConfig);
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        if (this.rateLimit.isLimitReached()) {
            this.getLogger().debug("Will not publish to Slack because rate limit has been reached");
            context.yield();
            return;
        }
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        String channelId = this.getChannelId(flowFile, session, context);
        if (channelId == null) {
            return;
        }
        String publishStrategy = context.getProperty(PUBLISH_STRATEGY).getValue();
        if (PUBLISH_STRATEGY_CONTENT_AS_MESSAGE.getValue().equalsIgnoreCase(publishStrategy)) {
            this.publishContentAsMessage(flowFile, channelId, context, session);
        } else if (context.getProperty(SEND_CONTENT_AS_ATTACHMENT).asBoolean().booleanValue()) {
            this.publishAsFile(flowFile, channelId, context, session);
        } else {
            String messageText = context.getProperty(MESSAGE_TEXT).evaluateAttributeExpressions(flowFile).getValue();
            this.publishAsMessage(flowFile, channelId, messageText, context, session);
        }
    }

    private String getChannelId(FlowFile flowFile, ProcessSession session, ProcessContext context) {
        String channelNameOrId = context.getProperty(CHANNEL).evaluateAttributeExpressions(flowFile).getValue();
        if (channelNameOrId.isEmpty()) {
            this.getLogger().error("No Channel ID was given for {}; routing to failure", new Object[]{flowFile});
            session.transfer(flowFile, REL_FAILURE);
            return null;
        }
        if (!channelNameOrId.startsWith("#")) {
            return channelNameOrId;
        }
        try {
            String channelId = this.channelMapper.lookupChannelId(channelNameOrId);
            if (channelId == null) {
                this.getLogger().error("Could not find Channel with name {} for {}; routing to failure", new Object[]{channelNameOrId, flowFile});
                session.transfer(flowFile, REL_FAILURE);
                return null;
            }
            return channelId;
        }
        catch (Exception e) {
            Relationship relationship = this.handleClientException(channelNameOrId, flowFile, session, context, e);
            this.getLogger().error("Failed to resolve Slack Channel ID for {}; transferring to {}", new Object[]{flowFile, relationship, e});
            return null;
        }
    }

    private void publishContentAsMessage(FlowFile flowFile, String channelId, ProcessContext context, ProcessSession session) {
        String messageText;
        if (flowFile.getSize() > 500000L) {
            this.getLogger().error("Cannot send contents of FlowFile {} to Slack because its length exceeds 500,000 bytes; routing to 'failure'", new Object[]{flowFile});
            session.transfer(flowFile, REL_FAILURE);
            return;
        }
        String charsetName = context.getProperty(CHARACTER_SET).evaluateAttributeExpressions(flowFile).getValue();
        byte[] buffer = new byte[(int)flowFile.getSize()];
        try (InputStream in = session.read(flowFile);){
            StreamUtils.fillBuffer((InputStream)in, (byte[])buffer, (boolean)true);
            messageText = new String(buffer, charsetName);
        }
        catch (IOException ioe) {
            this.getLogger().error("Failed to send contents of FlowFile {} to Slack; routing to failure", new Object[]{flowFile, ioe});
            session.transfer(flowFile, REL_FAILURE);
            return;
        }
        if (messageText.length() > 100000) {
            this.getLogger().error("Cannot send contents of FlowFile {} to Slack because its length exceeds 100,000 characters; routing to 'failure'", new Object[]{flowFile});
            session.transfer(flowFile, REL_FAILURE);
            return;
        }
        this.publishAsMessage(flowFile, channelId, messageText, context, session);
    }

    private void publishAsMessage(FlowFile flowFile, String channelId, String messageText, ProcessContext context, ProcessSession session) {
        ChatPostMessageResponse postMessageResponse;
        String threadTs = context.getProperty(THREAD_TS).evaluateAttributeExpressions(flowFile).getValue();
        ChatPostMessageRequest request = ChatPostMessageRequest.builder().channel(channelId).text(messageText).threadTs(threadTs).build();
        try {
            postMessageResponse = this.client.chatPostMessage(request);
        }
        catch (Exception e) {
            Relationship relationship = this.handleClientException(channelId, flowFile, session, context, e);
            this.getLogger().error("Failed to send message to Slack for {}; transferring to {}", new Object[]{flowFile, relationship, e});
            return;
        }
        if (!postMessageResponse.isOk()) {
            String errorMessage = PublishSlack.getErrorMessage(postMessageResponse.getError(), postMessageResponse.getNeeded(), postMessageResponse.getProvided(), postMessageResponse.getWarning());
            this.getLogger().error("Could not send message to Slack for {} - received error: {}", new Object[]{flowFile, errorMessage});
            session.transfer(flowFile, REL_FAILURE);
            return;
        }
        String ts = postMessageResponse.getTs();
        HashMap<String, String> attributes = new HashMap<String, String>();
        attributes.put("slack.ts", ts);
        attributes.put("slack.channel.id", channelId);
        flowFile = session.putAllAttributes(flowFile, attributes);
        session.getProvenanceReporter().send(flowFile, "https://slack.com/api/chat.postMessage");
        session.transfer(flowFile, REL_SUCCESS);
    }

    private void publishAsFile(FlowFile flowFile, String channelId, ProcessContext context, ProcessSession session) {
        FilesUploadV2Response uploadResponse;
        String filename = flowFile.getAttribute(CoreAttributes.FILENAME.key());
        long maxSize = context.getProperty(MAX_FILE_SIZE).asDataSize(DataUnit.B).longValue();
        if (flowFile.getSize() > maxSize) {
            this.getLogger().warn("{} exceeds max allowable file size. Max File Size = {}; FlowFile size = {}; routing to 'failure'", new Object[]{flowFile, FormatUtils.formatDataSize((double)maxSize), FormatUtils.formatDataSize((double)flowFile.getSize())});
            session.transfer(flowFile, REL_FAILURE);
            return;
        }
        try {
            byte[] buffer = new byte[(int)flowFile.getSize()];
            try (InputStream in = session.read(flowFile);){
                StreamUtils.fillBuffer((InputStream)in, (byte[])buffer, (boolean)true);
            }
            String message = context.getProperty(MESSAGE_TEXT).evaluateAttributeExpressions(flowFile).getValue();
            String threadTs = context.getProperty(THREAD_TS).evaluateAttributeExpressions(flowFile).getValue();
            FilesUploadV2Request uploadRequest = FilesUploadV2Request.builder().filename(filename).title(filename).initialComment(message).channel(channelId).threadTs(threadTs).fileData(buffer).build();
            uploadResponse = this.client.filesUploadV2(uploadRequest);
        }
        catch (Exception e) {
            Relationship relationship = this.handleClientException(channelId, flowFile, session, context, e);
            this.getLogger().error("Could not upload contents of {} to Slack; routing to {}", new Object[]{flowFile, relationship, e});
            return;
        }
        if (!uploadResponse.isOk()) {
            String errorMessage = PublishSlack.getErrorMessage(uploadResponse.getError(), uploadResponse.getNeeded(), uploadResponse.getProvided(), uploadResponse.getWarning());
            this.getLogger().error("Could not upload contents of {} to Slack - received error: {}", new Object[]{flowFile, errorMessage});
            session.transfer(flowFile, REL_FAILURE);
            return;
        }
        File file = uploadResponse.getFile();
        File.Shares shares = file.getShares();
        String ts = null;
        if (shares != null && (ts = this.getTs(shares.getPrivateChannels())) == null) {
            ts = this.getTs(shares.getPublicChannels());
        }
        HashMap<String, String> attributes = new HashMap<String, String>();
        attributes.put("slack.channel.id", channelId);
        if (ts != null) {
            attributes.put("slack.ts", ts);
        }
        flowFile = session.putAllAttributes(flowFile, attributes);
        session.getProvenanceReporter().send(flowFile, "https://slack.com/api/files.upload");
        session.transfer(flowFile, REL_SUCCESS);
    }

    private Relationship handleClientException(String channel, FlowFile flowFile, ProcessSession session, ProcessContext context, Exception cause) {
        boolean rateLimited = this.yieldOnRateLimit(cause, channel, context);
        Relationship relationship = rateLimited ? REL_RATE_LIMITED : REL_FAILURE;
        session.transfer(flowFile, relationship);
        return relationship;
    }

    private boolean yieldOnRateLimit(Throwable t, String channelId, ProcessContext context) {
        boolean rateLimited = PublishSlack.isRateLimited(t);
        if (rateLimited) {
            this.getLogger().warn("Slack indicated that the Rate Limit has been exceeded when attempting to publish messages to channel {}", new Object[]{channelId});
        } else {
            this.getLogger().error("Failed to retrieve messages for channel {}", new Object[]{channelId, t});
        }
        int retryAfterSeconds = PublishSlack.getRetryAfterSeconds(t);
        this.rateLimit.retryAfter(Duration.ofSeconds(retryAfterSeconds));
        context.yield();
        return rateLimited;
    }

    public static String getErrorMessage(String error, String needed, String provided, String warning) {
        String mainMessage;
        String string = mainMessage = error == null ? warning : error;
        if (needed != null && provided != null) {
            return mainMessage + ": Permission needed: " + needed + "; Permission granted to this bot: " + provided;
        }
        return mainMessage;
    }

    public static boolean isRateLimited(Throwable cause) {
        return cause instanceof SlackApiException && 429 == ((SlackApiException)cause).getResponse().code();
    }

    public static int getRetryAfterSeconds(Throwable cause) {
        if (!(cause instanceof SlackApiException)) {
            return 1;
        }
        SlackApiException slackApiException = (SlackApiException)cause;
        return Integer.parseInt(slackApiException.getResponse().header("Retry-After", "10"));
    }

    private String getTs(Map<String, List<File.ShareDetail>> shareDetails) {
        if (shareDetails == null) {
            return null;
        }
        for (List<File.ShareDetail> detailList : shareDetails.values()) {
            for (File.ShareDetail detail : detailList) {
                String ts = detail.getTs();
                if (ts == null) continue;
                return ts;
            }
        }
        return null;
    }
}

