/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.security.util.crypto;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.security.util.crypto.HashAlgorithm;
import org.bouncycastle.shaded.crypto.digests.Blake2bDigest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HashService {
    private static final Logger logger = LoggerFactory.getLogger(HashService.class);
    private static final int BUFFER_SIZE = 8192;
    private static final String UTF_16_DESCRIPTION = "This character set normally decodes using an optional BOM at the beginning of the data but encodes by inserting a BE BOM. For hashing, it will be replaced with UTF-16BE. ";

    public static AllowableValue[] buildHashAlgorithmAllowableValues() {
        HashAlgorithm[] hashAlgorithms = HashAlgorithm.values();
        ArrayList<AllowableValue> allowableValues = new ArrayList<AllowableValue>(hashAlgorithms.length);
        for (HashAlgorithm algorithm : hashAlgorithms) {
            allowableValues.add(new AllowableValue(algorithm.getName(), algorithm.getName(), algorithm.buildAllowableValueDescription()));
        }
        return allowableValues.toArray(new AllowableValue[0]);
    }

    public static AllowableValue[] buildCharacterSetAllowableValues() {
        List<Charset> charsets = HashService.getSupportedCharsets();
        return (AllowableValue[])charsets.stream().map(cs -> new AllowableValue(cs.name(), cs.displayName(), cs == StandardCharsets.UTF_16 ? UTF_16_DESCRIPTION : cs.displayName())).toArray(AllowableValue[]::new);
    }

    public static List<Charset> getSupportedCharsets() {
        return Arrays.asList(StandardCharsets.US_ASCII, StandardCharsets.ISO_8859_1, StandardCharsets.UTF_8, StandardCharsets.UTF_16BE, StandardCharsets.UTF_16LE, StandardCharsets.UTF_16);
    }

    public static String hashValueStreaming(HashAlgorithm algorithm, InputStream value) throws IOException {
        if (algorithm == null) {
            throw new IllegalArgumentException("The hash algorithm cannot be null");
        }
        if (value == null) {
            throw new IllegalArgumentException("The value cannot be null");
        }
        if (algorithm.isBlake2()) {
            return Hex.encodeHexString((byte[])HashService.blake2HashStreaming(algorithm, value));
        }
        return Hex.encodeHexString((byte[])HashService.traditionalHashStreaming(algorithm, value));
    }

    public static String hashValue(HashAlgorithm algorithm, String value, Charset charset) {
        byte[] rawHash = HashService.hashValueRaw(algorithm, value, charset);
        return Hex.encodeHexString((byte[])rawHash);
    }

    public static String hashValue(HashAlgorithm algorithm, String value) {
        return HashService.hashValue(algorithm, value, StandardCharsets.UTF_8);
    }

    public static byte[] hashValueRaw(HashAlgorithm algorithm, String value, Charset charset) {
        if (value == null) {
            throw new IllegalArgumentException("The value cannot be null");
        }
        if (charset == StandardCharsets.UTF_16) {
            logger.warn("The charset provided was UTF-16, but Java will insert a Big Endian BOM in the decoded message before hashing, so switching to UTF-16BE");
            charset = StandardCharsets.UTF_16BE;
        }
        return HashService.hashValueRaw(algorithm, value.getBytes(charset));
    }

    public static byte[] hashValueRaw(HashAlgorithm algorithm, String value) {
        return HashService.hashValueRaw(algorithm, value, StandardCharsets.UTF_8);
    }

    public static byte[] hashValueRaw(HashAlgorithm algorithm, byte[] value) {
        if (algorithm == null) {
            throw new IllegalArgumentException("The hash algorithm cannot be null");
        }
        if (value == null) {
            throw new IllegalArgumentException("The value cannot be null");
        }
        if (algorithm.isBlake2()) {
            return HashService.blake2Hash(algorithm, value);
        }
        return HashService.traditionalHash(algorithm, value);
    }

    private static byte[] traditionalHash(HashAlgorithm algorithm, byte[] value) {
        return DigestUtils.getDigest((String)algorithm.getName()).digest(value);
    }

    private static byte[] traditionalHashStreaming(HashAlgorithm algorithm, InputStream value) throws IOException {
        MessageDigest digest = DigestUtils.getDigest((String)algorithm.getName());
        return DigestUtils.digest((MessageDigest)digest, (InputStream)value);
    }

    private static byte[] blake2Hash(HashAlgorithm algorithm, byte[] value) {
        int digestLengthBytes = algorithm.getDigestBytesLength();
        Blake2bDigest blake2bDigest = new Blake2bDigest(digestLengthBytes * 8);
        byte[] rawHash = new byte[blake2bDigest.getDigestSize()];
        blake2bDigest.update(value, 0, value.length);
        blake2bDigest.doFinal(rawHash, 0);
        return rawHash;
    }

    private static byte[] blake2HashStreaming(HashAlgorithm algorithm, InputStream value) throws IOException {
        int digestLengthBytes = algorithm.getDigestBytesLength();
        Blake2bDigest blake2bDigest = new Blake2bDigest(digestLengthBytes * 8);
        byte[] rawHash = new byte[blake2bDigest.getDigestSize()];
        byte[] buffer = new byte[8192];
        int read = value.read(buffer, 0, 8192);
        while (read > -1) {
            blake2bDigest.update(buffer, 0, read);
            read = value.read(buffer, 0, 8192);
        }
        blake2bDigest.doFinal(rawHash, 0);
        return rawHash;
    }
}

