/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.remote.io.socket.ssl;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.SocketTimeoutException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.nio.channels.SocketChannel;
import java.util.concurrent.TimeUnit;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;
import javax.net.ssl.SSLSession;
import org.apache.nifi.remote.exception.TransmissionDisabledException;
import org.apache.nifi.remote.io.socket.BufferStateManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SSLSocketChannel
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(SSLSocketChannel.class);
    private static final int MINIMUM_READ_BUFFER_SIZE = 1;
    private static final int DISCARD_BUFFER_LENGTH = 8192;
    private static final int END_OF_STREAM = -1;
    private static final byte[] EMPTY_MESSAGE = new byte[0];
    private static final long BUFFER_FULL_EMPTY_WAIT_NANOS = TimeUnit.NANOSECONDS.convert(1L, TimeUnit.MILLISECONDS);
    private static final long FINISH_CONNECT_SLEEP = 50L;
    private static final long INITIAL_INCREMENTAL_SLEEP = 1L;
    private static final boolean CLIENT_AUTHENTICATION_REQUIRED = true;
    private final String remoteAddress;
    private final int port;
    private final SSLEngine engine;
    private final SocketAddress socketAddress;
    private final BufferStateManager streamInManager;
    private final BufferStateManager streamOutManager;
    private final BufferStateManager appDataManager;
    private final SocketChannel channel;
    private int timeoutMillis = 30000;
    private volatile boolean interrupted = false;
    private volatile ChannelStatus channelStatus = ChannelStatus.DISCONNECTED;

    public SSLSocketChannel(SSLContext sslContext, String remoteAddress, int port, InetAddress bindAddress, boolean useClientMode) throws IOException {
        this.engine = SSLSocketChannel.createEngine(sslContext, useClientMode);
        this.channel = SSLSocketChannel.createSocketChannel(bindAddress);
        this.socketAddress = new InetSocketAddress(remoteAddress, port);
        this.remoteAddress = remoteAddress;
        this.port = port;
        this.streamInManager = new BufferStateManager(ByteBuffer.allocate(this.engine.getSession().getPacketBufferSize()));
        this.streamOutManager = new BufferStateManager(ByteBuffer.allocate(this.engine.getSession().getPacketBufferSize()));
        this.appDataManager = new BufferStateManager(ByteBuffer.allocate(this.engine.getSession().getApplicationBufferSize()));
    }

    public SSLSocketChannel(SSLContext sslContext, SocketChannel socketChannel, boolean useClientMode) throws IOException {
        this(SSLSocketChannel.createEngine(sslContext, useClientMode), socketChannel);
    }

    public SSLSocketChannel(SSLEngine sslEngine, SocketChannel socketChannel) throws IOException {
        if (!socketChannel.isConnected()) {
            throw new IllegalArgumentException("Connected SocketChannel required");
        }
        socketChannel.configureBlocking(false);
        this.channel = socketChannel;
        this.socketAddress = socketChannel.getRemoteAddress();
        Socket socket = socketChannel.socket();
        this.remoteAddress = socket.getInetAddress().toString();
        this.port = socket.getPort();
        this.engine = sslEngine;
        this.streamInManager = new BufferStateManager(ByteBuffer.allocate(this.engine.getSession().getPacketBufferSize()));
        this.streamOutManager = new BufferStateManager(ByteBuffer.allocate(this.engine.getSession().getPacketBufferSize()));
        this.appDataManager = new BufferStateManager(ByteBuffer.allocate(this.engine.getSession().getApplicationBufferSize()));
    }

    public void setTimeout(int timeoutMillis) {
        this.timeoutMillis = timeoutMillis;
    }

    public int getTimeout() {
        return this.timeoutMillis;
    }

    public void connect() throws IOException {
        this.channelStatus = ChannelStatus.CONNECTING;
        try {
            if (!this.channel.isConnected()) {
                this.logOperation("Connection Started");
                long started = System.currentTimeMillis();
                if (!this.channel.connect(this.socketAddress)) {
                    while (!this.channel.finishConnect()) {
                        this.checkInterrupted();
                        this.checkTimeoutExceeded(started);
                        try {
                            TimeUnit.MILLISECONDS.sleep(50L);
                        }
                        catch (InterruptedException e) {
                            this.logOperation("Connection Interrupted");
                        }
                    }
                }
            }
            this.channelStatus = ChannelStatus.CONNECTED;
        }
        catch (Exception e) {
            this.close();
            throw new SSLException(String.format("[%s:%d] Connection Failed", this.remoteAddress, this.port), e);
        }
        try {
            this.performHandshake();
        }
        catch (IOException e) {
            this.close();
            throw new SSLException(String.format("[%s:%d] Handshake Failed", this.remoteAddress, this.port), e);
        }
    }

    public void consume() throws IOException {
        int readCount;
        this.channel.shutdownInput();
        byte[] byteBuffer = new byte[8192];
        ByteBuffer buffer = ByteBuffer.wrap(byteBuffer);
        do {
            readCount = this.channel.read(buffer);
            buffer.flip();
        } while (readCount > 0);
    }

    public boolean isClosed() {
        int bytesRead;
        if (ChannelStatus.CLOSED == this.channelStatus) {
            return true;
        }
        ByteBuffer inputBuffer = this.streamInManager.prepareForWrite(this.engine.getSession().getPacketBufferSize());
        try {
            bytesRead = this.channel.read(inputBuffer);
        }
        catch (IOException e) {
            LOGGER.warn("[{}:{}] Closed Status Read Failed", new Object[]{this.remoteAddress, this.port, e});
            bytesRead = -1;
        }
        this.logOperationBytes("Closed Status Read", bytesRead);
        if (bytesRead == 0) {
            return false;
        }
        if (bytesRead > 0) {
            try {
                SSLEngineResult unwrapResult = this.unwrap();
                if (SSLEngineResult.Status.CLOSED != unwrapResult.getStatus()) {
                    this.streamInManager.compact();
                    return false;
                }
                this.readChannelDiscard();
                this.engine.closeInbound();
            }
            catch (IOException e) {
                LOGGER.warn("[{}:{}] Closed Status Unwrap Failed", new Object[]{this.remoteAddress, this.port, e});
            }
        }
        try {
            this.close();
        }
        catch (IOException e) {
            LOGGER.warn("[{}:{}] Close Failed", new Object[]{this.remoteAddress, this.port, e});
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        block8: {
            this.logOperation("Close Requested");
            if (this.channelStatus == ChannelStatus.CLOSED) {
                return;
            }
            try {
                this.engine.closeOutbound();
                this.streamOutManager.clear();
                ByteBuffer inputBuffer = ByteBuffer.wrap(EMPTY_MESSAGE);
                ByteBuffer outputBuffer = this.streamOutManager.prepareForWrite(this.engine.getSession().getApplicationBufferSize());
                SSLEngineResult wrapResult = this.wrap(inputBuffer, outputBuffer);
                SSLEngineResult.Status status = wrapResult.getStatus();
                if (SSLEngineResult.Status.OK == status) {
                    this.logOperation("Clearing Outbound Buffer");
                    outputBuffer.clear();
                    wrapResult = this.wrap(inputBuffer, outputBuffer);
                    status = wrapResult.getStatus();
                }
                if (SSLEngineResult.Status.CLOSED == status) {
                    ByteBuffer streamOutputBuffer = this.streamOutManager.prepareForRead(1);
                    try {
                        this.writeChannel(streamOutputBuffer);
                    }
                    catch (IOException e) {
                        this.logOperation(String.format("Write Close Notification Failed: %s", e.getMessage()));
                    }
                    break block8;
                }
                throw new SSLException(String.format("[%s:%d] Invalid Wrap Result Status [%s]", new Object[]{this.remoteAddress, this.port, status}));
            }
            finally {
                this.channelStatus = ChannelStatus.CLOSED;
                this.readChannelDiscard();
                this.closeQuietly(this.channel.socket());
                this.closeQuietly(this.channel);
                this.logOperation("Close Completed");
            }
        }
    }

    public int available() throws IOException {
        ByteBuffer appDataBuffer = this.appDataManager.prepareForRead(1);
        return appDataBuffer.remaining();
    }

    public int read() throws IOException {
        byte[] buffer = new byte[1];
        int bytesRead = this.read(buffer);
        if (bytesRead == -1) {
            return -1;
        }
        return Byte.toUnsignedInt(buffer[0]);
    }

    public int read(byte[] buffer) throws IOException {
        return this.read(buffer, 0, buffer.length);
    }

    public int read(byte[] buffer, int offset, int len) throws IOException {
        this.logOperationBytes("Read Requested", len);
        this.checkChannelStatus();
        int applicationBytesRead = this.readApplicationBuffer(buffer, offset, len);
        if (applicationBytesRead > 0) {
            return applicationBytesRead;
        }
        this.appDataManager.clear();
        SSLEngineResult unwrapResult = this.unwrapBufferReadChannel();
        SSLEngineResult.Status status = unwrapResult.getStatus();
        if (SSLEngineResult.Status.CLOSED == status) {
            applicationBytesRead = this.readApplicationBuffer(buffer, offset, len);
            if (applicationBytesRead == 0) {
                return -1;
            }
            this.streamInManager.compact();
            return applicationBytesRead;
        }
        if (SSLEngineResult.Status.OK == status) {
            applicationBytesRead = this.readApplicationBuffer(buffer, offset, len);
            if (applicationBytesRead == 0) {
                throw new IOException("Read Application Buffer Failed");
            }
            this.streamInManager.compact();
            return applicationBytesRead;
        }
        throw new IllegalStateException(String.format("SSLEngineResult Status [%s] not expected from unwrap", new Object[]{status}));
    }

    public void write(int data) throws IOException {
        this.write(new byte[]{(byte)data}, 0, 1);
    }

    public void write(byte[] data) throws IOException {
        this.write(data, 0, data.length);
    }

    public void write(byte[] data, int offset, int len) throws IOException {
        this.logOperationBytes("Write Started", len);
        this.checkChannelStatus();
        int applicationBufferSize = this.engine.getSession().getApplicationBufferSize();
        this.logOperationBytes("Write Application Buffer Size", applicationBufferSize);
        int iterations = len / applicationBufferSize;
        if (len % applicationBufferSize > 0) {
            ++iterations;
        }
        block6: for (int i = 0; i < iterations; ++i) {
            this.streamOutManager.clear();
            int itrOffset = offset + i * applicationBufferSize;
            int itrLen = Math.min(len - itrOffset, applicationBufferSize);
            ByteBuffer byteBuffer = ByteBuffer.wrap(data, itrOffset, itrLen);
            BufferStateManager bufferStateManager = new BufferStateManager(byteBuffer, BufferStateManager.Direction.READ);
            SSLEngineResult.Status status = this.wrapWriteChannel(bufferStateManager);
            switch (status) {
                case BUFFER_OVERFLOW: {
                    this.streamOutManager.ensureSize(this.engine.getSession().getPacketBufferSize());
                    this.appDataManager.ensureSize(this.engine.getSession().getApplicationBufferSize());
                    continue block6;
                }
                case OK: {
                    continue block6;
                }
                case CLOSED: {
                    throw new IOException("Channel is closed");
                }
                case BUFFER_UNDERFLOW: {
                    throw new AssertionError((Object)"Got Buffer Underflow but should not have...");
                }
            }
        }
    }

    public void interrupt() {
        this.interrupted = true;
    }

    private void performHandshake() throws IOException {
        this.logOperation("Handshake Started");
        this.channelStatus = ChannelStatus.HANDSHAKING;
        this.engine.beginHandshake();
        SSLEngineResult.HandshakeStatus handshakeStatus = this.engine.getHandshakeStatus();
        while (true) {
            this.logHandshakeStatus(handshakeStatus);
            switch (handshakeStatus) {
                case FINISHED: 
                case NOT_HANDSHAKING: {
                    this.channelStatus = ChannelStatus.ESTABLISHED;
                    SSLSession session = this.engine.getSession();
                    LOGGER.debug("[{}:{}] [{}] Negotiated Protocol [{}] Cipher Suite [{}]", new Object[]{this.remoteAddress, this.port, this.channelStatus, session.getProtocol(), session.getCipherSuite()});
                    return;
                }
                case NEED_TASK: {
                    this.runDelegatedTasks();
                    handshakeStatus = this.engine.getHandshakeStatus();
                    break;
                }
                case NEED_UNWRAP: {
                    SSLEngineResult unwrapResult = this.unwrapBufferReadChannel();
                    handshakeStatus = unwrapResult.getHandshakeStatus();
                    if (unwrapResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                        throw this.getHandshakeException(handshakeStatus, "Channel Closed");
                    }
                    this.streamInManager.compact();
                    this.appDataManager.clear();
                    break;
                }
                case NEED_WRAP: {
                    ByteBuffer outboundBuffer = this.streamOutManager.prepareForWrite(this.engine.getSession().getApplicationBufferSize());
                    SSLEngineResult wrapResult = this.wrap(ByteBuffer.wrap(EMPTY_MESSAGE), outboundBuffer);
                    handshakeStatus = wrapResult.getHandshakeStatus();
                    SSLEngineResult.Status wrapResultStatus = wrapResult.getStatus();
                    if (wrapResultStatus == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                        this.streamOutManager.prepareForWrite(this.engine.getSession().getApplicationBufferSize());
                        break;
                    }
                    if (wrapResultStatus == SSLEngineResult.Status.OK) {
                        ByteBuffer streamBuffer = this.streamOutManager.prepareForRead(1);
                        int bytesRemaining = streamBuffer.remaining();
                        this.writeChannel(streamBuffer);
                        this.logOperationBytes("Handshake Channel Write Completed", bytesRemaining);
                        this.streamOutManager.clear();
                        break;
                    }
                    throw this.getHandshakeException(handshakeStatus, String.format("Wrap Failed [%s]", new Object[]{wrapResult.getStatus()}));
                }
            }
        }
    }

    private SSLEngineResult unwrapBufferReadChannel() throws IOException {
        SSLEngineResult unwrapResult = this.unwrap();
        while (SSLEngineResult.Status.BUFFER_UNDERFLOW == unwrapResult.getStatus()) {
            int channelBytesRead = this.readChannel();
            if (channelBytesRead == -1) {
                throw new EOFException("End of Stream found for Channel Read");
            }
            unwrapResult = this.unwrap();
            if (SSLEngineResult.HandshakeStatus.FINISHED != unwrapResult.getHandshakeStatus()) continue;
            this.logOperation("Processing Post-Handshake Messages");
            unwrapResult = this.unwrap();
        }
        return unwrapResult;
    }

    private int readChannel() throws IOException {
        int channelBytesRead;
        this.logOperation("Channel Read Started");
        ByteBuffer outputBuffer = this.streamInManager.prepareForWrite(this.engine.getSession().getPacketBufferSize());
        long started = System.currentTimeMillis();
        long sleepNanoseconds = 1L;
        while (true) {
            this.checkInterrupted();
            if (outputBuffer.remaining() == 0) {
                return 0;
            }
            channelBytesRead = this.channel.read(outputBuffer);
            if (channelBytesRead != 0) break;
            this.checkTimeoutExceeded(started);
            sleepNanoseconds = this.incrementalSleep(sleepNanoseconds);
        }
        this.logOperationBytes("Channel Read Completed", channelBytesRead);
        return channelBytesRead;
    }

    private void readChannelDiscard() {
        try {
            ByteBuffer readBuffer = ByteBuffer.allocate(8192);
            int bytesRead = this.channel.read(readBuffer);
            while (bytesRead > 0) {
                readBuffer.clear();
                bytesRead = this.channel.read(readBuffer);
            }
        }
        catch (IOException e) {
            LOGGER.debug("[{}:{}] Read Channel Discard Failed", new Object[]{this.remoteAddress, this.port, e});
        }
    }

    private void writeChannel(ByteBuffer inputBuffer) throws IOException {
        long lastWriteCompleted = System.currentTimeMillis();
        int totalBytes = 0;
        long sleepNanoseconds = 1L;
        while (inputBuffer.hasRemaining()) {
            this.checkInterrupted();
            int written = this.channel.write(inputBuffer);
            totalBytes += written;
            if (written > 0) {
                lastWriteCompleted = System.currentTimeMillis();
                continue;
            }
            this.checkTimeoutExceeded(lastWriteCompleted);
            sleepNanoseconds = this.incrementalSleep(sleepNanoseconds);
        }
        this.logOperationBytes("Channel Write Completed", totalBytes);
    }

    private long incrementalSleep(long nanoseconds) throws IOException {
        try {
            TimeUnit.NANOSECONDS.sleep(nanoseconds);
        }
        catch (InterruptedException e) {
            this.close();
            Thread.currentThread().interrupt();
            throw new ClosedByInterruptException();
        }
        return Math.min(nanoseconds * 2L, BUFFER_FULL_EMPTY_WAIT_NANOS);
    }

    private int readApplicationBuffer(byte[] buffer, int offset, int len) {
        this.logOperationBytes("Application Buffer Read Requested", len);
        ByteBuffer appDataBuffer = this.appDataManager.prepareForRead(len);
        int appDataRemaining = appDataBuffer.remaining();
        this.logOperationBytes("Application Buffer Remaining", appDataRemaining);
        if (appDataRemaining > 0) {
            int bytesToCopy = Math.min(len, appDataBuffer.remaining());
            appDataBuffer.get(buffer, offset, bytesToCopy);
            int bytesCopied = appDataRemaining - appDataBuffer.remaining();
            this.logOperationBytes("Application Buffer Copied", bytesCopied);
            return bytesCopied;
        }
        return 0;
    }

    private SSLEngineResult.Status wrapWriteChannel(BufferStateManager inputManager) throws IOException {
        SSLEngineResult result;
        ByteBuffer inputBuffer = inputManager.prepareForRead(0);
        ByteBuffer outputBuffer = this.streamOutManager.prepareForWrite(this.engine.getSession().getApplicationBufferSize());
        this.logOperationBytes("Wrap Started", inputBuffer.remaining());
        SSLEngineResult.Status status = SSLEngineResult.Status.OK;
        while (inputBuffer.remaining() > 0 && (status = (result = this.wrap(inputBuffer, outputBuffer)).getStatus()) == SSLEngineResult.Status.OK) {
            ByteBuffer readableOutBuff = this.streamOutManager.prepareForRead(0);
            this.writeChannel(readableOutBuff);
            this.streamOutManager.clear();
        }
        return status;
    }

    private SSLEngineResult wrap(ByteBuffer inputBuffer, ByteBuffer outputBuffer) throws SSLException {
        SSLEngineResult result = this.engine.wrap(inputBuffer, outputBuffer);
        this.logEngineResult(result, "WRAP Completed");
        return result;
    }

    private SSLEngineResult unwrap() throws IOException {
        ByteBuffer streamBuffer = this.streamInManager.prepareForRead(this.engine.getSession().getPacketBufferSize());
        ByteBuffer applicationBuffer = this.appDataManager.prepareForWrite(this.engine.getSession().getApplicationBufferSize());
        SSLEngineResult result = this.engine.unwrap(streamBuffer, applicationBuffer);
        this.logEngineResult(result, "UNWRAP Completed");
        return result;
    }

    private void runDelegatedTasks() {
        Runnable delegatedTask;
        while ((delegatedTask = this.engine.getDelegatedTask()) != null) {
            this.logOperation("Running Delegated Task");
            delegatedTask.run();
        }
    }

    private void closeQuietly(Closeable closeable) {
        try {
            closeable.close();
        }
        catch (Exception e) {
            this.logOperation(String.format("Close failed: %s", e.getMessage()));
        }
    }

    private SSLHandshakeException getHandshakeException(SSLEngineResult.HandshakeStatus handshakeStatus, String message) {
        String formatted = String.format("[%s:%d] Handshake Status [%s] %s", new Object[]{this.remoteAddress, this.port, handshakeStatus, message});
        return new SSLHandshakeException(formatted);
    }

    private void checkChannelStatus() throws IOException {
        if (ChannelStatus.ESTABLISHED != this.channelStatus) {
            this.connect();
        }
    }

    private void checkInterrupted() {
        if (this.interrupted) {
            throw new TransmissionDisabledException();
        }
    }

    private void checkTimeoutExceeded(long started) throws SocketTimeoutException {
        if (System.currentTimeMillis() > started + (long)this.timeoutMillis) {
            throw new SocketTimeoutException(String.format("Timeout Exceeded [%d ms] for [%s:%d]", this.timeoutMillis, this.remoteAddress, this.port));
        }
    }

    private void logOperation(String operation) {
        LOGGER.trace("[{}:{}] [{}] {}", new Object[]{this.remoteAddress, this.port, this.channelStatus, operation});
    }

    private void logOperationBytes(String operation, int bytes) {
        LOGGER.trace("[{}:{}] [{}] {} Bytes [{}]", new Object[]{this.remoteAddress, this.port, this.channelStatus, operation, bytes});
    }

    private void logHandshakeStatus(SSLEngineResult.HandshakeStatus handshakeStatus) {
        LOGGER.trace("[{}:{}] [{}] Handshake Status [{}]", new Object[]{this.remoteAddress, this.port, this.channelStatus, handshakeStatus});
    }

    private void logEngineResult(SSLEngineResult result, String method) {
        LOGGER.trace("[{}:{}] [{}] {} Status [{}] Handshake Status [{}] Produced [{}] Consumed [{}]", new Object[]{this.remoteAddress, this.port, this.channelStatus, method, result.getStatus(), result.getHandshakeStatus(), result.bytesProduced(), result.bytesConsumed()});
    }

    private static SocketChannel createSocketChannel(InetAddress bindAddress) throws IOException {
        SocketChannel socketChannel = SocketChannel.open();
        if (bindAddress != null) {
            InetSocketAddress socketAddress = new InetSocketAddress(bindAddress, 0);
            socketChannel.bind(socketAddress);
        }
        socketChannel.configureBlocking(false);
        return socketChannel;
    }

    private static SSLEngine createEngine(SSLContext sslContext, boolean useClientMode) {
        SSLEngine sslEngine = sslContext.createSSLEngine();
        sslEngine.setUseClientMode(useClientMode);
        sslEngine.setNeedClientAuth(true);
        return sslEngine;
    }

    private static enum ChannelStatus {
        DISCONNECTED,
        CONNECTING,
        CONNECTED,
        HANDSHAKING,
        ESTABLISHED,
        CLOSED;

    }
}

