/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.provenance.index.lucene;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Optional;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.nifi.provenance.RepositoryConfiguration;
import org.apache.nifi.provenance.index.lucene.IndexLocation;
import org.apache.nifi.provenance.util.DirectoryUtils;
import org.apache.nifi.util.Tuple;
import org.apache.nifi.util.file.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexDirectoryManager {
    private static final Logger logger = LoggerFactory.getLogger(IndexDirectoryManager.class);
    private static final Pattern LUCENE_8_AND_LATER_INDEX_PATTERN = Pattern.compile("lucene-\\d+-index-(.*)");
    private static final FileFilter LUCENE_8_AND_LATER_INDEX_DIRECTORY_FILTER = f -> LUCENE_8_AND_LATER_INDEX_PATTERN.matcher(f.getName()).matches();
    private static final Pattern INDEX_FILENAME_PATTERN = DirectoryUtils.INDEX_DIRECTORY_NAME_PATTERN;
    private static final FileFilter ALL_INDEX_FILE_FILTER = f -> INDEX_FILENAME_PATTERN.matcher(f.getName()).matches();
    private static final Pattern LUCENE_4_INDEX_PATTERN = Pattern.compile("index-(.*)");
    private static final FileFilter LUCENE_4_INDEX_FILE_FILTER = f -> LUCENE_4_INDEX_PATTERN.matcher(f.getName()).matches();
    private final RepositoryConfiguration repoConfig;
    private final SortedMap<Long, List<IndexLocation>> indexLocationByTimestamp = new TreeMap<Long, List<IndexLocation>>();
    private final Map<String, IndexLocation> activeIndices = new HashMap<String, IndexLocation>();

    public IndexDirectoryManager(RepositoryConfiguration repoConfig) {
        this.repoConfig = repoConfig;
    }

    public synchronized void initialize() {
        HashMap<File, Tuple> latestIndexByStorageDir = new HashMap<File, Tuple>();
        for (Map.Entry<String, File> entry : this.repoConfig.getStorageDirectories().entrySet()) {
            String partitionName = entry.getKey();
            File storageDir = entry.getValue();
            File[] indexDirs = storageDir.listFiles(LUCENE_8_AND_LATER_INDEX_DIRECTORY_FILTER);
            if (indexDirs == null) {
                logger.warn("Unable to access Provenance Repository storage directory {}", (Object)storageDir);
                continue;
            }
            for (File indexDir : indexDirs) {
                Matcher matcher = INDEX_FILENAME_PATTERN.matcher(indexDir.getName());
                if (!matcher.matches()) continue;
                long startTime = DirectoryUtils.getIndexTimestamp(indexDir);
                List dirsForTimestamp = this.indexLocationByTimestamp.computeIfAbsent(startTime, t -> new ArrayList());
                IndexLocation indexLoc = new IndexLocation(indexDir, startTime, partitionName);
                dirsForTimestamp.add(indexLoc);
                Tuple tuple = (Tuple)latestIndexByStorageDir.get(storageDir);
                if (tuple != null && startTime <= (Long)tuple.getKey()) continue;
                latestIndexByStorageDir.put(storageDir, new Tuple((Object)startTime, (Object)indexLoc));
            }
        }
        for (Tuple tuple : latestIndexByStorageDir.values()) {
            IndexLocation indexLoc = (IndexLocation)tuple.getValue();
            File indexDir = indexLoc.getIndexDirectory();
            if (!indexDir.exists()) continue;
            try {
                FSDirectory directory = FSDirectory.open((Path)indexDir.toPath());
                try {
                    DirectoryReader reader = DirectoryReader.open((Directory)directory);
                    try {
                        this.activeIndices.put(indexLoc.getPartitionName(), indexLoc);
                    }
                    finally {
                        if (reader == null) continue;
                        reader.close();
                    }
                }
                finally {
                    if (directory == null) continue;
                    directory.close();
                }
            }
            catch (IOException ioe) {
                logger.debug("Unable to open Lucene Index located at {} so assuming that it is defunct and will not use as the active index", (Object)indexDir, (Object)ioe);
            }
        }
    }

    public synchronized void removeDirectory(File directory) {
        Iterator<Map.Entry<Long, List<IndexLocation>>> itr = this.indexLocationByTimestamp.entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry<Long, List<IndexLocation>> entry = itr.next();
            List<IndexLocation> locations = entry.getValue();
            IndexLocation locToRemove = new IndexLocation(directory, DirectoryUtils.getIndexTimestamp(directory), directory.getName());
            locations.remove(locToRemove);
            if (!locations.isEmpty()) continue;
            itr.remove();
        }
    }

    public synchronized List<File> getAllIndexDirectories(boolean includeLucene4Directories, boolean includeLaterLuceneDirectories) {
        FileFilter directoryFilter;
        ArrayList<File> allDirectories = new ArrayList<File>();
        if (includeLucene4Directories && includeLaterLuceneDirectories) {
            directoryFilter = ALL_INDEX_FILE_FILTER;
        } else if (includeLucene4Directories) {
            directoryFilter = LUCENE_4_INDEX_FILE_FILTER;
        } else if (includeLaterLuceneDirectories) {
            directoryFilter = LUCENE_8_AND_LATER_INDEX_DIRECTORY_FILTER;
        } else {
            throw new IllegalArgumentException("Cannot list all directoreis but excluded Lucene 4 directories and later directories");
        }
        for (File storageDir : this.repoConfig.getStorageDirectories().values()) {
            File[] indexDirs = storageDir.listFiles(directoryFilter);
            if (indexDirs == null) {
                logger.warn("Unable to access Provenance Repository storage directory {}", (Object)storageDir);
                continue;
            }
            allDirectories.addAll(Arrays.asList(indexDirs));
        }
        return allDirectories;
    }

    public synchronized List<File> getDirectoriesBefore(long timestamp) {
        ArrayList<File> selected = new ArrayList<File>();
        Map<String, List<IndexLocation>> startTimeWithFileByStorageDirectory = this.flattenDirectoriesByTimestamp(true).stream().collect(Collectors.groupingBy(IndexLocation::getPartitionName));
        block0: for (List<IndexLocation> locationList : startTimeWithFileByStorageDirectory.values()) {
            for (int i = 0; i < locationList.size(); ++i) {
                String partition;
                IndexLocation activeLocation;
                IndexLocation indexLoc = locationList.get(i);
                if (indexLoc.equals(activeLocation = this.activeIndices.get(partition = indexLoc.getPartitionName()))) continue;
                long indexStartTime = indexLoc.getIndexStartTimestamp();
                if (indexStartTime > timestamp) continue block0;
                long indexEndTime = indexLoc.getIndexEndTimestamp();
                if (indexEndTime > timestamp) continue;
                logger.debug("Considering Index Location {} older than {} ({}) because its events have an EventTime ranging from {} ({}) to {} ({}) based on the following IndexLocations: {}", new Object[]{indexLoc, timestamp, new Date(timestamp), indexStartTime, new Date(indexStartTime), indexEndTime, new Date(indexEndTime), locationList});
                selected.add(indexLoc.getIndexDirectory());
            }
        }
        logger.debug("Returning the following list of index locations because they were finished being written to before {}: {}", (Object)timestamp, selected);
        return selected;
    }

    private List<IndexLocation> flattenDirectoriesByTimestamp(boolean includeOldIndices) {
        ArrayList<IndexLocation> startTimeWithFile = new ArrayList<IndexLocation>();
        for (Map.Entry<Long, List<IndexLocation>> entry : this.indexLocationByTimestamp.entrySet()) {
            if (includeOldIndices) {
                startTimeWithFile.addAll((Collection<IndexLocation>)entry.getValue());
                continue;
            }
            for (IndexLocation location : entry.getValue()) {
                if (!location.getIndexDirectory().getName().startsWith("lucene-")) continue;
                startTimeWithFile.add(location);
            }
        }
        return startTimeWithFile;
    }

    public synchronized List<File> getDirectories(Long startTime, Long endTime) {
        return this.getDirectories(startTime, endTime, true);
    }

    public synchronized List<File> getDirectories(Long startTime, Long endTime, boolean includeOldIndices) {
        ArrayList<File> selected = new ArrayList<File>();
        Map<String, List<IndexLocation>> startTimeWithFileByStorageDirectory = this.flattenDirectoriesByTimestamp(includeOldIndices).stream().collect(Collectors.groupingBy(IndexLocation::getPartitionName));
        for (List<IndexLocation> locationList : startTimeWithFileByStorageDirectory.values()) {
            selected.addAll(IndexDirectoryManager.getDirectories(startTime, endTime, locationList));
        }
        return selected;
    }

    public synchronized List<File> getDirectories(Long startTime, Long endTime, String partitionName) {
        Map<String, List<IndexLocation>> startTimeWithFileByStorageDirectory = this.flattenDirectoriesByTimestamp(true).stream().collect(Collectors.groupingBy(IndexLocation::getPartitionName));
        List<IndexLocation> indexLocations = startTimeWithFileByStorageDirectory.get(partitionName);
        if (indexLocations == null) {
            return Collections.emptyList();
        }
        return IndexDirectoryManager.getDirectories(startTime, endTime, indexLocations);
    }

    protected static List<File> getDirectories(Long startTime, Long endTime, List<IndexLocation> locations) {
        ArrayList<File> selected = new ArrayList<File>();
        int overlapCount = 0;
        for (int i = 0; i < locations.size(); ++i) {
            IndexLocation nextIndexLoc;
            Long indexEndTimestamp;
            IndexLocation indexLoc = locations.get(i);
            Long indexStartTimestamp = indexLoc.getIndexStartTimestamp();
            if (endTime != null && indexStartTimestamp > endTime) {
                if (overlapCount != 0) continue;
                ++overlapCount;
            }
            if (startTime != null && i < locations.size() - 1 && (indexEndTimestamp = Long.valueOf((nextIndexLoc = locations.get(i + 1)).getIndexStartTimestamp())) < startTime) continue;
            selected.add(indexLoc.getIndexDirectory());
        }
        return selected;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean onIndexCommitted(File indexDir) {
        long indexSize = this.getSize(indexDir);
        IndexDirectoryManager indexDirectoryManager = this;
        synchronized (indexDirectoryManager) {
            String partitionName = null;
            for (Map.Entry<String, IndexLocation> entry : this.activeIndices.entrySet()) {
                if (!indexDir.equals(entry.getValue().getIndexDirectory())) continue;
                partitionName = entry.getKey();
                break;
            }
            if (partitionName == null) {
                logger.debug("Size of Provenance Index at {} is now {}. However, was unable to find the appropriate Active Index to roll over.", (Object)indexDir, (Object)indexSize);
                return true;
            }
            if (indexSize >= this.repoConfig.getDesiredIndexSize()) {
                logger.info("Size of Provenance Index at {} is now {}. Will close this index and roll over to a new one.", (Object)indexDir, (Object)indexSize);
                this.activeIndices.remove(partitionName);
                return true;
            }
            return false;
        }
    }

    public synchronized Optional<File> getActiveIndexDirectory(String partitionName) {
        IndexLocation indexLocation = this.activeIndices.get(partitionName);
        if (indexLocation == null) {
            return Optional.empty();
        }
        return Optional.of(indexLocation.getIndexDirectory());
    }

    private long getSize(File indexDir) {
        if (!indexDir.exists()) {
            return 0L;
        }
        if (!indexDir.isDirectory()) {
            throw new IllegalArgumentException("Must specify a directory but specified " + indexDir);
        }
        File[] files = indexDir.listFiles();
        if (files == null) {
            return 0L;
        }
        long sum = 0L;
        for (File file : files) {
            sum += file.length();
        }
        return sum;
    }

    public synchronized File getWritableIndexingDirectory(long earliestTimestamp, String partitionName) {
        IndexLocation indexLoc = this.activeIndices.get(partitionName);
        if (indexLoc == null) {
            indexLoc = new IndexLocation(this.createIndex(earliestTimestamp, partitionName), earliestTimestamp, partitionName);
            logger.debug("Created new Index Directory {}", (Object)indexLoc);
            this.indexLocationByTimestamp.computeIfAbsent(earliestTimestamp, t -> new ArrayList()).add(indexLoc);
            this.activeIndices.put(partitionName, indexLoc);
        }
        return indexLoc.getIndexDirectory();
    }

    private File createIndex(long earliestTimestamp, String partitionName) {
        File storageDir = this.repoConfig.getStorageDirectories().entrySet().stream().filter(e -> ((String)e.getKey()).equals(partitionName)).map(Map.Entry::getValue).findFirst().orElseThrow(() -> new IllegalArgumentException("Invalid Partition: " + partitionName));
        File indexDir = new File(storageDir, "lucene-8-index-" + earliestTimestamp);
        return indexDir;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void replaceDirectory(File oldIndexDir, File newIndexDir, boolean destroyOldIndex) {
        boolean replaced = false;
        IndexDirectoryManager indexDirectoryManager = this;
        synchronized (indexDirectoryManager) {
            for (Map.Entry<Long, List<IndexLocation>> entry : this.indexLocationByTimestamp.entrySet()) {
                List<IndexLocation> locations = entry.getValue();
                ListIterator<IndexLocation> itr = locations.listIterator();
                while (itr.hasNext()) {
                    IndexLocation location = itr.next();
                    if (!location.getIndexDirectory().equals(oldIndexDir)) continue;
                    IndexLocation updatedLocation = new IndexLocation(newIndexDir, location.getIndexStartTimestamp(), location.getPartitionName());
                    itr.set(updatedLocation);
                    replaced = true;
                    logger.debug("Replaced {} with {}", (Object)location, (Object)updatedLocation);
                }
            }
        }
        if (!replaced) {
            this.insertIndexDirectory(newIndexDir);
        }
        if (destroyOldIndex) {
            try {
                FileUtils.deleteFile((File)oldIndexDir, (boolean)true);
            }
            catch (IOException e) {
                logger.warn("Failed to delete index directory {}; this directory should be cleaned up manually", (Object)oldIndexDir, (Object)e);
            }
        }
        this.removeDirectory(oldIndexDir);
        logger.info("Successfully replaced old index directory {} with new index directory {}", (Object)oldIndexDir, (Object)newIndexDir);
    }

    private void insertIndexDirectory(File indexDirectory) {
        long timestamp = DirectoryUtils.getIndexTimestamp(indexDirectory);
        if (timestamp < 0L) {
            logger.debug("Attempted to replace old index directory {} with new index directory but the old index directory did not exist and could not determine timestamp for new index directory", (Object)indexDirectory);
        } else {
            String partitionName = this.getPartitionName(indexDirectory);
            if (partitionName == null) {
                logger.debug("Attempted to replace old index directory {} with new index directory but the old index directory did not exist and could not determine partition name for new index directory", (Object)indexDirectory);
            } else {
                IndexLocation indexLocation = new IndexLocation(indexDirectory, timestamp, partitionName);
                this.indexLocationByTimestamp.computeIfAbsent(timestamp, key -> new ArrayList()).add(indexLocation);
                logger.debug("Successfully inserted new index directory {}", (Object)indexDirectory);
            }
        }
    }

    private String getPartitionName(File indexDir) {
        for (Map.Entry<String, File> entry : this.repoConfig.getStorageDirectories().entrySet()) {
            File storageDir = entry.getValue();
            if (!this.isParent(indexDir, storageDir)) continue;
            return entry.getKey();
        }
        return null;
    }

    private boolean isParent(File file, File potentialParent) {
        if (file == null) {
            return false;
        }
        File parentFile = file.getParentFile();
        if (parentFile != null && parentFile.equals(potentialParent)) {
            return true;
        }
        return this.isParent(parentFile, potentialParent);
    }
}

