/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.provenance.serialization;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.zip.GZIPInputStream;
import org.apache.nifi.provenance.ProvenanceEventRecord;
import org.apache.nifi.provenance.StandardProvenanceEventRecord;
import org.apache.nifi.provenance.serialization.RecordReader;
import org.apache.nifi.provenance.toc.TocReader;
import org.apache.nifi.stream.io.ByteCountingInputStream;
import org.apache.nifi.stream.io.LimitingInputStream;
import org.apache.nifi.stream.io.StreamUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class CompressableRecordReader
implements RecordReader {
    private static final Logger logger = LoggerFactory.getLogger(CompressableRecordReader.class);
    private final ByteCountingInputStream rawInputStream;
    private final String filename;
    private final int serializationVersion;
    private final boolean compressed;
    private final TocReader tocReader;
    private final int headerLength;
    private final int maxAttributeChars;
    private DataInputStream dis;
    private ByteCountingInputStream byteCountingIn;
    private StandardProvenanceEventRecord pushbackEvent = null;

    public CompressableRecordReader(InputStream in, String filename, int maxAttributeChars) throws IOException {
        this(in, filename, null, maxAttributeChars);
    }

    public CompressableRecordReader(InputStream in, String filename, TocReader tocReader, int maxAttributeChars) throws IOException {
        BufferedInputStream readableStream;
        long offset1;
        logger.trace("Creating RecordReader for {}", (Object)filename);
        this.rawInputStream = new ByteCountingInputStream(in);
        this.maxAttributeChars = maxAttributeChars;
        Object limitedStream = tocReader == null ? this.rawInputStream : ((offset1 = tocReader.getBlockOffset(1)) < 0L ? this.rawInputStream : new LimitingInputStream((InputStream)this.rawInputStream, offset1 - this.rawInputStream.getBytesConsumed()));
        if (filename.endsWith(".gz")) {
            readableStream = new BufferedInputStream(new GZIPInputStream((InputStream)limitedStream));
            this.compressed = true;
        } else {
            readableStream = new BufferedInputStream((InputStream)limitedStream);
            this.compressed = false;
        }
        this.byteCountingIn = new ByteCountingInputStream((InputStream)readableStream);
        this.dis = new DataInputStream((InputStream)this.byteCountingIn);
        String repoClassName = this.dis.readUTF();
        int serializationVersion = this.dis.readInt();
        this.headerLength = repoClassName.getBytes(StandardCharsets.UTF_8).length + 2 + 4;
        this.serializationVersion = serializationVersion;
        this.filename = filename;
        this.tocReader = tocReader;
        this.readHeader(this.dis, serializationVersion);
    }

    @Override
    public void skipToBlock(int blockIndex) throws IOException {
        if (this.tocReader == null) {
            throw new IllegalStateException("Cannot skip to block " + blockIndex + " for Provenance Log " + this.filename + " because no Table-of-Contents file was found for this Log");
        }
        if (blockIndex < 0) {
            throw new IllegalArgumentException("Cannot skip to block " + blockIndex + " because the value is negative");
        }
        if (blockIndex == this.getBlockIndex()) {
            return;
        }
        long offset = this.tocReader.getBlockOffset(blockIndex);
        if (offset < 0L) {
            throw new IOException("Unable to find block " + blockIndex + " in Provenance Log " + this.filename);
        }
        long curOffset = this.rawInputStream.getBytesConsumed();
        long bytesToSkip = offset - curOffset;
        if (bytesToSkip >= 0L) {
            try {
                StreamUtils.skip((InputStream)this.rawInputStream, (long)bytesToSkip);
                logger.debug("Skipped stream from offset {} to {} ({} bytes skipped)", new Object[]{curOffset, offset, bytesToSkip});
            }
            catch (EOFException eof) {
                throw new EOFException("Attempted to skip to byte offset " + offset + " for " + this.filename + " but file does not have that many bytes (TOC Reader=" + this.getTocReader() + ")");
            }
            catch (IOException e) {
                throw new IOException("Failed to skip to offset " + offset + " for block " + blockIndex + " of Provenance Log " + this.filename, e);
            }
            this.resetStreamForNextBlock();
        }
    }

    private void resetStreamForNextBlock() throws IOException {
        long offset;
        Object limitedStream = this.tocReader == null ? this.rawInputStream : ((offset = this.tocReader.getBlockOffset(1 + this.getBlockIndex())) < 0L ? this.rawInputStream : new LimitingInputStream((InputStream)this.rawInputStream, offset - this.rawInputStream.getBytesConsumed()));
        BufferedInputStream readableStream = this.compressed ? new BufferedInputStream(new GZIPInputStream((InputStream)limitedStream)) : new BufferedInputStream((InputStream)limitedStream);
        this.byteCountingIn = new ByteCountingInputStream((InputStream)readableStream, this.rawInputStream.getBytesConsumed());
        this.dis = new DataInputStream((InputStream)this.byteCountingIn);
    }

    @Override
    public TocReader getTocReader() {
        return this.tocReader;
    }

    @Override
    public boolean isBlockIndexAvailable() {
        return this.tocReader != null;
    }

    @Override
    public int getBlockIndex() {
        if (this.tocReader == null) {
            throw new IllegalStateException("Cannot determine Block Index because no Table-of-Contents could be found for Provenance Log " + this.filename);
        }
        return this.tocReader.getBlockIndex(this.rawInputStream.getBytesConsumed());
    }

    @Override
    public long getBytesConsumed() {
        return this.byteCountingIn.getBytesConsumed();
    }

    @Override
    public boolean isData() {
        try {
            this.byteCountingIn.mark(1);
            int nextByte = this.byteCountingIn.read();
            this.byteCountingIn.reset();
            if (nextByte < 0) {
                try {
                    this.resetStreamForNextBlock();
                }
                catch (EOFException eof) {
                    return false;
                }
                this.byteCountingIn.mark(1);
                nextByte = this.byteCountingIn.read();
                this.byteCountingIn.reset();
            }
            return nextByte >= 0;
        }
        catch (IOException ioe) {
            return false;
        }
    }

    @Override
    public long getMaxEventId() throws IOException {
        if (this.tocReader != null) {
            long lastBlockOffset = this.tocReader.getLastBlockOffset();
            this.skipToBlock(this.tocReader.getBlockIndex(lastBlockOffset));
        }
        StandardProvenanceEventRecord lastRecord = null;
        try {
            StandardProvenanceEventRecord record;
            while ((record = this.nextRecord()) != null) {
                lastRecord = record;
            }
        }
        catch (EOFException eOFException) {
            // empty catch block
        }
        return lastRecord == null ? -1L : lastRecord.getEventId();
    }

    @Override
    public void close() throws IOException {
        logger.trace("Closing Record Reader for {}", (Object)this.filename);
        try {
            this.dis.close();
        }
        finally {
            try {
                this.rawInputStream.close();
            }
            finally {
                if (this.tocReader != null) {
                    this.tocReader.close();
                }
            }
        }
    }

    @Override
    public void skip(long bytesToSkip) throws IOException {
        StreamUtils.skip((InputStream)this.dis, (long)bytesToSkip);
    }

    @Override
    public void skipTo(long position) throws IOException {
        long currentPosition = this.byteCountingIn.getBytesConsumed() - (long)this.headerLength;
        if (currentPosition == position) {
            return;
        }
        if (currentPosition > position) {
            throw new IOException("Cannot skip to byte offset " + position + " in stream because already at byte offset " + currentPosition);
        }
        long toSkip = position - currentPosition;
        StreamUtils.skip((InputStream)this.dis, (long)toSkip);
    }

    protected String getFilename() {
        return this.filename;
    }

    protected int getMaxAttributeLength() {
        return this.maxAttributeChars;
    }

    @Override
    public StandardProvenanceEventRecord nextRecord() throws IOException {
        if (this.pushbackEvent != null) {
            StandardProvenanceEventRecord toReturn = this.pushbackEvent;
            this.pushbackEvent = null;
            return toReturn;
        }
        if (this.isData()) {
            while (true) {
                try {
                    return this.nextRecord(this.dis, this.serializationVersion);
                }
                catch (IOException ioe) {
                    throw ioe;
                }
                catch (Exception e) {
                    logger.error("Failed to read Provenance Event from " + this.filename + "; will skip this event and continue reading subsequent events", (Throwable)e);
                    continue;
                }
                break;
            }
        }
        return null;
    }

    protected Optional<Integer> getBlockIndex(long eventId) {
        TocReader tocReader = this.getTocReader();
        if (tocReader == null) {
            return Optional.empty();
        }
        Integer blockIndex = tocReader.getBlockIndexForEventId(eventId);
        return Optional.ofNullable(blockIndex);
    }

    @Override
    public Optional<ProvenanceEventRecord> skipToEvent(long eventId) throws IOException {
        Optional<Integer> blockIndex;
        if (this.pushbackEvent != null) {
            StandardProvenanceEventRecord previousPushBack = this.pushbackEvent;
            if (previousPushBack.getEventId() >= eventId) {
                return Optional.of(previousPushBack);
            }
            this.pushbackEvent = null;
        }
        if ((blockIndex = this.getBlockIndex(eventId)).isPresent()) {
            this.skipToBlock(blockIndex.get());
        }
        try {
            boolean read = true;
            while (read) {
                Optional<StandardProvenanceEventRecord> eventOptional = this.readToEvent(eventId, this.dis, this.serializationVersion);
                if (eventOptional.isPresent()) {
                    this.pushbackEvent = eventOptional.get();
                    return Optional.of(this.pushbackEvent);
                }
                read = this.isData();
            }
            return Optional.empty();
        }
        catch (EOFException eof) {
            logger.error("Unexpectedly reached end of File when looking for Provenance Event with ID {} in {}", (Object)eventId, (Object)this.filename);
            return Optional.empty();
        }
    }

    protected Optional<StandardProvenanceEventRecord> readToEvent(long eventId, DataInputStream dis, int serializationVerison) throws IOException {
        StandardProvenanceEventRecord event;
        while ((event = this.nextRecord()) != null) {
            if (event.getEventId() < eventId) continue;
            return Optional.of(event);
        }
        return Optional.empty();
    }

    protected abstract StandardProvenanceEventRecord nextRecord(DataInputStream var1, int var2) throws IOException;

    protected void readHeader(DataInputStream in, int serializationVersion) throws IOException {
    }
}

