/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.aws.s3;

import com.amazonaws.AmazonWebServiceRequest;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazonaws.services.s3.model.AccessControlList;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.CopyPartRequest;
import com.amazonaws.services.s3.model.GetObjectMetadataRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadResult;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PartETag;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processors.aws.s3.AbstractS3Processor;
import org.apache.nifi.processors.aws.s3.DeleteS3Object;
import org.apache.nifi.processors.aws.s3.FetchS3Object;
import org.apache.nifi.processors.aws.s3.ListS3;
import org.apache.nifi.processors.aws.s3.PutS3Object;
import org.apache.nifi.processors.aws.s3.TagS3Object;
import org.apache.nifi.util.StringUtils;

@Tags(value={"Amazon", "S3", "AWS", "Archive", "Copy"})
@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@CapabilityDescription(value="Copies a file from one bucket and key to another in AWS S3")
@SeeAlso(value={PutS3Object.class, DeleteS3Object.class, ListS3.class, TagS3Object.class, DeleteS3Object.class, FetchS3Object.class})
public class CopyS3Object
extends AbstractS3Processor {
    public static final long MULTIPART_THRESHOLD = 0x140000000L;
    static final PropertyDescriptor SOURCE_BUCKET = new PropertyDescriptor.Builder().fromPropertyDescriptor(BUCKET).name("Source Bucket").displayName("Source Bucket").description("The bucket that contains the file to be copied.").build();
    static final PropertyDescriptor SOURCE_KEY = new PropertyDescriptor.Builder().fromPropertyDescriptor(KEY).name("Source Key").displayName("Source Key").description("The source key in the source bucket").build();
    static final PropertyDescriptor DESTINATION_BUCKET = new PropertyDescriptor.Builder().fromPropertyDescriptor(BUCKET).name("Destination Bucket").displayName("Destination Bucket").description("The bucket that will receive the copy.").build();
    static final PropertyDescriptor DESTINATION_KEY = new PropertyDescriptor.Builder().fromPropertyDescriptor(KEY).name("Destination Key").displayName("Destination Key").description("The target key in the target bucket").defaultValue("${filename}-1").build();
    static final List<PropertyDescriptor> properties = Arrays.asList(SOURCE_BUCKET, SOURCE_KEY, DESTINATION_BUCKET, DESTINATION_KEY, AWS_CREDENTIALS_PROVIDER_SERVICE, S3_REGION, TIMEOUT, FULL_CONTROL_USER_LIST, READ_USER_LIST, WRITE_USER_LIST, READ_ACL_LIST, WRITE_ACL_LIST, CANNED_ACL, OWNER, SSL_CONTEXT_SERVICE, ENDPOINT_OVERRIDE, SIGNER_OVERRIDE, S3_CUSTOM_SIGNER_CLASS_NAME, S3_CUSTOM_SIGNER_MODULE_LOCATION, PROXY_CONFIGURATION_SERVICE);

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return properties;
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        AmazonS3Client s3;
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        try {
            s3 = this.getS3Client(context, flowFile.getAttributes());
        }
        catch (Exception e) {
            this.getLogger().error("Failed to initialize S3 client", (Throwable)e);
            flowFile = session.penalize(flowFile);
            session.transfer(flowFile, REL_FAILURE);
            return;
        }
        String sourceBucket = context.getProperty(SOURCE_BUCKET).evaluateAttributeExpressions(flowFile).getValue();
        String sourceKey = context.getProperty(SOURCE_KEY).evaluateAttributeExpressions(flowFile).getValue();
        String destinationBucket = context.getProperty(DESTINATION_BUCKET).evaluateAttributeExpressions(flowFile).getValue();
        String destinationKey = context.getProperty(DESTINATION_KEY).evaluateAttributeExpressions(flowFile).getValue();
        AtomicReference<String> multipartIdRef = new AtomicReference<String>();
        boolean multipartUploadRequired = false;
        try {
            GetObjectMetadataRequest sourceMetadataRequest = new GetObjectMetadataRequest(sourceBucket, sourceKey);
            ObjectMetadata metadataResult = s3.getObjectMetadata(sourceMetadataRequest);
            long contentLength = metadataResult.getContentLength();
            multipartUploadRequired = contentLength > 0x140000000L;
            AccessControlList acl = this.createACL(context, flowFile);
            CannedAccessControlList cannedAccessControlList = this.createCannedACL(context, flowFile);
            if (multipartUploadRequired) {
                this.copyMultipart(s3, acl, cannedAccessControlList, sourceBucket, sourceKey, destinationBucket, destinationKey, multipartIdRef, contentLength);
            } else {
                this.copyObject(s3, acl, cannedAccessControlList, sourceBucket, sourceKey, destinationBucket, destinationKey);
            }
            session.getProvenanceReporter().send(flowFile, this.getTransitUrl(destinationBucket, destinationKey));
            session.transfer(flowFile, REL_SUCCESS);
        }
        catch (Exception e) {
            if (multipartUploadRequired && !StringUtils.isEmpty((String)((String)multipartIdRef.get()))) {
                try {
                    AbortMultipartUploadRequest abortRequest = new AbortMultipartUploadRequest(destinationBucket, destinationKey, (String)multipartIdRef.get());
                    s3.abortMultipartUpload(abortRequest);
                }
                catch (AmazonS3Exception s3e) {
                    this.getLogger().warn("Abort Multipart Upload failed for Bucket [{}] Key [{}]", new Object[]{destinationBucket, destinationKey, s3e});
                }
            }
            flowFile = this.extractExceptionDetails(e, session, flowFile);
            this.getLogger().error("Failed to copy S3 object from Bucket [{}] Key [{}]", new Object[]{sourceBucket, sourceKey, e});
            session.transfer(flowFile, REL_FAILURE);
        }
    }

    private void copyMultipart(AmazonS3Client s3, AccessControlList acl, CannedAccessControlList cannedAccessControlList, String sourceBucket, String sourceKey, String destinationBucket, String destinationKey, AtomicReference<String> multipartIdRef, long contentLength) {
        InitiateMultipartUploadRequest initRequest = new InitiateMultipartUploadRequest(destinationBucket, destinationKey);
        if (acl != null) {
            initRequest.setAccessControlList(acl);
        }
        if (cannedAccessControlList != null) {
            initRequest.setCannedACL(cannedAccessControlList);
        }
        InitiateMultipartUploadResult initResult = s3.initiateMultipartUpload(initRequest);
        multipartIdRef.set(initResult.getUploadId());
        int partNumber = 1;
        ArrayList copyPartResults = new ArrayList();
        for (long bytePosition = 0L; bytePosition < contentLength; bytePosition += 0x140000000L) {
            long lastByte = Math.min(bytePosition + 0x140000000L - 1L, contentLength - 1L);
            CopyPartRequest copyPartRequest = new CopyPartRequest().withSourceBucketName(sourceBucket).withSourceKey(sourceKey).withDestinationBucketName(destinationBucket).withDestinationKey(destinationKey).withUploadId(initResult.getUploadId()).withFirstByte(Long.valueOf(bytePosition)).withLastByte(Long.valueOf(lastByte)).withPartNumber(partNumber++);
            this.doRetryLoop(partRequest -> copyPartResults.add(s3.copyPart((CopyPartRequest)partRequest)), (AmazonWebServiceRequest)copyPartRequest);
        }
        CompleteMultipartUploadRequest completeRequest = new CompleteMultipartUploadRequest(destinationBucket, destinationKey, initResult.getUploadId(), copyPartResults.stream().map(response -> new PartETag(response.getPartNumber(), response.getETag())).collect(Collectors.toList()));
        this.doRetryLoop(complete -> s3.completeMultipartUpload(completeRequest), (AmazonWebServiceRequest)completeRequest);
    }

    private void doRetryLoop(Consumer<AmazonWebServiceRequest> consumer, AmazonWebServiceRequest request) {
        boolean requestComplete = false;
        int retryIndex = 0;
        while (!requestComplete) {
            try {
                consumer.accept(request);
                requestComplete = true;
            }
            catch (AmazonS3Exception e) {
                if (e.getStatusCode() == 503 && retryIndex < 3) {
                    ++retryIndex;
                    continue;
                }
                throw e;
            }
        }
    }

    private void copyObject(AmazonS3Client s3, AccessControlList acl, CannedAccessControlList cannedAcl, String sourceBucket, String sourceKey, String destinationBucket, String destinationKey) {
        CopyObjectRequest request = new CopyObjectRequest(sourceBucket, sourceKey, destinationBucket, destinationKey);
        if (acl != null) {
            request.setAccessControlList(acl);
        }
        if (cannedAcl != null) {
            request.setCannedAccessControlList(cannedAcl);
        }
        s3.copyObject(request);
    }

    private String getTransitUrl(String destinationBucket, String destinationKey) {
        String spacer = destinationKey.startsWith("/") ? "" : "/";
        return String.format("s3://%s%s%s", destinationBucket, spacer, destinationKey);
    }
}

