/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.amqp.processors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.rabbitmq.client.AMQP;
import com.rabbitmq.client.Connection;
import com.rabbitmq.client.Envelope;
import com.rabbitmq.client.GetResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.nifi.amqp.processors.AMQPConsumer;
import org.apache.nifi.amqp.processors.AMQPException;
import org.apache.nifi.amqp.processors.AbstractAMQPProcessor;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;

@Tags(value={"amqp", "rabbit", "get", "message", "receive", "consume"})
@InputRequirement(value=InputRequirement.Requirement.INPUT_FORBIDDEN)
@CapabilityDescription(value="Consumes AMQP Messages from an AMQP Broker using the AMQP 0.9.1 protocol. Each message that is received from the AMQP Broker will be emitted as its own FlowFile to the 'success' relationship.")
@WritesAttributes(value={@WritesAttribute(attribute="amqp$appId", description="The App ID field from the AMQP Message"), @WritesAttribute(attribute="amqp$contentEncoding", description="The Content Encoding reported by the AMQP Message"), @WritesAttribute(attribute="amqp$contentType", description="The Content Type reported by the AMQP Message"), @WritesAttribute(attribute="amqp$headers", description="The headers present on the AMQP Message. Added only if processor is configured to output this attribute."), @WritesAttribute(attribute="<Header Key Prefix>.<attribute>", description="Each message header will be inserted with this attribute name, if processor is configured to output headers as attribute"), @WritesAttribute(attribute="amqp$deliveryMode", description="The numeric indicator for the Message's Delivery Mode"), @WritesAttribute(attribute="amqp$priority", description="The Message priority"), @WritesAttribute(attribute="amqp$correlationId", description="The Message's Correlation ID"), @WritesAttribute(attribute="amqp$replyTo", description="The value of the Message's Reply-To field"), @WritesAttribute(attribute="amqp$expiration", description="The Message Expiration"), @WritesAttribute(attribute="amqp$messageId", description="The unique ID of the Message"), @WritesAttribute(attribute="amqp$timestamp", description="The timestamp of the Message, as the number of milliseconds since epoch"), @WritesAttribute(attribute="amqp$type", description="The type of message"), @WritesAttribute(attribute="amqp$userId", description="The ID of the user"), @WritesAttribute(attribute="amqp$clusterId", description="The ID of the AMQP Cluster"), @WritesAttribute(attribute="amqp$routingKey", description="The routingKey of the AMQP Message"), @WritesAttribute(attribute="amqp$exchange", description="The exchange from which AMQP Message was received")})
public class ConsumeAMQP
extends AbstractAMQPProcessor<AMQPConsumer> {
    private static final String ATTRIBUTES_PREFIX = "amqp$";
    public static final String DEFAULT_HEADERS_KEY_PREFIX = "consume.amqp";
    public static final AllowableValue HEADERS_FORMAT_COMMA_SEPARATED_STRING = new AllowableValue("Comma-Separated String", "Comma-Separated String", "Put all headers as a string with the specified separator in the attribute 'amqp$headers'.");
    public static final AllowableValue HEADERS_FORMAT_JSON_STRING = new AllowableValue("JSON String", "JSON String", "Format all headers as JSON string and output in the attribute 'amqp$headers'. It will include keys with null value as well.");
    public static final AllowableValue HEADERS_FORMAT_ATTRIBUTES = new AllowableValue("FlowFile Attributes", "FlowFile Attributes", "Put each header as attribute of the flow file with a prefix specified in the properties");
    public static final PropertyDescriptor QUEUE = new PropertyDescriptor.Builder().name("Queue").description("The name of the existing AMQP Queue from which messages will be consumed. Usually pre-defined by AMQP administrator. ").required(true).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor AUTO_ACKNOWLEDGE = new PropertyDescriptor.Builder().name("auto.acknowledge").displayName("Auto-Acknowledge Messages").description(" If false (Non-Auto-Acknowledge), the messages will be acknowledged by the processor after transferring the FlowFiles to success and committing the NiFi session. Non-Auto-Acknowledge mode provides 'at-least-once' delivery semantics. If true (Auto-Acknowledge), messages that are delivered to the AMQP Client will be auto-acknowledged by the AMQP Broker just after sending them out. This generally will provide better throughput but will also result in messages being lost upon restart/crash of the AMQP Broker, NiFi or the processor. Auto-Acknowledge mode provides 'at-most-once' delivery semantics and it is recommended only if loosing messages is acceptable.").allowableValues(new String[]{"true", "false"}).defaultValue("false").required(true).build();
    static final PropertyDescriptor BATCH_SIZE = new PropertyDescriptor.Builder().name("batch.size").displayName("Batch Size").description("The maximum number of messages that should be processed in a single session. Once this many messages have been received (or once no more messages are readily available), the messages received will be transferred to the 'success' relationship and the messages will be acknowledged to the AMQP Broker. Setting this value to a larger number could result in better performance, particularly for very small messages, but can also result in more messages being duplicated upon sudden restart of NiFi.").addValidator(StandardValidators.POSITIVE_INTEGER_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.NONE).defaultValue("10").required(true).build();
    static final PropertyDescriptor PREFETCH_COUNT = new PropertyDescriptor.Builder().name("prefetch.count").displayName("Prefetch Count").description("The maximum number of unacknowledged messages for the consumer. If consumer has this number of unacknowledged messages, AMQP broker will no longer send new messages until consumer acknowledges some of the messages already delivered to it.Allowed values: from 0 to 65535. 0 means no limit").addValidator(StandardValidators.createLongValidator((long)0L, (long)65535L, (boolean)true)).expressionLanguageSupported(ExpressionLanguageScope.NONE).defaultValue("0").required(true).build();
    public static final PropertyDescriptor HEADER_FORMAT = new PropertyDescriptor.Builder().name("header.format").displayName("Header Output Format").description("Defines how to output headers from the received message").allowableValues(new AllowableValue[]{HEADERS_FORMAT_COMMA_SEPARATED_STRING, HEADERS_FORMAT_JSON_STRING, HEADERS_FORMAT_ATTRIBUTES}).defaultValue(HEADERS_FORMAT_COMMA_SEPARATED_STRING.getValue()).required(true).build();
    public static final PropertyDescriptor HEADER_KEY_PREFIX = new PropertyDescriptor.Builder().name("header.key.prefix").displayName("Header Key Prefix").description("Text to be prefixed to header keys as the are added to the FlowFile attributes. Processor will append '.' to the value of this property").defaultValue("consume.amqp").addValidator(StandardValidators.NON_EMPTY_VALIDATOR).dependsOn(HEADER_FORMAT, new AllowableValue[]{HEADERS_FORMAT_ATTRIBUTES}).required(true).build();
    public static final PropertyDescriptor HEADER_SEPARATOR = new PropertyDescriptor.Builder().name("header.separator").displayName("Header Separator").description("The character that is used to separate key-value for header in String. The value must be only one character.").addValidator(StandardValidators.SINGLE_CHAR_VALIDATOR).defaultValue(",").dependsOn(HEADER_FORMAT, new AllowableValue[]{HEADERS_FORMAT_COMMA_SEPARATED_STRING}).required(false).build();
    static final PropertyDescriptor REMOVE_CURLY_BRACES = new PropertyDescriptor.Builder().name("remove.curly.braces").displayName("Remove Curly Braces").description("If true Remove Curly Braces, Curly Braces in the header will be automatically remove.").addValidator(StandardValidators.BOOLEAN_VALIDATOR).defaultValue("False").allowableValues(new String[]{"True", "False"}).dependsOn(HEADER_FORMAT, new AllowableValue[]{HEADERS_FORMAT_COMMA_SEPARATED_STRING}).required(false).build();
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("All FlowFiles that are received from the AMQP queue are routed to this relationship").build();
    private static final List<PropertyDescriptor> propertyDescriptors;
    private static final Set<Relationship> relationships;
    private static final ObjectMapper objectMapper;

    @Override
    protected void processResource(Connection connection, AMQPConsumer consumer, ProcessContext context, ProcessSession session) {
        GetResponse lastReceived = null;
        if (!connection.isOpen() || !consumer.getChannel().isOpen()) {
            throw new AMQPException("AMQP client has lost connection.");
        }
        for (int i = 0; i < context.getProperty(BATCH_SIZE).asInteger(); ++i) {
            GetResponse response = consumer.consume();
            if (response == null) {
                if (lastReceived != null) break;
                context.yield();
                break;
            }
            FlowFile flowFile = session.create();
            flowFile = session.write(flowFile, out -> out.write(response.getBody()));
            AMQP.BasicProperties amqpProperties = response.getProps();
            Envelope envelope = response.getEnvelope();
            String headerFormat = context.getProperty(HEADER_FORMAT).getValue();
            String headerKeyPrefix = context.getProperty(HEADER_KEY_PREFIX).getValue();
            Map<String, String> attributes = this.buildAttributes(amqpProperties, envelope, headerFormat, headerKeyPrefix, context.getProperty(REMOVE_CURLY_BRACES).asBoolean(), context.getProperty(HEADER_SEPARATOR).toString());
            flowFile = session.putAllAttributes(flowFile, attributes);
            session.getProvenanceReporter().receive(flowFile, connection.toString() + "/" + context.getProperty(QUEUE).getValue());
            session.transfer(flowFile, REL_SUCCESS);
            lastReceived = response;
        }
        if (lastReceived != null) {
            GetResponse finalGetResponse = lastReceived;
            session.commitAsync(() -> consumer.acknowledge(finalGetResponse), null);
        }
    }

    private Map<String, String> buildAttributes(AMQP.BasicProperties properties, Envelope envelope, String headersStringFormat, String headerAttributePrefix, boolean removeCurlyBraces, String valueSeparatorForHeaders) {
        AllowableValue headerFormat = new AllowableValue(headersStringFormat);
        HashMap<String, String> attributes = new HashMap<String, String>();
        this.addAttribute(attributes, "amqp$appId", properties.getAppId());
        this.addAttribute(attributes, "amqp$contentEncoding", properties.getContentEncoding());
        this.addAttribute(attributes, "amqp$contentType", properties.getContentType());
        this.addAttribute(attributes, "amqp$deliveryMode", properties.getDeliveryMode());
        this.addAttribute(attributes, "amqp$priority", properties.getPriority());
        this.addAttribute(attributes, "amqp$correlationId", properties.getCorrelationId());
        this.addAttribute(attributes, "amqp$replyTo", properties.getReplyTo());
        this.addAttribute(attributes, "amqp$expiration", properties.getExpiration());
        this.addAttribute(attributes, "amqp$messageId", properties.getMessageId());
        this.addAttribute(attributes, "amqp$timestamp", properties.getTimestamp() == null ? null : Long.valueOf(properties.getTimestamp().getTime()));
        this.addAttribute(attributes, "amqp$type", properties.getType());
        this.addAttribute(attributes, "amqp$userId", properties.getUserId());
        this.addAttribute(attributes, "amqp$clusterId", properties.getClusterId());
        this.addAttribute(attributes, "amqp$routingKey", envelope.getRoutingKey());
        this.addAttribute(attributes, "amqp$exchange", envelope.getExchange());
        Map headers = properties.getHeaders();
        if (headers != null) {
            if (HEADERS_FORMAT_ATTRIBUTES.equals((Object)headerFormat)) {
                headers.forEach((key, value) -> this.addAttribute(attributes, String.format("%s.%s", headerAttributePrefix, key), value));
            } else {
                this.addAttribute(attributes, "amqp$headers", this.buildHeaders(properties.getHeaders(), headerFormat, removeCurlyBraces, valueSeparatorForHeaders));
            }
        }
        return attributes;
    }

    private void addAttribute(Map<String, String> attributes, String attributeName, Object value) {
        if (value == null) {
            return;
        }
        attributes.put(attributeName, value.toString());
    }

    private String buildHeaders(Map<String, Object> headers, AllowableValue headerFormat, boolean removeCurlyBraces, String valueSeparatorForHeaders) {
        if (headers == null) {
            return null;
        }
        Object headerString = null;
        if (headerFormat.equals((Object)HEADERS_FORMAT_COMMA_SEPARATED_STRING)) {
            headerString = ConsumeAMQP.convertMapToString(headers, valueSeparatorForHeaders);
            if (!removeCurlyBraces) {
                headerString = "{" + (String)headerString + "}";
            }
        } else if (headerFormat.equals((Object)HEADERS_FORMAT_JSON_STRING)) {
            try {
                headerString = ConsumeAMQP.convertMapToJSONString(headers);
            }
            catch (JsonProcessingException e) {
                this.getLogger().warn("Header formatting as JSON failed", (Throwable)e);
            }
        }
        return headerString;
    }

    private static String convertMapToString(Map<String, Object> headers, String valueSeparatorForHeaders) {
        return headers.entrySet().stream().map(e -> e.getValue() != null ? (String)e.getKey() + "=" + e.getValue() : (String)e.getKey()).collect(Collectors.joining(valueSeparatorForHeaders));
    }

    private static String convertMapToJSONString(Map<String, Object> headers) throws JsonProcessingException {
        return objectMapper.writeValueAsString(headers);
    }

    @Override
    protected synchronized AMQPConsumer createAMQPWorker(ProcessContext context, Connection connection) {
        try {
            String queueName = context.getProperty(QUEUE).getValue();
            boolean autoAcknowledge = context.getProperty(AUTO_ACKNOWLEDGE).asBoolean();
            int prefetchCount = context.getProperty(PREFETCH_COUNT).asInteger();
            AMQPConsumer amqpConsumer = new AMQPConsumer(connection, queueName, autoAcknowledge, prefetchCount, this.getLogger());
            return amqpConsumer;
        }
        catch (IOException ioe) {
            throw new ProcessException("Failed to connect to AMQP Broker", (Throwable)ioe);
        }
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return propertyDescriptors;
    }

    public Set<Relationship> getRelationships() {
        return relationships;
    }

    static {
        ArrayList<PropertyDescriptor> properties = new ArrayList<PropertyDescriptor>();
        properties.add(QUEUE);
        properties.add(AUTO_ACKNOWLEDGE);
        properties.add(BATCH_SIZE);
        properties.add(PREFETCH_COUNT);
        properties.add(HEADER_FORMAT);
        properties.add(HEADER_KEY_PREFIX);
        properties.add(HEADER_SEPARATOR);
        properties.add(REMOVE_CURLY_BRACES);
        properties.addAll(ConsumeAMQP.getCommonPropertyDescriptors());
        propertyDescriptors = Collections.unmodifiableList(properties);
        relationships = Collections.singleton(REL_SUCCESS);
        objectMapper = new ObjectMapper();
    }
}

