/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hive.metastore;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.List;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.metastore.annotation.MetastoreUnitTest;
import org.apache.hadoop.hive.metastore.api.Database;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.api.NoSuchObjectException;
import org.apache.hadoop.hive.metastore.api.Partition;
import org.apache.hadoop.hive.metastore.api.PartitionSpec;
import org.apache.hadoop.hive.metastore.api.Table;
import org.apache.hadoop.hive.metastore.api.TableMeta;
import org.apache.hadoop.hive.metastore.client.builder.DatabaseBuilder;
import org.apache.hadoop.hive.metastore.client.builder.PartitionBuilder;
import org.apache.hadoop.hive.metastore.client.builder.TableBuilder;
import org.apache.hadoop.hive.metastore.conf.MetastoreConf;
import org.apache.hadoop.hive.metastore.conf.MetastoreConf.ConfVars;
import org.apache.hadoop.hive.metastore.security.HadoopThriftAuthBridge;
import org.apache.hadoop.util.StringUtils;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

/**
 * Test the filtering behavior at HMS client and HMS server. The configuration at each test
 * changes, and therefore HMS client and server are created for each test case
 */

@Category(MetastoreUnitTest.class)
public class TestFilterHooks {
  private static final Logger LOG = LoggerFactory.getLogger(TestFilterHooks.class);

  public static class DummyMetaStoreFilterHookImpl  implements MetaStoreFilterHook {
    private static boolean blockResults = false;

    public DummyMetaStoreFilterHookImpl(Configuration conf) {
    }

    @Override
    public List<String> filterDatabases(List<String> dbList) throws MetaException  {
      if (blockResults) {
        return new ArrayList<>();
      }
      return dbList;
    }

    @Override
    public Database filterDatabase(Database dataBase) throws NoSuchObjectException {
      if (blockResults) {
        throw new NoSuchObjectException("Blocked access");
      }
      return dataBase;
    }

    @Override
    public List<String> filterTableNames(String catName, String dbName, List<String> tableList)
        throws MetaException {
      if (blockResults) {
        return new ArrayList<>();
      }
      return tableList;
    }

    @Override
    public Table filterTable(Table table) throws NoSuchObjectException {
      if (blockResults) {
        throw new NoSuchObjectException("Blocked access");
      }
      return table;
    }

    @Override
    public List<Table> filterTables(List<Table> tableList) throws MetaException {
      if (blockResults) {
        return new ArrayList<>();
      }
      return tableList;
    }

    @Override
    public List<TableMeta> filterTableMetas(List<TableMeta> tableMetas) throws MetaException {
      return tableMetas;
    }

    @Override
    public List<Partition> filterPartitions(List<Partition> partitionList) throws MetaException {
      if (blockResults) {
        return new ArrayList<>();
      }
      return partitionList;
    }

    @Override
    public List<PartitionSpec> filterPartitionSpecs(
        List<PartitionSpec> partitionSpecList) throws MetaException {
      if (blockResults) {
        return new ArrayList<>();
      }
      return partitionSpecList;
    }

    @Override
    public Partition filterPartition(Partition partition) throws NoSuchObjectException {
      if (blockResults) {
        throw new NoSuchObjectException("Blocked access");
      }
      return partition;
    }

    @Override
    public List<String> filterPartitionNames(String catName, String dbName, String tblName,
        List<String> partitionNames) throws MetaException {
      if (blockResults) {
        return new ArrayList<>();
      }
      return partitionNames;
    }
  }

  protected static HiveMetaStoreClient client;
  protected static Configuration conf;
  protected static Warehouse warehouse;

  private static final int DEFAULT_LIMIT_PARTITION_REQUEST = 100;

  private static String DBNAME1 = "testdb1";
  private static String DBNAME2 = "testdb2";
  private static final String TAB1 = "tab1";
  private static final String TAB2 = "tab2";

  protected HiveMetaStoreClient createClient(Configuration metaStoreConf) throws Exception {
    try {
      return new HiveMetaStoreClient(metaStoreConf);
    } catch (Throwable e) {
      System.err.println("Unable to open the metastore");
      System.err.println(StringUtils.stringifyException(e));
      throw new Exception(e);
    }
  }

  @BeforeClass
  public static void setUp() throws Exception {
    DummyMetaStoreFilterHookImpl.blockResults = true;
  }

  @Before
  public void setUpForTest() throws Exception {

    conf = MetastoreConf.newMetastoreConf();
    conf.set("fs.defaultFS", "file:///");
    MetastoreConf.setLongVar(conf, ConfVars.THRIFT_CONNECTION_RETRIES, 3);
    MetastoreConf.setBoolVar(conf, ConfVars.HIVE_SUPPORT_CONCURRENCY, false);
    MetastoreConf.setClass(conf, ConfVars.FILTER_HOOK, DummyMetaStoreFilterHookImpl.class,
        MetaStoreFilterHook.class);
    MetastoreConf.setBoolVar(conf, ConfVars.METRICS_ENABLED, true);
    conf.set("hive.key1", "value1");
    conf.set("hive.key2", "http://www.example.com");
    conf.set("hive.key3", "");
    conf.set("hive.key4", "0");
    conf.set("datanucleus.autoCreateTables", "false");
    conf.set("hive.in.test", "true");

    MetastoreConf.setLongVar(conf, ConfVars.BATCH_RETRIEVE_MAX, 2);
    MetastoreConf.setLongVar(conf, ConfVars.LIMIT_PARTITION_REQUEST, DEFAULT_LIMIT_PARTITION_REQUEST);
    MetastoreConf.setVar(conf, ConfVars.STORAGE_SCHEMA_READER_IMPL, "no.such.class");
    MetaStoreTestUtils.setConfForStandloneMode(conf);

    warehouse = new Warehouse(conf);
  }

  /**
   * This is called in each test after the configuration is set in each test case
   * @throws Exception
   */
  protected void creatEnv(Configuration conf) throws Exception {
    client = createClient(conf);

    client.dropDatabase(DBNAME1, true, true, true);
    client.dropDatabase(DBNAME2, true, true, true);
    Database db1 = new DatabaseBuilder()
        .setName(DBNAME1)
        .setCatalogName(Warehouse.DEFAULT_CATALOG_NAME)
        .create(client, conf);
    Database db2 = new DatabaseBuilder()
        .setName(DBNAME2)
        .setCatalogName(Warehouse.DEFAULT_CATALOG_NAME)
        .create(client, conf);
    new TableBuilder()
        .setDbName(DBNAME1)
        .setTableName(TAB1)
        .addCol("id", "int")
        .addCol("name", "string")
        .create(client, conf);
    Table tab2 = new TableBuilder()
        .setDbName(DBNAME1)
        .setTableName(TAB2)
        .addCol("id", "int")
        .addPartCol("name", "string")
        .create(client, conf);
    new PartitionBuilder()
        .inTable(tab2)
        .addValue("value1")
        .addToTable(client, conf);
    new PartitionBuilder()
        .inTable(tab2)
        .addValue("value2")
        .addToTable(client, conf);
  }

  /**
   * The default configuration should be disable filtering at HMS server
   * Disable the HMS client side filtering in order to see HMS server filtering behavior
   * @throws Exception
   */
  @Test
  public void testHMSServerWithoutFilter() throws Exception {
    MetastoreConf.setBoolVar(conf, ConfVars.METASTORE_CLIENT_FILTER_ENABLED, false);
    DBNAME1 = "db_testHMSServerWithoutFilter_1";
    DBNAME2 = "db_testHMSServerWithoutFilter_2";
    creatEnv(conf);

    assertNotNull(client.getTable(DBNAME1, TAB1));
    assertEquals(2, client.getTables(DBNAME1, "*").size());
    assertEquals(2, client.getAllTables(DBNAME1).size());
    assertEquals(1, client.getTables(DBNAME1, TAB2).size());
    assertEquals(0, client.getAllTables(DBNAME2).size());

    assertNotNull(client.getDatabase(DBNAME1));
    assertEquals(2, client.getDatabases("*testHMSServerWithoutFilter*").size());
    assertEquals(1, client.getDatabases(DBNAME1).size());

    assertNotNull(client.getPartition(DBNAME1, TAB2, "name=value1"));
    assertEquals(1, client.getPartitionsByNames(DBNAME1, TAB2, Lists.newArrayList("name=value1")).size());
  }

  /**
   * Enable the HMS server side filtering
   * Disable the HMS client side filtering in order to see HMS server filtering behavior
   * @throws Exception
   */

  @Test
  public void testHMSServerWithFilter() throws Exception {
    MetastoreConf.setBoolVar(conf, ConfVars.METASTORE_CLIENT_FILTER_ENABLED, false);
    MetastoreConf.setBoolVar(conf, ConfVars.METASTORE_SERVER_FILTER_ENABLED, true);
    DBNAME1 = "db_testHMSServerWithFilter_1";
    DBNAME2 = "db_testHMSServerWithFilter_2";
    creatEnv(conf);

    testFilterForDb(true);
    testFilterForTables(true);
    testFilterForPartition(true);
  }

  /**
   * Disable filtering at HMS client
   * By default, the HMS server side filtering is diabled, so we can see HMS client filtering behavior
   * @throws Exception
   */
  @Test
  public void testHMSClientWithoutFilter() throws Exception {
    MetastoreConf.setBoolVar(conf, ConfVars.METASTORE_CLIENT_FILTER_ENABLED, false);
    DBNAME1 = "db_testHMSClientWithoutFilter_1";
    DBNAME2 = "db_testHMSClientWithoutFilter_2";
    creatEnv(conf);

    assertNotNull(client.getTable(DBNAME1, TAB1));
    assertEquals(2, client.getTables(DBNAME1, "*").size());
    assertEquals(2, client.getAllTables(DBNAME1).size());
    assertEquals(1, client.getTables(DBNAME1, TAB2).size());
    assertEquals(0, client.getAllTables(DBNAME2).size());

    assertNotNull(client.getDatabase(DBNAME1));
    assertEquals(2, client.getDatabases("*testHMSClientWithoutFilter*").size());
    assertEquals(1, client.getDatabases(DBNAME1).size());

    assertNotNull(client.getPartition(DBNAME1, TAB2, "name=value1"));
    assertEquals(1, client.getPartitionsByNames(DBNAME1, TAB2, Lists.newArrayList("name=value1")).size());
  }

  /**
   * By default, the HMS Client side filtering is enabled
   * Disable the HMS server side filtering in order to see HMS client filtering behavior
   * @throws Exception
   */
  @Test
  public void testHMSClientWithFilter() throws Exception {
    MetastoreConf.setBoolVar(conf, ConfVars.METASTORE_SERVER_FILTER_ENABLED, false);
    DBNAME1 = "db_testHMSClientWithFilter_1";
    DBNAME2 = "db_testHMSClientWithFilter_2";
    creatEnv(conf);

    testFilterForDb(false);
    testFilterForTables(false);
    testFilterForPartition(false);
  }

  protected void testFilterForDb(boolean filterAtServer) throws Exception {

    // Skip this call when testing filter hook at HMS server because HMS server calls authorization
    // API for getDatabase(), and does not call filter hook
    if (!filterAtServer) {
      try {
        assertNotNull(client.getDatabase(DBNAME1));
        fail("getDatabase() should fail with blocking mode");
      } catch (NoSuchObjectException e) {
        // Excepted
      }
    }

    assertEquals(0, client.getDatabases("*").size());
    assertEquals(0, client.getAllDatabases().size());
    assertEquals(0, client.getDatabases(DBNAME1).size());
  }

  protected void testFilterForTables(boolean filterAtServer) throws Exception {

    // Skip this call when testing filter hook at HMS server because HMS server calls authorization
    // API for getTable(), and does not call filter hook
    if (!filterAtServer) {
      try {
        client.getTable(DBNAME1, TAB1);
        fail("getTable() should fail with blocking mode");
      } catch (NoSuchObjectException e) {
        // Excepted
      }
    }

    assertEquals(0, client.getTables(DBNAME1, "*").size());
    assertEquals(0, client.getAllTables(DBNAME1).size());
    assertEquals(0, client.getTables(DBNAME1, TAB2).size());
  }

  protected void testFilterForPartition(boolean filterAtServer) throws Exception {
    try {
      assertNotNull(client.getPartition(DBNAME1, TAB2, "name=value1"));
      fail("getPartition() should fail with blocking mode");
    } catch (NoSuchObjectException e) {
      // Excepted
    }

    if (filterAtServer) {
      // at HMS server, the table of the partitions should be filtered out and result in
      // NoSuchObjectException
      try {
        client.getPartitionsByNames(DBNAME1, TAB2,
            Lists.newArrayList("name=value1")).size();
        fail("getPartitionsByNames() should fail with blocking mode at server side");
      } catch (NoSuchObjectException e) {
        // Excepted
      }
    } else {
      // at HMS client, we cannot filter the table of the partitions due to
      // HIVE-21227: HIVE-20776 causes view access regression
      assertEquals(0, client.getPartitionsByNames(DBNAME1, TAB2,
          Lists.newArrayList("name=value1")).size());
    }
  }
}
