/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hive.beeline;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.URI;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.io.output.NullOutputStream;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.HiveMetaException;
import org.apache.hadoop.hive.metastore.IMetaStoreSchemaInfo;
import org.apache.hadoop.hive.metastore.MetaStoreSchemaInfoFactory;
import org.apache.hadoop.hive.metastore.TableType;
import org.apache.hadoop.hive.metastore.conf.MetastoreConf;
import org.apache.hadoop.hive.metastore.tools.HiveSchemaHelper;
import org.apache.hadoop.hive.shims.ShimLoader;
import org.apache.hive.beeline.BeeLine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HiveSchemaTool {
    private String userName = null;
    private String passWord = null;
    private boolean dryRun = false;
    private boolean verbose = false;
    private boolean silent = false;
    private String dbOpts = null;
    private String url = null;
    private String driver = null;
    private URI[] validationServers = null;
    private final HiveConf hiveConf;
    private final String dbType;
    private final String metaDbType;
    private final IMetaStoreSchemaInfo metaStoreSchemaInfo;
    private boolean needsQuotedIdentifier;
    private String quoteCharacter;
    private static final Logger LOG = LoggerFactory.getLogger((String)HiveSchemaTool.class.getName());

    public HiveSchemaTool(String hiveHome, HiveConf hiveConf, String dbType, String metaDbType) throws HiveMetaException {
        if (hiveHome == null || hiveHome.isEmpty()) {
            throw new HiveMetaException("No Hive home directory provided");
        }
        this.hiveConf = hiveConf;
        this.dbType = dbType;
        this.metaDbType = metaDbType;
        HiveSchemaHelper.NestedScriptParser parser = this.getDbCommandParser(dbType, metaDbType);
        this.needsQuotedIdentifier = parser.needsQuotedIdentifier();
        this.quoteCharacter = parser.getQuoteCharacter();
        this.metaStoreSchemaInfo = MetaStoreSchemaInfoFactory.get((Configuration)hiveConf, (String)hiveHome, (String)dbType);
        if (dbType.equalsIgnoreCase("hive")) {
            this.url = "jdbc:hive2://?hive.conf.restricted.list=;hive.security.authorization.sqlstd.confwhitelist=.*;hive.security.authorization.sqlstd.confwhitelist.append=.*;hive.security.authorization.enabled=false;hive.metastore.uris=;hive.security.authorization.manager=org.apache.hadoop.hive.ql.security.authorization.plugin.sqlstd.SQLStdConfOnlyAuthorizerFactory;hive.support.concurrency=false;hive.txn.manager=org.apache.hadoop.hive.ql.lockmgr.DummyTxnManager;hive.metastore.rawstore.impl=org.apache.hadoop.hive.metastore.ObjectStore";
            this.driver = "org.apache.hive.jdbc.HiveDriver";
        }
    }

    public HiveConf getHiveConf() {
        return this.hiveConf;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    public void setDriver(String driver) {
        this.driver = driver;
    }

    public void setUserName(String userName) {
        this.userName = userName;
    }

    public void setPassWord(String passWord) {
        this.passWord = passWord;
    }

    public void setDryRun(boolean dryRun) {
        this.dryRun = dryRun;
    }

    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    public void setSilent(boolean silent) {
        this.silent = silent;
    }

    public void setDbOpts(String dbOpts) {
        this.dbOpts = dbOpts;
    }

    public void setValidationServers(String servers) {
        if (StringUtils.isNotEmpty((String)servers)) {
            String[] strServers = servers.split(",");
            this.validationServers = new URI[strServers.length];
            for (int i = 0; i < this.validationServers.length; ++i) {
                this.validationServers[i] = new Path(strServers[i]).toUri();
            }
        }
    }

    private static void printAndExit(Options cmdLineOptions) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp("schemaTool", cmdLineOptions);
        System.exit(1);
    }

    Connection getConnectionToMetastore(boolean printInfo) throws HiveMetaException {
        return HiveSchemaHelper.getConnectionToMetastore((String)this.userName, (String)this.passWord, (String)this.url, (String)this.driver, (boolean)printInfo, (Configuration)this.hiveConf, null);
    }

    private HiveSchemaHelper.NestedScriptParser getDbCommandParser(String dbType, String metaDbType) {
        return HiveSchemaHelper.getDbCommandParser((String)dbType, (String)this.dbOpts, (String)this.userName, (String)this.passWord, (Configuration)this.hiveConf, (String)metaDbType, (boolean)false);
    }

    public void showInfo() throws HiveMetaException {
        String hiveVersion = this.metaStoreSchemaInfo.getHiveSchemaVersion();
        String dbVersion = this.metaStoreSchemaInfo.getMetaStoreSchemaVersion(this.getConnectionInfo(true));
        this.logAndPrintToOut("Hive distribution version:\t " + hiveVersion);
        this.logAndPrintToOut("Metastore schema version:\t " + dbVersion);
        this.assertCompatibleVersion(hiveVersion, dbVersion);
    }

    boolean validateLocations(Connection conn, URI[] defaultServers) throws HiveMetaException {
        this.logAndPrintToOut("Validating DFS locations");
        boolean rtn = this.checkMetaStoreDBLocation(conn, defaultServers);
        rtn = this.checkMetaStoreTableLocation(conn, defaultServers) && rtn;
        rtn = this.checkMetaStorePartitionLocation(conn, defaultServers) && rtn;
        rtn = this.checkMetaStoreSkewedColumnsLocation(conn, defaultServers) && rtn;
        this.logAndPrintToOut((rtn ? "Succeeded" : "Failed") + " in DFS location validation.");
        return rtn;
    }

    private String getNameOrID(ResultSet res, int nameInx, int idInx) throws SQLException {
        String itemName = res.getString(nameInx);
        return itemName == null || itemName.isEmpty() ? "ID: " + res.getString(idInx) : "Name: " + itemName;
    }

    private boolean checkMetaStoreDBLocation(Connection conn, URI[] defaultServers) throws HiveMetaException {
        boolean isValid = true;
        int numOfInvalid = 0;
        String dbLoc = this.needsQuotedIdentifier ? "select dbt.\"DB_ID\", dbt.\"NAME\", dbt.\"DB_LOCATION_URI\" from \"DBS\" dbt order by dbt.\"DB_ID\" " : "select dbt.DB_ID, dbt.NAME, dbt.DB_LOCATION_URI from DBS dbt order by dbt.DB_ID";
        try (Statement stmt = conn.createStatement();
             ResultSet res = stmt.executeQuery(dbLoc);){
            while (res.next()) {
                String locValue = res.getString(3);
                String dbName = this.getNameOrID(res, 2, 1);
                if (this.checkLocation("Database " + dbName, locValue, defaultServers)) continue;
                ++numOfInvalid;
            }
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to get DB Location Info.", (Throwable)e);
        }
        if (numOfInvalid > 0) {
            isValid = false;
        }
        return isValid;
    }

    private boolean checkMetaStoreTableLocation(Connection conn, URI[] defaultServers) throws HiveMetaException {
        boolean isValid = true;
        int numOfInvalid = 0;
        String tabIDRange = this.needsQuotedIdentifier ? "select max(\"TBL_ID\"), min(\"TBL_ID\") from \"TBLS\" " : "select max(TBL_ID), min(TBL_ID) from TBLS";
        String tabLoc = this.needsQuotedIdentifier ? "select tbl.\"TBL_ID\", tbl.\"TBL_NAME\", sd.\"LOCATION\", dbt.\"DB_ID\", dbt.\"NAME\" from \"TBLS\" tbl inner join \"SDS\" sd on tbl.\"SD_ID\" = sd.\"SD_ID\" and tbl.\"TBL_TYPE\" != '" + TableType.VIRTUAL_VIEW + "' and tbl.\"TBL_ID\" >= ? and tbl.\"TBL_ID\"<= ? inner join \"DBS\" dbt on tbl.\"DB_ID\" = dbt.\"DB_ID\" order by tbl.\"TBL_ID\" " : "select tbl.TBL_ID, tbl.TBL_NAME, sd.LOCATION, dbt.DB_ID, dbt.NAME from TBLS tbl join SDS sd on tbl.SD_ID = sd.SD_ID and tbl.TBL_TYPE !='" + TableType.VIRTUAL_VIEW + "' and tbl.TBL_ID >= ? and tbl.TBL_ID <= ?  inner join DBS dbt on tbl.DB_ID = dbt.DB_ID order by tbl.TBL_ID";
        long maxID = 0L;
        long minID = 0L;
        long rtnSize = 2000L;
        try {
            Statement stmt = conn.createStatement();
            ResultSet res = stmt.executeQuery(tabIDRange);
            if (res.next()) {
                maxID = res.getLong(1);
                minID = res.getLong(2);
            }
            res.close();
            stmt.close();
            PreparedStatement pStmt = conn.prepareStatement(tabLoc);
            while (minID <= maxID) {
                pStmt.setLong(1, minID);
                pStmt.setLong(2, minID + rtnSize);
                res = pStmt.executeQuery();
                while (res.next()) {
                    String locValue = res.getString(3);
                    String entity = "Database " + this.getNameOrID(res, 5, 4) + ", Table " + this.getNameOrID(res, 2, 1);
                    if (this.checkLocation(entity, locValue, defaultServers)) continue;
                    ++numOfInvalid;
                }
                res.close();
                minID += rtnSize + 1L;
            }
            pStmt.close();
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to get Table Location Info.", (Throwable)e);
        }
        if (numOfInvalid > 0) {
            isValid = false;
        }
        return isValid;
    }

    private boolean checkMetaStorePartitionLocation(Connection conn, URI[] defaultServers) throws HiveMetaException {
        boolean isValid = true;
        int numOfInvalid = 0;
        String partIDRange = this.needsQuotedIdentifier ? "select max(\"PART_ID\"), min(\"PART_ID\") from \"PARTITIONS\" " : "select max(PART_ID), min(PART_ID) from PARTITIONS";
        String partLoc = this.needsQuotedIdentifier ? "select pt.\"PART_ID\", pt.\"PART_NAME\", sd.\"LOCATION\", tbl.\"TBL_ID\", tbl.\"TBL_NAME\",dbt.\"DB_ID\", dbt.\"NAME\" from \"PARTITIONS\" pt inner join \"SDS\" sd on pt.\"SD_ID\" = sd.\"SD_ID\" and pt.\"PART_ID\" >= ? and pt.\"PART_ID\"<= ?  inner join \"TBLS\" tbl on pt.\"TBL_ID\" = tbl.\"TBL_ID\" inner join \"DBS\" dbt on tbl.\"DB_ID\" = dbt.\"DB_ID\" order by tbl.\"TBL_ID\" " : "select pt.PART_ID, pt.PART_NAME, sd.LOCATION, tbl.TBL_ID, tbl.TBL_NAME, dbt.DB_ID, dbt.NAME from PARTITIONS pt inner join SDS sd on pt.SD_ID = sd.SD_ID and pt.PART_ID >= ? and pt.PART_ID <= ?  inner join TBLS tbl on tbl.TBL_ID = pt.TBL_ID inner join DBS dbt on tbl.DB_ID = dbt.DB_ID order by tbl.TBL_ID ";
        long maxID = 0L;
        long minID = 0L;
        long rtnSize = 2000L;
        try {
            Statement stmt = conn.createStatement();
            ResultSet res = stmt.executeQuery(partIDRange);
            if (res.next()) {
                maxID = res.getLong(1);
                minID = res.getLong(2);
            }
            res.close();
            stmt.close();
            PreparedStatement pStmt = conn.prepareStatement(partLoc);
            while (minID <= maxID) {
                pStmt.setLong(1, minID);
                pStmt.setLong(2, minID + rtnSize);
                res = pStmt.executeQuery();
                while (res.next()) {
                    String locValue = res.getString(3);
                    String entity = "Database " + this.getNameOrID(res, 7, 6) + ", Table " + this.getNameOrID(res, 5, 4) + ", Partition " + this.getNameOrID(res, 2, 1);
                    if (this.checkLocation(entity, locValue, defaultServers)) continue;
                    ++numOfInvalid;
                }
                res.close();
                minID += rtnSize + 1L;
            }
            pStmt.close();
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to get Partition Location Info.", (Throwable)e);
        }
        if (numOfInvalid > 0) {
            isValid = false;
        }
        return isValid;
    }

    private boolean checkMetaStoreSkewedColumnsLocation(Connection conn, URI[] defaultServers) throws HiveMetaException {
        boolean isValid = true;
        int numOfInvalid = 0;
        String skewedColIDRange = this.needsQuotedIdentifier ? "select max(\"STRING_LIST_ID_KID\"), min(\"STRING_LIST_ID_KID\") from \"SKEWED_COL_VALUE_LOC_MAP\" " : "select max(STRING_LIST_ID_KID), min(STRING_LIST_ID_KID) from SKEWED_COL_VALUE_LOC_MAP";
        String skewedColLoc = this.needsQuotedIdentifier ? "select t.\"TBL_NAME\", t.\"TBL_ID\", sk.\"STRING_LIST_ID_KID\", sk.\"LOCATION\", db.\"NAME\", db.\"DB_ID\"  from \"TBLS\" t, \"SDS\" s, \"DBS\" db, \"SKEWED_COL_VALUE_LOC_MAP\" sk where sk.\"SD_ID\" = s.\"SD_ID\" and s.\"SD_ID\" = t.\"SD_ID\" and t.\"DB_ID\" = db.\"DB_ID\" and sk.\"STRING_LIST_ID_KID\" >= ? and sk.\"STRING_LIST_ID_KID\" <= ? order by t.\"TBL_ID\" " : "select t.TBL_NAME, t.TBL_ID, sk.STRING_LIST_ID_KID, sk.LOCATION, db.NAME, db.DB_ID from TBLS t, SDS s, DBS db, SKEWED_COL_VALUE_LOC_MAP sk where sk.SD_ID = s.SD_ID and s.SD_ID = t.SD_ID and t.DB_ID = db.DB_ID and sk.STRING_LIST_ID_KID >= ? and sk.STRING_LIST_ID_KID <= ? order by t.TBL_ID ";
        long maxID = 0L;
        long minID = 0L;
        long rtnSize = 2000L;
        try {
            Statement stmt = conn.createStatement();
            ResultSet res = stmt.executeQuery(skewedColIDRange);
            if (res.next()) {
                maxID = res.getLong(1);
                minID = res.getLong(2);
            }
            res.close();
            stmt.close();
            PreparedStatement pStmt = conn.prepareStatement(skewedColLoc);
            while (minID <= maxID) {
                pStmt.setLong(1, minID);
                pStmt.setLong(2, minID + rtnSize);
                res = pStmt.executeQuery();
                while (res.next()) {
                    String locValue = res.getString(4);
                    String entity = "Database " + this.getNameOrID(res, 5, 6) + ", Table " + this.getNameOrID(res, 1, 2) + ", String list " + res.getString(3);
                    if (this.checkLocation(entity, locValue, defaultServers)) continue;
                    ++numOfInvalid;
                }
                res.close();
                minID += rtnSize + 1L;
            }
            pStmt.close();
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to get skewed columns location info.", (Throwable)e);
        }
        if (numOfInvalid > 0) {
            isValid = false;
        }
        return isValid;
    }

    private boolean checkLocation(String entity, String entityLocation, URI[] defaultServers) {
        boolean isValid = true;
        if (entityLocation == null) {
            System.err.println(entity + ", Error: empty location");
            isValid = false;
        } else {
            try {
                URI currentUri = new Path(entityLocation).toUri();
                String scheme = currentUri.getScheme();
                String path = currentUri.getPath();
                if (StringUtils.isEmpty((String)scheme)) {
                    System.err.println(entity + ", Location: " + entityLocation + ", Error: missing location scheme.");
                    isValid = false;
                } else if (StringUtils.isEmpty((String)path)) {
                    System.err.println(entity + ", Location: " + entityLocation + ", Error: missing location path.");
                    isValid = false;
                } else if (ArrayUtils.isNotEmpty((Object[])defaultServers) && currentUri.getAuthority() != null) {
                    String authority = currentUri.getAuthority();
                    boolean matchServer = false;
                    for (URI server : defaultServers) {
                        if (!StringUtils.equalsIgnoreCase((String)server.getScheme(), (String)scheme) || !StringUtils.equalsIgnoreCase((String)server.getAuthority(), (String)authority)) continue;
                        matchServer = true;
                        break;
                    }
                    if (!matchServer) {
                        System.err.println(entity + ", Location: " + entityLocation + ", Error: mismatched server.");
                        isValid = false;
                    }
                }
                if (isValid && StringUtils.containsOnly((String)path, (String)"/")) {
                    System.err.println(entity + ", Location: " + entityLocation + ", Warn: location set to root, not a recommended config.");
                }
            }
            catch (Exception pe) {
                System.err.println(entity + ", Error: invalid location - " + pe.getMessage());
                isValid = false;
            }
        }
        return isValid;
    }

    private void testConnectionToMetastore() throws HiveMetaException {
        Connection conn = this.getConnectionToMetastore(!this.silent);
        try {
            conn.close();
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to close metastore connection", (Throwable)e);
        }
    }

    public void verifySchemaVersion() throws HiveMetaException {
        if (this.dryRun) {
            return;
        }
        String newSchemaVersion = this.metaStoreSchemaInfo.getMetaStoreSchemaVersion(this.getConnectionInfo(false));
        this.assertCompatibleVersion(this.metaStoreSchemaInfo.getHiveSchemaVersion(), newSchemaVersion);
    }

    private void assertCompatibleVersion(String hiveSchemaVersion, String dbSchemaVersion) throws HiveMetaException {
        if (!this.metaStoreSchemaInfo.isVersionCompatible(hiveSchemaVersion, dbSchemaVersion)) {
            throw new HiveMetaException("Metastore schema version is not compatible. Hive Version: " + hiveSchemaVersion + ", Database Schema Version: " + dbSchemaVersion);
        }
    }

    public void doUpgrade() throws HiveMetaException {
        String fromVersion = this.metaStoreSchemaInfo.getMetaStoreSchemaVersion(this.getConnectionInfo(false));
        if (fromVersion == null || fromVersion.isEmpty()) {
            throw new HiveMetaException("Schema version not stored in the metastore. Metastore schema is too old or corrupt. Try specifying the version manually");
        }
        this.doUpgrade(fromVersion);
    }

    private HiveSchemaHelper.MetaStoreConnectionInfo getConnectionInfo(boolean printInfo) {
        return new HiveSchemaHelper.MetaStoreConnectionInfo(this.userName, this.passWord, this.url, this.driver, printInfo, (Configuration)this.hiveConf, this.dbType, this.metaDbType);
    }

    public void doUpgrade(String fromSchemaVer) throws HiveMetaException {
        if (this.metaStoreSchemaInfo.getHiveSchemaVersion().equals(fromSchemaVer)) {
            this.logAndPrintToOut("No schema upgrade required from version " + fromSchemaVer);
            return;
        }
        List upgradeScripts = this.metaStoreSchemaInfo.getUpgradeScripts(fromSchemaVer);
        this.testConnectionToMetastore();
        this.logAndPrintToOut("Starting upgrade metastore schema from version " + fromSchemaVer + " to " + this.metaStoreSchemaInfo.getHiveSchemaVersion());
        String scriptDir = this.metaStoreSchemaInfo.getMetaStoreScriptDir();
        try {
            for (String scriptFile : upgradeScripts) {
                this.logAndPrintToOut("Upgrade script " + scriptFile);
                if (this.dryRun) continue;
                this.runPreUpgrade(scriptDir, scriptFile);
                this.runBeeLine(scriptDir, scriptFile);
                this.logAndPrintToOut("Completed " + scriptFile);
            }
        }
        catch (IOException eIO) {
            throw new HiveMetaException("Upgrade FAILED! Metastore state would be inconsistent !!", (Throwable)eIO);
        }
        this.verifySchemaVersion();
    }

    public void doInit() throws HiveMetaException {
        this.doInit(this.metaStoreSchemaInfo.getHiveSchemaVersion());
        this.verifySchemaVersion();
    }

    public void doInit(String toVersion) throws HiveMetaException {
        this.testConnectionToMetastore();
        this.logAndPrintToOut("Starting metastore schema initialization to " + toVersion);
        String initScriptDir = this.metaStoreSchemaInfo.getMetaStoreScriptDir();
        String initScriptFile = this.metaStoreSchemaInfo.generateInitFileName(toVersion);
        try {
            this.logAndPrintToOut("Initialization script " + initScriptFile);
            if (!this.dryRun) {
                this.runBeeLine(initScriptDir, initScriptFile);
                this.logAndPrintToOut("Initialization script completed");
            }
        }
        catch (IOException e) {
            throw new HiveMetaException("Schema initialization FAILED! Metastore state would be inconsistent !!", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void doValidate() throws HiveMetaException {
        this.logAndPrintToOut("Starting metastore validation\n");
        Connection conn = this.getConnectionToMetastore(false);
        boolean success = true;
        try {
            if (this.validateSchemaVersions()) {
                this.logAndPrintToOut("[SUCCESS]\n");
            } else {
                success = false;
                this.logAndPrintToOut("[FAIL]\n");
            }
            if (this.validateSequences(conn)) {
                this.logAndPrintToOut("[SUCCESS]\n");
            } else {
                success = false;
                this.logAndPrintToOut("[FAIL]\n");
            }
            if (this.validateSchemaTables(conn)) {
                this.logAndPrintToOut("[SUCCESS]\n");
            } else {
                success = false;
                this.logAndPrintToOut("[FAIL]\n");
            }
            if (this.validateLocations(conn, this.validationServers)) {
                this.logAndPrintToOut("[SUCCESS]\n");
            } else {
                this.logAndPrintToOut("[WARN]\n");
            }
            if (this.validateColumnNullValues(conn)) {
                this.logAndPrintToOut("[SUCCESS]\n");
            } else {
                this.logAndPrintToOut("[WARN]\n");
            }
        }
        finally {
            if (conn != null) {
                try {
                    conn.close();
                }
                catch (SQLException e) {
                    throw new HiveMetaException("Failed to close metastore connection", (Throwable)e);
                }
            }
        }
        this.logAndPrintToOut("Done with metastore validation: ");
        if (!success) {
            this.logAndPrintToOut("[FAIL]");
            System.exit(1);
        } else {
            this.logAndPrintToOut("[SUCCESS]");
        }
    }

    boolean validateSequences(Connection conn) throws HiveMetaException {
        ImmutableMap seqNameToTable = new ImmutableMap.Builder().put((Object)"MDatabase", (Object)Pair.of((Object)"DBS", (Object)"DB_ID")).put((Object)"MRole", (Object)Pair.of((Object)"ROLES", (Object)"ROLE_ID")).put((Object)"MGlobalPrivilege", (Object)Pair.of((Object)"GLOBAL_PRIVS", (Object)"USER_GRANT_ID")).put((Object)"MTable", (Object)Pair.of((Object)"TBLS", (Object)"TBL_ID")).put((Object)"MStorageDescriptor", (Object)Pair.of((Object)"SDS", (Object)"SD_ID")).put((Object)"MSerDeInfo", (Object)Pair.of((Object)"SERDES", (Object)"SERDE_ID")).put((Object)"MColumnDescriptor", (Object)Pair.of((Object)"CDS", (Object)"CD_ID")).put((Object)"MTablePrivilege", (Object)Pair.of((Object)"TBL_PRIVS", (Object)"TBL_GRANT_ID")).put((Object)"MTableColumnStatistics", (Object)Pair.of((Object)"TAB_COL_STATS", (Object)"CS_ID")).put((Object)"MPartition", (Object)Pair.of((Object)"PARTITIONS", (Object)"PART_ID")).put((Object)"MPartitionColumnStatistics", (Object)Pair.of((Object)"PART_COL_STATS", (Object)"CS_ID")).put((Object)"MFunction", (Object)Pair.of((Object)"FUNCS", (Object)"FUNC_ID")).put((Object)"MIndex", (Object)Pair.of((Object)"IDXS", (Object)"INDEX_ID")).put((Object)"MStringList", (Object)Pair.of((Object)"SKEWED_STRING_LIST", (Object)"STRING_LIST_ID")).build();
        this.logAndPrintToOut("Validating sequence number for SEQUENCE_TABLE");
        boolean isValid = true;
        try {
            Statement stmt = conn.createStatement();
            for (String seqName : seqNameToTable.keySet()) {
                long maxId;
                String tableName = (String)((Pair)seqNameToTable.get(seqName)).getLeft();
                String tableKey = (String)((Pair)seqNameToTable.get(seqName)).getRight();
                String fullSequenceName = "org.apache.hadoop.hive.metastore.model." + seqName;
                String seqQuery = this.needsQuotedIdentifier ? "select t.\"NEXT_VAL\" from \"SEQUENCE_TABLE\" t WHERE t.\"SEQUENCE_NAME\"=? order by t.\"SEQUENCE_NAME\" " : "select t.NEXT_VAL from SEQUENCE_TABLE t WHERE t.SEQUENCE_NAME=? order by t.SEQUENCE_NAME ";
                String maxIdQuery = this.needsQuotedIdentifier ? "select max(\"" + tableKey + "\") from \"" + tableName + "\"" : "select max(" + tableKey + ") from " + tableName;
                ResultSet res = stmt.executeQuery(maxIdQuery);
                if (!res.next() || (maxId = res.getLong(1)) <= 0L) continue;
                PreparedStatement pStmt = conn.prepareStatement(seqQuery);
                pStmt.setString(1, fullSequenceName);
                ResultSet resSeq = pStmt.executeQuery();
                if (!resSeq.next()) {
                    isValid = false;
                    System.err.println("Missing SEQUENCE_NAME " + seqName + " from SEQUENCE_TABLE");
                    continue;
                }
                if (resSeq.getLong(1) >= maxId) continue;
                isValid = false;
                System.err.println("NEXT_VAL for " + seqName + " in SEQUENCE_TABLE < max(" + tableKey + ") in " + tableName);
            }
            this.logAndPrintToOut((isValid ? "Succeeded" : "Failed") + " in sequence number validation for SEQUENCE_TABLE.");
            return isValid;
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to validate sequence number for SEQUENCE_TABLE", (Throwable)e);
        }
    }

    boolean validateSchemaVersions() throws HiveMetaException {
        this.logAndPrintToOut("Validating schema version");
        try {
            String newSchemaVersion = this.metaStoreSchemaInfo.getMetaStoreSchemaVersion(this.getConnectionInfo(false));
            this.assertCompatibleVersion(this.metaStoreSchemaInfo.getHiveSchemaVersion(), newSchemaVersion);
        }
        catch (HiveMetaException hme) {
            if (hme.getMessage().contains("Metastore schema version is not compatible") || hme.getMessage().contains("Multiple versions were found in metastore") || hme.getMessage().contains("Could not find version info in metastore VERSION table")) {
                System.err.println(hme.getMessage());
                this.logAndPrintToOut("Failed in schema version validation.");
                return false;
            }
            throw hme;
        }
        this.logAndPrintToOut("Succeeded in schema version validation.");
        return true;
    }

    boolean validateSchemaTables(Connection conn) throws HiveMetaException {
        String version = null;
        ResultSet rs = null;
        DatabaseMetaData metadata = null;
        ArrayList<String> dbTables = new ArrayList<String>();
        ArrayList<String> schemaTables = new ArrayList<String>();
        ArrayList<String> subScripts = new ArrayList<String>();
        Connection hmsConn = this.getConnectionToMetastore(false);
        this.logAndPrintToOut("Validating metastore schema tables");
        try {
            version = this.metaStoreSchemaInfo.getMetaStoreSchemaVersion(this.getConnectionInfo(false));
        }
        catch (HiveMetaException he) {
            System.err.println("Failed to determine schema version from Hive Metastore DB. " + he.getMessage());
            this.logAndPrintToOut("Failed in schema table validation.");
            LOG.debug("Failed to determine schema version from Hive Metastore DB," + he.getMessage());
            return false;
        }
        hmsConn = this.getConnectionToMetastore(false);
        LOG.debug("Validating tables in the schema for version " + version);
        try {
            String schema = null;
            try {
                schema = hmsConn.getSchema();
            }
            catch (SQLFeatureNotSupportedException e) {
                LOG.debug("schema is not supported");
            }
            metadata = conn.getMetaData();
            String[] types = new String[]{"TABLE"};
            rs = metadata.getTables(null, schema, "%", types);
            String table = null;
            while (rs.next()) {
                table = rs.getString("TABLE_NAME");
                dbTables.add(table.toLowerCase());
                LOG.debug("Found table " + table + " in HMS dbstore");
            }
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to retrieve schema tables from Hive Metastore DB", (Throwable)e);
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                }
                catch (SQLException e) {
                    throw new HiveMetaException("Failed to close resultset", (Throwable)e);
                }
            }
        }
        String baseDir = new File(this.metaStoreSchemaInfo.getMetaStoreScriptDir()).getParent();
        Object schemaFile = new File(this.metaStoreSchemaInfo.getMetaStoreScriptDir(), this.metaStoreSchemaInfo.generateInitFileName(version)).getPath();
        try {
            LOG.debug("Parsing schema script " + (String)schemaFile);
            subScripts.addAll(this.findCreateTable((String)schemaFile, schemaTables));
            while (subScripts.size() > 0) {
                schemaFile = baseDir + "/" + this.dbType + "/" + (String)subScripts.remove(0);
                LOG.debug("Parsing subscript " + (String)schemaFile);
                subScripts.addAll(this.findCreateTable((String)schemaFile, schemaTables));
            }
        }
        catch (Exception e) {
            System.err.println("Exception in parsing schema file. Cause:" + e.getMessage());
            this.logAndPrintToOut("Failed in schema table validation.");
            return false;
        }
        LOG.debug("Schema tables:[ " + Arrays.toString(schemaTables.toArray()) + " ]");
        LOG.debug("DB tables:[ " + Arrays.toString(dbTables.toArray()) + " ]");
        schemaTables.removeAll(dbTables);
        if (schemaTables.size() > 0) {
            Collections.sort(schemaTables);
            System.err.println("Table(s) [ " + Arrays.toString(schemaTables.toArray()) + " ] are missing from the metastore database schema.");
            this.logAndPrintToOut("Failed in schema table validation.");
            return false;
        }
        this.logAndPrintToOut("Succeeded in schema table validation.");
        return true;
    }

    private List<String> findCreateTable(String path, List<String> tableList) throws Exception {
        HiveSchemaHelper.NestedScriptParser sp = HiveSchemaHelper.getDbCommandParser((String)this.dbType, (boolean)false);
        Matcher matcher = null;
        Pattern regexp = null;
        ArrayList<String> subs = new ArrayList<String>();
        int groupNo = 2;
        regexp = Pattern.compile("CREATE TABLE(\\s+IF NOT EXISTS)?\\s+(\\S+).*");
        if (!new File(path).exists()) {
            throw new Exception(path + " does not exist. Potentially incorrect version in the metastore VERSION table");
        }
        try (BufferedReader reader = new BufferedReader(new FileReader(path));){
            String line = null;
            while ((line = reader.readLine()) != null) {
                if (sp.isNestedScript(line)) {
                    String subScript = null;
                    subScript = sp.getScriptName(line);
                    LOG.debug("Schema subscript " + subScript + " found");
                    subs.add(subScript);
                    continue;
                }
                line = line.replaceAll("( )+", " ");
                line = line.replaceAll("\\(", " ");
                line = line.replaceAll("IF NOT EXISTS ", "");
                line = line.replaceAll("`", "");
                line = line.replaceAll("'", "");
                matcher = regexp.matcher(line = line.replaceAll("\"", ""));
                if (!matcher.find()) continue;
                String table = matcher.group(groupNo);
                if (this.dbType.equals("derby")) {
                    table = table.replaceAll("APP\\.", "");
                }
                tableList.add(table.toLowerCase());
                LOG.debug("Found table " + table + " in the schema");
            }
        }
        catch (IOException ex) {
            throw new Exception(ex.getMessage());
        }
        return subs;
    }

    boolean validateColumnNullValues(Connection conn) throws HiveMetaException {
        this.logAndPrintToOut("Validating columns for incorrect NULL values.");
        boolean isValid = true;
        try {
            Statement stmt = conn.createStatement();
            String tblQuery = this.needsQuotedIdentifier ? "select t.* from \"TBLS\" t WHERE t.\"SD_ID\" IS NULL and (t.\"TBL_TYPE\"='" + TableType.EXTERNAL_TABLE + "' or t.\"TBL_TYPE\"='" + TableType.MANAGED_TABLE + "') order by t.\"TBL_ID\" " : "select t.* from TBLS t WHERE t.SD_ID IS NULL and (t.TBL_TYPE='" + TableType.EXTERNAL_TABLE + "' or t.TBL_TYPE='" + TableType.MANAGED_TABLE + "') order by t.TBL_ID ";
            ResultSet res = stmt.executeQuery(tblQuery);
            while (res.next()) {
                long tableId = res.getLong("TBL_ID");
                String tableName = res.getString("TBL_NAME");
                String tableType = res.getString("TBL_TYPE");
                isValid = false;
                System.err.println("SD_ID in TBLS should not be NULL for Table Name=" + tableName + ", Table ID=" + tableId + ", Table Type=" + tableType);
            }
            this.logAndPrintToOut((isValid ? "Succeeded" : "Failed") + " in column validation for incorrect NULL values.");
            return isValid;
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to validate columns for incorrect NULL values", (Throwable)e);
        }
    }

    @VisibleForTesting
    void createCatalog(String catName, String location, String description, boolean ifNotExists) throws HiveMetaException {
        catName = org.apache.hadoop.hive.metastore.utils.StringUtils.normalizeIdentifier((String)catName);
        this.logAndPrintToOut("Create catalog " + catName + " at location " + location);
        Connection conn = this.getConnectionToMetastore(!this.silent);
        boolean success = false;
        try {
            conn.setAutoCommit(false);
            try (Statement stmt = conn.createStatement();){
                int floor;
                ResultSet rs;
                String query;
                if (ifNotExists) {
                    query = "select " + this.quoteIf("NAME") + " from " + this.quoteIf("CTLGS") + " where " + this.quoteIf("NAME") + " = '" + catName + "'";
                    LOG.debug("Going to run " + (String)query);
                    rs = stmt.executeQuery(query);
                    if (rs.next()) {
                        this.logAndPrintToOut("Catalog " + catName + " already exists");
                        return;
                    }
                }
                query = "select max(" + this.quoteIf("CTLG_ID") + ") from " + this.quoteIf("CTLGS");
                LOG.debug("Going to run " + query);
                rs = stmt.executeQuery(query);
                if (!rs.next()) {
                    throw new HiveMetaException("No catalogs found, have you upgraded the database?");
                }
                int catNum = rs.getInt(1) + 1;
                if (catNum < (floor = 0x40000000)) {
                    catNum = floor;
                }
                String update = "insert into " + this.quoteIf("CTLGS") + "(" + this.quoteIf("CTLG_ID") + ", " + this.quoteIf("NAME") + ", " + this.quoteAlways("DESC") + ", " + this.quoteIf("LOCATION_URI") + ")  values (" + catNum + ", '" + catName + "', '" + description + "', '" + location + "')";
                LOG.debug("Going to run " + update);
                stmt.execute(update);
                conn.commit();
                success = true;
            }
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to add catalog", (Throwable)e);
        }
        finally {
            try {
                if (!success) {
                    conn.rollback();
                }
            }
            catch (SQLException e) {
                LOG.error("Failed to rollback, everything will probably go bad from here.", (Throwable)e);
            }
        }
    }

    @VisibleForTesting
    void alterCatalog(String catName, String location, String description) throws HiveMetaException {
        if (location == null && description == null) {
            throw new HiveMetaException("Asked to update catalog " + catName + " but not given any changes to update");
        }
        catName = org.apache.hadoop.hive.metastore.utils.StringUtils.normalizeIdentifier((String)catName);
        this.logAndPrintToOut("Updating catalog " + catName);
        Connection conn = this.getConnectionToMetastore(!this.silent);
        boolean success = false;
        try {
            conn.setAutoCommit(false);
            try (Statement stmt = conn.createStatement();){
                StringBuilder update = new StringBuilder("update ").append(this.quoteIf("CTLGS")).append(" set ");
                if (location != null) {
                    update.append(this.quoteIf("LOCATION_URI")).append(" = '").append(location).append("' ");
                }
                if (description != null) {
                    if (location != null) {
                        update.append(", ");
                    }
                    update.append(this.quoteAlways("DESC")).append(" = '").append(description).append("'");
                }
                update.append(" where ").append(this.quoteIf("NAME")).append(" = '").append(catName).append("'");
                LOG.debug("Going to run " + update.toString());
                int count = stmt.executeUpdate(update.toString());
                if (count != 1) {
                    throw new HiveMetaException("Failed to find catalog " + catName + " to update");
                }
                conn.commit();
                success = true;
            }
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to update catalog", (Throwable)e);
        }
        finally {
            try {
                if (!success) {
                    conn.rollback();
                }
            }
            catch (SQLException e) {
                LOG.error("Failed to rollback, everything will probably go bad from here.", (Throwable)e);
            }
        }
    }

    @VisibleForTesting
    void moveDatabase(String fromCatName, String toCatName, String dbName) throws HiveMetaException {
        fromCatName = org.apache.hadoop.hive.metastore.utils.StringUtils.normalizeIdentifier((String)fromCatName);
        toCatName = org.apache.hadoop.hive.metastore.utils.StringUtils.normalizeIdentifier((String)toCatName);
        dbName = org.apache.hadoop.hive.metastore.utils.StringUtils.normalizeIdentifier((String)dbName);
        this.logAndPrintToOut("Moving database " + dbName + " from catalog " + fromCatName + " to catalog " + toCatName);
        Connection conn = this.getConnectionToMetastore(!this.silent);
        boolean success = false;
        try {
            conn.setAutoCommit(false);
            try (Statement stmt = conn.createStatement();){
                this.updateCatalogNameInTable(stmt, "DBS", "CTLG_NAME", "NAME", fromCatName, toCatName, dbName, false);
                this.updateCatalogNameInTable(stmt, "TAB_COL_STATS", "CAT_NAME", "DB_NAME", fromCatName, toCatName, dbName, true);
                this.updateCatalogNameInTable(stmt, "PART_COL_STATS", "CAT_NAME", "DB_NAME", fromCatName, toCatName, dbName, true);
                this.updateCatalogNameInTable(stmt, "PARTITION_EVENTS", "CAT_NAME", "DB_NAME", fromCatName, toCatName, dbName, true);
                this.updateCatalogNameInTable(stmt, "NOTIFICATION_LOG", "CAT_NAME", "DB_NAME", fromCatName, toCatName, dbName, true);
                conn.commit();
                success = true;
            }
        }
        catch (SQLException e) {
            throw new HiveMetaException("Failed to move database", (Throwable)e);
        }
        finally {
            try {
                if (!success) {
                    conn.rollback();
                }
            }
            catch (SQLException e) {
                LOG.error("Failed to rollback, everything will probably go bad from here.");
            }
        }
    }

    private void updateCatalogNameInTable(Statement stmt, String tableName, String catColName, String dbColName, String fromCatName, String toCatName, String dbName, boolean zeroUpdatesOk) throws HiveMetaException, SQLException {
        String update = "update " + this.quoteIf(tableName) + " set " + this.quoteIf(catColName) + " = '" + toCatName + "' where " + this.quoteIf(catColName) + " = '" + fromCatName + "' and " + this.quoteIf(dbColName) + " = '" + dbName + "'";
        LOG.debug("Going to run " + update);
        int numUpdated = stmt.executeUpdate(update);
        if (!(numUpdated == 1 || zeroUpdatesOk && numUpdated == 0)) {
            throw new HiveMetaException("Failed to properly update the " + tableName + " table.  Expected to update 1 row but instead updated " + numUpdated);
        }
    }

    @VisibleForTesting
    void moveTable(String fromCat, String toCat, String fromDb, String toDb, String tableName) throws HiveMetaException {
        fromCat = org.apache.hadoop.hive.metastore.utils.StringUtils.normalizeIdentifier((String)fromCat);
        toCat = org.apache.hadoop.hive.metastore.utils.StringUtils.normalizeIdentifier((String)toCat);
        fromDb = org.apache.hadoop.hive.metastore.utils.StringUtils.normalizeIdentifier((String)fromDb);
        toDb = org.apache.hadoop.hive.metastore.utils.StringUtils.normalizeIdentifier((String)toDb);
        tableName = org.apache.hadoop.hive.metastore.utils.StringUtils.normalizeIdentifier((String)tableName);
        Connection conn = this.getConnectionToMetastore(!this.silent);
        boolean success = false;
        try {
            conn.setAutoCommit(false);
            try (Statement stmt = conn.createStatement();){
                String query = "select " + this.quoteIf("DB_ID") + " from " + this.quoteIf("DBS") + " where " + this.quoteIf("NAME") + " = '" + fromDb + "' and " + this.quoteIf("CTLG_NAME") + " = '" + fromCat + "'";
                LOG.debug("Going to run " + query);
                ResultSet rs = stmt.executeQuery(query);
                if (!rs.next()) {
                    throw new HiveMetaException("Unable to find database " + fromDb);
                }
                long oldDbId = rs.getLong(1);
                query = "select " + this.quoteIf("DB_ID") + " from " + this.quoteIf("DBS") + " where " + this.quoteIf("NAME") + " = '" + toDb + "' and " + this.quoteIf("CTLG_NAME") + " = '" + toCat + "'";
                LOG.debug("Going to run " + query);
                rs = stmt.executeQuery(query);
                if (!rs.next()) {
                    throw new HiveMetaException("Unable to find database " + toDb);
                }
                long newDbId = rs.getLong(1);
                String update = "update " + this.quoteIf("TBLS") + " set " + this.quoteIf("DB_ID") + " = " + newDbId + " where " + this.quoteIf("DB_ID") + " = " + oldDbId + " and " + this.quoteIf("TBL_NAME") + " = '" + tableName + "'";
                LOG.debug("Going to run " + update);
                int numUpdated = stmt.executeUpdate(update);
                if (numUpdated != 1) {
                    throw new HiveMetaException("Failed to properly update TBLS table.  Expected to update 1 row but instead updated " + numUpdated);
                }
                this.updateDbNameForTable(stmt, "TAB_COL_STATS", "TABLE_NAME", fromCat, toCat, fromDb, toDb, tableName);
                this.updateDbNameForTable(stmt, "PART_COL_STATS", "TABLE_NAME", fromCat, toCat, fromDb, toDb, tableName);
                this.updateDbNameForTable(stmt, "PARTITION_EVENTS", "TBL_NAME", fromCat, toCat, fromDb, toDb, tableName);
                this.updateDbNameForTable(stmt, "NOTIFICATION_LOG", "TBL_NAME", fromCat, toCat, fromDb, toDb, tableName);
                conn.commit();
                success = true;
            }
        }
        catch (SQLException se) {
            throw new HiveMetaException("Failed to move table", (Throwable)se);
        }
        finally {
            try {
                if (!success) {
                    conn.rollback();
                }
            }
            catch (SQLException e) {
                LOG.error("Failed to rollback, everything will probably go bad from here.");
            }
        }
    }

    private void updateDbNameForTable(Statement stmt, String tableName, String tableColumnName, String fromCat, String toCat, String fromDb, String toDb, String hiveTblName) throws HiveMetaException, SQLException {
        String update = "update " + this.quoteIf(tableName) + " set " + this.quoteIf("CAT_NAME") + " = '" + toCat + "', " + this.quoteIf("DB_NAME") + " = '" + toDb + "' where " + this.quoteIf("CAT_NAME") + " = '" + fromCat + "' and " + this.quoteIf("DB_NAME") + " = '" + fromDb + "' and " + this.quoteIf(tableColumnName) + " = '" + hiveTblName + "'";
        LOG.debug("Going to run " + update);
        int numUpdated = stmt.executeUpdate(update);
        if (numUpdated > 1 || numUpdated < 0) {
            throw new HiveMetaException("Failed to properly update the " + tableName + " table.  Expected to update 1 row but instead updated " + numUpdated);
        }
    }

    private String quoteIf(String identifier) {
        return this.needsQuotedIdentifier ? this.quoteCharacter + identifier + this.quoteCharacter : identifier;
    }

    private String quoteAlways(String identifier) {
        return this.quoteCharacter + identifier + this.quoteCharacter;
    }

    private void runPreUpgrade(String scriptDir, String scriptFile) {
        String preUpgradeScript;
        File preUpgradeScriptFile;
        int i = 0;
        while ((preUpgradeScriptFile = new File(scriptDir, preUpgradeScript = this.metaStoreSchemaInfo.getPreUpgradeScriptName(i, scriptFile))).isFile()) {
            block3: {
                try {
                    this.runBeeLine(scriptDir, preUpgradeScript);
                    this.logAndPrintToOut("Completed " + preUpgradeScript);
                }
                catch (Exception e) {
                    System.err.println("Warning in pre-upgrade script " + preUpgradeScript + ": " + e.getMessage());
                    if (!this.verbose) break block3;
                    e.printStackTrace();
                }
            }
            ++i;
        }
    }

    private void runBeeLine(String scriptDir, String scriptFile) throws IOException, HiveMetaException {
        HiveSchemaHelper.NestedScriptParser dbCommandParser = this.getDbCommandParser(this.dbType, this.metaDbType);
        String sqlCommands = dbCommandParser.buildCommand(scriptDir, scriptFile, this.metaDbType != null);
        File tmpFile = File.createTempFile("schematool", ".sql");
        tmpFile.deleteOnExit();
        FileWriter fstream = new FileWriter(tmpFile.getPath());
        BufferedWriter out = new BufferedWriter(fstream);
        out.write("!autocommit on" + System.getProperty("line.separator"));
        out.write(sqlCommands);
        out.write("!closeall" + System.getProperty("line.separator"));
        out.close();
        this.runBeeLine(tmpFile.getPath());
    }

    public void runBeeLine(String sqlScriptFile) throws IOException {
        CommandBuilder builder = new CommandBuilder(this.hiveConf, this.url, this.driver, this.userName, this.passWord, sqlScriptFile);
        try (BeeLine beeLine = new BeeLine();){
            if (!this.verbose) {
                beeLine.setOutputStream(new PrintStream((OutputStream)new NullOutputStream()));
                beeLine.getOpts().setSilent(true);
            }
            beeLine.getOpts().setAllowMultiLineCommand(false);
            beeLine.getOpts().setIsolation("TRANSACTION_READ_COMMITTED");
            beeLine.getOpts().setEntireLineAsCommand(true);
            LOG.debug("Going to run command <" + builder.buildToLog() + ">");
            int status = beeLine.begin(builder.buildToRun(), null, false);
            if (status != 0) {
                throw new IOException("Schema script failed, errorcode " + status);
            }
        }
    }

    private static void initOptions(Options cmdLineOptions) {
        Option help = new Option("help", "print this message");
        Option upgradeOpt = new Option("upgradeSchema", "Schema upgrade");
        OptionBuilder.withArgName((String)"upgradeFrom");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Schema upgrade from a version");
        Option upgradeFromOpt = OptionBuilder.create((String)"upgradeSchemaFrom");
        Option initOpt = new Option("initSchema", "Schema initialization");
        OptionBuilder.withArgName((String)"initTo");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Schema initialization to a version");
        Option initToOpt = OptionBuilder.create((String)"initSchemaTo");
        Option infoOpt = new Option("info", "Show config and schema details");
        Option validateOpt = new Option("validate", "Validate the database");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Create a catalog, requires --catalogLocation parameter as well");
        Option createCatalog = OptionBuilder.create((String)"createCatalog");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Alter a catalog, requires --catalogLocation and/or --catalogDescription parameter as well");
        Option alterCatalog = OptionBuilder.create((String)"alterCatalog");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Move a database between catalogs.  Argument is the database name. Requires --fromCatalog and --toCatalog parameters as well");
        Option moveDatabase = OptionBuilder.create((String)"moveDatabase");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Move a table to a different database.  Argument is the table name. Requires --fromCatalog, --toCatalog, --fromDatabase, and --toDatabase  parameters as well.");
        Option moveTable = OptionBuilder.create((String)"moveTable");
        OptionGroup optGroup = new OptionGroup();
        optGroup.addOption(upgradeOpt).addOption(initOpt).addOption(help).addOption(upgradeFromOpt).addOption(initToOpt).addOption(infoOpt).addOption(validateOpt).addOption(createCatalog).addOption(alterCatalog).addOption(moveDatabase).addOption(moveTable);
        optGroup.setRequired(true);
        OptionBuilder.withArgName((String)"user");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"Override config file user name");
        Option userNameOpt = OptionBuilder.create((String)"userName");
        OptionBuilder.withArgName((String)"password");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"Override config file password");
        Option passwdOpt = OptionBuilder.create((String)"passWord");
        OptionBuilder.withArgName((String)"databaseType");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"Metastore database type");
        Option dbTypeOpt = OptionBuilder.create((String)"dbType");
        OptionBuilder.withArgName((String)"metaDatabaseType");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"Used only if upgrading the system catalog for hive");
        Option metaDbTypeOpt = OptionBuilder.create((String)"metaDbType");
        OptionBuilder.withArgName((String)"url");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"connection url to the database");
        Option urlOpt = OptionBuilder.create((String)"url");
        OptionBuilder.withArgName((String)"driver");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"driver name for connection");
        Option driverOpt = OptionBuilder.create((String)"driver");
        OptionBuilder.withArgName((String)"databaseOpts");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"Backend DB specific options");
        Option dbOpts = OptionBuilder.create((String)"dbOpts");
        Option dryRunOpt = new Option("dryRun", "list SQL scripts (no execute)");
        Option verboseOpt = new Option("verbose", "only print SQL statements");
        Option silentOpt = new Option("silent", "print info output only to the log");
        OptionBuilder.withArgName((String)"serverList");
        OptionBuilder.hasArgs();
        OptionBuilder.withDescription((String)"a comma-separated list of servers used in location validation in the format of scheme://authority (e.g. hdfs://localhost:8000)");
        Option serversOpt = OptionBuilder.create((String)"servers");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Location of new catalog, required when adding a catalog");
        Option catalogLocation = OptionBuilder.create((String)"catalogLocation");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Description of new catalog");
        Option catalogDescription = OptionBuilder.create((String)"catalogDescription");
        OptionBuilder.withDescription((String)"If passed then it is not an error to create an existing catalog");
        Option ifNotExists = OptionBuilder.create((String)"ifNotExists");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Catalog a moving database or table is going to.  This is required if you are moving a database or table.");
        Option toCatalog = OptionBuilder.create((String)"toCatalog");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Catalog a moving database or table is coming from.  This is required if you are moving a database or table.");
        Option fromCatalog = OptionBuilder.create((String)"fromCatalog");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Database a moving table is going to.  This is required if you are moving a table.");
        Option toDatabase = OptionBuilder.create((String)"toDatabase");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Database a moving table is coming from.  This is required if you are moving a table.");
        Option fromDatabase = OptionBuilder.create((String)"fromDatabase");
        cmdLineOptions.addOption(help);
        cmdLineOptions.addOption(dryRunOpt);
        cmdLineOptions.addOption(userNameOpt);
        cmdLineOptions.addOption(passwdOpt);
        cmdLineOptions.addOption(dbTypeOpt);
        cmdLineOptions.addOption(verboseOpt);
        cmdLineOptions.addOption(silentOpt);
        cmdLineOptions.addOption(metaDbTypeOpt);
        cmdLineOptions.addOption(urlOpt);
        cmdLineOptions.addOption(driverOpt);
        cmdLineOptions.addOption(dbOpts);
        cmdLineOptions.addOption(serversOpt);
        cmdLineOptions.addOption(catalogLocation);
        cmdLineOptions.addOption(catalogDescription);
        cmdLineOptions.addOption(ifNotExists);
        cmdLineOptions.addOption(toCatalog);
        cmdLineOptions.addOption(fromCatalog);
        cmdLineOptions.addOption(toDatabase);
        cmdLineOptions.addOption(fromDatabase);
        cmdLineOptions.addOptionGroup(optGroup);
    }

    private void logAndPrintToOut(String msg) {
        LOG.info(msg);
        if (!this.silent) {
            System.out.println(msg);
        }
    }

    public static void main(String[] args) {
        HiveSchemaTool.main(args, new HiveConf(HiveSchemaTool.class), System.getenv("HIVE_HOME"));
    }

    public static void main(String[] args, HiveConf hiveConf, String hiveHome) {
        GnuParser parser = new GnuParser();
        CommandLine line = null;
        String dbType = null;
        String metaDbType = null;
        String schemaVer = null;
        Options cmdLineOptions = new Options();
        HiveSchemaTool.initOptions(cmdLineOptions);
        try {
            line = parser.parse(cmdLineOptions, args);
        }
        catch (ParseException e) {
            System.err.println("HiveSchemaTool:Parsing failed.  Reason: " + e.getLocalizedMessage());
            HiveSchemaTool.printAndExit(cmdLineOptions);
        }
        if (line.hasOption("help")) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("schemaTool", cmdLineOptions);
            return;
        }
        if (line.hasOption("dbType")) {
            dbType = line.getOptionValue("dbType");
            if (!(dbType.equalsIgnoreCase("derby") || dbType.equalsIgnoreCase("hive") || dbType.equalsIgnoreCase("mssql") || dbType.equalsIgnoreCase("mysql") || dbType.equalsIgnoreCase("postgres") || dbType.equalsIgnoreCase("oracle"))) {
                System.err.println("Unsupported dbType " + dbType);
                HiveSchemaTool.printAndExit(cmdLineOptions);
            }
        } else {
            System.err.println("no dbType supplied");
            HiveSchemaTool.printAndExit(cmdLineOptions);
        }
        if (line.hasOption("metaDbType")) {
            metaDbType = line.getOptionValue("metaDbType");
            if (!dbType.equals("hive")) {
                System.err.println("metaDbType only supported for dbType = hive");
                HiveSchemaTool.printAndExit(cmdLineOptions);
            }
            if (!(metaDbType.equalsIgnoreCase("derby") || metaDbType.equalsIgnoreCase("mssql") || metaDbType.equalsIgnoreCase("mysql") || metaDbType.equalsIgnoreCase("postgres") || metaDbType.equalsIgnoreCase("oracle"))) {
                System.err.println("Unsupported metaDbType " + metaDbType);
                HiveSchemaTool.printAndExit(cmdLineOptions);
            }
        } else if (dbType.equalsIgnoreCase("hive")) {
            System.err.println("no metaDbType supplied");
            HiveSchemaTool.printAndExit(cmdLineOptions);
        }
        System.setProperty(HiveConf.ConfVars.METASTORE_SCHEMA_VERIFICATION.varname, "true");
        try {
            HiveSchemaTool schemaTool = new HiveSchemaTool(hiveHome, hiveConf, dbType, metaDbType);
            if (line.hasOption("userName")) {
                schemaTool.setUserName(line.getOptionValue("userName"));
            } else {
                schemaTool.setUserName(schemaTool.getHiveConf().get(HiveConf.ConfVars.METASTORE_CONNECTION_USER_NAME.varname));
            }
            if (line.hasOption("passWord")) {
                schemaTool.setPassWord(line.getOptionValue("passWord"));
            } else {
                try {
                    schemaTool.setPassWord(ShimLoader.getHadoopShims().getPassword((Configuration)schemaTool.getHiveConf(), HiveConf.ConfVars.METASTOREPWD.varname));
                }
                catch (IOException err) {
                    throw new HiveMetaException("Error getting metastore password", (Throwable)err);
                }
            }
            if (line.hasOption("url")) {
                schemaTool.setUrl(line.getOptionValue("url"));
            }
            if (line.hasOption("driver")) {
                schemaTool.setDriver(line.getOptionValue("driver"));
            }
            if (line.hasOption("dryRun")) {
                schemaTool.setDryRun(true);
            }
            if (line.hasOption("verbose")) {
                schemaTool.setVerbose(true);
            }
            if (line.hasOption("silent")) {
                schemaTool.setSilent(true);
            }
            if (line.hasOption("dbOpts")) {
                schemaTool.setDbOpts(line.getOptionValue("dbOpts"));
            }
            if (line.hasOption("validate") && line.hasOption("servers")) {
                schemaTool.setValidationServers(line.getOptionValue("servers"));
            }
            if (line.hasOption("info")) {
                schemaTool.showInfo();
            } else if (line.hasOption("upgradeSchema")) {
                schemaTool.doUpgrade();
            } else if (line.hasOption("upgradeSchemaFrom")) {
                schemaVer = line.getOptionValue("upgradeSchemaFrom");
                schemaTool.doUpgrade(schemaVer);
            } else if (line.hasOption("initSchema")) {
                schemaTool.doInit();
            } else if (line.hasOption("initSchemaTo")) {
                schemaVer = line.getOptionValue("initSchemaTo");
                schemaTool.doInit(schemaVer);
            } else if (line.hasOption("validate")) {
                schemaTool.doValidate();
            } else if (line.hasOption("createCatalog")) {
                schemaTool.createCatalog(line.getOptionValue("createCatalog"), line.getOptionValue("catalogLocation"), line.getOptionValue("catalogDescription"), line.hasOption("ifNotExists"));
            } else if (line.hasOption("alterCatalog")) {
                schemaTool.alterCatalog(line.getOptionValue("alterCatalog"), line.getOptionValue("catalogLocation"), line.getOptionValue("catalogDescription"));
            } else if (line.hasOption("moveDatabase")) {
                schemaTool.moveDatabase(line.getOptionValue("fromCatalog"), line.getOptionValue("toCatalog"), line.getOptionValue("moveDatabase"));
            } else if (line.hasOption("moveTable")) {
                schemaTool.moveTable(line.getOptionValue("fromCatalog"), line.getOptionValue("toCatalog"), line.getOptionValue("fromDatabase"), line.getOptionValue("toDatabase"), line.getOptionValue("moveTable"));
            } else {
                System.err.println("no valid option supplied");
                HiveSchemaTool.printAndExit(cmdLineOptions);
            }
        }
        catch (HiveMetaException e) {
            System.err.println((Object)e);
            if (e.getCause() != null) {
                Throwable t = e.getCause();
                System.err.println("Underlying cause: " + t.getClass().getName() + " : " + t.getMessage());
                if (e.getCause() instanceof SQLException) {
                    System.err.println("SQL Error code: " + ((SQLException)t).getErrorCode());
                }
            }
            if (line.hasOption("verbose")) {
                e.printStackTrace();
            } else {
                System.err.println("Use --verbose for detailed stacktrace.");
            }
            System.err.println("*** schemaTool failed ***");
            System.exit(1);
        }
        if (!line.hasOption("silent")) {
            System.out.println("schemaTool completed");
        }
        System.exit(0);
    }

    static class CommandBuilder {
        private final HiveConf hiveConf;
        private final String userName;
        private final String password;
        private final String sqlScriptFile;
        private final String driver;
        private final String url;

        CommandBuilder(HiveConf hiveConf, String url, String driver, String userName, String password, String sqlScriptFile) {
            this.hiveConf = hiveConf;
            this.userName = userName;
            this.password = password;
            this.url = url;
            this.driver = driver;
            this.sqlScriptFile = sqlScriptFile;
        }

        String[] buildToRun() throws IOException {
            return this.argsWith(this.password);
        }

        String buildToLog() throws IOException {
            this.logScript();
            return StringUtils.join((Object[])this.argsWith("[passwd stripped]"), (String)" ");
        }

        private String[] argsWith(String password) throws IOException {
            return new String[]{"-u", this.url == null ? HiveSchemaHelper.getValidConfVar((MetastoreConf.ConfVars)MetastoreConf.ConfVars.CONNECT_URL_KEY, (Configuration)this.hiveConf) : this.url, "-d", this.driver == null ? HiveSchemaHelper.getValidConfVar((MetastoreConf.ConfVars)MetastoreConf.ConfVars.CONNECTION_DRIVER, (Configuration)this.hiveConf) : this.driver, "-n", this.userName, "-p", password, "-f", this.sqlScriptFile};
        }

        private void logScript() throws IOException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Going to invoke file that contains:");
                try (BufferedReader reader = new BufferedReader(new FileReader(this.sqlScriptFile));){
                    String line;
                    while ((line = reader.readLine()) != null) {
                        LOG.debug("script: " + line);
                    }
                }
            }
        }
    }
}

