/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.tools;

import java.io.BufferedReader;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.Stack;
import java.util.StringTokenizer;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.Trash;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.io.BinaryComparable;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.io.NullWritable;
import org.apache.hadoop.io.RawComparator;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.WritableComparable;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.mapred.FileOutputFormat;
import org.apache.hadoop.mapred.FileSplit;
import org.apache.hadoop.mapred.InputFormat;
import org.apache.hadoop.mapred.InputSplit;
import org.apache.hadoop.mapred.InvalidInputException;
import org.apache.hadoop.mapred.JobClient;
import org.apache.hadoop.mapred.JobConf;
import org.apache.hadoop.mapred.Mapper;
import org.apache.hadoop.mapred.OutputCollector;
import org.apache.hadoop.mapred.RecordReader;
import org.apache.hadoop.mapred.Reporter;
import org.apache.hadoop.mapred.SequenceFileRecordReader;
import org.apache.hadoop.mapreduce.Cluster;
import org.apache.hadoop.mapreduce.JobSubmissionFiles;
import org.apache.hadoop.mapreduce.security.TokenCache;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DistCpV1
implements Tool {
    public static final Logger LOG = LoggerFactory.getLogger(DistCpV1.class);
    private static final String NAME = "distcp";
    private static final String usage = "distcp [OPTIONS] <srcurl>* <desturl>\n\nOPTIONS:\n-p[rbugpt]             Preserve status\n                       r: replication number\n                       b: block size\n                       u: user\n                       g: group\n                       p: permission\n                       t: modification and access times\n                       -p alone is equivalent to -prbugpt\n-i                     Ignore failures\n-basedir <basedir>     Use <basedir> as the base directory when copying files from <srcurl>\n-log <logdir>          Write logs to <logdir>\n-m <num_maps>          Maximum number of simultaneous copies\n-overwrite             Overwrite destination\n-update                Overwrite if src size different from dst size\n-skipcrccheck          Do not use CRC check to determine if src is \n                       different from dest. Relevant only if -update\n                       is specified\n-f <urilist_uri>       Use list at <urilist_uri> as src list\n-filelimit <n>         Limit the total number of files to be <= n\n-sizelimit <n>         Limit the total size to be <= n bytes\n-delete                Delete the files existing in the dst but not in src\n-dryrun                Display count of files and total size of files\n                        in src and then exit. Copy is not done at all.\n                        desturl should not be speicified with out -update.\n-mapredSslConf <f>     Filename of SSL configuration for mapper task\n\nNOTE 1: if -overwrite or -update are set, each source URI is \n      interpreted as an isomorphic update to an existing directory.\nFor example:\nhadoop distcp -p -update \"hdfs://A:8020/user/foo/bar\" \"hdfs://B:8020/user/foo/baz\"\n\n     would update all descendants of 'baz' also in 'bar'; it would \n     *not* update /user/foo/baz/bar\n\nNOTE 2: The parameter <n> in -filelimit and -sizelimit can be \n     specified with symbolic representation.  For examples,\n       1230k = 1230 * 1024 = 1259520\n       891g = 891 * 1024^3 = 956703965184\n";
    private static final long BYTES_PER_MAP = 0x10000000L;
    private static final int MAX_MAPS_PER_NODE = 20;
    private static final int SYNC_FILE_MAX = 10;
    private static final int DEFAULT_FILE_RETRIES = 3;
    static final String TMP_DIR_LABEL = "distcp.tmp.dir";
    static final String DST_DIR_LABEL = "distcp.dest.path";
    static final String JOB_DIR_LABEL = "distcp.job.dir";
    static final String MAX_MAPS_LABEL = "distcp.max.map.tasks";
    static final String SRC_LIST_LABEL = "distcp.src.list";
    static final String SRC_COUNT_LABEL = "distcp.src.count";
    static final String TOTAL_SIZE_LABEL = "distcp.total.size";
    static final String DST_DIR_LIST_LABEL = "distcp.dst.dir.list";
    static final String BYTES_PER_MAP_LABEL = "distcp.bytes.per.map";
    static final String PRESERVE_STATUS_LABEL = Options.PRESERVE_STATUS.propertyname + ".value";
    static final String FILE_RETRIES_LABEL = "distcp.file.retries";
    private JobConf conf;
    private static final Random RANDOM = new Random();

    public void setConf(Configuration conf) {
        this.conf = conf instanceof JobConf ? (JobConf)conf : new JobConf(conf);
    }

    public Configuration getConf() {
        return this.conf;
    }

    public DistCpV1(Configuration conf) {
        this.setConf(conf);
    }

    private static List<Path> fetchFileList(Configuration conf, Path srcList) throws IOException {
        ArrayList<Path> result = new ArrayList<Path>();
        FileSystem fs = srcList.getFileSystem(conf);
        try (BufferedReader input = new BufferedReader(new InputStreamReader((InputStream)fs.open(srcList), Charset.forName("UTF-8")));){
            String line = input.readLine();
            while (line != null) {
                result.add(new Path(line));
                line = input.readLine();
            }
        }
        return result;
    }

    @Deprecated
    public static void copy(Configuration conf, String srcPath, String destPath, Path logPath, boolean srcAsList, boolean ignoreReadFailures) throws IOException {
        Path src = new Path(srcPath);
        ArrayList<Path> tmp = new ArrayList<Path>();
        if (srcAsList) {
            tmp.addAll(DistCpV1.fetchFileList(conf, src));
        } else {
            tmp.add(src);
        }
        EnumSet<Options> flags = ignoreReadFailures ? EnumSet.of(Options.IGNORE_READ_FAILURES) : EnumSet.noneOf(Options.class);
        Path dst = new Path(destPath);
        DistCpV1.copy(conf, new Arguments(tmp, null, dst, logPath, flags, null, Long.MAX_VALUE, Long.MAX_VALUE, null, false));
    }

    private static void checkSrcPath(JobConf jobConf, List<Path> srcPaths) throws IOException {
        ArrayList<IOException> rslt = new ArrayList<IOException>();
        LinkedList<Path> unglobbed = new LinkedList<Path>();
        Path[] ps = new Path[srcPaths.size()];
        ps = srcPaths.toArray(ps);
        TokenCache.obtainTokensForNamenodes((Credentials)jobConf.getCredentials(), (Path[])ps, (Configuration)jobConf);
        for (Path p : srcPaths) {
            FileSystem fs = p.getFileSystem((Configuration)jobConf);
            FileStatus[] inputs = fs.globStatus(p);
            if (inputs != null && inputs.length > 0) {
                for (FileStatus onePath : inputs) {
                    unglobbed.add(onePath.getPath());
                }
                continue;
            }
            rslt.add(new IOException("Input source " + p + " does not exist."));
        }
        if (!rslt.isEmpty()) {
            throw new InvalidInputException(rslt);
        }
        srcPaths.clear();
        srcPaths.addAll(unglobbed);
    }

    static void copy(Configuration conf, Arguments args) throws IOException {
        LOG.info("srcPaths=" + args.srcs);
        if (!args.dryrun || args.flags.contains((Object)Options.UPDATE)) {
            LOG.info("destPath=" + args.dst);
        }
        JobConf job = DistCpV1.createJobConf(conf);
        DistCpV1.checkSrcPath(job, args.srcs);
        if (args.preservedAttributes != null) {
            job.set(PRESERVE_STATUS_LABEL, args.preservedAttributes);
        }
        if (args.mapredSslConf != null) {
            job.set("dfs.https.client.keystore.resource", args.mapredSslConf);
        }
        try {
            if (DistCpV1.setup(conf, job, args)) {
                JobClient.runJob((JobConf)job);
            }
            if (!args.dryrun) {
                DistCpV1.finalize(conf, job, args.dst, args.preservedAttributes);
            }
        }
        finally {
            if (!args.dryrun) {
                DistCpV1.fullyDelete(job.get(TMP_DIR_LABEL), (Configuration)job);
            }
            DistCpV1.fullyDelete(job.get(JOB_DIR_LABEL), (Configuration)job);
        }
    }

    private static void updateDestStatus(FileStatus src, FileStatus dst, EnumSet<FileAttribute> preseved, FileSystem destFileSys) throws IOException {
        String owner = null;
        String group = null;
        if (preseved.contains((Object)FileAttribute.USER) && !src.getOwner().equals(dst.getOwner())) {
            owner = src.getOwner();
        }
        if (preseved.contains((Object)FileAttribute.GROUP) && !src.getGroup().equals(dst.getGroup())) {
            group = src.getGroup();
        }
        if (owner != null || group != null) {
            destFileSys.setOwner(dst.getPath(), owner, group);
        }
        if (preseved.contains((Object)FileAttribute.PERMISSION) && !src.getPermission().equals((Object)dst.getPermission())) {
            destFileSys.setPermission(dst.getPath(), src.getPermission());
        }
        if (preseved.contains((Object)FileAttribute.TIMES)) {
            destFileSys.setTimes(dst.getPath(), src.getModificationTime(), src.getAccessTime());
        }
    }

    private static void finalize(Configuration conf, JobConf jobconf, Path destPath, String presevedAttributes) throws IOException {
        if (presevedAttributes == null) {
            return;
        }
        EnumSet<FileAttribute> preseved = FileAttribute.parse(presevedAttributes);
        if (!(preseved.contains((Object)FileAttribute.USER) || preseved.contains((Object)FileAttribute.GROUP) || preseved.contains((Object)FileAttribute.PERMISSION))) {
            return;
        }
        FileSystem dstfs = destPath.getFileSystem(conf);
        Path dstdirlist = new Path(jobconf.get(DST_DIR_LIST_LABEL));
        try (SequenceFile.Reader in = new SequenceFile.Reader((Configuration)jobconf, new SequenceFile.Reader.Option[]{SequenceFile.Reader.file((Path)dstdirlist)});){
            Text dsttext = new Text();
            FilePair pair = new FilePair();
            while (in.next((Writable)dsttext, (Writable)pair)) {
                Path absdst = new Path(destPath, pair.output);
                DistCpV1.updateDestStatus(pair.input, dstfs.getFileStatus(absdst), preseved, dstfs);
            }
        }
    }

    public int run(String[] args) {
        try {
            DistCpV1.copy((Configuration)this.conf, Arguments.valueOf(args, (Configuration)this.conf));
            return 0;
        }
        catch (IllegalArgumentException e) {
            System.err.println(StringUtils.stringifyException((Throwable)e) + "\ndistcp [OPTIONS] <srcurl>* <desturl>\n\nOPTIONS:\n-p[rbugpt]             Preserve status\n                       r: replication number\n                       b: block size\n                       u: user\n                       g: group\n                       p: permission\n                       t: modification and access times\n                       -p alone is equivalent to -prbugpt\n-i                     Ignore failures\n-basedir <basedir>     Use <basedir> as the base directory when copying files from <srcurl>\n-log <logdir>          Write logs to <logdir>\n-m <num_maps>          Maximum number of simultaneous copies\n-overwrite             Overwrite destination\n-update                Overwrite if src size different from dst size\n-skipcrccheck          Do not use CRC check to determine if src is \n                       different from dest. Relevant only if -update\n                       is specified\n-f <urilist_uri>       Use list at <urilist_uri> as src list\n-filelimit <n>         Limit the total number of files to be <= n\n-sizelimit <n>         Limit the total size to be <= n bytes\n-delete                Delete the files existing in the dst but not in src\n-dryrun                Display count of files and total size of files\n                        in src and then exit. Copy is not done at all.\n                        desturl should not be speicified with out -update.\n-mapredSslConf <f>     Filename of SSL configuration for mapper task\n\nNOTE 1: if -overwrite or -update are set, each source URI is \n      interpreted as an isomorphic update to an existing directory.\nFor example:\nhadoop distcp -p -update \"hdfs://A:8020/user/foo/bar\" \"hdfs://B:8020/user/foo/baz\"\n\n     would update all descendants of 'baz' also in 'bar'; it would \n     *not* update /user/foo/baz/bar\n\nNOTE 2: The parameter <n> in -filelimit and -sizelimit can be \n     specified with symbolic representation.  For examples,\n       1230k = 1230 * 1024 = 1259520\n       891g = 891 * 1024^3 = 956703965184\n");
            ToolRunner.printGenericCommandUsage((PrintStream)System.err);
            return -1;
        }
        catch (DuplicationException e) {
            System.err.println(StringUtils.stringifyException((Throwable)e));
            return -2;
        }
        catch (RemoteException e) {
            IOException unwrapped = e.unwrapRemoteException(new Class[]{FileNotFoundException.class, AccessControlException.class, QuotaExceededException.class});
            System.err.println(StringUtils.stringifyException((Throwable)unwrapped));
            return -3;
        }
        catch (Exception e) {
            System.err.println("With failures, global counters are inaccurate; consider running with -i");
            System.err.println("Copy failed: " + StringUtils.stringifyException((Throwable)e));
            return -999;
        }
    }

    public static void main(String[] args) throws Exception {
        JobConf job = new JobConf(DistCpV1.class);
        DistCpV1 distcp = new DistCpV1((Configuration)job);
        int res = ToolRunner.run((Tool)distcp, (String[])args);
        System.exit(res);
    }

    static String makeRelative(Path root, Path absPath) {
        if (!absPath.isAbsolute()) {
            throw new IllegalArgumentException("!absPath.isAbsolute(), absPath=" + absPath);
        }
        String p = absPath.toUri().getPath();
        StringTokenizer pathTokens = new StringTokenizer(p, "/");
        StringTokenizer rootTokens = new StringTokenizer(root.toUri().getPath(), "/");
        while (rootTokens.hasMoreTokens()) {
            if (rootTokens.nextToken().equals(pathTokens.nextToken())) continue;
            return null;
        }
        StringBuilder sb = new StringBuilder();
        while (pathTokens.hasMoreTokens()) {
            sb.append(pathTokens.nextToken());
            if (!pathTokens.hasMoreTokens()) continue;
            sb.append("/");
        }
        return sb.length() == 0 ? "." : sb.toString();
    }

    private static int setMapCount(long totalBytes, JobConf job) throws IOException {
        int numMaps = (int)(totalBytes / job.getLong(BYTES_PER_MAP_LABEL, 0x10000000L));
        numMaps = Math.min(numMaps, job.getInt(MAX_MAPS_LABEL, 20 * new JobClient(job).getClusterStatus().getTaskTrackers()));
        numMaps = Math.max(numMaps, 1);
        job.setNumMapTasks(numMaps);
        return numMaps;
    }

    static void fullyDelete(String dir, Configuration conf) throws IOException {
        Path tmp;
        boolean success;
        if (dir != null && !(success = (tmp = new Path(dir)).getFileSystem(conf).delete(tmp, true))) {
            LOG.warn("Could not fully delete " + tmp);
        }
    }

    private static JobConf createJobConf(Configuration conf) {
        JobConf jobconf = new JobConf(conf, DistCpV1.class);
        jobconf.setJobName(conf.get("mapred.job.name", NAME));
        jobconf.setMapSpeculativeExecution(false);
        jobconf.setInputFormat(CopyInputFormat.class);
        jobconf.setOutputKeyClass(Text.class);
        jobconf.setOutputValueClass(Text.class);
        jobconf.setMapperClass(CopyFilesMapper.class);
        jobconf.setNumReduceTasks(0);
        return jobconf;
    }

    public static String getRandomId() {
        return Integer.toString(RANDOM.nextInt(Integer.MAX_VALUE), 36);
    }

    private static void setReplication(Configuration conf, JobConf jobConf, Path srcfilelist, int numMaps) throws IOException {
        int numMaxMaps = new JobClient(jobConf).getClusterStatus().getMaxMapTasks();
        short replication = (short)Math.ceil(Math.sqrt(Math.min(numMaxMaps, numMaps)));
        FileSystem fs = srcfilelist.getFileSystem(conf);
        FileStatus srcStatus = fs.getFileStatus(srcfilelist);
        if (srcStatus.getReplication() < replication && !fs.setReplication(srcfilelist, replication)) {
            throw new IOException("Unable to increase the replication of file " + srcfilelist);
        }
    }

    private static boolean dirExists(Configuration conf, Path dst) throws IOException {
        FileSystem destFileSys = dst.getFileSystem(conf);
        FileStatus status = null;
        try {
            status = destFileSys.getFileStatus(dst);
        }
        catch (FileNotFoundException e) {
            return false;
        }
        if (status.isFile()) {
            throw new FileAlreadyExistsException("Not a dir: " + dst + " is a file.");
        }
        return true;
    }

    static boolean setup(Configuration conf, JobConf jobConf, Arguments args) throws IOException {
        Path logPath;
        Path stagingArea;
        jobConf.set(DST_DIR_LABEL, args.dst.toUri().toString());
        boolean update = args.flags.contains((Object)Options.UPDATE);
        boolean skipCRCCheck = args.flags.contains((Object)Options.SKIPCRC);
        boolean overwrite = !update && args.flags.contains((Object)Options.OVERWRITE) && !args.dryrun;
        jobConf.setBoolean(Options.UPDATE.propertyname, update);
        jobConf.setBoolean(Options.SKIPCRC.propertyname, skipCRCCheck);
        jobConf.setBoolean(Options.OVERWRITE.propertyname, overwrite);
        jobConf.setBoolean(Options.IGNORE_READ_FAILURES.propertyname, args.flags.contains((Object)Options.IGNORE_READ_FAILURES));
        jobConf.setBoolean(Options.PRESERVE_STATUS.propertyname, args.flags.contains((Object)Options.PRESERVE_STATUS));
        String randomId = DistCpV1.getRandomId();
        JobClient jClient = new JobClient(jobConf);
        try {
            stagingArea = JobSubmissionFiles.getStagingDir((Cluster)jClient.getClusterHandle(), (Configuration)conf);
        }
        catch (InterruptedException ie) {
            throw new IOException(ie);
        }
        Path jobDirectory = new Path(stagingArea + "distcp_" + randomId);
        FsPermission mapredSysPerms = new FsPermission(JobSubmissionFiles.JOB_DIR_PERMISSION);
        FileSystem.mkdirs((FileSystem)jClient.getFs(), (Path)jobDirectory, (FsPermission)mapredSysPerms);
        jobConf.set(JOB_DIR_LABEL, jobDirectory.toString());
        long maxBytesPerMap = conf.getLong(BYTES_PER_MAP_LABEL, 0x10000000L);
        FileSystem dstfs = args.dst.getFileSystem(conf);
        TokenCache.obtainTokensForNamenodes((Credentials)jobConf.getCredentials(), (Path[])new Path[]{args.dst}, (Configuration)conf);
        boolean dstExists = dstfs.exists(args.dst);
        boolean dstIsDir = false;
        if (dstExists) {
            dstIsDir = dstfs.getFileStatus(args.dst).isDirectory();
        }
        if ((logPath = args.log) == null) {
            String filename = "_distcp_logs_" + randomId;
            if (!dstExists || !dstIsDir) {
                Path parent = args.dst.getParent();
                if (null == parent) {
                    parent = args.dst;
                }
                if (!dstfs.exists(parent)) {
                    dstfs.mkdirs(parent);
                }
                logPath = new Path(parent, filename);
            } else {
                logPath = new Path(args.dst, filename);
            }
        }
        FileOutputFormat.setOutputPath((JobConf)jobConf, (Path)logPath);
        FileSystem jobfs = jobDirectory.getFileSystem((Configuration)jobConf);
        Path srcfilelist = new Path(jobDirectory, "_distcp_src_files");
        Path dstfilelist = new Path(jobDirectory, "_distcp_dst_files");
        Path dstdirlist = new Path(jobDirectory, "_distcp_dst_dirs");
        jobConf.set(SRC_LIST_LABEL, srcfilelist.toString());
        jobConf.set(DST_DIR_LIST_LABEL, dstdirlist.toString());
        int srcCount = 0;
        int cnsyncf = 0;
        int dirsyn = 0;
        long fileCount = 0L;
        long dirCount = 0L;
        long byteCount = 0L;
        long cbsyncs = 0L;
        long skipFileCount = 0L;
        long skipByteCount = 0L;
        try (SequenceFile.Writer src_writer = SequenceFile.createWriter((Configuration)jobConf, (SequenceFile.Writer.Option[])new SequenceFile.Writer.Option[]{SequenceFile.Writer.file((Path)srcfilelist), SequenceFile.Writer.keyClass(LongWritable.class), SequenceFile.Writer.valueClass(FilePair.class), SequenceFile.Writer.compression((SequenceFile.CompressionType)SequenceFile.CompressionType.NONE)});
             SequenceFile.Writer dst_writer = SequenceFile.createWriter((Configuration)jobConf, (SequenceFile.Writer.Option[])new SequenceFile.Writer.Option[]{SequenceFile.Writer.file((Path)dstfilelist), SequenceFile.Writer.keyClass(Text.class), SequenceFile.Writer.valueClass(Text.class), SequenceFile.Writer.compression((SequenceFile.CompressionType)SequenceFile.CompressionType.NONE)});
             SequenceFile.Writer dir_writer = SequenceFile.createWriter((Configuration)jobConf, (SequenceFile.Writer.Option[])new SequenceFile.Writer.Option[]{SequenceFile.Writer.file((Path)dstdirlist), SequenceFile.Writer.keyClass(Text.class), SequenceFile.Writer.valueClass(FilePair.class), SequenceFile.Writer.compression((SequenceFile.CompressionType)SequenceFile.CompressionType.NONE)});){
            FileSystem basefs;
            boolean special = args.srcs.size() == 1 && !dstExists || update || overwrite;
            Path basedir = null;
            HashSet<Path> parentDirsToCopy = new HashSet<Path>();
            if (args.basedir != null && !(basefs = args.basedir.getFileSystem(conf)).isDirectory(basedir = args.basedir.makeQualified(basefs.getUri(), basefs.getWorkingDirectory()))) {
                throw new IOException("Basedir " + basedir + " is not a directory.");
            }
            for (Path src : args.srcs) {
                Path root;
                FileSystem srcfs = src.getFileSystem(conf);
                FileStatus srcfilestat = srcfs.getFileStatus(src);
                Path path = root = special && srcfilestat.isDirectory() ? src : src.getParent();
                if (dstExists && !dstIsDir && (args.srcs.size() > 1 || srcfilestat.isDirectory())) {
                    throw new IOException("Destination " + args.dst + " should be a dir if multiple source paths are there OR if the source path is a dir");
                }
                if (basedir != null) {
                    Path parent;
                    root = basedir;
                    for (parent = src.getParent().makeQualified(srcfs.getUri(), srcfs.getWorkingDirectory()); parent != null && !parent.equals((Object)basedir); parent = parent.getParent()) {
                        if (parentDirsToCopy.contains(parent)) continue;
                        parentDirsToCopy.add(parent);
                        String dst = DistCpV1.makeRelative(root, parent);
                        FileStatus pst = srcfs.getFileStatus(parent);
                        src_writer.append((Writable)new LongWritable(0L), (Writable)new FilePair(pst, dst));
                        dst_writer.append((Writable)new Text(dst), (Writable)new Text(parent.toString()));
                        dir_writer.append((Writable)new Text(dst), (Writable)new FilePair(pst, dst));
                        if (++dirsyn <= 10) continue;
                        dirsyn = 0;
                        dir_writer.sync();
                    }
                    if (parent == null) {
                        throw new IOException("Basedir " + basedir + " is not a prefix of source path " + src);
                    }
                }
                if (srcfilestat.isDirectory()) {
                    ++srcCount;
                    String dst = DistCpV1.makeRelative(root, src);
                    if (!update || !DistCpV1.dirExists(conf, new Path(args.dst, dst))) {
                        ++dirCount;
                        src_writer.append((Writable)new LongWritable(0L), (Writable)new FilePair(srcfilestat, dst));
                    }
                    dst_writer.append((Writable)new Text(dst), (Writable)new Text(src.toString()));
                }
                Stack<FileStatus> pathstack = new Stack<FileStatus>();
                pathstack.push(srcfilestat);
                while (!pathstack.empty()) {
                    FileStatus cur = (FileStatus)pathstack.pop();
                    FileStatus[] children = srcfs.listStatus(cur.getPath());
                    for (int i = 0; i < children.length; ++i) {
                        boolean skipPath = false;
                        FileStatus child = children[i];
                        String dst = DistCpV1.makeRelative(root, child.getPath());
                        ++srcCount;
                        if (child.isDirectory()) {
                            pathstack.push(child);
                            if (!update || !DistCpV1.dirExists(conf, new Path(args.dst, dst))) {
                                ++dirCount;
                            } else {
                                skipPath = true;
                            }
                        } else {
                            Path destPath = new Path(args.dst, dst);
                            if (cur.isFile() && args.srcs.size() == 1) {
                                Path dstparent = destPath.getParent();
                                FileSystem destFileSys = destPath.getFileSystem((Configuration)jobConf);
                                if (!destFileSys.exists(dstparent) || !destFileSys.getFileStatus(dstparent).isDirectory()) {
                                    destPath = dstparent;
                                }
                            }
                            skipPath = update && DistCpV1.sameFile(srcfs, child, dstfs, destPath, skipCRCCheck);
                            if (!(skipPath |= fileCount == args.filelimit || byteCount + child.getLen() > args.sizelimit)) {
                                ++fileCount;
                                byteCount += child.getLen();
                                if (LOG.isTraceEnabled()) {
                                    LOG.trace("adding file " + child.getPath());
                                }
                                if (++cnsyncf > 10 || (cbsyncs += child.getLen()) > maxBytesPerMap) {
                                    src_writer.sync();
                                    dst_writer.sync();
                                    cnsyncf = 0;
                                    cbsyncs = 0L;
                                }
                            } else {
                                ++skipFileCount;
                                skipByteCount += child.getLen();
                                if (LOG.isTraceEnabled()) {
                                    LOG.trace("skipping file " + child.getPath());
                                }
                            }
                        }
                        if (!skipPath) {
                            src_writer.append((Writable)new LongWritable(child.isDirectory() ? 0L : child.getLen()), (Writable)new FilePair(child, dst));
                        }
                        dst_writer.append((Writable)new Text(dst), (Writable)new Text(child.getPath().toString()));
                    }
                    if (!cur.isDirectory()) continue;
                    String dst = DistCpV1.makeRelative(root, cur.getPath());
                    dir_writer.append((Writable)new Text(dst), (Writable)new FilePair(cur, dst));
                    if (++dirsyn <= 10) continue;
                    dirsyn = 0;
                    dir_writer.sync();
                }
            }
        }
        LOG.info("sourcePathsCount(files+directories)=" + srcCount);
        LOG.info("filesToCopyCount=" + fileCount);
        LOG.info("bytesToCopyCount=" + StringUtils.TraditionalBinaryPrefix.long2String((long)byteCount, (String)"", (int)1));
        if (update) {
            LOG.info("filesToSkipCopyCount=" + skipFileCount);
            LOG.info("bytesToSkipCopyCount=" + StringUtils.TraditionalBinaryPrefix.long2String((long)skipByteCount, (String)"", (int)1));
        }
        if (args.dryrun) {
            return false;
        }
        int mapCount = DistCpV1.setMapCount(byteCount, jobConf);
        DistCpV1.setReplication(conf, jobConf, srcfilelist, mapCount);
        FileStatus dststatus = null;
        try {
            dststatus = dstfs.getFileStatus(args.dst);
        }
        catch (FileNotFoundException fnfe) {
            LOG.info(args.dst + " does not exist.");
        }
        if (dststatus == null && srcCount > 1 && !dstfs.mkdirs(args.dst)) {
            throw new IOException("Failed to create" + args.dst);
        }
        Path sorted = new Path(jobDirectory, "_distcp_sorted");
        DistCpV1.checkDuplication(jobfs, dstfilelist, sorted, conf);
        if (dststatus != null && args.flags.contains((Object)Options.DELETE)) {
            long deletedPathsCount = DistCpV1.deleteNonexisting(dstfs, dststatus, sorted, jobfs, jobDirectory, jobConf, conf);
            LOG.info("deletedPathsFromDestCount(files+directories)=" + deletedPathsCount);
        }
        Path tmpDir = new Path(dstExists && !dstIsDir || !dstExists && srcCount == 1 ? args.dst.getParent() : args.dst, "_distcp_tmp_" + randomId);
        jobConf.set(TMP_DIR_LABEL, tmpDir.toUri().toString());
        tmpDir.getFileSystem(conf).mkdirs(tmpDir);
        LOG.info("sourcePathsCount=" + srcCount);
        LOG.info("filesToCopyCount=" + fileCount);
        LOG.info("bytesToCopyCount=" + StringUtils.TraditionalBinaryPrefix.long2String((long)byteCount, (String)"", (int)1));
        jobConf.setInt(SRC_COUNT_LABEL, srcCount);
        jobConf.setLong(TOTAL_SIZE_LABEL, byteCount);
        return fileCount + dirCount > 0L;
    }

    private static boolean sameFile(FileSystem srcfs, FileStatus srcstatus, FileSystem dstfs, Path dstpath, boolean skipCRCCheck) throws IOException {
        FileChecksum srccs;
        FileStatus dststatus;
        try {
            dststatus = dstfs.getFileStatus(dstpath);
        }
        catch (FileNotFoundException fnfe) {
            return false;
        }
        if (srcstatus.getLen() != dststatus.getLen()) {
            return false;
        }
        if (skipCRCCheck) {
            LOG.debug("Skipping the CRC check");
            return true;
        }
        try {
            srccs = srcfs.getFileChecksum(srcstatus.getPath());
        }
        catch (FileNotFoundException fnfe) {
            return true;
        }
        try {
            FileChecksum dstcs = dstfs.getFileChecksum(dststatus.getPath());
            return srccs == null || dstcs == null || srccs.equals((Object)dstcs);
        }
        catch (FileNotFoundException fnfe) {
            return false;
        }
    }

    private static long deleteNonexisting(FileSystem dstfs, FileStatus dstroot, Path dstsorted, FileSystem jobfs, Path jobdir, JobConf jobconf, Configuration conf) throws IOException {
        if (dstroot.isFile()) {
            throw new IOException("dst must be a directory when option " + Options.DELETE.cmd + " is set, but dst (= " + dstroot.getPath() + ") is not a directory.");
        }
        Path dstlsr = new Path(jobdir, "_distcp_dst_lsr");
        try (SequenceFile.Writer writer = SequenceFile.createWriter((Configuration)jobconf, (SequenceFile.Writer.Option[])new SequenceFile.Writer.Option[]{SequenceFile.Writer.file((Path)dstlsr), SequenceFile.Writer.keyClass(Text.class), SequenceFile.Writer.valueClass(NullWritable.class), SequenceFile.Writer.compression((SequenceFile.CompressionType)SequenceFile.CompressionType.NONE)});){
            Stack<FileStatus> lsrstack = new Stack<FileStatus>();
            lsrstack.push(dstroot);
            while (!lsrstack.isEmpty()) {
                FileStatus status = (FileStatus)lsrstack.pop();
                if (!status.isDirectory()) continue;
                for (FileStatus child : dstfs.listStatus(status.getPath())) {
                    String relative = DistCpV1.makeRelative(dstroot.getPath(), child.getPath());
                    writer.append((Writable)new Text(relative), (Writable)NullWritable.get());
                    lsrstack.push(child);
                }
            }
        }
        Path sortedlsr = new Path(jobdir, "_distcp_dst_lsr_sorted");
        SequenceFile.Sorter sorter = new SequenceFile.Sorter(jobfs, (RawComparator)new Text.Comparator(), Text.class, NullWritable.class, (Configuration)jobconf);
        sorter.sort(dstlsr, sortedlsr);
        long deletedPathsCount = 0L;
        try (SequenceFile.Reader lsrin = new SequenceFile.Reader((Configuration)jobconf, new SequenceFile.Reader.Option[]{SequenceFile.Reader.file((Path)sortedlsr)});
             SequenceFile.Reader dstin = new SequenceFile.Reader((Configuration)jobconf, new SequenceFile.Reader.Option[]{SequenceFile.Reader.file((Path)dstsorted)});){
            Text lsrpath = new Text();
            Text dstpath = new Text();
            Text dstfrom = new Text();
            Trash trash = new Trash(dstfs, conf);
            Path lastpath = null;
            boolean hasnext = dstin.next((Writable)dstpath, (Writable)dstfrom);
            while (lsrin.next((Writable)lsrpath, (Writable)NullWritable.get())) {
                int dst_cmp_lsr = dstpath.compareTo((BinaryComparable)lsrpath);
                while (hasnext && dst_cmp_lsr < 0) {
                    hasnext = dstin.next((Writable)dstpath, (Writable)dstfrom);
                    dst_cmp_lsr = dstpath.compareTo((BinaryComparable)lsrpath);
                }
                if (dst_cmp_lsr == 0) {
                    hasnext = dstin.next((Writable)dstpath, (Writable)dstfrom);
                    continue;
                }
                Path rmpath = new Path(dstroot.getPath(), lsrpath.toString());
                ++deletedPathsCount;
                if (lastpath != null && DistCpV1.isAncestorPath(lastpath, rmpath)) continue;
                if (!trash.moveToTrash(rmpath) && !dstfs.delete(rmpath, true)) {
                    throw new IOException("Failed to delete " + rmpath);
                }
                lastpath = rmpath;
            }
        }
        return deletedPathsCount;
    }

    private static boolean isAncestorPath(Path xp, Path yp) {
        String x = xp.toString();
        String y = yp.toString();
        if (!y.startsWith(x)) {
            return false;
        }
        int len = x.length();
        return y.length() == len || y.charAt(len) == '/';
    }

    private static void checkDuplication(FileSystem fs, Path file, Path sorted, Configuration conf) throws IOException {
        SequenceFile.Sorter sorter = new SequenceFile.Sorter(fs, (RawComparator)new Text.Comparator(), Text.class, Text.class, conf);
        sorter.sort(file, sorted);
        try (SequenceFile.Reader in = new SequenceFile.Reader(conf, new SequenceFile.Reader.Option[]{SequenceFile.Reader.file((Path)sorted)});){
            Text prevdst = null;
            Text curdst = new Text();
            Text prevsrc = null;
            Text cursrc = new Text();
            while (in.next((Writable)curdst, (Writable)cursrc)) {
                if (prevdst != null && curdst.equals((Object)prevdst)) {
                    throw new DuplicationException("Invalid input, there are duplicated files in the sources: " + prevsrc + ", " + cursrc);
                }
                prevdst = curdst;
                curdst = new Text();
                prevsrc = cursrc;
                cursrc = new Text();
            }
        }
    }

    public static class DuplicationException
    extends IOException {
        private static final long serialVersionUID = 1L;
        public static final int ERROR_CODE = -2;

        DuplicationException(String message) {
            super(message);
        }
    }

    static class Arguments {
        final List<Path> srcs;
        final Path basedir;
        final Path dst;
        final Path log;
        final EnumSet<Options> flags;
        final String preservedAttributes;
        final long filelimit;
        final long sizelimit;
        final String mapredSslConf;
        final boolean dryrun;

        Arguments(List<Path> srcs, Path basedir, Path dst, Path log, EnumSet<Options> flags, String preservedAttributes, long filelimit, long sizelimit, String mapredSslConf, boolean dryrun) {
            this.srcs = srcs;
            this.basedir = basedir;
            this.dst = dst;
            this.log = log;
            this.flags = flags;
            this.preservedAttributes = preservedAttributes;
            this.filelimit = filelimit;
            this.sizelimit = sizelimit;
            this.mapredSslConf = mapredSslConf;
            this.dryrun = dryrun;
            if (LOG.isTraceEnabled()) {
                LOG.trace("this = " + this);
            }
        }

        static Arguments valueOf(String[] args, Configuration conf) throws IOException {
            ArrayList<Path> srcs = new ArrayList<Path>();
            Path dst = null;
            Path log = null;
            Path basedir = null;
            EnumSet<Options> flags = EnumSet.noneOf(Options.class);
            String presevedAttributes = null;
            String mapredSslConf = null;
            long filelimit = Long.MAX_VALUE;
            long sizelimit = Long.MAX_VALUE;
            boolean dryrun = false;
            for (int idx = 0; idx < args.length; ++idx) {
                int i;
                Options[] opt = Options.values();
                for (i = 0; i < opt.length && !args[idx].startsWith(opt[i].cmd); ++i) {
                }
                if (i < opt.length) {
                    flags.add(opt[i]);
                    if (opt[i] == Options.PRESERVE_STATUS) {
                        presevedAttributes = args[idx].substring(2);
                        FileAttribute.parse(presevedAttributes);
                        continue;
                    }
                    if (opt[i] == Options.FILE_LIMIT) {
                        filelimit = Options.FILE_LIMIT.parseLong(args, ++idx);
                        continue;
                    }
                    if (opt[i] != Options.SIZE_LIMIT) continue;
                    sizelimit = Options.SIZE_LIMIT.parseLong(args, ++idx);
                    continue;
                }
                if ("-f".equals(args[idx])) {
                    if (++idx == args.length) {
                        throw new IllegalArgumentException("urilist_uri not specified in -f");
                    }
                    srcs.addAll(DistCpV1.fetchFileList(conf, new Path(args[idx])));
                    continue;
                }
                if ("-log".equals(args[idx])) {
                    if (++idx == args.length) {
                        throw new IllegalArgumentException("logdir not specified in -log");
                    }
                    log = new Path(args[idx]);
                    continue;
                }
                if ("-basedir".equals(args[idx])) {
                    if (++idx == args.length) {
                        throw new IllegalArgumentException("basedir not specified in -basedir");
                    }
                    basedir = new Path(args[idx]);
                    continue;
                }
                if ("-mapredSslConf".equals(args[idx])) {
                    if (++idx == args.length) {
                        throw new IllegalArgumentException("ssl conf file not specified in -mapredSslConf");
                    }
                    mapredSslConf = args[idx];
                    continue;
                }
                if ("-dryrun".equals(args[idx])) {
                    dryrun = true;
                    dst = new Path("/tmp/distcp_dummy_dest");
                    continue;
                }
                if ("-m".equals(args[idx])) {
                    if (++idx == args.length) {
                        throw new IllegalArgumentException("num_maps not specified in -m");
                    }
                    try {
                        conf.setInt(DistCpV1.MAX_MAPS_LABEL, Integer.parseInt(args[idx]));
                        continue;
                    }
                    catch (NumberFormatException e) {
                        throw new IllegalArgumentException("Invalid argument to -m: " + args[idx]);
                    }
                }
                if (45 == args[idx].codePointAt(0)) {
                    throw new IllegalArgumentException("Invalid switch " + args[idx]);
                }
                if (idx == args.length - 1 && (!dryrun || flags.contains((Object)Options.UPDATE))) {
                    dst = new Path(args[idx]);
                    continue;
                }
                srcs.add(new Path(args[idx]));
            }
            if (srcs.isEmpty() || dst == null) {
                throw new IllegalArgumentException("Missing " + (dst == null ? "dst path" : "src"));
            }
            boolean isOverwrite = flags.contains((Object)Options.OVERWRITE);
            boolean isUpdate = flags.contains((Object)Options.UPDATE);
            boolean isDelete = flags.contains((Object)Options.DELETE);
            boolean skipCRC = flags.contains((Object)Options.SKIPCRC);
            if (isOverwrite && isUpdate) {
                throw new IllegalArgumentException("Conflicting overwrite policies");
            }
            if (!isUpdate && skipCRC) {
                throw new IllegalArgumentException(Options.SKIPCRC.cmd + " is relevant only with the " + Options.UPDATE.cmd + " option");
            }
            if (isDelete && !isOverwrite && !isUpdate) {
                throw new IllegalArgumentException(Options.DELETE.cmd + " must be specified with " + Options.OVERWRITE + " or " + Options.UPDATE + ".");
            }
            return new Arguments(srcs, basedir, dst, log, flags, presevedAttributes, filelimit, sizelimit, mapredSslConf, dryrun);
        }

        public String toString() {
            return this.getClass().getName() + "{\n  srcs = " + this.srcs + "\n  dst = " + this.dst + "\n  log = " + this.log + "\n  flags = " + this.flags + "\n  preservedAttributes = " + this.preservedAttributes + "\n  filelimit = " + this.filelimit + "\n  sizelimit = " + this.sizelimit + "\n  mapredSslConf = " + this.mapredSslConf + "\n}";
        }
    }

    static class CopyFilesMapper
    implements Mapper<LongWritable, FilePair, WritableComparable<?>, Text> {
        private int sizeBuf = 131072;
        private FileSystem destFileSys = null;
        private boolean ignoreReadFailures;
        private boolean preserve_status;
        private EnumSet<FileAttribute> preseved;
        private boolean overwrite;
        private boolean update;
        private Path destPath = null;
        private byte[] buffer = null;
        private JobConf job;
        private boolean skipCRCCheck = false;
        private int failcount = 0;
        private int skipcount = 0;
        private int copycount = 0;

        CopyFilesMapper() {
        }

        private String getCountString() {
            return "Copied: " + this.copycount + " Skipped: " + this.skipcount + " Failed: " + this.failcount;
        }

        private void updateStatus(Reporter reporter) {
            reporter.setStatus(this.getCountString());
        }

        private boolean needsUpdate(FileStatus srcstatus, FileSystem dstfs, Path dstpath) throws IOException {
            return this.update && !DistCpV1.sameFile(srcstatus.getPath().getFileSystem((Configuration)this.job), srcstatus, dstfs, dstpath, this.skipCRCCheck);
        }

        private FSDataOutputStream create(Path f, Reporter reporter, FileStatus srcstat) throws IOException {
            if (this.destFileSys.exists(f)) {
                this.destFileSys.delete(f, false);
            }
            if (!this.preserve_status) {
                return this.destFileSys.create(f, true, this.sizeBuf, (Progressable)reporter);
            }
            FsPermission permission = this.preseved.contains((Object)FileAttribute.PERMISSION) ? srcstat.getPermission() : null;
            short replication = this.preseved.contains((Object)FileAttribute.REPLICATION) ? srcstat.getReplication() : this.destFileSys.getDefaultReplication(f);
            long blockSize = this.preseved.contains((Object)FileAttribute.BLOCK_SIZE) ? srcstat.getBlockSize() : this.destFileSys.getDefaultBlockSize(f);
            return this.destFileSys.create(f, permission, true, this.sizeBuf, replication, blockSize, (Progressable)reporter);
        }

        private boolean validateCopy(FileStatus srcstat, Path absdst) throws IOException {
            return this.destFileSys.exists(absdst) && DistCpV1.sameFile(srcstat.getPath().getFileSystem((Configuration)this.job), srcstat, this.destFileSys, absdst, this.skipCRCCheck);
        }

        void updateCopyStatus(FileStatus srcstat, Reporter reporter) {
            ++this.copycount;
            reporter.incrCounter((Enum)Counter.BYTESCOPIED, srcstat.getLen());
            reporter.incrCounter((Enum)Counter.COPY, 1L);
            this.updateStatus(reporter);
        }

        private boolean skipCopyFile(FileStatus srcstat, Path absdst, OutputCollector<WritableComparable<?>, Text> outc, Reporter reporter) throws IOException {
            if (this.destFileSys.exists(absdst) && !this.overwrite && !this.needsUpdate(srcstat, this.destFileSys, absdst)) {
                outc.collect(null, (Object)new Text("SKIP: " + srcstat.getPath()));
                ++this.skipcount;
                reporter.incrCounter((Enum)Counter.SKIP, 1L);
                this.updateStatus(reporter);
                return true;
            }
            return false;
        }

        private long doCopyFile(FileStatus srcstat, Path tmpfile, Path absdst, Reporter reporter) throws IOException {
            long bytesCopied = 0L;
            Path srcPath = srcstat.getPath();
            try (FSDataInputStream in = srcPath.getFileSystem((Configuration)this.job).open(srcPath);){
                reporter.incrCounter((Enum)Counter.BYTESEXPECTED, srcstat.getLen());
                try (FSDataOutputStream out = this.create(tmpfile, reporter, srcstat);){
                    int bytesRead;
                    LOG.info("Copying file " + srcPath + " of size " + srcstat.getLen() + " bytes...");
                    while ((bytesRead = in.read(this.buffer)) >= 0) {
                        out.write(this.buffer, 0, bytesRead);
                        reporter.setStatus(String.format("%.2f ", (double)(bytesCopied += (long)bytesRead) * 100.0 / (double)srcstat.getLen()) + absdst + " [ " + StringUtils.TraditionalBinaryPrefix.long2String((long)bytesCopied, (String)"", (int)1) + " / " + StringUtils.TraditionalBinaryPrefix.long2String((long)srcstat.getLen(), (String)"", (int)1) + " ]");
                    }
                }
            }
            return bytesCopied;
        }

        private void copy(FileStatus srcstat, Path relativedst, OutputCollector<WritableComparable<?>, Text> outc, Reporter reporter) throws IOException {
            Path dstparent;
            Path absdst = new Path(this.destPath, relativedst);
            int totfiles = this.job.getInt(DistCpV1.SRC_COUNT_LABEL, -1);
            assert (totfiles >= 0) : "Invalid file count " + totfiles;
            if (!(totfiles != 1 || this.destFileSys.exists(dstparent = absdst.getParent()) && this.destFileSys.getFileStatus(dstparent).isDirectory())) {
                absdst = dstparent;
            }
            if (srcstat.isDirectory()) {
                if (this.destFileSys.exists(absdst)) {
                    if (this.destFileSys.getFileStatus(absdst).isFile()) {
                        throw new IOException("Failed to mkdirs: " + absdst + " is a file.");
                    }
                } else if (!this.destFileSys.mkdirs(absdst)) {
                    throw new IOException("Failed to mkdirs " + absdst);
                }
                return;
            }
            if (this.skipCopyFile(srcstat, absdst, outc, reporter)) {
                return;
            }
            Path tmpfile = new Path(this.job.get(DistCpV1.TMP_DIR_LABEL), relativedst);
            long bytesCopied = this.doCopyFile(srcstat, tmpfile, absdst, reporter);
            if (bytesCopied != srcstat.getLen()) {
                throw new IOException("File size not matched: copied " + CopyFilesMapper.bytesString(bytesCopied) + " to tmpfile (=" + tmpfile + ") but expected " + CopyFilesMapper.bytesString(srcstat.getLen()) + " from " + srcstat.getPath());
            }
            if (this.destFileSys.exists(absdst) && this.destFileSys.getFileStatus(absdst).isDirectory()) {
                throw new IOException(absdst + " is a directory");
            }
            if (!this.destFileSys.mkdirs(absdst.getParent())) {
                throw new IOException("Failed to create parent dir: " + absdst.getParent());
            }
            this.rename(tmpfile, absdst);
            if (!this.validateCopy(srcstat, absdst)) {
                this.destFileSys.delete(absdst, false);
                throw new IOException("Validation of copy of file " + srcstat.getPath() + " failed.");
            }
            this.updateDestStatus(srcstat, this.destFileSys.getFileStatus(absdst));
            this.updateCopyStatus(srcstat, reporter);
        }

        private void rename(Path tmp, Path dst) throws IOException {
            try {
                if (this.destFileSys.exists(dst)) {
                    this.destFileSys.delete(dst, true);
                }
                if (!this.destFileSys.rename(tmp, dst)) {
                    throw new IOException();
                }
            }
            catch (IOException cause) {
                throw (IOException)new IOException("Fail to rename tmp file (=" + tmp + ") to destination file (=" + dst + ")").initCause(cause);
            }
        }

        private void updateDestStatus(FileStatus src, FileStatus dst) throws IOException {
            if (this.preserve_status) {
                DistCpV1.updateDestStatus(src, dst, this.preseved, this.destFileSys);
            }
        }

        static String bytesString(long b) {
            return b + " bytes (" + StringUtils.TraditionalBinaryPrefix.long2String((long)b, (String)"", (int)1) + ")";
        }

        void copyWithRetries(FileStatus srcstat, Path relativedst, OutputCollector<WritableComparable<?>, Text> out, Reporter reporter) throws IOException {
            int maxRetries = this.job.getInt(DistCpV1.FILE_RETRIES_LABEL, 3);
            boolean saveUpdate = this.update;
            for (int retryCnt = 1; retryCnt <= maxRetries; ++retryCnt) {
                try {
                    this.copy(srcstat, relativedst, out, reporter);
                    break;
                }
                catch (IOException e) {
                    LOG.warn("Copy of " + srcstat.getPath() + " failed.", (Throwable)e);
                    if (retryCnt >= maxRetries) {
                        this.update = saveUpdate;
                        throw new IOException("Copy of file failed even with " + retryCnt + " tries.", e);
                    }
                    LOG.info("Retrying copy of file " + srcstat.getPath());
                    this.update = true;
                    continue;
                }
            }
        }

        public void configure(JobConf job) {
            this.destPath = new Path(job.get(DistCpV1.DST_DIR_LABEL, "/"));
            try {
                this.destFileSys = this.destPath.getFileSystem((Configuration)job);
            }
            catch (IOException ex) {
                throw new RuntimeException("Unable to get the named file system.", ex);
            }
            this.sizeBuf = job.getInt("copy.buf.size", 131072);
            this.buffer = new byte[this.sizeBuf];
            this.ignoreReadFailures = job.getBoolean(Options.IGNORE_READ_FAILURES.propertyname, false);
            this.preserve_status = job.getBoolean(Options.PRESERVE_STATUS.propertyname, false);
            if (this.preserve_status) {
                this.preseved = FileAttribute.parse(job.get(PRESERVE_STATUS_LABEL));
            }
            this.update = job.getBoolean(Options.UPDATE.propertyname, false);
            this.overwrite = !this.update && job.getBoolean(Options.OVERWRITE.propertyname, false);
            this.skipCRCCheck = job.getBoolean(Options.SKIPCRC.propertyname, false);
            this.job = job;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void map(LongWritable key, FilePair value, OutputCollector<WritableComparable<?>, Text> out, Reporter reporter) throws IOException {
            FileStatus srcstat = value.input;
            Path relativedst = new Path(value.output);
            try {
                this.copyWithRetries(srcstat, relativedst, out, reporter);
            }
            catch (IOException e) {
                ++this.failcount;
                reporter.incrCounter((Enum)Counter.FAIL, 1L);
                this.updateStatus(reporter);
                String sfailure = "FAIL " + relativedst + " : " + StringUtils.stringifyException((Throwable)e);
                out.collect(null, (Object)new Text(sfailure));
                LOG.info(sfailure);
                if (e instanceof FileNotFoundException) {
                    String s = "Possible Cause for failure: Either the filesystem " + srcstat.getPath().getFileSystem((Configuration)this.job) + " is not accessible or the file is deleted";
                    LOG.error(s);
                    out.collect(null, (Object)new Text(s));
                }
                try {
                    for (int i = 0; i < 3; ++i) {
                        block12: {
                            try {
                                Path tmp = new Path(this.job.get(DistCpV1.TMP_DIR_LABEL), relativedst);
                                if (!this.destFileSys.delete(tmp, true)) break block12;
                                break;
                            }
                            catch (Throwable ex) {
                                LOG.debug("Ignoring cleanup exception", ex);
                            }
                        }
                        this.updateStatus(reporter);
                        Thread.sleep(3000L);
                    }
                }
                catch (InterruptedException inte) {
                    throw (IOException)new IOException().initCause(inte);
                }
            }
            finally {
                this.updateStatus(reporter);
            }
        }

        public void close() throws IOException {
            if (0 == this.failcount || this.ignoreReadFailures) {
                return;
            }
            throw new IOException(this.getCountString());
        }
    }

    static class CopyInputFormat
    implements InputFormat<Text, Text> {
        CopyInputFormat() {
        }

        public InputSplit[] getSplits(JobConf job, int numSplits) throws IOException {
            int cnfiles = job.getInt(DistCpV1.SRC_COUNT_LABEL, -1);
            long cbsize = job.getLong(DistCpV1.TOTAL_SIZE_LABEL, -1L);
            String srcfilelist = job.get(DistCpV1.SRC_LIST_LABEL, "");
            if (cnfiles < 0 || cbsize < 0L || "".equals(srcfilelist)) {
                throw new RuntimeException("Invalid metadata: #files(" + cnfiles + ") total_size(" + cbsize + ") listuri(" + srcfilelist + ")");
            }
            Path src = new Path(srcfilelist);
            FileSystem fs = src.getFileSystem((Configuration)job);
            FileStatus srcst = fs.getFileStatus(src);
            ArrayList<FileSplit> splits = new ArrayList<FileSplit>(numSplits);
            LongWritable key = new LongWritable();
            FilePair value = new FilePair();
            long targetsize = cbsize / (long)numSplits;
            long pos = 0L;
            long last = 0L;
            long acc = 0L;
            long cbrem = srcst.getLen();
            try (SequenceFile.Reader sl = new SequenceFile.Reader((Configuration)job, new SequenceFile.Reader.Option[]{SequenceFile.Reader.file((Path)src)});){
                while (sl.next((Writable)key, (Writable)value)) {
                    if (acc + key.get() > targetsize && acc != 0L) {
                        long splitsize = last - pos;
                        splits.add(new FileSplit(src, pos, splitsize, (String[])null));
                        cbrem -= splitsize;
                        pos = last;
                        acc = 0L;
                    }
                    acc += key.get();
                    last = sl.getPosition();
                }
            }
            if (cbrem != 0L) {
                splits.add(new FileSplit(src, pos, cbrem, (String[])null));
            }
            return (InputSplit[])splits.toArray(new FileSplit[splits.size()]);
        }

        public RecordReader<Text, Text> getRecordReader(InputSplit split, JobConf job, Reporter reporter) throws IOException {
            return new SequenceFileRecordReader((Configuration)job, (FileSplit)split);
        }
    }

    static class FilePair
    implements Writable {
        FileStatus input = new FileStatus();
        String output;

        FilePair() {
        }

        FilePair(FileStatus input, String output) {
            this.input = input;
            this.output = output;
        }

        public void readFields(DataInput in) throws IOException {
            this.input.readFields(in);
            this.output = Text.readString((DataInput)in);
        }

        public void write(DataOutput out) throws IOException {
            this.input.write(out);
            Text.writeString((DataOutput)out, (String)this.output);
        }

        public String toString() {
            return this.input + " : " + this.output;
        }
    }

    static enum FileAttribute {
        BLOCK_SIZE,
        REPLICATION,
        USER,
        GROUP,
        PERMISSION,
        TIMES;

        final char symbol = StringUtils.toLowerCase((String)this.toString()).charAt(0);

        static EnumSet<FileAttribute> parse(String s) {
            if (s == null || s.length() == 0) {
                return EnumSet.allOf(FileAttribute.class);
            }
            EnumSet<FileAttribute> set = EnumSet.noneOf(FileAttribute.class);
            FileAttribute[] attributes = FileAttribute.values();
            for (char c : s.toCharArray()) {
                int i;
                for (i = 0; i < attributes.length && c != attributes[i].symbol; ++i) {
                }
                if (i < attributes.length) {
                    if (set.contains((Object)attributes[i])) {
                        throw new IllegalArgumentException("There are more than one '" + attributes[i].symbol + "' in " + s);
                    }
                } else {
                    throw new IllegalArgumentException("'" + c + "' in " + s + " is undefined.");
                }
                set.add(attributes[i]);
            }
            return set;
        }
    }

    static enum Options {
        DELETE("-delete", "distcp.delete"),
        FILE_LIMIT("-filelimit", "distcp.limit.file"),
        SIZE_LIMIT("-sizelimit", "distcp.limit.size"),
        IGNORE_READ_FAILURES("-i", "distcp.ignore.read.failures"),
        PRESERVE_STATUS("-p", "distcp.preserve.status"),
        OVERWRITE("-overwrite", "distcp.overwrite.always"),
        UPDATE("-update", "distcp.overwrite.ifnewer"),
        SKIPCRC("-skipcrccheck", "distcp.skip.crc.check");

        final String cmd;
        final String propertyname;

        private Options(String cmd, String propertyname) {
            this.cmd = cmd;
            this.propertyname = propertyname;
        }

        private long parseLong(String[] args, int offset) {
            if (offset == args.length) {
                throw new IllegalArgumentException("<n> not specified in " + this.cmd);
            }
            long n = StringUtils.TraditionalBinaryPrefix.string2long((String)args[offset]);
            if (n <= 0L) {
                throw new IllegalArgumentException("n = " + n + " <= 0 in " + this.cmd);
            }
            return n;
        }
    }

    static enum Counter {
        COPY,
        SKIP,
        FAIL,
        BYTESCOPIED,
        BYTESEXPECTED;

    }
}

