/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ress;

import java.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.IOException;
import java.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.OutputStream;

import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceStability;

/**
 * A {@link org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ress.CompressorStream} which works
 * with 'block-based' based org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ression algorithms, as opposed to 
 * 'stream-based' org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ression algorithms.
 *
 * It should be noted that this wrapper does not guarantee that blocks will
 * be sized for the org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor. If the
 * {@link org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ress.Compressor} requires buffering to
 * effect meaningful org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ression, it is responsible for it.
 */
@InterfaceAudience.Public
@InterfaceStability.Evolving
public class BlockCompressorStream extends CompressorStream {

  // The 'maximum' size of input data to be org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressed, to account
  // for the overhead of the org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ression algorithm.
  private final int MAX_INPUT_SIZE;

  /**
   * Create a {@link BlockCompressorStream}.
   * 
   * @param out stream
   * @param org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor to be used
   * @param bufferSize size of buffer
   * @param org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressionOverhead maximum 'overhead' of the org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ression 
   *                            algorithm with given bufferSize
   */
  public BlockCompressorStream(OutputStream out, Compressor org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor, 
                               int bufferSize, int org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressionOverhead) {
    super(out, org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor, bufferSize);
    MAX_INPUT_SIZE = bufferSize - org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressionOverhead;
  }

  /**
   * Create a {@link BlockCompressorStream} with given output-stream and 
   * org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.
   * Use default of 512 as bufferSize and org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressionOverhead of 
   * (1% of bufferSize + 12 bytes) =  18 bytes (zlib algorithm).
   * 
   * @param out stream
   * @param org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor to be used
   */
  public BlockCompressorStream(OutputStream out, Compressor org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor) {
    this(out, org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor, 512, 18);
  }

  /**
   * Write the data provided to the org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ression codec, org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressing no more
   * than the buffer size less the org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ression overhead as specified during
   * construction for each block.
   *
   * Each block contains the uncompressed length for the block, followed by
   * one or more length-prefixed blocks of org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressed data.
   */
  @Override
  public void write(byte[] b, int off, int len) throws IOException {
    // Sanity checks
    if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.finished()) {
      throw new IOException("write beyond end of stream");
    }
    if (b == null) {
      throw new NullPointerException();
    } else if ((off < 0) || (off > b.length) || (len < 0) ||
               ((off + len) > b.length)) {
      throw new IndexOutOfBoundsException();
    } else if (len == 0) {
      return;
    }

    long limlen = org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.getBytesRead();
    if (len + limlen > MAX_INPUT_SIZE && limlen > 0) {
      // Adding this segment would exceed the maximum size.
      // Flush data if we have it.
      finish();
      org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.reset();
    }

    if (len > MAX_INPUT_SIZE) {
      // The data we're given exceeds the maximum size. Any data
      // we had have been flushed, so we write out this chunk in segments
      // not exceeding the maximum size until it is exhausted.
      rawWriteInt(len);
      do {
        int bufLen = Math.min(len, MAX_INPUT_SIZE);
        
        org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.setInput(b, off, bufLen);
        org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.finish();
        while (!org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.finished()) {
          org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ress();
        }
        org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.reset();
        off += bufLen;
        len -= bufLen;
      } while (len > 0);
      return;
    }

    // Give data to the org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor
    org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.setInput(b, off, len);
    if (!org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.needsInput()) {
      // org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor buffer size might be smaller than the maximum
      // size, so we permit it to flush if required.
      rawWriteInt((int)org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.getBytesRead());
      do {
        org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ress();
      } while (!org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.needsInput());
    }
  }

  @Override
  public void finish() throws IOException {
    if (!org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.finished()) {
      rawWriteInt((int)org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.getBytesRead());
      org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.finish();
      while (!org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.finished()) {
        org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ress();
      }
    }
  }

  @Override
  protected void org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ress() throws IOException {
    int len = org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressor.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ress(buffer, 0, buffer.length);
    if (len > 0) {
      // Write out the org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.ressed chunk
      rawWriteInt(len);
      out.write(buffer, 0, len);
    }
  }
  
  private void rawWriteInt(int v) throws IOException {
    out.write((v >>> 24) & 0xFF);
    out.write((v >>> 16) & 0xFF);
    out.write((v >>>  8) & 0xFF);
    out.write((v >>>  0) & 0xFF);
  }

}
