/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.

import java.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.DataInput;
import java.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.DataOutput;
import java.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.IOException;
import java.lang.reflect.Array;
import java.util.HashMap;
import java.util.Map;

import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceStability;

/**
 * This is a wrapper class.  It wraps a Writable implementation around
 * an array of primitives (e.g., int[], long[], etc.), with optimized 
 * wire format, and without creating new objects per element.
 * 
 * This is a wrapper class only; it does not make a copy of the 
 * underlying array.
 */
@InterfaceAudience.Public
@InterfaceStability.Stable
public class ArrayPrimitiveWritable implements Writable {
  
  //org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType is determined from the org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onent type of the value array 
  //during a "set" operation.  It must be primitive.
  private Class<?> org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType = null; 
  //declaredComponentType need not be declared, but if you do (by using the
  //ArrayPrimitiveWritable(Class<?>) constructor), it will provide typechecking
  //for all "set" operations.
  private Class<?> declaredComponentType = null;
  private int length;
  private Object value; //must be an array of <org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType>[length]
  
  private static final Map<String, Class<?>> PRIMITIVE_NAMES = 
    new HashMap<String, Class<?>>(16);
  static {
    PRIMITIVE_NAMES.put(boolean.class.getName(), boolean.class);
    PRIMITIVE_NAMES.put(byte.class.getName(), byte.class);
    PRIMITIVE_NAMES.put(char.class.getName(), char.class);
    PRIMITIVE_NAMES.put(short.class.getName(), short.class);
    PRIMITIVE_NAMES.put(int.class.getName(), int.class);
    PRIMITIVE_NAMES.put(long.class.getName(), long.class);
    PRIMITIVE_NAMES.put(float.class.getName(), float.class);
    PRIMITIVE_NAMES.put(double.class.getName(), double.class);
  }
  
  private static Class<?> getPrimitiveClass(String className) {
    return PRIMITIVE_NAMES.get(className);
  }
  
  private static void checkPrimitive(Class<?> org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType) {
    if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == null) { 
      throw new HadoopIllegalArgumentException("null org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onent type not allowed"); 
    }
    if (! PRIMITIVE_NAMES.containsKey(org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType.getName())) {
      throw new HadoopIllegalArgumentException("input array org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onent type "
          + org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType.getName() + " is not a candidate primitive type");
    }
  }
  
  private void checkDeclaredComponentType(Class<?> org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType) {
    if ((declaredComponentType != null) 
        && (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType != declaredComponentType)) {
      throw new HadoopIllegalArgumentException("input array org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onent type "
          + org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType.getName() + " does not match declared type "
          + declaredComponentType.getName());     
    }
  }
  
  private static void checkArray(Object value) {
    if (value == null) { 
      throw new HadoopIllegalArgumentException("null value not allowed"); 
    }
    if (! value.getClass().isArray()) {
      throw new HadoopIllegalArgumentException("non-array value of class "
          + value.getClass() + " not allowed");             
    }
  }
  
  /**
   * Construct an empty instance, for use during Writable read
   */
  public ArrayPrimitiveWritable() {
    //empty constructor
  }
  
  /**
   * Construct an instance of known type but no value yet
   * for use with type-specific wrapper classes.
   *
   * @param org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType.
   */
  public ArrayPrimitiveWritable(Class<?> org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType) {
    checkPrimitive(org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType);
    this.declaredComponentType = org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType;
  }
  
  /**
   * Wrap an existing array of primitives
   * @param value - array of primitives
   */
  public ArrayPrimitiveWritable(Object value) {
    set(value);
  }
  
  /**
   * Get the original array.  
   * Client must cast it back to type org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType[]
   * (or may use type-specific wrapper classes).
   * @return - original array as Object
   */
  public Object get() { return value; }
  
  public Class<?> getComponentType() { return org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType; }
  
  public Class<?> getDeclaredComponentType() { return declaredComponentType; }
  
  public boolean isDeclaredComponentType(Class<?> org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType) {
    return org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == declaredComponentType;
  }
  
  public void set(Object value) {
    checkArray(value);
    Class<?> org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType = value.getClass().getComponentType();
    checkPrimitive(org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType);
    checkDeclaredComponentType(org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType);
    this.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType = org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType;
    this.value = value;
    this.length = Array.getLength(value);
  }
  
  /**
   * Do not use this class.
   * This is an internal class, purely for ObjectWritable to use as
   * a label class for transparent conversions of arrays of primitives
   * during wire protocol reads and writes.
   */
  static class Internal extends ArrayPrimitiveWritable {
    Internal() {             //use for reads
      super(); 
    }
    
    Internal(Object value) { //use for writes
      super(value);
    }
  } //end Internal subclass declaration

  /* 
   * @see org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.Writable#write(java.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.DataOutput)
   */
  @Override
  @SuppressWarnings("deprecation")
  public void write(DataOutput out) throws IOException {
    // write org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType 
    UTF8.writeString(out, org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType.getName());      
    // write length
    out.writeInt(length);

    // do the inner loop.  Walk the decision tree only once.
    if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Boolean.TYPE) {          // boolean
      writeBooleanArray(out);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Character.TYPE) { // char
      writeCharArray(out);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Byte.TYPE) {      // byte
      writeByteArray(out);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Short.TYPE) {     // short
      writeShortArray(out);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Integer.TYPE) {   // int
      writeIntArray(out);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Long.TYPE) {      // long
      writeLongArray(out);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Float.TYPE) {     // float
      writeFloatArray(out);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Double.TYPE) {    // double
      writeDoubleArray(out);
    } else {
      throw new IOException("Component type " + org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType.toString()
          + " is set as the output type, but no encoding is implemented for this type.");
    }
  }

  /* 
   * @see org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.Writable#readFields(java.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.io.DataInput)
   */
  @Override
  public void readFields(DataInput in) throws IOException {
    
    // read and set the org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onent type of the array
    @SuppressWarnings("deprecation")
    String className = UTF8.readString(in);
    Class<?> org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType = getPrimitiveClass(className);
    if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == null) {
      throw new IOException("encoded array org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onent type "
          + className + " is not a candidate primitive type");
    }
    checkDeclaredComponentType(org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType);
    this.org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType = org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType;
  
    // read and set the length of the array
    int length = in.readInt();
    if (length < 0) {
      throw new IOException("encoded array length is negative " + length);
    }
    this.length = length;
    
    // construct and read in the array
    value = Array.newInstance(org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType, length);

    // do the inner loop.  Walk the decision tree only once.
    if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Boolean.TYPE) {             // boolean
      readBooleanArray(in);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Character.TYPE) {    // char
      readCharArray(in);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Byte.TYPE) {         // byte
      readByteArray(in);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Short.TYPE) {        // short
      readShortArray(in);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Integer.TYPE) {      // int
      readIntArray(in);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Long.TYPE) {         // long
      readLongArray(in);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Float.TYPE) {        // float
      readFloatArray(in);
    } else if (org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType == Double.TYPE) {       // double
      readDoubleArray(in);
    } else {
      throw new IOException("Encoded type " + className
          + " converted to valid org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onent type " + org.apache.hadoop.shaded.org.apache.hadoop.shaded.org.apache.hadoop.shaded.com.onentType.toString()
          + " but no encoding is implemented for this type.");
    }
  }
  
  //For efficient implementation, there's no way around
  //the following massive code duplication.
  
  private void writeBooleanArray(DataOutput out) throws IOException {
    boolean[] v = (boolean[]) value;
    for (int i = 0; i < length; i++)
      out.writeBoolean(v[i]);
  }
  
  private void writeCharArray(DataOutput out) throws IOException {
    char[] v = (char[]) value;
    for (int i = 0; i < length; i++)
      out.writeChar(v[i]);
  }
  
  private void writeByteArray(DataOutput out) throws IOException {
    out.write((byte[]) value, 0, length);
  }
  
  private void writeShortArray(DataOutput out) throws IOException {
    short[] v = (short[]) value;
    for (int i = 0; i < length; i++)
      out.writeShort(v[i]);
  }
  
  private void writeIntArray(DataOutput out) throws IOException {
    int[] v = (int[]) value;
    for (int i = 0; i < length; i++)
      out.writeInt(v[i]);
  }
  
  private void writeLongArray(DataOutput out) throws IOException {
    long[] v = (long[]) value;
    for (int i = 0; i < length; i++)
      out.writeLong(v[i]);
  }
  
  private void writeFloatArray(DataOutput out) throws IOException {
    float[] v = (float[]) value;
    for (int i = 0; i < length; i++)
      out.writeFloat(v[i]);
  }
  
  private void writeDoubleArray(DataOutput out) throws IOException {
    double[] v = (double[]) value;
    for (int i = 0; i < length; i++)
      out.writeDouble(v[i]);
  }

  private void readBooleanArray(DataInput in) throws IOException {
    boolean[] v = (boolean[]) value;
    for (int i = 0; i < length; i++)
      v[i] = in.readBoolean(); 
  }
  
  private void readCharArray(DataInput in) throws IOException {
    char[] v = (char[]) value;
    for (int i = 0; i < length; i++)
      v[i] = in.readChar(); 
  }
  
  private void readByteArray(DataInput in) throws IOException {
    in.readFully((byte[]) value, 0, length);
  }
  
  private void readShortArray(DataInput in) throws IOException {
    short[] v = (short[]) value;
    for (int i = 0; i < length; i++)
      v[i] = in.readShort(); 
  }
  
  private void readIntArray(DataInput in) throws IOException {
    int[] v = (int[]) value;
    for (int i = 0; i < length; i++)
      v[i] = in.readInt(); 
  }
  
  private void readLongArray(DataInput in) throws IOException {
    long[] v = (long[]) value;
    for (int i = 0; i < length; i++)
      v[i] = in.readLong(); 
  }
  
  private void readFloatArray(DataInput in) throws IOException {
    float[] v = (float[]) value;
    for (int i = 0; i < length; i++)
      v[i] = in.readFloat(); 
  }
  
  private void readDoubleArray(DataInput in) throws IOException {
    double[] v = (double[]) value;
    for (int i = 0; i < length; i++)
      v[i] = in.readDouble(); 
  }
}

