/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azure;

import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.EnumSet;
import java.util.Random;
import java.util.concurrent.Callable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FutureDataInputStreamBuilder;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.azure.AzureBlobStorageTestAccount;
import org.apache.hadoop.fs.azure.NativeAzureFileSystem;
import org.apache.hadoop.fs.azure.integration.AbstractAzureScaleTest;
import org.apache.hadoop.fs.azure.integration.AzureTestUtils;
import org.apache.hadoop.fs.contract.ContractTestUtils;
import org.apache.hadoop.test.LambdaTestUtils;
import org.junit.Assume;
import org.junit.FixMethodOrder;
import org.junit.Test;
import org.junit.runners.MethodSorters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@FixMethodOrder(value=MethodSorters.NAME_ASCENDING)
public class ITestBlockBlobInputStream
extends AbstractAzureScaleTest {
    private static final Logger LOG = LoggerFactory.getLogger(ITestBlockBlobInputStream.class);
    private static final int KILOBYTE = 1024;
    private static final int MEGABYTE = 0x100000;
    private static final int TEST_FILE_SIZE = 0x600000;
    private static final Path TEST_FILE_PATH = new Path("TestBlockBlobInputStream.txt");
    private AzureBlobStorageTestAccount accountUsingInputStreamV1;
    private AzureBlobStorageTestAccount accountUsingInputStreamV2;
    private long testFileLength;
    private FileStatus testFileStatus;
    private Path hugefile;

    @Override
    public void setUp() throws Exception {
        super.setUp();
        Configuration conf = new Configuration();
        conf.setInt("fs.azure.input.stream.version.for.internal.use.only", 1);
        this.accountUsingInputStreamV1 = AzureBlobStorageTestAccount.create("testblockblobinputstream", EnumSet.of(AzureBlobStorageTestAccount.CreateOptions.CreateContainer), conf, true);
        this.accountUsingInputStreamV2 = AzureBlobStorageTestAccount.create("testblockblobinputstream", EnumSet.noneOf(AzureBlobStorageTestAccount.CreateOptions.class), null, true);
        Assume.assumeNotNull((Object[])new Object[]{this.accountUsingInputStreamV1});
        Assume.assumeNotNull((Object[])new Object[]{this.accountUsingInputStreamV2});
        this.hugefile = this.fs.makeQualified(TEST_FILE_PATH);
        try {
            this.testFileStatus = this.fs.getFileStatus(TEST_FILE_PATH);
            this.testFileLength = this.testFileStatus.getLen();
        }
        catch (FileNotFoundException e) {
            this.testFileLength = 0L;
        }
    }

    @Override
    protected AzureBlobStorageTestAccount createTestAccount() throws Exception {
        Configuration conf = new Configuration();
        conf.setInt("fs.azure.input.stream.version.for.internal.use.only", 1);
        this.accountUsingInputStreamV1 = AzureBlobStorageTestAccount.create("testblockblobinputstream", EnumSet.of(AzureBlobStorageTestAccount.CreateOptions.CreateContainer), conf, true);
        this.accountUsingInputStreamV2 = AzureBlobStorageTestAccount.create("testblockblobinputstream", EnumSet.noneOf(AzureBlobStorageTestAccount.CreateOptions.class), null, true);
        Assume.assumeNotNull((Object[])new Object[]{this.accountUsingInputStreamV1});
        Assume.assumeNotNull((Object[])new Object[]{this.accountUsingInputStreamV2});
        return this.accountUsingInputStreamV1;
    }

    private void createTestFileAndSetLength() throws IOException {
        NativeAzureFileSystem fs = this.accountUsingInputStreamV1.getFileSystem();
        if (fs.exists(TEST_FILE_PATH)) {
            this.testFileStatus = fs.getFileStatus(TEST_FILE_PATH);
            this.testFileLength = this.testFileStatus.getLen();
            LOG.info("Reusing test file: {}", (Object)this.testFileStatus);
            return;
        }
        int sizeOfAlphabet = 26;
        byte[] buffer = new byte[26624];
        int character = 97;
        for (int i = 0; i < buffer.length; ++i) {
            buffer[i] = (byte)character;
            character = (char)(character == 122 ? 97 : (char)(character + 1));
        }
        LOG.info("Creating test file {} of size: {}", (Object)TEST_FILE_PATH, (Object)0x600000);
        ContractTestUtils.NanoTimer timer = new ContractTestUtils.NanoTimer();
        try (FSDataOutputStream outputStream = fs.create(TEST_FILE_PATH);){
            for (int bytesWritten = 0; bytesWritten < 0x600000; bytesWritten += buffer.length) {
                outputStream.write(buffer);
            }
            LOG.info("Closing stream {}", (Object)outputStream);
            ContractTestUtils.NanoTimer closeTimer = new ContractTestUtils.NanoTimer();
            outputStream.close();
            closeTimer.end("time to close() output stream", new Object[0]);
        }
        timer.end("time to write %d KB", new Object[]{6144});
        this.testFileLength = fs.getFileStatus(TEST_FILE_PATH).getLen();
    }

    void assumeHugeFileExists() throws IOException {
        ContractTestUtils.assertPathExists((FileSystem)this.fs, (String)"huge file not created", (Path)this.hugefile);
        FileStatus status = this.fs.getFileStatus(this.hugefile);
        ContractTestUtils.assertIsFile((Path)this.hugefile, (FileStatus)status);
        ITestBlockBlobInputStream.assertTrue((String)("File " + this.hugefile + " is empty"), (status.getLen() > 0L ? 1 : 0) != 0);
    }

    private static double toMbps(long bytes, long milliseconds) {
        return (double)bytes / 1000.0 * 8.0 / (double)milliseconds;
    }

    @Test
    public void test_0100_CreateHugeFile() throws IOException {
        this.createTestFileAndSetLength();
    }

    @Test
    public void test_0200_BasicReadTest() throws Exception {
        this.assumeHugeFileExists();
        try (FSDataInputStream inputStreamV1 = this.accountUsingInputStreamV1.getFileSystem().open(TEST_FILE_PATH);
             FSDataInputStream inputStreamV2 = this.accountUsingInputStreamV2.getFileSystem().open(TEST_FILE_PATH);){
            byte[] bufferV1 = new byte[0x300000];
            byte[] bufferV2 = new byte[bufferV1.length];
            inputStreamV1.seek(0x500000L);
            int numBytesReadV1 = inputStreamV1.read(bufferV1, 0, 1024);
            ITestBlockBlobInputStream.assertEquals((long)1024L, (long)numBytesReadV1);
            inputStreamV2.seek(0x500000L);
            int numBytesReadV2 = inputStreamV2.read(bufferV2, 0, 1024);
            ITestBlockBlobInputStream.assertEquals((long)1024L, (long)numBytesReadV2);
            ITestBlockBlobInputStream.assertArrayEquals((byte[])bufferV1, (byte[])bufferV2);
            int len = 0x100000;
            int offset = bufferV1.length - len;
            inputStreamV1.seek(0x300000L);
            numBytesReadV1 = inputStreamV1.read(bufferV1, offset, len);
            ITestBlockBlobInputStream.assertEquals((long)len, (long)numBytesReadV1);
            inputStreamV2.seek(0x300000L);
            numBytesReadV2 = inputStreamV2.read(bufferV2, offset, len);
            ITestBlockBlobInputStream.assertEquals((long)len, (long)numBytesReadV2);
            ITestBlockBlobInputStream.assertArrayEquals((byte[])bufferV1, (byte[])bufferV2);
        }
    }

    @Test
    public void test_0201_RandomReadTest() throws Exception {
        this.assumeHugeFileExists();
        try (FSDataInputStream inputStreamV1 = this.accountUsingInputStreamV1.getFileSystem().open(TEST_FILE_PATH);
             FSDataInputStream inputStreamV2 = this.accountUsingInputStreamV2.getFileSystem().open(TEST_FILE_PATH);){
            int bufferSize = 4096;
            byte[] bufferV1 = new byte[4096];
            byte[] bufferV2 = new byte[bufferV1.length];
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, bufferV1, bufferV2);
            inputStreamV1.seek(0L);
            inputStreamV2.seek(0L);
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, bufferV1, bufferV2);
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, bufferV1, bufferV2);
            int seekPosition = 2048;
            inputStreamV1.seek((long)seekPosition);
            inputStreamV2.seek((long)seekPosition);
            inputStreamV1.seek(0L);
            inputStreamV2.seek(0L);
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, bufferV1, bufferV2);
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, bufferV1, bufferV2);
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, bufferV1, bufferV2);
            seekPosition = 5120;
            inputStreamV1.seek((long)seekPosition);
            inputStreamV2.seek((long)seekPosition);
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, bufferV1, bufferV2);
            seekPosition = 10240;
            inputStreamV1.seek((long)seekPosition);
            inputStreamV2.seek((long)seekPosition);
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, bufferV1, bufferV2);
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, bufferV1, bufferV2);
            seekPosition = 0x401000;
            inputStreamV1.seek((long)seekPosition);
            inputStreamV2.seek((long)seekPosition);
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, bufferV1, bufferV2);
        }
    }

    private void verifyConsistentReads(FSDataInputStream inputStreamV1, FSDataInputStream inputStreamV2, byte[] bufferV1, byte[] bufferV2) throws IOException {
        int size = bufferV1.length;
        int numBytesReadV1 = inputStreamV1.read(bufferV1, 0, size);
        ITestBlockBlobInputStream.assertEquals((String)"Bytes read from V1 stream", (long)size, (long)numBytesReadV1);
        int numBytesReadV2 = inputStreamV2.read(bufferV2, 0, size);
        ITestBlockBlobInputStream.assertEquals((String)"Bytes read from V2 stream", (long)size, (long)numBytesReadV2);
        ITestBlockBlobInputStream.assertArrayEquals((String)"Mismatch in read data", (byte[])bufferV1, (byte[])bufferV2);
    }

    @Test
    public void test_202_PosReadTest() throws Exception {
        this.assumeHugeFileExists();
        FutureDataInputStreamBuilder builder = this.accountUsingInputStreamV2.getFileSystem().openFile(TEST_FILE_PATH);
        builder.opt("fs.azure.block.blob.buffered.pread.disable", true);
        try (FSDataInputStream inputStreamV1 = this.accountUsingInputStreamV1.getFileSystem().open(TEST_FILE_PATH);
             FSDataInputStream inputStreamV2 = this.accountUsingInputStreamV2.getFileSystem().open(TEST_FILE_PATH);
             FSDataInputStream inputStreamV2NoBuffer = (FSDataInputStream)builder.build().get();){
            int bufferSize = 4096;
            byte[] bufferV1 = new byte[4096];
            byte[] bufferV2 = new byte[4096];
            byte[] bufferV2NoBuffer = new byte[4096];
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, inputStreamV2NoBuffer, 0, bufferV1, bufferV2, bufferV2NoBuffer);
            int pos = 2048;
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, inputStreamV2NoBuffer, pos, bufferV1, bufferV2, bufferV2NoBuffer);
            pos = 10240;
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, inputStreamV2NoBuffer, pos, bufferV1, bufferV2, bufferV2NoBuffer);
            pos = 0x401000;
            this.verifyConsistentReads(inputStreamV1, inputStreamV2, inputStreamV2NoBuffer, pos, bufferV1, bufferV2, bufferV2NoBuffer);
        }
    }

    private void verifyConsistentReads(FSDataInputStream inputStreamV1, FSDataInputStream inputStreamV2, FSDataInputStream inputStreamV2NoBuffer, int pos, byte[] bufferV1, byte[] bufferV2, byte[] bufferV2NoBuffer) throws IOException {
        int size = bufferV1.length;
        int numBytesReadV1 = inputStreamV1.read((long)pos, bufferV1, 0, size);
        ITestBlockBlobInputStream.assertEquals((String)"Bytes read from V1 stream", (long)size, (long)numBytesReadV1);
        int numBytesReadV2 = inputStreamV2.read((long)pos, bufferV2, 0, size);
        ITestBlockBlobInputStream.assertEquals((String)"Bytes read from V2 stream", (long)size, (long)numBytesReadV2);
        int numBytesReadV2NoBuffer = inputStreamV2NoBuffer.read((long)pos, bufferV2NoBuffer, 0, size);
        ITestBlockBlobInputStream.assertEquals((String)"Bytes read from V2 stream (buffered pread disabled)", (long)size, (long)numBytesReadV2NoBuffer);
        ITestBlockBlobInputStream.assertArrayEquals((String)"Mismatch in read data", (byte[])bufferV1, (byte[])bufferV2);
        ITestBlockBlobInputStream.assertArrayEquals((String)"Mismatch in read data", (byte[])bufferV2, (byte[])bufferV2NoBuffer);
    }

    @Test
    public void test_0301_MarkSupportedV1() throws IOException {
        this.validateMarkSupported((FileSystem)this.accountUsingInputStreamV1.getFileSystem());
    }

    @Test
    public void test_0302_MarkSupportedV2() throws IOException {
        this.validateMarkSupported((FileSystem)this.accountUsingInputStreamV1.getFileSystem());
    }

    private void validateMarkSupported(FileSystem fs) throws IOException {
        this.assumeHugeFileExists();
        try (FSDataInputStream inputStream = fs.open(TEST_FILE_PATH);){
            ITestBlockBlobInputStream.assertTrue((String)"mark is not supported", (boolean)inputStream.markSupported());
        }
    }

    @Test
    public void test_0303_MarkAndResetV1() throws Exception {
        this.validateMarkAndReset((FileSystem)this.accountUsingInputStreamV1.getFileSystem());
    }

    @Test
    public void test_0304_MarkAndResetV2() throws Exception {
        this.validateMarkAndReset((FileSystem)this.accountUsingInputStreamV2.getFileSystem());
    }

    private void validateMarkAndReset(FileSystem fs) throws Exception {
        this.assumeHugeFileExists();
        try (final FSDataInputStream inputStream = fs.open(TEST_FILE_PATH);){
            inputStream.mark(1023);
            byte[] buffer = new byte[1024];
            int bytesRead = inputStream.read(buffer);
            ITestBlockBlobInputStream.assertEquals((long)buffer.length, (long)bytesRead);
            inputStream.reset();
            ITestBlockBlobInputStream.assertEquals((String)"rest -> pos 0", (long)0L, (long)inputStream.getPos());
            inputStream.mark(8191);
            buffer = new byte[8192];
            bytesRead = inputStream.read(buffer);
            ITestBlockBlobInputStream.assertEquals((long)buffer.length, (long)bytesRead);
            LambdaTestUtils.intercept(IOException.class, (String)"Resetting to invalid mark", (Callable)new Callable<FSDataInputStream>(){

                @Override
                public FSDataInputStream call() throws Exception {
                    inputStream.reset();
                    return inputStream;
                }
            });
        }
    }

    @Test
    public void test_0305_SeekToNewSourceV1() throws IOException {
        this.validateSeekToNewSource((FileSystem)this.accountUsingInputStreamV1.getFileSystem());
    }

    @Test
    public void test_0306_SeekToNewSourceV2() throws IOException {
        this.validateSeekToNewSource((FileSystem)this.accountUsingInputStreamV2.getFileSystem());
    }

    private void validateSeekToNewSource(FileSystem fs) throws IOException {
        this.assumeHugeFileExists();
        try (FSDataInputStream inputStream = fs.open(TEST_FILE_PATH);){
            ITestBlockBlobInputStream.assertFalse((boolean)inputStream.seekToNewSource(0L));
        }
    }

    @Test
    public void test_0307_SkipBoundsV1() throws Exception {
        this.validateSkipBounds((FileSystem)this.accountUsingInputStreamV1.getFileSystem());
    }

    @Test
    public void test_0308_SkipBoundsV2() throws Exception {
        this.validateSkipBounds((FileSystem)this.accountUsingInputStreamV2.getFileSystem());
    }

    private void validateSkipBounds(FileSystem fs) throws Exception {
        this.assumeHugeFileExists();
        try (final FSDataInputStream inputStream = fs.open(TEST_FILE_PATH);){
            ContractTestUtils.NanoTimer timer = new ContractTestUtils.NanoTimer();
            long skipped = inputStream.skip(-1L);
            ITestBlockBlobInputStream.assertEquals((long)0L, (long)skipped);
            skipped = inputStream.skip(0L);
            ITestBlockBlobInputStream.assertEquals((long)0L, (long)skipped);
            ITestBlockBlobInputStream.assertTrue((this.testFileLength > 0L ? 1 : 0) != 0);
            skipped = inputStream.skip(this.testFileLength);
            ITestBlockBlobInputStream.assertEquals((long)this.testFileLength, (long)skipped);
            LambdaTestUtils.intercept(EOFException.class, (Callable)new Callable<Long>(){

                @Override
                public Long call() throws Exception {
                    return inputStream.skip(1L);
                }
            });
            long elapsedTimeMs = timer.elapsedTimeMs();
            ITestBlockBlobInputStream.assertTrue((String)String.format("There should not be any network I/O (elapsedTimeMs=%1$d).", elapsedTimeMs), (elapsedTimeMs < 20L ? 1 : 0) != 0);
        }
    }

    @Test
    public void test_0309_SeekBoundsV1() throws Exception {
        this.validateSeekBounds((FileSystem)this.accountUsingInputStreamV1.getFileSystem());
    }

    @Test
    public void test_0310_SeekBoundsV2() throws Exception {
        this.validateSeekBounds((FileSystem)this.accountUsingInputStreamV2.getFileSystem());
    }

    private void validateSeekBounds(FileSystem fs) throws Exception {
        this.assumeHugeFileExists();
        try (final FSDataInputStream inputStream = fs.open(TEST_FILE_PATH);){
            ContractTestUtils.NanoTimer timer = new ContractTestUtils.NanoTimer();
            inputStream.seek(0L);
            ITestBlockBlobInputStream.assertEquals((long)0L, (long)inputStream.getPos());
            LambdaTestUtils.intercept(EOFException.class, (String)"Cannot seek to a negative offset", (Callable)new Callable<FSDataInputStream>(){

                @Override
                public FSDataInputStream call() throws Exception {
                    inputStream.seek(-1L);
                    return inputStream;
                }
            });
            ITestBlockBlobInputStream.assertTrue((String)("Test file length only " + this.testFileLength), (this.testFileLength > 0L ? 1 : 0) != 0);
            inputStream.seek(this.testFileLength);
            ITestBlockBlobInputStream.assertEquals((long)this.testFileLength, (long)inputStream.getPos());
            LambdaTestUtils.intercept(EOFException.class, (String)"Attempted to seek or read past the end of the file", (Callable)new Callable<FSDataInputStream>(){

                @Override
                public FSDataInputStream call() throws Exception {
                    inputStream.seek(ITestBlockBlobInputStream.this.testFileLength + 1L);
                    return inputStream;
                }
            });
            long elapsedTimeMs = timer.elapsedTimeMs();
            ITestBlockBlobInputStream.assertTrue((String)String.format("There should not be any network I/O (elapsedTimeMs=%1$d).", elapsedTimeMs), (elapsedTimeMs < 20L ? 1 : 0) != 0);
        }
    }

    @Test
    public void test_0311_SeekAndAvailableAndPositionV1() throws Exception {
        this.validateSeekAndAvailableAndPosition((FileSystem)this.accountUsingInputStreamV1.getFileSystem());
    }

    @Test
    public void test_0312_SeekAndAvailableAndPositionV2() throws Exception {
        this.validateSeekAndAvailableAndPosition((FileSystem)this.accountUsingInputStreamV2.getFileSystem());
    }

    private void validateSeekAndAvailableAndPosition(FileSystem fs) throws Exception {
        this.assumeHugeFileExists();
        try (FSDataInputStream inputStream = fs.open(TEST_FILE_PATH);){
            byte[] expected1 = new byte[]{97, 98, 99};
            byte[] expected2 = new byte[]{100, 101, 102};
            byte[] expected3 = new byte[]{98, 99, 100};
            byte[] expected4 = new byte[]{103, 104, 105};
            byte[] buffer = new byte[3];
            int bytesRead = inputStream.read(buffer);
            ITestBlockBlobInputStream.assertEquals((long)buffer.length, (long)bytesRead);
            ITestBlockBlobInputStream.assertArrayEquals((byte[])expected1, (byte[])buffer);
            ITestBlockBlobInputStream.assertEquals((long)buffer.length, (long)inputStream.getPos());
            ITestBlockBlobInputStream.assertEquals((long)(this.testFileLength - inputStream.getPos()), (long)inputStream.available());
            bytesRead = inputStream.read(buffer);
            ITestBlockBlobInputStream.assertEquals((long)buffer.length, (long)bytesRead);
            ITestBlockBlobInputStream.assertArrayEquals((byte[])expected2, (byte[])buffer);
            ITestBlockBlobInputStream.assertEquals((long)(2 * buffer.length), (long)inputStream.getPos());
            ITestBlockBlobInputStream.assertEquals((long)(this.testFileLength - inputStream.getPos()), (long)inputStream.available());
            int seekPos = 0;
            inputStream.seek((long)seekPos);
            bytesRead = inputStream.read(buffer);
            ITestBlockBlobInputStream.assertEquals((long)buffer.length, (long)bytesRead);
            ITestBlockBlobInputStream.assertArrayEquals((byte[])expected1, (byte[])buffer);
            ITestBlockBlobInputStream.assertEquals((long)(buffer.length + seekPos), (long)inputStream.getPos());
            ITestBlockBlobInputStream.assertEquals((long)(this.testFileLength - inputStream.getPos()), (long)inputStream.available());
            seekPos = 1;
            inputStream.seek((long)seekPos);
            bytesRead = inputStream.read(buffer);
            ITestBlockBlobInputStream.assertEquals((long)buffer.length, (long)bytesRead);
            ITestBlockBlobInputStream.assertArrayEquals((byte[])expected3, (byte[])buffer);
            ITestBlockBlobInputStream.assertEquals((long)(buffer.length + seekPos), (long)inputStream.getPos());
            ITestBlockBlobInputStream.assertEquals((long)(this.testFileLength - inputStream.getPos()), (long)inputStream.available());
            seekPos = 6;
            inputStream.seek((long)seekPos);
            bytesRead = inputStream.read(buffer);
            ITestBlockBlobInputStream.assertEquals((long)buffer.length, (long)bytesRead);
            ITestBlockBlobInputStream.assertArrayEquals((byte[])expected4, (byte[])buffer);
            ITestBlockBlobInputStream.assertEquals((long)(buffer.length + seekPos), (long)inputStream.getPos());
            ITestBlockBlobInputStream.assertEquals((long)(this.testFileLength - inputStream.getPos()), (long)inputStream.available());
        }
    }

    @Test
    public void test_0313_SkipAndAvailableAndPositionV1() throws IOException {
        this.validateSkipAndAvailableAndPosition((FileSystem)this.accountUsingInputStreamV1.getFileSystem());
    }

    @Test
    public void test_0314_SkipAndAvailableAndPositionV2() throws IOException {
        this.validateSkipAndAvailableAndPosition((FileSystem)this.accountUsingInputStreamV1.getFileSystem());
    }

    private void validateSkipAndAvailableAndPosition(FileSystem fs) throws IOException {
        this.assumeHugeFileExists();
        try (FSDataInputStream inputStream = fs.open(TEST_FILE_PATH);){
            byte[] expected1 = new byte[]{97, 98, 99};
            byte[] expected2 = new byte[]{100, 101, 102};
            byte[] expected3 = new byte[]{98, 99, 100};
            byte[] expected4 = new byte[]{103, 104, 105};
            ITestBlockBlobInputStream.assertEquals((long)this.testFileLength, (long)inputStream.available());
            ITestBlockBlobInputStream.assertEquals((long)0L, (long)inputStream.getPos());
            int n = 3;
            long skipped = inputStream.skip((long)n);
            ITestBlockBlobInputStream.assertEquals((long)skipped, (long)inputStream.getPos());
            ITestBlockBlobInputStream.assertEquals((long)(this.testFileLength - inputStream.getPos()), (long)inputStream.available());
            ITestBlockBlobInputStream.assertEquals((long)skipped, (long)n);
            byte[] buffer = new byte[3];
            int bytesRead = inputStream.read(buffer);
            ITestBlockBlobInputStream.assertEquals((long)buffer.length, (long)bytesRead);
            ITestBlockBlobInputStream.assertArrayEquals((byte[])expected2, (byte[])buffer);
            ITestBlockBlobInputStream.assertEquals((long)((long)buffer.length + skipped), (long)inputStream.getPos());
            ITestBlockBlobInputStream.assertEquals((long)(this.testFileLength - inputStream.getPos()), (long)inputStream.available());
            int seekPos = 1;
            inputStream.seek((long)seekPos);
            bytesRead = inputStream.read(buffer);
            ITestBlockBlobInputStream.assertEquals((long)buffer.length, (long)bytesRead);
            ITestBlockBlobInputStream.assertArrayEquals((byte[])expected3, (byte[])buffer);
            ITestBlockBlobInputStream.assertEquals((long)(buffer.length + seekPos), (long)inputStream.getPos());
            ITestBlockBlobInputStream.assertEquals((long)(this.testFileLength - inputStream.getPos()), (long)inputStream.available());
            long currentPosition = inputStream.getPos();
            n = 2;
            skipped = inputStream.skip((long)n);
            ITestBlockBlobInputStream.assertEquals((long)(currentPosition + skipped), (long)inputStream.getPos());
            ITestBlockBlobInputStream.assertEquals((long)(this.testFileLength - inputStream.getPos()), (long)inputStream.available());
            ITestBlockBlobInputStream.assertEquals((long)skipped, (long)n);
            bytesRead = inputStream.read(buffer);
            ITestBlockBlobInputStream.assertEquals((long)buffer.length, (long)bytesRead);
            ITestBlockBlobInputStream.assertArrayEquals((byte[])expected4, (byte[])buffer);
            ITestBlockBlobInputStream.assertEquals((long)((long)buffer.length + skipped + currentPosition), (long)inputStream.getPos());
            ITestBlockBlobInputStream.assertEquals((long)(this.testFileLength - inputStream.getPos()), (long)inputStream.available());
        }
    }

    @Test
    public void test_0315_SequentialReadPerformance() throws IOException {
        this.assumeHugeFileExists();
        int maxAttempts = 10;
        double maxAcceptableRatio = 1.01;
        double v1ElapsedMs = 0.0;
        double v2ElapsedMs = 0.0;
        double ratio = Double.MAX_VALUE;
        for (int i = 0; i < 10 && ratio >= 1.01; ++i) {
            v1ElapsedMs = this.sequentialRead(1, (FileSystem)this.accountUsingInputStreamV1.getFileSystem(), false);
            v2ElapsedMs = this.sequentialRead(2, (FileSystem)this.accountUsingInputStreamV2.getFileSystem(), false);
            ratio = v2ElapsedMs / v1ElapsedMs;
            LOG.info(String.format("v1ElapsedMs=%1$d, v2ElapsedMs=%2$d, ratio=%3$.2f", (long)v1ElapsedMs, (long)v2ElapsedMs, ratio));
        }
        ITestBlockBlobInputStream.assertTrue((String)String.format("Performance of version 2 is not acceptable: v1ElapsedMs=%1$d, v2ElapsedMs=%2$d, ratio=%3$.2f", (long)v1ElapsedMs, (long)v2ElapsedMs, ratio), (ratio < 1.01 ? 1 : 0) != 0);
    }

    @Test
    public void test_0316_SequentialReadAfterReverseSeekPerformanceV2() throws IOException {
        this.assumeHugeFileExists();
        int maxAttempts = 10;
        double maxAcceptableRatio = 1.01;
        double beforeSeekElapsedMs = 0.0;
        double afterSeekElapsedMs = 0.0;
        double ratio = Double.MAX_VALUE;
        for (int i = 0; i < 10 && ratio >= 1.01; ++i) {
            beforeSeekElapsedMs = this.sequentialRead(2, (FileSystem)this.accountUsingInputStreamV2.getFileSystem(), false);
            afterSeekElapsedMs = this.sequentialRead(2, (FileSystem)this.accountUsingInputStreamV2.getFileSystem(), true);
            ratio = afterSeekElapsedMs / beforeSeekElapsedMs;
            LOG.info(String.format("beforeSeekElapsedMs=%1$d, afterSeekElapsedMs=%2$d, ratio=%3$.2f", (long)beforeSeekElapsedMs, (long)afterSeekElapsedMs, ratio));
        }
        ITestBlockBlobInputStream.assertTrue((String)String.format("Performance of version 2 after reverse seek is not acceptable: beforeSeekElapsedMs=%1$d, afterSeekElapsedMs=%2$d, ratio=%3$.2f", (long)beforeSeekElapsedMs, (long)afterSeekElapsedMs, ratio), (ratio < 1.01 ? 1 : 0) != 0);
    }

    private long sequentialRead(int version, FileSystem fs, boolean afterReverseSeek) throws IOException {
        byte[] buffer = new byte[16384];
        long bytesRead = 0L;
        try (FSDataInputStream inputStream = fs.open(TEST_FILE_PATH);){
            long totalBytesRead;
            if (afterReverseSeek) {
                for (totalBytesRead = 0L; bytesRead > 0L && totalBytesRead < 0x400000L; totalBytesRead += bytesRead) {
                    bytesRead = inputStream.read(buffer);
                }
                totalBytesRead = 0L;
                inputStream.seek(0L);
            }
            ContractTestUtils.NanoTimer timer = new ContractTestUtils.NanoTimer();
            while ((bytesRead = (long)inputStream.read(buffer)) > 0L) {
                totalBytesRead += bytesRead;
            }
            long elapsedTimeMs = timer.elapsedTimeMs();
            LOG.info(String.format("v%1$d: bytesRead=%2$d, elapsedMs=%3$d, Mbps=%4$.2f, afterReverseSeek=%5$s", version, totalBytesRead, elapsedTimeMs, ITestBlockBlobInputStream.toMbps(totalBytesRead, elapsedTimeMs), afterReverseSeek));
            ITestBlockBlobInputStream.assertEquals((long)this.testFileLength, (long)totalBytesRead);
            inputStream.close();
            long l = elapsedTimeMs;
            return l;
        }
    }

    @Test
    public void test_0317_RandomReadPerformance() throws IOException {
        this.assumeHugeFileExists();
        int maxAttempts = 10;
        double maxAcceptableRatio = 0.1;
        double v1ElapsedMs = 0.0;
        double v2ElapsedMs = 0.0;
        double ratio = Double.MAX_VALUE;
        for (int i = 0; i < 10 && ratio >= 0.1; ++i) {
            v1ElapsedMs = this.randomRead(1, (FileSystem)this.accountUsingInputStreamV1.getFileSystem());
            v2ElapsedMs = this.randomRead(2, (FileSystem)this.accountUsingInputStreamV2.getFileSystem());
            ratio = v2ElapsedMs / v1ElapsedMs;
            LOG.info(String.format("v1ElapsedMs=%1$d, v2ElapsedMs=%2$d, ratio=%3$.2f", (long)v1ElapsedMs, (long)v2ElapsedMs, ratio));
        }
        ITestBlockBlobInputStream.assertTrue((String)String.format("Performance of version 2 is not acceptable: v1ElapsedMs=%1$d, v2ElapsedMs=%2$d, ratio=%3$.2f", (long)v1ElapsedMs, (long)v2ElapsedMs, ratio), (ratio < 0.1 ? 1 : 0) != 0);
    }

    private long randomRead(int version, FileSystem fs) throws IOException {
        this.assumeHugeFileExists();
        int minBytesToRead = 0x200000;
        Random random = new Random();
        byte[] buffer = new byte[8192];
        long totalBytesRead = 0L;
        long bytesRead = 0L;
        try (FSDataInputStream inputStream = fs.open(TEST_FILE_PATH);){
            ContractTestUtils.NanoTimer timer = new ContractTestUtils.NanoTimer();
            do {
                bytesRead = inputStream.read(buffer);
                inputStream.seek((long)random.nextInt((int)(this.testFileLength - (long)buffer.length)));
            } while (bytesRead > 0L && (totalBytesRead += bytesRead) < 0x200000L);
            long elapsedTimeMs = timer.elapsedTimeMs();
            inputStream.close();
            LOG.info(String.format("v%1$d: totalBytesRead=%2$d, elapsedTimeMs=%3$d, Mbps=%4$.2f", version, totalBytesRead, elapsedTimeMs, ITestBlockBlobInputStream.toMbps(totalBytesRead, elapsedTimeMs)));
            ITestBlockBlobInputStream.assertTrue((0x200000L <= totalBytesRead ? 1 : 0) != 0);
            long l = elapsedTimeMs;
            return l;
        }
    }

    @Test
    public void test_999_DeleteHugeFiles() throws IOException {
        try {
            ContractTestUtils.NanoTimer timer = new ContractTestUtils.NanoTimer();
            NativeAzureFileSystem fs = this.getFileSystem();
            fs.delete(TEST_FILE_PATH, false);
            timer.end("time to delete %s", new Object[]{TEST_FILE_PATH});
        }
        finally {
            AzureTestUtils.cleanupTestAccount(this.accountUsingInputStreamV1);
        }
    }
}

