/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.security.authentication.util;

import com.auth0.jwk.JwkException;
import com.auth0.jwk.UrlJwkProvider;
import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.interfaces.DecodedJWT;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.security.InvalidParameterException;
import java.security.interfaces.RSAPublicKey;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import org.apache.hadoop.security.authentication.util.SsoConfigurationUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JWTUtils {
    private static Logger LOG = LoggerFactory.getLogger(JWTUtils.class);

    public static boolean validateToken(DecodedJWT jwtToken) throws InvalidParameterException {
        try {
            boolean expValid;
            boolean audValid;
            DecodedJWT verifiedToken = JWTUtils.verifyToken(jwtToken);
            if (verifiedToken == null) {
                LOG.warn("Token validation failed.");
            }
            if (!(audValid = JWTUtils.validateAudiences(jwtToken))) {
                LOG.warn("Audience validation failed.");
            }
            if (!(expValid = JWTUtils.validateExpiration(jwtToken))) {
                LOG.info("Expiration validation failed.");
            }
            return verifiedToken != null && audValid && expValid;
        }
        catch (Exception e) {
            LOG.error("Exception while validating/introspecting jwt token, check debug logs for more details");
            if (LOG.isDebugEnabled()) {
                e.printStackTrace();
            }
            return false;
        }
    }

    public static DecodedJWT verifyToken(DecodedJWT jwt) throws InvalidParameterException {
        try {
            RSAPublicKey publicKey = JWTUtils.loadPublicKey(jwt);
            Algorithm algorithm = JWTUtils.getSigntureAlgorithm(SsoConfigurationUtil.getInstance().getJwsSsoAlgorithm(), publicKey);
            JWTVerifier verifier = JWT.require((Algorithm)algorithm).withIssuer(jwt.getIssuer()).build();
            return verifier.verify(jwt);
        }
        catch (Exception e) {
            if (LOG.isDebugEnabled()) {
                e.printStackTrace();
            }
            LOG.error("Unable to authenticate: {}", (Object)e.getMessage());
            throw new InvalidParameterException("Unable to authenticate: " + e.getMessage());
        }
    }

    private static Algorithm getSigntureAlgorithm(String algorithmConf, RSAPublicKey publicKey) {
        Algorithm alg;
        switch (algorithmConf) {
            case "RS384": {
                alg = Algorithm.RSA384((RSAPublicKey)publicKey, null);
                break;
            }
            case "RS512": {
                alg = Algorithm.RSA512((RSAPublicKey)publicKey, null);
                break;
            }
            default: {
                alg = Algorithm.RSA256((RSAPublicKey)publicKey, null);
            }
        }
        return alg;
    }

    private static RSAPublicKey loadPublicKey(DecodedJWT token) throws JwkException, MalformedURLException {
        String url = JWTUtils.getKeycloakCertificateUrl(token);
        UrlJwkProvider provider = new UrlJwkProvider(new URL(url));
        return (RSAPublicKey)provider.get(token.getKeyId()).getPublicKey();
    }

    private static String getKeycloakCertificateUrl(DecodedJWT token) {
        return token.getIssuer() + "/protocol/openid-connect/certs";
    }

    private static boolean validateExpiration(DecodedJWT jwtToken) {
        boolean valid = false;
        Date expires = jwtToken.getClaim("exp").asDate();
        if (expires == null || new Date().before(expires)) {
            LOG.debug("JWT token expiration date has been successfully validated");
            valid = true;
        } else {
            LOG.warn("JWT expiration date validation failed.");
        }
        return valid;
    }

    private static boolean validateAudiences(DecodedJWT jwtToken) {
        boolean valid = false;
        List tokenAudienceList = jwtToken.getClaim("aud").asList(String.class);
        if (SsoConfigurationUtil.getInstance().getAudiences().isEmpty()) {
            valid = true;
        } else {
            for (String aud : tokenAudienceList) {
                if (!SsoConfigurationUtil.getInstance().getAudiences().contains(aud)) continue;
                LOG.debug("JWT token audience has been successfully validated");
                valid = true;
                break;
            }
            if (!valid) {
                LOG.warn("JWT audience validation failed.");
            }
        }
        return valid;
    }

    public static String getJWTFromCookie(HttpServletRequest req) {
        String serializedJWT = null;
        Cookie[] cookies = req.getCookies();
        String cookieName = SsoConfigurationUtil.getInstance().getCookieName();
        if (cookies != null) {
            for (Cookie cookie : cookies) {
                if (!cookieName.equals(cookie.getName())) continue;
                LOG.info("{} cookie has been found and is being processed", (Object)cookieName);
                serializedJWT = cookie.getValue();
                break;
            }
        }
        return serializedJWT;
    }

    public static String constructURLWithHostname(String originalUrl) {
        try {
            URI originalUri = new URI(originalUrl);
            InetAddress address = InetAddress.getByName(new URL(originalUrl).getHost());
            if (originalUrl.contains(address.getHostAddress())) {
                return JWTUtils.replaceHostInUrl(originalUri, address.getHostName() + ":" + originalUri.getPort());
            }
        }
        catch (Exception ex) {
            LOG.warn("Can't create new URL from request hostname {}. Use URL from request.", (Object)originalUrl);
        }
        return originalUrl;
    }

    public static String replaceHostInUrl(URI originalUri, String newAuthority) {
        URI uri;
        try {
            uri = new URI(originalUri.getScheme().toLowerCase(Locale.US), newAuthority, originalUri.getPath(), originalUri.getQuery(), originalUri.getFragment());
        }
        catch (URISyntaxException ex) {
            LOG.warn("Can't create new URI with hostname for host {}", (Object)newAuthority);
            ex.printStackTrace();
            return originalUri.toString();
        }
        return uri.toString();
    }
}

