/*
 * Decompiled with CFR 0.152.
 */
package oadd.org.apache.drill.exec.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinTask;
import java.util.concurrent.RecursiveTask;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import oadd.org.apache.drill.common.exceptions.ErrorHelper;
import oadd.org.apache.drill.common.exceptions.UserException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemUtil {
    private static final Logger logger = LoggerFactory.getLogger(FileSystemUtil.class);
    public static final String RECURSIVE_FILE_LISTING_MAX_SIZE = "drill.exec.recursive_file_listing_max_size";
    public static final PathFilter DUMMY_FILTER = path -> true;

    public static List<FileStatus> listDirectories(FileSystem fs, Path path, boolean recursive, PathFilter ... filters) throws IOException {
        return FileSystemUtil.list(fs, path, Scope.DIRECTORIES, recursive, false, filters);
    }

    public static List<FileStatus> listDirectoriesSafe(FileSystem fs, Path path, boolean recursive, PathFilter ... filters) {
        try {
            return FileSystemUtil.list(fs, path, Scope.DIRECTORIES, recursive, true, filters);
        }
        catch (Exception e) {
            return Collections.emptyList();
        }
    }

    public static List<FileStatus> listFiles(FileSystem fs, Path path, boolean recursive, PathFilter ... filters) throws IOException {
        return FileSystemUtil.list(fs, path, Scope.FILES, recursive, false, filters);
    }

    public static List<FileStatus> listFilesSafe(FileSystem fs, Path path, boolean recursive, PathFilter ... filters) {
        try {
            return FileSystemUtil.list(fs, path, Scope.FILES, recursive, true, filters);
        }
        catch (Exception e) {
            return Collections.emptyList();
        }
    }

    public static List<FileStatus> listAll(FileSystem fs, Path path, boolean recursive, PathFilter ... filters) throws IOException {
        return FileSystemUtil.list(fs, path, Scope.ALL, recursive, false, filters);
    }

    public static List<FileStatus> listAllSafe(FileSystem fs, Path path, boolean recursive, PathFilter ... filters) {
        try {
            return FileSystemUtil.list(fs, path, Scope.ALL, recursive, true, filters);
        }
        catch (Exception e) {
            return Collections.emptyList();
        }
    }

    public static PathFilter mergeFilters(PathFilter filter, PathFilter[] filters) {
        if (filters == null || filters.length == 0) {
            return filter;
        }
        int length = filters.length;
        PathFilter[] newFilters = Arrays.copyOf(filters, length + 1);
        newFilters[length] = filter;
        return FileSystemUtil.mergeFilters(newFilters);
    }

    public static PathFilter mergeFilters(PathFilter ... filters) {
        if (filters.length == 0) {
            return DUMMY_FILTER;
        }
        return path -> Stream.of(filters).allMatch(filter -> filter.accept(path));
    }

    private static List<FileStatus> list(FileSystem fs, Path path, Scope scope, boolean recursive, boolean suppressExceptions, PathFilter ... filters) throws IOException {
        PathFilter filter = FileSystemUtil.mergeFilters(filters);
        return recursive ? FileSystemUtil.listRecursive(fs, path, scope, suppressExceptions, filter) : FileSystemUtil.listNonRecursive(fs, path, scope, suppressExceptions, filter);
    }

    private static List<FileStatus> listNonRecursive(FileSystem fs, Path path, Scope scope, boolean suppressExceptions, PathFilter filter) throws IOException {
        try {
            return Stream.of(fs.listStatus(path, filter)).filter(status -> FileSystemUtil.isStatusApplicable(status, scope)).collect(Collectors.toList());
        }
        catch (Exception e) {
            if (suppressExceptions) {
                logger.debug("Exception during listing file statuses", e);
                return Collections.emptyList();
            }
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static List<FileStatus> listRecursive(FileSystem fs, Path path, Scope scope, boolean suppressExceptions, PathFilter filter) {
        ForkJoinPool pool = new ForkJoinPool();
        AtomicInteger fileCounter = new AtomicInteger(0);
        int recursiveListingMaxSize = fs.getConf().getInt(RECURSIVE_FILE_LISTING_MAX_SIZE, 0);
        try {
            RecursiveListing task = new RecursiveListing(fs, path, scope, suppressExceptions, filter, fileCounter, recursiveListingMaxSize, pool);
            List<FileStatus> list = pool.invoke(task);
            return list;
        }
        catch (CancellationException ex) {
            logger.debug("RecursiveListing task to list {} was cancelled.", (Object)path);
            List<FileStatus> list = Collections.emptyList();
            return list;
        }
        finally {
            pool.shutdown();
        }
    }

    private static boolean isStatusApplicable(FileStatus status, Scope scope) {
        switch (scope) {
            case DIRECTORIES: {
                return status.isDirectory();
            }
            case FILES: {
                return status.isFile();
            }
            case ALL: {
                return true;
            }
        }
        return false;
    }

    private static class RecursiveListing
    extends RecursiveTask<List<FileStatus>> {
        private final FileSystem fs;
        private final Path path;
        private final Scope scope;
        private final boolean suppressExceptions;
        private final PathFilter filter;
        private final AtomicInteger fileCounter;
        private final int recursiveListingMaxSize;
        private final ForkJoinPool pool;

        RecursiveListing(FileSystem fs, Path path, Scope scope, boolean suppressExceptions, PathFilter filter, AtomicInteger fileCounter, int recursiveListingMaxSize, ForkJoinPool pool) {
            this.fs = fs;
            this.path = path;
            this.scope = scope;
            this.suppressExceptions = suppressExceptions;
            this.filter = filter;
            this.fileCounter = fileCounter;
            this.recursiveListingMaxSize = recursiveListingMaxSize;
            this.pool = pool;
        }

        @Override
        protected List<FileStatus> compute() {
            ArrayList<FileStatus> statuses = new ArrayList<FileStatus>();
            ArrayList<RecursiveListing> tasks = new ArrayList<RecursiveListing>();
            try {
                FileStatus[] dirFs = this.fs.listStatus(this.path, this.filter);
                if (this.fileCounter.addAndGet(dirFs.length) > this.recursiveListingMaxSize && this.recursiveListingMaxSize > 0) {
                    try {
                        throw UserException.resourceError().message("File listing size limit of %d exceeded recursing through path %s, see JVM system property %s", this.recursiveListingMaxSize, this.path, FileSystemUtil.RECURSIVE_FILE_LISTING_MAX_SIZE).build(logger);
                    }
                    catch (Throwable throwable) {
                        this.pool.shutdownNow();
                        throw throwable;
                    }
                }
                for (FileStatus status : dirFs) {
                    if (FileSystemUtil.isStatusApplicable(status, this.scope)) {
                        statuses.add(status);
                    }
                    if (!status.isDirectory()) continue;
                    RecursiveListing task = new RecursiveListing(this.fs, status.getPath(), this.scope, this.suppressExceptions, this.filter, this.fileCounter, this.recursiveListingMaxSize, this.pool);
                    task.fork();
                    tasks.add(task);
                }
            }
            catch (Exception e) {
                if (this.suppressExceptions) {
                    logger.debug("Exception during listing file statuses", e);
                }
                ErrorHelper.sneakyThrow(e);
            }
            tasks.stream().map(ForkJoinTask::join).forEach(statuses::addAll);
            return statuses;
        }
    }

    private static enum Scope {
        DIRECTORIES,
        FILES,
        ALL;

    }
}

