/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.drill.exec.store;

import com.google.common.collect.Lists;
import com.google.common.primitives.Ints;
import com.google.common.primitives.Longs;
import org.apache.drill.common.types.TypeProtos.MinorType;
import org.apache.drill.exec.expr.TypeHelper;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.exec.record.BatchSchema;
import org.apache.drill.exec.store.EventBasedRecordWriter.FieldConverter;
import org.apache.drill.exec.store.parquet.ParquetTypeHelper;
import org.apache.drill.exec.store.parquet.decimal.DecimalValueWriter;
import org.apache.drill.exec.vector.*;
import org.apache.drill.exec.util.DecimalUtility;
import org.apache.drill.exec.vector.complex.reader.FieldReader;
import org.apache.parquet.io.api.RecordConsumer;
import org.apache.parquet.schema.MessageType;
import org.apache.parquet.schema.PrimitiveType;
import org.apache.parquet.io.api.Binary;
import io.netty.buffer.DrillBuf;
import org.apache.drill.exec.record.BatchSchema;
import org.apache.drill.exec.record.MaterializedField;


import org.apache.drill.common.types.TypeProtos;

import org.joda.time.DateTimeConstants;

import java.io.IOException;
import java.lang.UnsupportedOperationException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * Abstract implementation of RecordWriter interface which exposes interface:
 *    {@link #writeHeader(List)}
 *    {@link #addField(int,String)}
 * to output the data in string format instead of implementing addField for each type holder.
 *
 * This is useful for text format writers such as CSV, TSV etc.
 *
 * NB: Source code generated using FreeMarker template ParquetOutputRecordWriter.java
 */
public abstract class ParquetOutputRecordWriter extends AbstractRecordWriter implements RecordWriter {

  /**
   * Name of nested group for Parquet's {@code LIST} type.
   * @see <a href="https://github.com/apache/parquet-format/blob/master/LogicalTypes.md#lists">LIST logical type</a>
   */
  protected static final String LIST = "list";

  /**
   * Name of Parquet's {@code LIST} element type.
   * @see #LIST
   */
  protected static final String ELEMENT = "element";
  protected static final int ZERO_IDX = 0;

  private RecordConsumer consumer;
  private MessageType schema;

  public void setUp(MessageType schema, RecordConsumer consumer) {
    this.schema = schema;
    this.consumer = consumer;
  }

  protected abstract PrimitiveType getPrimitiveType(MaterializedField field);

  public abstract class BaseFieldConverter extends FieldConverter {

    public BaseFieldConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    public abstract void read();

    public abstract void read(int i);

    public abstract void consume();

    @Override
    public void writeField() throws IOException {
      read();
      consume();
    }
  }

  public class NullableFieldConverter extends FieldConverter {
    private BaseFieldConverter delegate;

    public NullableFieldConverter(int fieldId, String fieldName, FieldReader reader, BaseFieldConverter delegate) {
      super(fieldId, fieldName, reader);
      this.delegate = delegate;
    }

    @Override
    public void writeField() throws IOException {
      if (!reader.isSet()) {
        return;
      }
      consumer.startField(fieldName, fieldId);
      delegate.writeField();
      consumer.endField(fieldName, fieldId);
    }

    public void setPosition(int index) {
      delegate.setPosition(index);
    }

    public void startField() throws IOException {
      delegate.startField();
    }

    public void endField() throws IOException {
      delegate.endField();
    }
  }

  public class RequiredFieldConverter extends FieldConverter {
    private BaseFieldConverter delegate;

    public RequiredFieldConverter(int fieldId, String fieldName, FieldReader reader, BaseFieldConverter delegate) {
      super(fieldId, fieldName, reader);
      this.delegate = delegate;
    }

    @Override
    public void writeField() throws IOException {
      consumer.startField(fieldName, fieldId);
      delegate.writeField();
      consumer.endField(fieldName, fieldId);
    }

    public void setPosition(int index) {
      delegate.setPosition(index);
    }

    public void startField() throws IOException {
      delegate.startField();
    }

    public void endField() throws IOException {
      delegate.endField();
    }
  }

  public class RepeatedFieldConverter extends FieldConverter {

    private BaseFieldConverter delegate;

    public RepeatedFieldConverter(int fieldId, String fieldName, FieldReader reader, BaseFieldConverter delegate) {
      super(fieldId, fieldName, reader);
      this.delegate = delegate;
    }

    @Override
    public void writeField() throws IOException {
      // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
      if (reader.size() == 0) {
        return;
      }
      consumer.startField(fieldName, fieldId);
      for (int i = 0; i < reader.size(); i++) {
        delegate.read(i);
        delegate.consume();
      }
      consumer.endField(fieldName, fieldId);
    }

    @Override
    public void writeListField() {
      if (reader.size() == 0) {
        return;
      }
      consumer.startField(LIST, ZERO_IDX);
      for (int i = 0; i < reader.size(); i++) {
        consumer.startGroup();
        consumer.startField(ELEMENT, ZERO_IDX);

        delegate.read(i);
        delegate.consume();

        consumer.endField(ELEMENT, ZERO_IDX);
        consumer.endGroup();
      }
      consumer.endField(LIST, ZERO_IDX);
    }

    public void setPosition(int index) {
      delegate.setPosition(index);
    }

    public void startField() throws IOException {
      delegate.startField();
    }

    public void endField() throws IOException {
      delegate.endField();
    }
  }

  @Override
  public FieldConverter getNewNullableTinyIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new TinyIntParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewTinyIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new TinyIntParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedTinyIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new TinyIntParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class TinyIntParquetConverter extends BaseFieldConverter {
    private NullableTinyIntHolder holder = new NullableTinyIntHolder();

    public TinyIntParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addInteger(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableUInt1Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt1ParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewUInt1Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt1ParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedUInt1Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt1ParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class UInt1ParquetConverter extends BaseFieldConverter {
    private NullableUInt1Holder holder = new NullableUInt1Holder();

    public UInt1ParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addInteger(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableUInt2Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt2ParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewUInt2Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt2ParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedUInt2Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt2ParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class UInt2ParquetConverter extends BaseFieldConverter {
    private NullableUInt2Holder holder = new NullableUInt2Holder();

    public UInt2ParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addInteger(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableSmallIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new SmallIntParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewSmallIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new SmallIntParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedSmallIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new SmallIntParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class SmallIntParquetConverter extends BaseFieldConverter {
    private NullableSmallIntHolder holder = new NullableSmallIntHolder();

    public SmallIntParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addInteger(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class IntParquetConverter extends BaseFieldConverter {
    private NullableIntHolder holder = new NullableIntHolder();

    public IntParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addInteger(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableUInt4Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt4ParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewUInt4Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt4ParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedUInt4Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt4ParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class UInt4ParquetConverter extends BaseFieldConverter {
    private NullableUInt4Holder holder = new NullableUInt4Holder();

    public UInt4ParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addInteger(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableFloat4Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Float4ParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewFloat4Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Float4ParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedFloat4Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Float4ParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class Float4ParquetConverter extends BaseFieldConverter {
    private NullableFloat4Holder holder = new NullableFloat4Holder();

    public Float4ParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addFloat(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableTimeConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new TimeParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewTimeConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new TimeParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedTimeConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new TimeParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class TimeParquetConverter extends BaseFieldConverter {
    private NullableTimeHolder holder = new NullableTimeHolder();

    public TimeParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addInteger(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableIntervalYearConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntervalYearParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewIntervalYearConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntervalYearParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedIntervalYearConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntervalYearParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class IntervalYearParquetConverter extends BaseFieldConverter {
    private NullableIntervalYearHolder holder = new NullableIntervalYearHolder();
    private final byte[] output = new byte[12];

    public IntervalYearParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
        IntervalUtility.intToLEByteArray(holder.value, output, 0);
        Arrays.fill(output, 4, 8, (byte) 0);
        Arrays.fill(output, 8, 12, (byte) 0);
      consumer.addBinary(Binary.fromByteArray(output));
    }

  }
  @Override
  public FieldConverter getNewNullableDecimal9Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal9ParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewDecimal9Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal9ParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedDecimal9Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal9ParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class Decimal9ParquetConverter extends BaseFieldConverter {
    private NullableDecimal9Holder holder = new NullableDecimal9Holder();

    public Decimal9ParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addInteger(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableBigIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new BigIntParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewBigIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new BigIntParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedBigIntConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new BigIntParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class BigIntParquetConverter extends BaseFieldConverter {
    private NullableBigIntHolder holder = new NullableBigIntHolder();

    public BigIntParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addLong(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableUInt8Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt8ParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewUInt8Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt8ParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedUInt8Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new UInt8ParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class UInt8ParquetConverter extends BaseFieldConverter {
    private NullableUInt8Holder holder = new NullableUInt8Holder();

    public UInt8ParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addLong(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableFloat8Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Float8ParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewFloat8Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Float8ParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedFloat8Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Float8ParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class Float8ParquetConverter extends BaseFieldConverter {
    private NullableFloat8Holder holder = new NullableFloat8Holder();

    public Float8ParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addDouble(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableDateConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new DateParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewDateConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new DateParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedDateConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new DateParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class DateParquetConverter extends BaseFieldConverter {
    private NullableDateHolder holder = new NullableDateHolder();

    public DateParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      // convert from internal Drill date format to Julian Day centered around Unix Epoc
      consumer.addInteger((int) (holder.value / DateTimeConstants.MILLIS_PER_DAY));
    }

  }
  @Override
  public FieldConverter getNewNullableTimeStampConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new TimeStampParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewTimeStampConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new TimeStampParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedTimeStampConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new TimeStampParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class TimeStampParquetConverter extends BaseFieldConverter {
    private NullableTimeStampHolder holder = new NullableTimeStampHolder();

    public TimeStampParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addLong(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableDecimal18Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal18ParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewDecimal18Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal18ParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedDecimal18Converter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal18ParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class Decimal18ParquetConverter extends BaseFieldConverter {
    private NullableDecimal18Holder holder = new NullableDecimal18Holder();

    public Decimal18ParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addLong(holder.value);
    }

  }
  @Override
  public FieldConverter getNewNullableIntervalDayConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntervalDayParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewIntervalDayConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntervalDayParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedIntervalDayConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntervalDayParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class IntervalDayParquetConverter extends BaseFieldConverter {
    private NullableIntervalDayHolder holder = new NullableIntervalDayHolder();
    private final byte[] output = new byte[12];

    public IntervalDayParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
        Arrays.fill(output, 0, 4, (byte) 0);
        IntervalUtility.intToLEByteArray(holder.days, output, 4);
        IntervalUtility.intToLEByteArray(holder.milliseconds, output, 8);
      consumer.addBinary(Binary.fromByteArray(output));
    }

  }
  @Override
  public FieldConverter getNewNullableIntervalConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntervalParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewIntervalConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntervalParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedIntervalConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new IntervalParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class IntervalParquetConverter extends BaseFieldConverter {
    private NullableIntervalHolder holder = new NullableIntervalHolder();
    private final byte[] output = new byte[12];

    public IntervalParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
        IntervalUtility.intToLEByteArray(holder.months, output, 0);
        IntervalUtility.intToLEByteArray(holder.days, output, 4);
        IntervalUtility.intToLEByteArray(holder.milliseconds, output, 8);
      consumer.addBinary(Binary.fromByteArray(output));
    }

  }
  @Override
  public FieldConverter getNewNullableDecimal28DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal28DenseParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewDecimal28DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal28DenseParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedDecimal28DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal28DenseParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class Decimal28DenseParquetConverter extends BaseFieldConverter {
    private NullableDecimal28DenseHolder holder = new NullableDecimal28DenseHolder();

    public Decimal28DenseParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
    }

  }
  @Override
  public FieldConverter getNewNullableDecimal38DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal38DenseParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewDecimal38DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal38DenseParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedDecimal38DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal38DenseParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class Decimal38DenseParquetConverter extends BaseFieldConverter {
    private NullableDecimal38DenseHolder holder = new NullableDecimal38DenseHolder();

    public Decimal38DenseParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
    }

  }
  @Override
  public FieldConverter getNewNullableDecimal38SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal38SparseParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewDecimal38SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal38SparseParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedDecimal38SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal38SparseParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class Decimal38SparseParquetConverter extends BaseFieldConverter {
    private NullableDecimal38SparseHolder holder = new NullableDecimal38SparseHolder();

    public Decimal38SparseParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      byte[] bytes = DecimalUtility.getBigDecimalFromSparse(
              holder.buffer, holder.start, Decimal38SparseHolder.nDecimalDigits, holder.scale).unscaledValue().toByteArray();
      byte[] output = new byte[ParquetTypeHelper.getLengthForMinorType(MinorType.DECIMAL38SPARSE)];
      if (holder.getSign(holder.start, holder.buffer)) {
        Arrays.fill(output, 0, output.length - bytes.length, (byte) -1);
      } else {
        Arrays.fill(output, 0, output.length - bytes.length, (byte) 0);
      }
      System.arraycopy(bytes, 0, output, output.length - bytes.length, bytes.length);
      consumer.addBinary(Binary.fromByteArray(output));
    }

  }
  @Override
  public FieldConverter getNewNullableDecimal28SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal28SparseParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewDecimal28SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal28SparseParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedDecimal28SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Decimal28SparseParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class Decimal28SparseParquetConverter extends BaseFieldConverter {
    private NullableDecimal28SparseHolder holder = new NullableDecimal28SparseHolder();

    public Decimal28SparseParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      byte[] bytes = DecimalUtility.getBigDecimalFromSparse(
              holder.buffer, holder.start, Decimal28SparseHolder.nDecimalDigits, holder.scale).unscaledValue().toByteArray();
      byte[] output = new byte[ParquetTypeHelper.getLengthForMinorType(MinorType.DECIMAL28SPARSE)];
      if (holder.getSign(holder.start, holder.buffer)) {
        Arrays.fill(output, 0, output.length - bytes.length, (byte) -1);
      } else {
        Arrays.fill(output, 0, output.length - bytes.length, (byte) 0);
      }
      System.arraycopy(bytes, 0, output, output.length - bytes.length, bytes.length);
      consumer.addBinary(Binary.fromByteArray(output));
    }

  }
  @Override
  public FieldConverter getNewNullableVarBinaryConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new VarBinaryParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewVarBinaryConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new VarBinaryParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedVarBinaryConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new VarBinaryParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class VarBinaryParquetConverter extends BaseFieldConverter {
    private NullableVarBinaryHolder holder = new NullableVarBinaryHolder();

    public VarBinaryParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addBinary(Binary.fromByteBuffer(holder.buffer.nioBuffer(holder.start, holder.end - holder.start)));
    }

  }
  @Override
  public FieldConverter getNewNullableVarCharConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new VarCharParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewVarCharConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new VarCharParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedVarCharConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new VarCharParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class VarCharParquetConverter extends BaseFieldConverter {
    private NullableVarCharHolder holder = new NullableVarCharHolder();

    public VarCharParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addBinary(Binary.fromByteBuffer(holder.buffer.nioBuffer(holder.start, holder.end - holder.start)));
    }

  }
  @Override
  public FieldConverter getNewNullableVar16CharConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Var16CharParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewVar16CharConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Var16CharParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedVar16CharConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new Var16CharParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class Var16CharParquetConverter extends BaseFieldConverter {
    private NullableVar16CharHolder holder = new NullableVar16CharHolder();

    public Var16CharParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addBinary(Binary.fromByteBuffer(holder.buffer.nioBuffer(holder.start, holder.end - holder.start)));
    }

  }
  @Override
  public FieldConverter getNewNullableVarDecimalConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new VarDecimalParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewVarDecimalConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new VarDecimalParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedVarDecimalConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new VarDecimalParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class VarDecimalParquetConverter extends BaseFieldConverter {
    private NullableVarDecimalHolder holder = new NullableVarDecimalHolder();
    private final DecimalValueWriter decimalValueWriter;

    public VarDecimalParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
      decimalValueWriter = DecimalValueWriter.
          getDecimalValueWriterForType(getPrimitiveType(reader.getField()).getPrimitiveTypeName());
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      decimalValueWriter.writeValue(consumer, holder.buffer,
          holder.start, holder.end, reader.getField().getPrecision());
    }

  }
  @Override
  public FieldConverter getNewNullableBitConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new BitParquetConverter(fieldId, fieldName, reader);
    return new NullableFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewBitConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new BitParquetConverter(fieldId, fieldName, reader);
    return new RequiredFieldConverter(fieldId, fieldName, reader, converter);
  }

  @Override
  public FieldConverter getNewRepeatedBitConverter(int fieldId, String fieldName, FieldReader reader) {
    BaseFieldConverter converter = new BitParquetConverter(fieldId, fieldName, reader);
    return new RepeatedFieldConverter(fieldId, fieldName, reader, converter);
  }


  public class BitParquetConverter extends BaseFieldConverter {
    private NullableBitHolder holder = new NullableBitHolder();

    public BitParquetConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void read() {
      reader.read(holder);
    }

    @Override
    public void read(int i) {
      reader.read(i, holder);
    }

    @Override
    public void consume() {
      consumer.addBoolean(holder.value == 1);
    }

  }

  private static class IntervalUtility {
    private static void intToLEByteArray(final int value, final byte[] output, final int outputIndex) {
      int shiftOrder = 0;
      for (int i = outputIndex; i < outputIndex + 4; i++) {
        output[i] = (byte) (value >> shiftOrder);
        shiftOrder += 8;
      }
    }
  }
}
