/*
 * Decompiled with CFR 0.152.
 */
package org.apache.drill.yarn.client;

import com.typesafe.config.Config;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.drill.yarn.client.ClientException;
import org.apache.drill.yarn.core.DfsFacade;
import org.apache.drill.yarn.core.DoYUtil;
import org.apache.drill.yarn.core.DoyConfigException;
import org.apache.drill.yarn.core.DrillOnYarnConfig;
import org.apache.hadoop.yarn.api.records.LocalResource;

public abstract class FileUploader {
    protected DrillOnYarnConfig doyConfig;
    protected Config config;
    protected DfsFacade dfs;
    protected boolean dryRun;
    protected boolean verbose;
    protected File localDrillHome;
    protected File localSiteDir;
    protected File localDrillArchivePath;
    public Map<String, LocalResource> resources = new HashMap<String, LocalResource>();
    public String drillArchivePath;
    public String siteArchivePath;
    public String remoteDrillHome;
    public String remoteSiteDir;

    public FileUploader(boolean dryRun, boolean verbose) {
        this.doyConfig = DrillOnYarnConfig.instance();
        this.config = this.doyConfig.getConfig();
        this.dryRun = dryRun;
        this.verbose = verbose;
    }

    public abstract void run() throws ClientException;

    protected void setup() throws ClientException {
        this.localDrillHome = this.doyConfig.getLocalDrillHome();
        try {
            this.remoteDrillHome = this.doyConfig.getRemoteDrillHome();
        }
        catch (DoyConfigException e) {
            throw new ClientException(e);
        }
        this.localSiteDir = this.doyConfig.getLocalSiteDir();
        if (this.hasSiteDir()) {
            if (!this.localSiteDir.isDirectory()) {
                throw new ClientException("Drill site dir not a directory: " + this.localSiteDir);
            }
            this.remoteSiteDir = this.doyConfig.getRemoteSiteDir();
        }
        if (this.dryRun) {
            System.out.println("Dry run only.");
        }
    }

    public boolean hasSiteDir() {
        return this.localSiteDir != null;
    }

    public boolean isLocalized() {
        return this.config.getBoolean(DrillOnYarnConfig.LOCALIZE_DRILL);
    }

    protected void connectToDfs() throws ClientException {
        try {
            System.out.print("Connecting to DFS...");
            this.dfs = new DfsFacade(this.config);
            this.dfs.connect();
            System.out.println(" Connected.");
        }
        catch (DfsFacade.DfsFacadeException e) {
            System.out.println("Failed.");
            throw new ClientException("Failed to connect to DFS", e);
        }
    }

    protected DfsFacade.Localizer makeDrillLocalizer() throws ClientException {
        String localArchivePath = this.config.getString(DrillOnYarnConfig.DRILL_ARCHIVE_PATH);
        if (DoYUtil.isBlank(localArchivePath)) {
            throw new ClientException("Drill archive path (" + DrillOnYarnConfig.DRILL_ARCHIVE_PATH + ") is not set.");
        }
        this.localDrillArchivePath = new File(localArchivePath);
        if (!this.localDrillArchivePath.isAbsolute()) {
            this.localDrillArchivePath = new File(DrillOnYarnConfig.instance().getLocalDrillHome(), localArchivePath);
        }
        DfsFacade.Localizer localizer = new DfsFacade.Localizer(this.dfs, this.localDrillArchivePath, "Drill");
        this.drillArchivePath = localizer.getDestPath();
        return localizer;
    }

    protected DfsFacade.Localizer makeSiteLocalizer(File siteArchive) {
        DfsFacade.Localizer localizer = new DfsFacade.Localizer(this.dfs, siteArchive, "site.tar.gz", "Site");
        this.siteArchivePath = localizer.getDestPath();
        return localizer;
    }

    protected void upload(DfsFacade.Localizer localizer) throws ClientException {
        try {
            localizer.upload();
        }
        catch (DfsFacade.DfsFacadeException e) {
            System.out.println("Failed.");
            throw new ClientException("Failed to upload " + localizer.getLabel() + " archive", e);
        }
        System.out.println("Uploaded.");
    }

    protected void defineResources(DfsFacade.Localizer localizer, String keyProp) throws ClientException {
        String key = this.config.getString(keyProp);
        try {
            localizer.defineResources(this.resources, key);
        }
        catch (DfsFacade.DfsFacadeException e) {
            throw new ClientException("Failed to get DFS status for " + localizer.getLabel() + " archive", e);
        }
    }

    protected void dump(PrintStream out) {
        out.print("Localized: ");
        out.println(this.isLocalized() ? "Yes" : "No");
        out.print("Has Site Dir: ");
        out.println(this.hasSiteDir() ? "Yes" : "No");
        out.print("Local Drill home: ");
        out.println(this.localDrillHome.getAbsolutePath());
        out.print("Remote Drill home: ");
        out.println(this.remoteDrillHome);
        if (this.hasSiteDir()) {
            out.print("Local Site dir: ");
            out.println(this.localSiteDir.getAbsolutePath());
            out.print("Remote Site dir: ");
            out.println(this.remoteSiteDir);
        }
        if (this.isLocalized()) {
            out.print("Drill archive DFS path: ");
            out.println(this.drillArchivePath);
            if (this.hasSiteDir()) {
                out.print("Site archive DFS path: ");
                out.println(this.siteArchivePath);
            }
        }
    }

    public static class UploadFiles
    extends FileUploader {
        private boolean force;

        public UploadFiles(boolean force, boolean dryRun, boolean verbose) {
            super(dryRun, verbose);
            this.force = force;
        }

        @Override
        public void run() throws ClientException {
            this.setup();
            this.uploadDrillArchive();
            if (this.hasSiteDir()) {
                this.uploadSite();
            }
            if (this.verbose || this.dryRun) {
                this.dump(System.out);
            }
        }

        private void uploadSite() throws ClientException {
            File siteArchive = this.createSiteArchive();
            try {
                this.uploadSiteArchive(siteArchive);
            }
            finally {
                siteArchive.delete();
            }
        }

        private void uploadDrillArchive() throws ClientException {
            boolean needsUpload;
            this.connectToDfs();
            DfsFacade.Localizer localizer = this.makeDrillLocalizer();
            boolean bl = needsUpload = this.force || !localizer.filesMatch();
            if (needsUpload) {
                if (!this.localDrillArchivePath.exists()) {
                    throw new ClientException("Drill archive not found: " + this.localDrillArchivePath.getAbsolutePath());
                }
                if (!this.localDrillArchivePath.canRead()) {
                    throw new ClientException("Drill archive is not readable: " + this.localDrillArchivePath.getAbsolutePath());
                }
                if (this.localDrillArchivePath.isDirectory()) {
                    throw new ClientException("Drill archive cannot be a directory: " + this.localDrillArchivePath.getAbsolutePath());
                }
            }
            this.drillArchivePath = localizer.getDestPath();
            if (needsUpload) {
                if (this.dryRun) {
                    System.out.print("Upload " + this.localDrillArchivePath.getAbsolutePath() + " to " + this.drillArchivePath);
                } else {
                    System.out.print("Uploading " + this.localDrillArchivePath.getAbsolutePath() + " to " + this.drillArchivePath + " ... ");
                    this.upload(localizer);
                }
            } else {
                System.out.println("Using existing Drill archive in DFS: " + this.drillArchivePath);
            }
            this.defineResources(localizer, DrillOnYarnConfig.DRILL_ARCHIVE_KEY);
        }

        private File createSiteArchive() throws ClientException {
            Process proc;
            File siteArchiveFile;
            try {
                siteArchiveFile = File.createTempFile("drill-site-", ".tar.gz");
            }
            catch (IOException e) {
                throw new ClientException("Failed to create site archive temp file", e);
            }
            String[] cmd = new String[]{"tar", "-C", this.localSiteDir.getAbsolutePath(), "-czf", siteArchiveFile.getAbsolutePath(), "."};
            List<String> cmdList = Arrays.asList(cmd);
            String cmdLine = DoYUtil.join(" ", cmdList);
            if (this.dryRun) {
                System.out.print("Site archive command: ");
                System.out.println(cmdLine);
                return siteArchiveFile;
            }
            ProcessBuilder builder = new ProcessBuilder(cmdList);
            builder.redirectErrorStream(true);
            try {
                proc = builder.start();
            }
            catch (IOException e) {
                throw new ClientException("Failed to launch tar process: " + cmdLine, e);
            }
            BufferedReader br = new BufferedReader(new InputStreamReader(proc.getInputStream()));
            StringBuilder buf = new StringBuilder();
            try {
                String line;
                while ((line = br.readLine()) != null) {
                    buf.append(line);
                    buf.append("\n");
                }
                br.close();
            }
            catch (IOException e) {
                throw new ClientException("Failed to read output from tar command", e);
            }
            try {
                proc.waitFor();
            }
            catch (InterruptedException e) {
                // empty catch block
            }
            if (proc.exitValue() != 0) {
                String msg = buf.toString().trim();
                throw new ClientException("Tar of site directory failed: " + msg);
            }
            return siteArchiveFile;
        }

        private void uploadSiteArchive(File siteArchive) throws ClientException {
            DfsFacade.Localizer localizer = this.makeSiteLocalizer(siteArchive);
            if (this.dryRun) {
                System.out.println("Upload site archive to " + this.siteArchivePath);
            } else {
                System.out.print("Uploading site directory " + this.localSiteDir.getAbsolutePath() + " to " + this.siteArchivePath + " ... ");
                this.upload(localizer);
            }
            this.defineResources(localizer, DrillOnYarnConfig.SITE_ARCHIVE_KEY);
        }
    }

    public static class ReuseFiles
    extends FileUploader {
        public ReuseFiles(boolean dryRun, boolean verbose) {
            super(dryRun, verbose);
        }

        @Override
        public void run() throws ClientException {
            this.setup();
            this.checkDrillArchive();
            if (this.hasSiteDir()) {
                this.checkSiteArchive();
            }
            if (this.verbose || this.dryRun) {
                this.dump(System.out);
            }
        }

        private void checkDrillArchive() throws ClientException {
            DfsFacade.Localizer localizer = this.makeDrillLocalizer();
            this.connectToDfs();
            try {
                if (!localizer.destExists()) {
                    throw new ClientException("Drill archive not found in DFS: " + this.drillArchivePath);
                }
            }
            catch (IOException e) {
                throw new ClientException("Failed to check existence of " + this.drillArchivePath, e);
            }
            if (!this.localDrillArchivePath.exists()) {
                return;
            }
            if (!localizer.filesMatch()) {
                System.out.println("Warning: Drill archive on DFS does not match the local version.");
            }
            this.defineResources(localizer, DrillOnYarnConfig.DRILL_ARCHIVE_KEY);
        }

        private void checkSiteArchive() throws ClientException {
            DfsFacade.Localizer localizer = this.makeSiteLocalizer(null);
            try {
                if (!localizer.destExists()) {
                    throw new ClientException("Drill archive not found in DFS: " + this.drillArchivePath);
                }
            }
            catch (IOException e) {
                throw new ClientException("Failed to check existence of " + this.drillArchivePath, e);
            }
            this.defineResources(localizer, DrillOnYarnConfig.SITE_ARCHIVE_KEY);
        }
    }

    public static class NonLocalized
    extends FileUploader {
        public NonLocalized(boolean dryRun, boolean verbose) {
            super(dryRun, verbose);
        }

        @Override
        public void run() throws ClientException {
            this.setup();
            this.prepareDrillHome();
            if (this.hasSiteDir()) {
                this.prepareSiteDir();
            }
            if (this.verbose || this.dryRun) {
                this.dump(System.out);
            }
        }

        private void prepareDrillHome() throws ClientException {
            String drillHomeProp = this.config.getString(DrillOnYarnConfig.DRILL_HOME);
            if (DoYUtil.isBlank(drillHomeProp)) {
                System.out.println("Warning: non-localized run " + DrillOnYarnConfig.DRILL_HOME + " is not set.");
                System.out.println("Assuming remote Drill home is the same as the local location: " + this.localDrillHome.getAbsolutePath());
            }
        }

        private void prepareSiteDir() throws ClientException {
            String siteDirProp = this.config.getString(DrillOnYarnConfig.SITE_DIR);
            if (DoYUtil.isBlank(siteDirProp)) {
                System.out.println("Warning: non-localized run " + DrillOnYarnConfig.SITE_DIR + " is not set.");
                System.out.println("Assuming remote Drill site is the same as the local location: " + this.localSiteDir.getAbsolutePath());
            }
        }
    }
}

