/*
 * Decompiled with CFR 0.152.
 */
package org.ojai.json.impl;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.util.LinkedList;
import java.util.Stack;
import org.ojai.DocumentReader;
import org.ojai.Value;
import org.ojai.annotation.API;
import org.ojai.base.DocumentReaderBase;
import org.ojai.exceptions.DecodingException;
import org.ojai.exceptions.TypeException;
import org.ojai.json.impl.ContainerContext;
import org.ojai.json.impl.JsonDocumentStream;
import org.ojai.types.ODate;
import org.ojai.types.OInterval;
import org.ojai.types.OTime;
import org.ojai.types.OTimestamp;
import org.ojai.util.Values;

@API.Internal
public class JsonStreamDocumentReader
extends DocumentReaderBase {
    private final JsonDocumentStream documentStream;
    private int mapLevel;
    private boolean eor;
    private long currentLongValue = 0L;
    private Object currentObjValue = null;
    private double currentDoubleValue = 0.0;
    private DocumentReader.EventType currentEvent;
    private String fieldName;
    private LinkedList<JsonToken> cachedTokens;
    private Stack<ContainerContext> containerStack = new Stack();
    private ContainerContext currentContainer;
    private boolean isExtended;

    JsonStreamDocumentReader(JsonDocumentStream stream) {
        this.cachedTokens = new LinkedList();
        this.documentStream = stream;
        this.mapLevel = 0;
        this.eor = false;
    }

    private DocumentReader.EventType parseMap() {
        JsonToken nextToken = this.peekToken();
        if (nextToken == JsonToken.END_OBJECT) {
            return DocumentReader.EventType.START_MAP;
        }
        if (nextToken != JsonToken.FIELD_NAME) {
            throw new DecodingException("Encountered " + nextToken + " while looking for a field name.");
        }
        String field_name = this.getCurrentName();
        if (field_name.startsWith("$")) {
            this.isExtended = true;
            switch (field_name) {
                case "$numberLong": {
                    this.setCurrentEventType(DocumentReader.EventType.LONG);
                    break;
                }
                case "$decimal": {
                    this.setCurrentEventType(DocumentReader.EventType.DECIMAL);
                    break;
                }
                case "$dateDay": {
                    this.setCurrentEventType(DocumentReader.EventType.DATE);
                    break;
                }
                case "$time": {
                    this.setCurrentEventType(DocumentReader.EventType.TIME);
                    break;
                }
                case "$date": {
                    this.setCurrentEventType(DocumentReader.EventType.TIMESTAMP);
                    break;
                }
                case "$interval": {
                    this.setCurrentEventType(DocumentReader.EventType.INTERVAL);
                    break;
                }
                case "$binary": {
                    this.setCurrentEventType(DocumentReader.EventType.BINARY);
                    break;
                }
                default: {
                    this.isExtended = false;
                    return DocumentReader.EventType.START_MAP;
                }
            }
            this.nextToken();
            nextToken = this.nextToken();
            this.cacheCurrentValue();
            nextToken = this.nextToken();
            if (nextToken != JsonToken.END_OBJECT) {
                throw new DecodingException("Encountered " + nextToken + " while looking for end object token.");
            }
            return this.currentEvent;
        }
        return DocumentReader.EventType.START_MAP;
    }

    protected void cacheCurrentValue() {
        try {
            switch (this.currentEvent) {
                case BOOLEAN: {
                    this.currentObjValue = this.isEventBoolean() ? this.getParser().getBooleanValue() : Boolean.valueOf(this.getValueAsString()).booleanValue();
                    break;
                }
                case STRING: {
                    this.currentObjValue = this.getParser().getText();
                    break;
                }
                case BYTE: {
                    this.currentLongValue = this.getValueAsLong() & 0xFFL;
                    break;
                }
                case SHORT: {
                    this.currentLongValue = this.getValueAsLong() & 0xFFFFL;
                    break;
                }
                case INT: {
                    this.currentLongValue = this.getValueAsLong() & 0xFFFFFFFFFFFFFFFFL;
                    break;
                }
                case LONG: {
                    this.currentLongValue = this.getValueAsLong();
                    break;
                }
                case FLOAT: 
                case DOUBLE: {
                    this.currentDoubleValue = this.getValueAsDouble();
                    break;
                }
                case DECIMAL: {
                    this.currentObjValue = Values.parseBigDecimal(this.getParser().getText());
                    break;
                }
                case DATE: {
                    this.currentObjValue = ODate.parse(this.getParser().getText());
                    break;
                }
                case TIME: {
                    this.currentObjValue = OTime.parse(this.getParser().getText());
                    break;
                }
                case TIMESTAMP: {
                    this.currentObjValue = OTimestamp.parse(this.getParser().getText());
                    break;
                }
                case INTERVAL: {
                    this.currentLongValue = this.getValueAsLong();
                    break;
                }
                case BINARY: {
                    this.currentObjValue = ByteBuffer.wrap(this.getParser().getBinaryValue());
                    break;
                }
            }
        }
        catch (IOException ie) {
            throw new DecodingException(ie);
        }
    }

    private void checkEventType(DocumentReader.EventType eventType) throws TypeException {
        if (this.currentEvent != eventType) {
            throw new TypeException(String.format("Event type mismatch, expected %s, got %s", new Object[]{eventType, this.currentEvent}));
        }
    }

    private void checkNumericEventType() throws TypeException {
        switch (this.currentEvent) {
            case BYTE: 
            case SHORT: 
            case INT: 
            case LONG: 
            case FLOAT: 
            case DOUBLE: {
                return;
            }
        }
        throw new TypeException("Event type mismatch. Expected numeric type, found " + (Object)((Object)this.currentEvent));
    }

    private JsonParser getParser() {
        return this.documentStream.getParser();
    }

    private boolean isEventNumeric() {
        return this.getParser().getCurrentToken().isNumeric();
    }

    private boolean isEventBoolean() {
        return this.getParser().getCurrentToken().isBoolean();
    }

    private long getValueAsLong() throws NumberFormatException, IOException {
        return this.isEventNumeric() ? this.getParser().getLongValue() : Long.valueOf(this.getValueAsString()).longValue();
    }

    private double getValueAsDouble() throws NumberFormatException, IOException {
        return this.isEventNumeric() ? this.getParser().getDoubleValue() : Double.valueOf(this.getValueAsString()).doubleValue();
    }

    protected String getValueAsString() {
        try {
            return this.getParser().getText();
        }
        catch (IOException e) {
            throw new DecodingException(e);
        }
    }

    protected boolean eor() {
        return this.eor || !this.hasMoreTokens();
    }

    @Override
    public DocumentReader.EventType next() {
        this.isExtended = false;
        if (this.eor()) {
            return null;
        }
        JsonToken currentToken = this.nextToken();
        if (currentToken == JsonToken.FIELD_NAME) {
            this.fieldName = this.getCurrentName();
            currentToken = this.nextToken();
        }
        this.updateCurrentContainer();
        switch (currentToken) {
            case START_OBJECT: {
                this.setCurrentEventType(this.parseMap());
                if (this.currentEvent != DocumentReader.EventType.START_MAP) break;
                this.containerStack.push(new ContainerContext(Value.Type.MAP, this.fieldName));
                break;
            }
            case END_OBJECT: {
                this.setCurrentEventType(DocumentReader.EventType.END_MAP);
                ContainerContext lastContainer = this.containerStack.pop();
                if (lastContainer.getType() == Value.Type.MAP) {
                    this.fieldName = lastContainer.getFieldName();
                }
                this.updateCurrentContainer();
                break;
            }
            case START_ARRAY: {
                this.setCurrentEventType(DocumentReader.EventType.START_ARRAY);
                if (!this.inMap()) {
                    this.currentContainer.incrementIndex();
                }
                this.containerStack.push(new ContainerContext(Value.Type.ARRAY));
                break;
            }
            case END_ARRAY: {
                this.setCurrentEventType(DocumentReader.EventType.END_ARRAY);
                this.containerStack.pop();
                this.updateCurrentContainer();
                break;
            }
            case VALUE_NULL: {
                this.setCurrentEventType(DocumentReader.EventType.NULL).cacheCurrentValue();
                break;
            }
            case VALUE_TRUE: 
            case VALUE_FALSE: {
                this.setCurrentEventType(DocumentReader.EventType.BOOLEAN).cacheCurrentValue();
                break;
            }
            case VALUE_STRING: {
                this.setCurrentEventType(DocumentReader.EventType.STRING).cacheCurrentValue();
                break;
            }
            case VALUE_NUMBER_INT: 
            case VALUE_NUMBER_FLOAT: {
                this.setCurrentEventType(DocumentReader.EventType.DOUBLE).cacheCurrentValue();
                break;
            }
            default: {
                throw new DecodingException("Encountered unexpected token of type: " + currentToken);
            }
        }
        if (!this.inMap() && this.currentEvent != DocumentReader.EventType.END_MAP && this.currentEvent != DocumentReader.EventType.START_ARRAY && this.currentEvent != DocumentReader.EventType.END_ARRAY) {
            this.currentContainer.incrementIndex();
        }
        if (this.currentEvent == DocumentReader.EventType.START_MAP) {
            ++this.mapLevel;
        } else if (this.currentEvent == DocumentReader.EventType.END_MAP) {
            --this.mapLevel;
        }
        if (this.mapLevel == 0) {
            this.eor = true;
        }
        return this.currentEvent;
    }

    void readFully() {
        JsonToken token;
        if (this.eor) {
            return;
        }
        while ((token = this.nextToken()) != null) {
            switch (token) {
                case START_OBJECT: {
                    ++this.mapLevel;
                    break;
                }
                case END_OBJECT: {
                    --this.mapLevel;
                    break;
                }
            }
            if (this.mapLevel != 0) continue;
        }
        this.eor = true;
    }

    @Override
    public boolean inMap() {
        return this.currentContainer == null || this.currentContainer.getType() == Value.Type.MAP;
    }

    @Override
    public String getFieldName() {
        if (!this.inMap()) {
            throw new IllegalStateException("Not traversing a map!");
        }
        return this.fieldName;
    }

    @Override
    public int getArrayIndex() {
        if (this.inMap()) {
            throw new IllegalStateException("Not traversing an array!");
        }
        return this.currentContainer.getIndex();
    }

    @Override
    public byte getByte() {
        this.checkNumericEventType();
        return (byte)this.currentLongValue;
    }

    @Override
    public short getShort() {
        this.checkNumericEventType();
        return (short)this.currentLongValue;
    }

    @Override
    public int getInt() {
        this.checkNumericEventType();
        return (int)this.currentLongValue;
    }

    @Override
    public long getLong() {
        this.checkEventType(DocumentReader.EventType.LONG);
        return this.currentLongValue;
    }

    @Override
    public float getFloat() {
        this.checkEventType(DocumentReader.EventType.FLOAT);
        return (float)this.currentDoubleValue;
    }

    @Override
    public double getDouble() {
        this.checkEventType(DocumentReader.EventType.DOUBLE);
        return this.currentDoubleValue;
    }

    @Override
    public BigDecimal getDecimal() {
        this.checkEventType(DocumentReader.EventType.DECIMAL);
        return (BigDecimal)this.currentObjValue;
    }

    @Override
    public int getDecimalPrecision() {
        BigDecimal decimal = this.getDecimal();
        if (decimal != null) {
            return decimal.precision();
        }
        return 0;
    }

    @Override
    public int getDecimalScale() {
        BigDecimal decimal = this.getDecimal();
        if (decimal != null) {
            return decimal.scale();
        }
        return 0;
    }

    @Override
    public int getDecimalValueAsInt() {
        BigDecimal decimal = this.getDecimal();
        if (decimal != null) {
            return decimal.intValueExact();
        }
        return 0;
    }

    @Override
    public long getDecimalValueAsLong() {
        BigDecimal decimal = this.getDecimal();
        if (decimal != null) {
            return decimal.longValueExact();
        }
        return 0L;
    }

    @Override
    public ByteBuffer getDecimalValueAsBytes() {
        BigDecimal decimal = this.getDecimal();
        if (decimal != null) {
            BigInteger decimalInteger = decimal.unscaledValue();
            byte[] bytearray = decimalInteger.toByteArray();
            return ByteBuffer.wrap(bytearray);
        }
        return null;
    }

    @Override
    public boolean getBoolean() {
        this.checkEventType(DocumentReader.EventType.BOOLEAN);
        return (Boolean)this.currentObjValue;
    }

    @Override
    public String getString() {
        this.checkEventType(DocumentReader.EventType.STRING);
        return (String)this.currentObjValue;
    }

    @Override
    public long getTimestampLong() {
        this.checkEventType(DocumentReader.EventType.TIMESTAMP);
        return ((OTimestamp)this.currentObjValue).getMillis();
    }

    @Override
    public OTimestamp getTimestamp() {
        this.checkEventType(DocumentReader.EventType.TIMESTAMP);
        return (OTimestamp)this.currentObjValue;
    }

    @Override
    public int getDateInt() {
        this.checkEventType(DocumentReader.EventType.DATE);
        return ((ODate)this.currentObjValue).toDaysSinceEpoch();
    }

    @Override
    public ODate getDate() {
        this.checkEventType(DocumentReader.EventType.DATE);
        return (ODate)this.currentObjValue;
    }

    @Override
    public int getTimeInt() {
        this.checkEventType(DocumentReader.EventType.TIME);
        return ((OTime)this.currentObjValue).toTimeInMillis();
    }

    @Override
    public OTime getTime() {
        this.checkEventType(DocumentReader.EventType.TIME);
        return (OTime)this.currentObjValue;
    }

    @Override
    public OInterval getInterval() {
        this.checkEventType(DocumentReader.EventType.INTERVAL);
        return new OInterval(this.currentLongValue);
    }

    @Override
    public int getIntervalDays() {
        this.checkEventType(DocumentReader.EventType.INTERVAL);
        return (int)(this.currentLongValue / 86400000L);
    }

    @Override
    public long getIntervalMillis() {
        this.checkEventType(DocumentReader.EventType.INTERVAL);
        return this.currentLongValue;
    }

    @Override
    public ByteBuffer getBinary() {
        this.checkEventType(DocumentReader.EventType.BINARY);
        return (ByteBuffer)this.currentObjValue;
    }

    private void updateCurrentContainer() {
        this.currentContainer = this.containerStack.isEmpty() ? null : this.containerStack.peek();
    }

    protected JsonToken peekToken() {
        if (this.hasMoreTokens()) {
            return this.cachedTokens.peek();
        }
        throw new DecodingException("No more Json tokens.");
    }

    protected JsonToken nextToken() {
        if (this.hasMoreTokens()) {
            return this.cachedTokens.remove();
        }
        throw new DecodingException("No more Json tokens.");
    }

    protected boolean hasMoreTokens() {
        try {
            if (!this.cachedTokens.isEmpty()) {
                return true;
            }
            JsonToken token = this.getParser().nextToken();
            if (token == null) {
                return false;
            }
            if (this.mapLevel == 0) {
                if (token == JsonToken.START_ARRAY) {
                    token = this.getParser().nextToken();
                }
                if (token == JsonToken.END_ARRAY && (token = this.getParser().nextToken()) == JsonToken.START_ARRAY) {
                    token = this.getParser().nextToken();
                }
                if (token == null) {
                    return false;
                }
            }
            this.cachedTokens.add(token);
            return true;
        }
        catch (IOException e) {
            throw new DecodingException(e);
        }
    }

    protected String getCurrentName() {
        try {
            return this.getParser().getCurrentName();
        }
        catch (IOException ie) {
            throw new DecodingException(ie);
        }
    }

    protected Object getCurrentObj() {
        return this.currentObjValue;
    }

    protected JsonStreamDocumentReader setArrayIndex(int index) {
        if (this.inMap()) {
            throw new IllegalStateException("Not traversing an array!");
        }
        this.currentContainer.setIndex(index);
        return this;
    }

    protected JsonStreamDocumentReader setFieldName(String fieldName) {
        if (!this.inMap()) {
            throw new IllegalStateException("Not traversing a map!");
        }
        this.fieldName = fieldName;
        return this;
    }

    protected JsonStreamDocumentReader setCurrentObj(Object obj) {
        this.currentObjValue = obj;
        return this;
    }

    protected DocumentReader.EventType getCurrentEventType() {
        return this.currentEvent;
    }

    protected JsonStreamDocumentReader setCurrentEventType(DocumentReader.EventType eventType) {
        this.currentEvent = eventType;
        return this;
    }

    protected long getCurrentLongValue() {
        return this.currentLongValue;
    }

    protected JsonStreamDocumentReader setCurrentLongValue(long value) {
        this.currentLongValue = value;
        return this;
    }

    protected double getCurrentDoubleValue() {
        return this.currentDoubleValue;
    }

    protected JsonStreamDocumentReader setCurrentDoubleValue(double value) {
        this.currentDoubleValue = value;
        return this;
    }

    protected boolean isExtended() {
        return this.isExtended;
    }

    @Override
    public DocumentReader.EventType getCurrentEvent() {
        return this.currentEvent;
    }
}

