/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.schemaregistry.json;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.BooleanNode;
import com.fasterxml.jackson.databind.node.FloatNode;
import com.fasterxml.jackson.databind.node.IntNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.NullNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.google.common.collect.ImmutableSet;
import io.confluent.kafka.schemaregistry.ParsedSchema;
import io.confluent.kafka.schemaregistry.client.rest.entities.Metadata;
import io.confluent.kafka.schemaregistry.client.rest.entities.SchemaEntity;
import io.confluent.kafka.schemaregistry.client.rest.entities.SchemaReference;
import io.confluent.kafka.schemaregistry.json.JsonSchema;
import io.confluent.kafka.schemaregistry.json.JsonSchemaProvider;
import io.confluent.kafka.schemaregistry.json.JsonSchemaUtils;
import io.confluent.kafka.schemaregistry.json.SpecificationVersion;
import io.confluent.kafka.schemaregistry.json.diff.SchemaDiff;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import org.everit.json.schema.Schema;
import org.everit.json.schema.ValidationException;
import org.junit.Assert;
import org.junit.Test;

public class JsonSchemaTest {
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private static final String recordSchemaString = "{\"properties\": {\n     \"null\": {\"type\": \"null\"},\n     \"boolean\": {\"type\": \"boolean\"},\n     \"number\": {\"type\": \"number\"},\n     \"string\": {\"type\": \"string\"}\n  },\n  \"additionalProperties\": false\n}";
    private static final JsonSchema recordSchema = new JsonSchema("{\"properties\": {\n     \"null\": {\"type\": \"null\"},\n     \"boolean\": {\"type\": \"boolean\"},\n     \"number\": {\"type\": \"number\"},\n     \"string\": {\"type\": \"string\"}\n  },\n  \"additionalProperties\": false\n}");
    private static final String recordWithDefaultsSchemaString = "{\"properties\": {\n     \"null\": {\"type\": \"null\", \"default\": null},\n     \"boolean\": {\"type\": \"boolean\", \"default\": true},\n     \"number\": {\"type\": \"number\", \"default\": 123},\n     \"string\": {\"type\": \"string\", \"default\": \"abc\"}\n  },\n  \"additionalProperties\": false\n}";
    private static final JsonSchema recordWithDefaultsSchema = new JsonSchema("{\"properties\": {\n     \"null\": {\"type\": \"null\", \"default\": null},\n     \"boolean\": {\"type\": \"boolean\", \"default\": true},\n     \"number\": {\"type\": \"number\", \"default\": 123},\n     \"string\": {\"type\": \"string\", \"default\": \"abc\"}\n  },\n  \"additionalProperties\": false\n}");
    private static final String arraySchemaString = "{\"type\": \"array\", \"items\": { \"type\": \"string\" } }";
    private static final JsonSchema arraySchema = new JsonSchema("{\"type\": \"array\", \"items\": { \"type\": \"string\" } }");
    private static final String unionSchemaString = "{\n  \"oneOf\": [\n    { \"type\": \"string\", \"maxLength\": 5 },\n    { \"type\": \"number\", \"minimum\": 0 }\n  ]\n}";
    private static final JsonSchema unionSchema = new JsonSchema("{\n  \"oneOf\": [\n    { \"type\": \"string\", \"maxLength\": 5 },\n    { \"type\": \"number\", \"minimum\": 0 }\n  ]\n}");
    private static final String enumSchemaString = "{ \"type\": \"string\", \"enum\": [\"red\", \"amber\", \"green\"] }";
    private static final JsonSchema enumSchema = new JsonSchema("{ \"type\": \"string\", \"enum\": [\"red\", \"amber\", \"green\"] }");
    private static final String invalidSchemaString = "{\"properties\": {\n  \"string\": {\"type\": \"str\"}\n  }  \"additionalProperties\": false\n}";
    private static final String schema = "{\n  \"$schema\": \"http://json-schema.org/draft-07/schema#\",\n  \"$id\": \"task.schema.json\",\n  \"title\": \"Task\",\n  \"description\": \"A task\",\n  \"type\": [\"null\", \"object\"],\n  \"properties\": {\n    \"parent\": {\n        \"$ref\": \"task.schema.json\"\n    },    \n    \"title\": {\n        \"description\": \"Task title\",\n        \"type\": \"string\"\n    }\n  }\n}";

    @Test
    public void testHasTopLevelField() {
        JsonSchema parsedSchema = new JsonSchema(schema);
        Assert.assertTrue((boolean)parsedSchema.hasTopLevelField("parent"));
        Assert.assertFalse((boolean)parsedSchema.hasTopLevelField("doesNotExist"));
    }

    @Test
    public void testGetReservedFields() {
        Metadata reservedFieldMetadata = new Metadata(Collections.emptyMap(), Collections.singletonMap("confluent:reserved", "name, city"), Collections.emptySet());
        JsonSchema parsedSchema = new JsonSchema(schema, Collections.emptyList(), Collections.emptyMap(), reservedFieldMetadata, null, null);
        Assert.assertEquals((Object)ImmutableSet.of((Object)"name", (Object)"city"), (Object)parsedSchema.getReservedFields());
    }

    @Test
    public void testPrimitiveTypesToJsonSchema() throws Exception {
        Object envelope = JsonSchemaUtils.toObject((String)null, (JsonSchema)JsonSchemaTest.createPrimitiveSchema("null"));
        JsonNode result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((Object)NullNode.getInstance(), (Object)result);
        envelope = JsonSchemaUtils.toObject((String)"true", (JsonSchema)JsonSchemaTest.createPrimitiveSchema("boolean"));
        result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertTrue((boolean)result.asBoolean());
        envelope = JsonSchemaUtils.toObject((JsonNode)BooleanNode.getTrue(), (JsonSchema)JsonSchemaTest.createPrimitiveSchema("boolean"));
        result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((Object)BooleanNode.getTrue(), (Object)result);
        envelope = JsonSchemaUtils.toObject((String)"false", (JsonSchema)JsonSchemaTest.createPrimitiveSchema("boolean"));
        result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertFalse((boolean)result.asBoolean());
        envelope = JsonSchemaUtils.toObject((JsonNode)BooleanNode.getFalse(), (JsonSchema)JsonSchemaTest.createPrimitiveSchema("boolean"));
        result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((Object)BooleanNode.getFalse(), (Object)result);
        envelope = JsonSchemaUtils.toObject((String)"12", (JsonSchema)JsonSchemaTest.createPrimitiveSchema("number"));
        result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((long)12L, (long)result.asInt());
        envelope = JsonSchemaUtils.toObject((JsonNode)IntNode.valueOf((int)12), (JsonSchema)JsonSchemaTest.createPrimitiveSchema("number"));
        result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((Object)IntNode.valueOf((int)12), (Object)result);
        envelope = JsonSchemaUtils.toObject((String)"23.2", (JsonSchema)JsonSchemaTest.createPrimitiveSchema("number"));
        result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((double)23.2, (double)result.asDouble(), (double)0.1);
        envelope = JsonSchemaUtils.toObject((JsonNode)FloatNode.valueOf((float)23.2f), (JsonSchema)JsonSchemaTest.createPrimitiveSchema("number"));
        result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((Object)FloatNode.valueOf((float)23.2f), (Object)result);
        envelope = JsonSchemaUtils.toObject((String)"\"a string\"", (JsonSchema)JsonSchemaTest.createPrimitiveSchema("string"));
        result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((Object)"a string", (Object)result.asText());
        envelope = JsonSchemaUtils.toObject((JsonNode)TextNode.valueOf((String)"a string"), (JsonSchema)JsonSchemaTest.createPrimitiveSchema("string"));
        result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((Object)TextNode.valueOf((String)"a string"), (Object)result);
    }

    @Test
    public void testRecordToJsonSchema() throws Exception {
        String json = "{\n    \"null\": null,\n    \"boolean\": true,\n    \"number\": 12,\n    \"string\": \"string\"\n}";
        JsonNode envelope = (JsonNode)JsonSchemaUtils.toObject((String)json, (JsonSchema)recordSchema);
        JsonNode result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertTrue((boolean)result.get("null").isNull());
        Assert.assertTrue((boolean)result.get("boolean").booleanValue());
        Assert.assertEquals((long)12L, (long)result.get("number").intValue());
        Assert.assertEquals((Object)"string", (Object)result.get("string").textValue());
    }

    @Test
    public void testRecordWithDefaultsToJsonSchema() throws Exception {
        String json = "{}";
        JsonNode envelope = (JsonNode)JsonSchemaUtils.toObject((String)json, (JsonSchema)recordWithDefaultsSchema);
        JsonNode result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((Object)true, (Object)result.get("null").isNull());
        Assert.assertEquals((Object)true, (Object)result.get("boolean").booleanValue());
        Assert.assertEquals((long)123L, (long)result.get("number").intValue());
        Assert.assertEquals((Object)"abc", (Object)result.get("string").textValue());
    }

    @Test(expected=ValidationException.class)
    public void testInvalidRecordToJsonSchema() throws Exception {
        String json = "{\n    \"null\": null,\n    \"boolean\": true,\n    \"number\": 12,\n    \"string\": \"string\",\n    \"badString\": \"string\"\n}";
        JsonNode envelope = (JsonNode)JsonSchemaUtils.toObject((String)json, (JsonSchema)recordSchema);
        JsonNode result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertTrue((boolean)result.get("null").isNull());
        Assert.assertTrue((boolean)result.get("boolean").booleanValue());
        Assert.assertEquals((long)12L, (long)result.get("number").intValue());
        Assert.assertEquals((Object)"string", (Object)result.get("string").textValue());
    }

    @Test
    public void testArrayToJsonSchema() throws Exception {
        String json = "[\"one\", \"two\", \"three\"]";
        Object envelope = JsonSchemaUtils.toObject((String)json, (JsonSchema)arraySchema);
        JsonNode result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        ArrayNode arrayNode = (ArrayNode)result;
        Iterator elements = arrayNode.elements();
        ArrayList<String> strings = new ArrayList<String>();
        while (elements.hasNext()) {
            strings.add(((JsonNode)elements.next()).textValue());
        }
        Assert.assertArrayEquals((Object[])new String[]{"one", "two", "three"}, (Object[])strings.toArray());
    }

    @Test
    public void testUnionToJsonSchema() throws Exception {
        Object envelope = JsonSchemaUtils.toObject((String)"\"test\"", (JsonSchema)unionSchema);
        JsonNode result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((Object)"test", (Object)result.asText());
        envelope = JsonSchemaUtils.toObject((String)"12", (JsonSchema)unionSchema);
        result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((long)12L, (long)result.asInt());
        try {
            JsonSchemaUtils.toObject((String)"-1", (JsonSchema)unionSchema);
            Assert.fail((String)"Trying to use negative number should fail");
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @Test
    public void testEnumToJsonSchema() throws Exception {
        Object envelope = JsonSchemaUtils.toObject((String)"\"red\"", (JsonSchema)enumSchema);
        JsonNode result = (JsonNode)JsonSchemaUtils.getValue((Object)envelope);
        Assert.assertEquals((Object)"red", (Object)result.asText());
        try {
            JsonSchemaUtils.toObject((String)"\"yellow\"", (JsonSchema)enumSchema);
            Assert.fail((String)"Trying to use non-enum should fail");
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @Test
    public void testPrimitiveTypesToJson() throws Exception {
        JsonNode result = objectMapper.readTree(JsonSchemaUtils.toJson((Object)0));
        Assert.assertTrue((boolean)result.isNumber());
        result = objectMapper.readTree(JsonSchemaUtils.toJson((Object)0L));
        Assert.assertTrue((boolean)result.isNumber());
        result = objectMapper.readTree(JsonSchemaUtils.toJson((Object)Float.valueOf(0.1f)));
        Assert.assertTrue((boolean)result.isNumber());
        result = objectMapper.readTree(JsonSchemaUtils.toJson((Object)0.1));
        Assert.assertTrue((boolean)result.isNumber());
        result = objectMapper.readTree(JsonSchemaUtils.toJson((Object)true));
        Assert.assertTrue((boolean)result.isBoolean());
        result = objectMapper.readTree(JsonSchemaUtils.toJson((Object)"abcdefg"));
        Assert.assertTrue((boolean)result.isTextual());
        Assert.assertEquals((Object)"abcdefg", (Object)result.textValue());
    }

    @Test
    public void testRecordToJson() throws Exception {
        String json = "{\n    \"null\": null,\n    \"boolean\": true,\n    \"number\": 12,\n    \"string\": \"string\"\n}";
        JsonNode data = new ObjectMapper().readTree(json);
        JsonNode result = objectMapper.readTree(JsonSchemaUtils.toJson((Object)data));
        Assert.assertTrue((boolean)result.isObject());
        Assert.assertTrue((boolean)result.get("null").isNull());
        Assert.assertTrue((boolean)result.get("boolean").isBoolean());
        Assert.assertTrue((boolean)result.get("boolean").booleanValue());
        Assert.assertTrue((boolean)result.get("number").isIntegralNumber());
        Assert.assertEquals((long)12L, (long)result.get("number").intValue());
        Assert.assertTrue((boolean)result.get("string").isTextual());
        Assert.assertEquals((Object)"string", (Object)result.get("string").textValue());
    }

    @Test
    public void testArrayToJson() throws Exception {
        String json = "[\"one\", \"two\", \"three\"]";
        JsonNode data = new ObjectMapper().readTree(json);
        JsonNode result = objectMapper.readTree(JsonSchemaUtils.toJson((Object)data));
        Assert.assertTrue((boolean)result.isArray());
        Assert.assertEquals((long)3L, (long)result.size());
        Assert.assertEquals((Object)JsonNodeFactory.instance.textNode("one"), (Object)result.get(0));
        Assert.assertEquals((Object)JsonNodeFactory.instance.textNode("two"), (Object)result.get(1));
        Assert.assertEquals((Object)JsonNodeFactory.instance.textNode("three"), (Object)result.get(2));
    }

    @Test
    public void testSchemaWithDraft4() throws Exception {
        TestObj testObj = new TestObj();
        String actual = JsonSchemaUtils.getSchema((Object)testObj, (SpecificationVersion)SpecificationVersion.DRAFT_4, (boolean)true, null).toString();
        String expected = "{\"$schema\":\"http://json-schema.org/draft-04/schema#\",\"title\":\"Test Obj\",\"type\":\"object\",\"additionalProperties\":false,\"properties\":{\"prop\":{\"oneOf\":[{\"type\":\"null\",\"title\":\"Not included\"},{\"type\":\"string\"}]}}}";
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testSchemaWithOneofs() throws Exception {
        TestObj testObj = new TestObj();
        String actual = JsonSchemaUtils.getSchema((Object)testObj).toString();
        String expected = "{\"$schema\":\"http://json-schema.org/draft-07/schema#\",\"title\":\"Test Obj\",\"type\":\"object\",\"additionalProperties\":false,\"properties\":{\"prop\":{\"oneOf\":[{\"type\":\"null\",\"title\":\"Not included\"},{\"type\":\"string\"}]}}}";
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testSchemaWithoutOneofs() throws Exception {
        TestObj testObj = new TestObj();
        String actual = JsonSchemaUtils.getSchema((Object)testObj, (SpecificationVersion)SpecificationVersion.DRAFT_7, (boolean)false, null).toString();
        String expected = "{\"$schema\":\"http://json-schema.org/draft-07/schema#\",\"title\":\"Test Obj\",\"type\":\"object\",\"additionalProperties\":false,\"properties\":{\"prop\":{\"type\":\"string\"}}}";
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testSchemaWithAdditionalProperties() throws Exception {
        TestObj testObj = new TestObj();
        String actual = JsonSchemaUtils.getSchema((Object)testObj, (SpecificationVersion)SpecificationVersion.DRAFT_7, (boolean)false, (boolean)false, null).toString();
        String expected = "{\"$schema\":\"http://json-schema.org/draft-07/schema#\",\"title\":\"Test Obj\",\"type\":\"object\",\"additionalProperties\":true,\"properties\":{\"prop\":{\"type\":\"string\"}}}";
        Assert.assertEquals((Object)expected, (Object)actual);
    }

    @Test
    public void testEnvelopeWithReferences() throws Exception {
        String draft = "\"$schema\": \"http://json-schema.org/draft-07/schema#\"";
        this.testEnvelopeWithReferences(draft);
    }

    @Test
    public void testEnvelopeWithReferencesDraft_2020_12() throws Exception {
        String draft = "\"$schema\": \"https://json-schema.org/draft/2020-12/schema\"";
        this.testEnvelopeWithReferences(draft);
    }

    private void testEnvelopeWithReferences(String draft) throws Exception {
        Map<String, String> schemas = JsonSchemaTest.getJsonSchemaWithReferences(draft);
        SchemaReference ref = new SchemaReference("ref.json", "reference", Integer.valueOf(1));
        JsonSchema schema = new JsonSchema(schemas.get("main.json"), Collections.singletonList(ref), Collections.singletonMap("ref.json", schemas.get("ref.json")), null);
        schema.validate(true);
        ObjectNode envelope = JsonSchemaUtils.envelope((JsonSchema)schema, null);
        JsonSchema schema2 = JsonSchemaUtils.getSchema((Object)envelope);
        schema2.validate(true);
        Assert.assertEquals((Object)schema, (Object)schema2);
    }

    @Test
    public void testRecursiveSchema() {
        String schema = schema;
        JsonSchema jsonSchema = new JsonSchema(schema);
        List diff = SchemaDiff.compare((Schema)jsonSchema.rawSchema(), (Schema)jsonSchema.rawSchema());
        Assert.assertEquals((long)0L, (long)diff.size());
    }

    @Test
    public void testParseSchema() {
        JsonSchemaProvider jsonSchemaProvider = new JsonSchemaProvider();
        ParsedSchema parsedSchema = jsonSchemaProvider.parseSchemaOrElseThrow(new io.confluent.kafka.schemaregistry.client.rest.entities.Schema(null, null, null, "JSON", new ArrayList(), recordSchemaString), false, false);
        Optional parsedSchemaOptional = jsonSchemaProvider.parseSchema(recordSchemaString, new ArrayList(), false, false);
        Assert.assertNotNull((Object)parsedSchema);
        Assert.assertTrue((boolean)parsedSchemaOptional.isPresent());
    }

    @Test(expected=IllegalArgumentException.class)
    public void testParseSchemaThrowException() {
        JsonSchemaProvider jsonSchemaProvider = new JsonSchemaProvider();
        jsonSchemaProvider.parseSchemaOrElseThrow(new io.confluent.kafka.schemaregistry.client.rest.entities.Schema(null, null, null, "JSON", new ArrayList(), invalidSchemaString), false, false);
    }

    @Test
    public void testParseSchemaSuppressException() {
        JsonSchemaProvider jsonSchemaProvider = new JsonSchemaProvider();
        Optional parsedSchema = jsonSchemaProvider.parseSchema(invalidSchemaString, new ArrayList(), false, false);
        Assert.assertFalse((boolean)parsedSchema.isPresent());
    }

    @Test
    public void testSchemasDifferentFieldOrder() {
        String schema1 = "{\n  \"title\": \"Person\",\n  \"type\": \"object\",\n  \"properties\": {\n    \"lastName\": {\n      \"type\": \"string\",\n      \"description\": \"The person's last name.\"\n    },\n    \"firstName\": {\n      \"type\": \"string\",\n      \"description\": \"The person's first name.\"\n    }\n  }\n}";
        String schema2 = "{\n  \"title\": \"Person\",\n  \"type\": \"object\",\n  \"properties\": {\n    \"firstName\": {\n      \"type\": \"string\",\n      \"description\": \"The person's first name.\"\n    },\n    \"lastName\": {\n      \"type\": \"string\",\n      \"description\": \"The person's last name.\"\n    }\n  }\n}";
        JsonSchema jsonSchema1 = new JsonSchema(schema1);
        JsonSchema jsonSchema2 = new JsonSchema(schema2);
        Assert.assertNotEquals((Object)jsonSchema1, (Object)jsonSchema2);
    }

    @Test
    public void testBasicAddAndRemoveTags() {
        String schemaString = "{\n  \"$schema\": \"http://json-schema.org/draft-07/schema#\",\n  \"$id\": \"http://example.com/myURI.schema.json\",\n  \"title\": \"SampleRecord\",\n  \"description\": \"Sample schema to help you get started.\",\n  \"type\": \"object\",\n  \"additionalProperties\": false,\n  \"properties\": {\n    \"myfield1\": {\n      \"type\": \"array\",\n      \"items\": {\n        \"type\": \"object\",\n        \"title\": \"arrayRecord\",\n        \"properties\": {\n          \"field1\" : {\n            \"type\": \"string\"\n          },\n          \"field2\": {\n            \"type\": \"number\"\n          }\n        }\n      }\n    },\n    \"myfield2\": {\n      \"allOf\": [\n        { \"type\": \"string\" },\n        { \"type\": \"object\",\n          \"title\": \"nestedUnion\",\n          \"properties\": {\n            \"nestedUnionField1\": { \"type\": \"boolean\"},\n            \"nestedUnionField2\": { \"type\": \"number\"}\n          }\n        }\n      ]\n    },\n    \"myfield3\": {\n      \"not\": {\n        \"type\": \"object\",\n        \"title\": \"nestedNot\",\n        \"properties\": {\n          \"nestedNotField1\": { \"type\": \"boolean\"},\n          \"nestedNotField2\": { \"type\": \"string\"}\n        }\n      }\n    },\n    \"myfield4\": { \"enum\": [\"red\", \"amber\", \"green\"]}\n  }\n}";
        String addedTagSchema = "{\n  \"$schema\": \"http://json-schema.org/draft-07/schema#\",\n  \"$id\": \"http://example.com/myURI.schema.json\",\n  \"title\": \"SampleRecord\",\n  \"description\": \"Sample schema to help you get started.\",\n  \"type\": \"object\",\n  \"additionalProperties\": false,\n  \"properties\": {\n    \"myfield1\": {\n      \"type\": \"array\",\n      \"items\": {\n        \"type\": \"object\",\n        \"title\": \"arrayRecord\",\n        \"properties\": {\n          \"field1\" : {\n            \"type\": \"string\",\n            \"confluent:tags\": [ \"PII\" ]\n          },\n          \"field2\": {\n            \"type\": \"number\"\n          }\n        },\n        \"confluent:tags\": [ \"record\" ]\n      }\n    },\n    \"myfield2\": {\n      \"allOf\": [\n        { \"type\": \"string\" },\n        { \"type\": \"object\",\n          \"title\": \"nestedUnion\",\n          \"properties\": {\n            \"nestedUnionField1\": { \"type\": \"boolean\"},\n            \"nestedUnionField2\": { \n              \"type\": \"number\", \n              \"confluent:tags\": [ \"PII\" ]\n            }\n          },\n          \"confluent:tags\": [ \"record\" ]\n        }\n      ]\n    },\n    \"myfield3\": {\n      \"not\": {\n        \"type\": \"object\",\n        \"title\": \"nestedNot\",\n        \"properties\": {\n          \"nestedNotField1\": { \"type\": \"boolean\" },\n          \"nestedNotField2\": {\n            \"type\": \"string\",\n            \"confluent:tags\": [ \"PII\" ]\n          }\n        },\n        \"confluent:tags\": [ \"record\" ]\n      }\n    },\n    \"myfield4\": { \n      \"enum\": [\"red\", \"amber\", \"green\"],\n      \"confluent:tags\": [ \"PII\" ]\n    }\n  }\n}\n";
        JsonSchema schema = new JsonSchema(schemaString);
        JsonSchema expectSchema = new JsonSchema(addedTagSchema);
        HashMap<SchemaEntity, Set<String>> tags = new HashMap<SchemaEntity, Set<String>>();
        tags.put(new SchemaEntity("object.myfield1.array.object.field1", SchemaEntity.EntityType.SR_FIELD), Collections.singleton("PII"));
        tags.put(new SchemaEntity("object.myfield2.allof.0.object.nestedUnionField2", SchemaEntity.EntityType.SR_FIELD), Collections.singleton("PII"));
        tags.put(new SchemaEntity("object.myfield3.not.object.nestedNotField2", SchemaEntity.EntityType.SR_FIELD), Collections.singleton("PII"));
        tags.put(new SchemaEntity("object.myfield4", SchemaEntity.EntityType.SR_FIELD), Collections.singleton("PII"));
        tags.put(new SchemaEntity("object.myfield1.array.object", SchemaEntity.EntityType.SR_RECORD), Collections.singleton("record"));
        tags.put(new SchemaEntity("object.myfield2.allof.0.object", SchemaEntity.EntityType.SR_RECORD), Collections.singleton("record"));
        tags.put(new SchemaEntity("object.myfield3.not.object", SchemaEntity.EntityType.SR_RECORD), Collections.singleton("record"));
        ParsedSchema resultSchema = schema.copy(tags, Collections.emptyMap());
        Assert.assertEquals((Object)expectSchema.canonicalString(), (Object)resultSchema.canonicalString());
        Assert.assertEquals((Object)ImmutableSet.of((Object)"record", (Object)"PII"), (Object)resultSchema.inlineTags());
        resultSchema = resultSchema.copy(Collections.emptyMap(), tags);
        Assert.assertEquals((Object)schema.canonicalString(), (Object)resultSchema.canonicalString());
        Assert.assertEquals((Object)ImmutableSet.of(), (Object)resultSchema.inlineTags());
        Map<String, Set<String>> pathTags = Collections.singletonMap("some.path", Collections.singleton("EXTERNAL"));
        Metadata metadata = new Metadata(pathTags, null, null);
        resultSchema = resultSchema.copy(metadata, null);
        Assert.assertEquals((Object)ImmutableSet.of((Object)"EXTERNAL"), (Object)resultSchema.tags());
    }

    @Test
    public void testAddTagsToConditional() {
        String schemaString = "{\n  \"else\": {\n    \"properties\": {\n      \"postal_code\": {\n        \"pattern\": \"[A-Z][0-9][A-Z] [0-9][A-Z][0-9]\"\n      }\n    }\n  },\n  \"if\": {\n    \"properties\": {\n      \"country\": {\n        \"const\": \"United States of America\"\n      }\n    }\n  },\n  \"properties\": {\n    \"country\": {\n      \"default\": \"United States of America\",\n      \"enum\": [\n        \"United States of America\",\n        \"Canada\"\n      ]\n    },\n    \"street_address\": {\n      \"type\": \"string\"\n    }\n  },\n  \"then\": {\n    \"properties\": {\n      \"postal_code\": {\n        \"pattern\": \"[0-9]{5}(-[0-9]{4})?\"\n      }\n    }\n  },\n  \"type\": \"object\"\n}\n";
        String addedTagSchema = "{\n  \"else\": {\n    \"properties\": {\n      \"postal_code\": {\n        \"pattern\": \"[A-Z][0-9][A-Z] [0-9][A-Z][0-9]\",\n        \"confluent:tags\": [ \"testConditional\" ]\n      }\n    }\n  },\n  \"if\": {\n    \"properties\": {\n      \"country\": {\n        \"const\": \"United States of America\"\n      }\n    },\n    \"confluent:tags\": [ \"record\" ]\n  },\n  \"properties\": {\n    \"country\": {\n      \"default\": \"United States of America\",\n      \"enum\": [\n        \"United States of America\",\n        \"Canada\"\n      ],\n      \"confluent:tags\": [ \"testConditional\" ]\n    },\n    \"street_address\": {\n      \"type\": \"string\"\n    }\n  },\n  \"then\": {\n    \"properties\": {\n      \"postal_code\": {\n        \"pattern\": \"[0-9]{5}(-[0-9]{4})?\"\n      }\n    }\n  },\n  \"type\": \"object\",\n  \"confluent:tags\": [ \"record\" ]\n}\n";
        JsonSchema schema = new JsonSchema(schemaString);
        JsonSchema expectSchema = new JsonSchema(addedTagSchema);
        HashMap<SchemaEntity, Set<String>> tags = new HashMap<SchemaEntity, Set<String>>();
        tags.put(new SchemaEntity("allof.0.conditional.else.object.postal_code", SchemaEntity.EntityType.SR_FIELD), Collections.singleton("testConditional"));
        tags.put(new SchemaEntity("allof.1.object.country", SchemaEntity.EntityType.SR_FIELD), Collections.singleton("testConditional"));
        tags.put(new SchemaEntity("allof.1.object", SchemaEntity.EntityType.SR_RECORD), Collections.singleton("record"));
        tags.put(new SchemaEntity("allof.0.conditional.if.object", SchemaEntity.EntityType.SR_RECORD), Collections.singleton("record"));
        ParsedSchema resultSchema = schema.copy(tags, Collections.emptyMap());
        Assert.assertEquals((Object)expectSchema.canonicalString(), (Object)resultSchema.canonicalString());
        Assert.assertEquals((Object)ImmutableSet.of((Object)"record", (Object)"testConditional"), (Object)resultSchema.inlineTags());
    }

    @Test
    public void testAddTagToRecursiveSchema() {
        String schema = "{\n  \"$id\": \"task.schema.json\",\n  \"$schema\": \"http://json-schema.org/draft-07/schema#\",\n  \"description\": \"A task\",\n  \"properties\": {\n    \"parent\": {\n      \"$ref\": \"task.schema.json\",\n      \"confluent:tags\": [ \"testRecursive\" ]\n    },\n    \"title\": {\n      \"description\": \"Task title\",\n      \"type\": \"string\"\n    }\n  },\n  \"title\": \"Task\",\n  \"type\": [\n    \"null\",\n    \"object\"\n  ]\n}\n";
        String addedTagSchema = "{\n  \"$id\": \"task.schema.json\",\n  \"$schema\": \"http://json-schema.org/draft-07/schema#\",\n  \"description\": \"A task\",\n  \"properties\": {\n    \"parent\": {\n      \"$ref\": \"task.schema.json\",\n      \"confluent:tags\": [ \"testRecursive\", \"PII\" ]\n    },\n    \"title\": {\n      \"description\": \"Task title\",\n      \"type\": \"string\"\n    }\n  },\n  \"title\": \"Task\",\n  \"type\": [\n    \"null\",\n    \"object\"\n  ]\n}\n";
        JsonSchema jsonSchema = new JsonSchema(schema);
        JsonSchema expectSchema = new JsonSchema(addedTagSchema);
        HashMap<SchemaEntity, Set<String>> tags = new HashMap<SchemaEntity, Set<String>>();
        tags.put(new SchemaEntity("anyof.1.object.parent", SchemaEntity.EntityType.SR_FIELD), Collections.singleton("PII"));
        ParsedSchema resultSchema = jsonSchema.copy(tags, Collections.emptyMap());
        Assert.assertEquals((Object)expectSchema.canonicalString(), (Object)resultSchema.canonicalString());
        Assert.assertEquals((Object)ImmutableSet.of((Object)"testRecursive", (Object)"PII"), (Object)resultSchema.inlineTags());
    }

    @Test
    public void testAddTagToCompositeField() {
        String schema = "{\n    \"$schema\": \"http://json-schema.org/draft-07/schema#\",\n    \"title\": \"Customer\",\n    \"type\": \"object\",\n    \"additionalProperties\": false,\n    \"properties\": {\n        \"cc_details\": {\n            \"oneOf\": [\n                {\n                    \"type\": \"null\",\n                    \"title\": \"Not included\"\n                },\n                {\n                    \"$ref\": \"#/definitions/CardDetails\"\n                }\n            ]\n        }\n    },\n    \"definitions\": {\n        \"Another.Details\": {\n            \"additionalProperties\": false,\n            \"properties\": {\n              \"additional.field1\": {\n                \"type\": \"string\"\n              },\n              \"field2\": {\n                \"type\": \"number\"\n              }\n            },\n            \"type\": \"object\"\n          },\n        \"CardDetails\": {\n            \"type\": \"object\",\n            \"additionalProperties\": false,\n            \"properties\": {\n                \"credit_card\": {\n                    \"oneOf\": [\n                        {\n                            \"type\": \"null\",\n                            \"title\": \"Not included\"\n                        },\n                        {\n                            \"type\": \"string\"\n                        }\n                    ]\n                }\n            }\n        }\n    }\n}";
        String addedTagSchema = "{\n    \"$schema\": \"http://json-schema.org/draft-07/schema#\",\n    \"title\": \"Customer\",\n    \"type\": \"object\",\n    \"additionalProperties\": false,\n    \"properties\": {\n        \"cc_details\": {\n            \"oneOf\": [\n                {\n                    \"type\": \"null\",\n                    \"title\": \"Not included\"\n                },\n                {\n                    \"$ref\": \"#/definitions/CardDetails\"\n                }\n            ]\n        }\n    },\n    \"definitions\": {\n        \"Another.Details\": {\n            \"additionalProperties\": false,\n            \"properties\": {\n              \"additional.field1\": {\n                \"type\": \"string\",\n                \"confluent:tags\": [ \"TEST2\" ]\n              },\n              \"field2\": {\n                \"type\": \"number\"\n              }\n            },\n            \"type\": \"object\"\n          },\n        \"CardDetails\": {\n            \"type\": \"object\",\n            \"additionalProperties\": false,\n            \"properties\": {\n                \"credit_card\": {\n                    \"oneOf\": [\n                        {\n                            \"type\": \"null\",\n                            \"title\": \"Not included\"\n                        },\n                        {\n                            \"type\": \"string\"\n                        }\n                    ],\n                    \"confluent:tags\": [ \"PII\" ]\n                }\n            },\n            \"confluent:tags\": [ \"TEST3\" ]\n        }\n    }\n}";
        JsonSchema jsonSchema = new JsonSchema(schema);
        JsonSchema expectSchema = new JsonSchema(addedTagSchema);
        HashMap<SchemaEntity, Set<String>> tags = new HashMap<SchemaEntity, Set<String>>();
        tags.put(new SchemaEntity("object.definitions.CardDetails.object.credit_card", SchemaEntity.EntityType.SR_FIELD), Collections.singleton("PII"));
        tags.put(new SchemaEntity("object.definitions.Another.Details.object.additional.field1", SchemaEntity.EntityType.SR_FIELD), Collections.singleton("TEST2"));
        tags.put(new SchemaEntity("object.definitions.CardDetails.object", SchemaEntity.EntityType.SR_RECORD), Collections.singleton("TEST3"));
        ParsedSchema resultSchema = jsonSchema.copy(tags, Collections.emptyMap());
        Assert.assertEquals((Object)expectSchema.canonicalString(), (Object)resultSchema.canonicalString());
        Assert.assertEquals((Object)ImmutableSet.of((Object)"PII", (Object)"TEST2", (Object)"TEST3"), (Object)resultSchema.inlineTags());
    }

    @Test
    public void testRestrictedFields() {
        String schema = "{\n  \"$schema\": \"http://json-schema.org/draft-07/schema#\",\n  \"$id\": \"task.schema.json\",\n  \"title\": \"Task\",\n  \"description\": \"A task\",\n  \"type\": \"object\",\n  \"properties\": {\n    \"$id\": {\n        \"type\": \"string\"\n    },    \n    \"$$title\": {\n        \"description\": \"Task title\",\n        \"type\": \"string\"\n    },    \n    \"status\": {\n        \"type\": \"string\"\n    }\n  }\n}";
        JsonSchema jsonSchema = new JsonSchema(schema);
        jsonSchema.validate(false);
        Assert.assertThrows(ValidationException.class, () -> jsonSchema.validate(true));
        String stringSchema = "{\n  \"$schema\": \"http://json-schema.org/draft-07/schema#\",\n  \"$id\": \"task.schema.json\",\n  \"title\": \"Task\",\n  \"description\": \"A task\",\n  \"type\": \"object\",\n  \"properties\": {\n    \"$id\": {\n        \"type\": \"string\"\n    },    \n    \"title\": {\n        \"description\": \"Task title\",\n        \"type\": \"string\"\n    },    \n    \"status\": {\n        \"type\": \"string\"\n    }\n  }\n}";
        JsonSchema validSchema = new JsonSchema(stringSchema);
        validSchema.validate(true);
    }

    @Test
    public void testLocalReferenceDraft_2020_12() {
        String parent = "{\n    \"$id\": \"acme.webhooks.checkout-application_updated.jsonschema.json\",\n    \"$schema\": \"https://json-schema.org/draft/2020-12/schema\",\n    \"$scope\": \"r:application\",\n    \"title\": \"ApplicationUpdatedEvent\",\n    \"description\": \"Application updated event representing a state change in application data.\",\n    \"type\": \"object\",\n    \"properties\": {\n        \"identity\": {\n            \"$ref\": \"https://getbread.github.io/docs/oas/v2/models.openapi3.json#/components/schemas/Identity\"\n        },\n        \"application\": {\n            \"$ref\": \"./checkout.common.webhooks.jsonschema.json#/components/schemas/Application\"\n        }\n    },\n    \"required\": [\n        \"identity\",\n        \"application\"\n    ]\n}";
        String child = "{\n  \"$schema\": \"https://json-schema.org/draft/2020-12/schema\",\n  \"components\": {\n    \"schemas\": {\n      \"Application\": {\n        \"properties\": {\n          \"id\": {\n            \"description\": \"The unique identifier of the Application.\",\n            \"format\": \"uuid\",\n            \"readOnly\": true,\n            \"type\": \"string\"\n          },\n          \"shippingContact\": {\n            \"$ref\": \"https://getbread.github.io/docs/oas/v2/models.openapi3.json#/components/schemas/Contact\"\n          }\n        },\n        \"title\": \"Application\",\n        \"type\": \"object\"\n      }\n    }\n  }\n}";
        SchemaReference ref = new SchemaReference("checkout.common.webhooks.jsonschema.json", "reference", Integer.valueOf(1));
        JsonSchema jsonSchema = new JsonSchema(parent, Collections.singletonList(ref), Collections.singletonMap("checkout.common.webhooks.jsonschema.json", child), null);
        jsonSchema.validate(true);
    }

    private static Map<String, String> getJsonSchemaWithReferences(String draft) {
        HashMap<String, String> schemas = new HashMap<String, String>();
        String reference = "{" + draft + ",\"type\":\"object\",\"additionalProperties\":false,\"definitions\":{\"ExternalType\":{\"type\":\"object\",\"properties\":{\"name\":{\"type\":\"string\"}},\"additionalProperties\":false}}}";
        schemas.put("ref.json", new JsonSchema(reference).canonicalString());
        String schemaString = "{" + draft + ",\"$id\": \"https://acme.com/referrer.json\",\"type\":\"object\",\"properties\":{\"Ref\":{\"$ref\":\"ref.json#/definitions/ExternalType\"}},\"additionalProperties\":false}";
        schemas.put("main.json", schemaString);
        return schemas;
    }

    private static JsonSchema createPrimitiveSchema(String type) {
        String schemaString = String.format("{\"type\" : \"%s\"}", type);
        return new JsonSchema(schemaString);
    }

    static class TestObj {
        private String prop;

        TestObj() {
        }

        public String getProp() {
            return this.prop;
        }

        public void setProp(String prop) {
            this.prop = prop;
        }
    }
}

