/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.Date;
import java.util.Random;
import java.util.StringTokenizer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.AccumulatingReducer;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.IOMapperBase;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PositionedReadable;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.compress.CompressionCodec;
import org.apache.hadoop.mapred.FileInputFormat;
import org.apache.hadoop.mapred.FileOutputFormat;
import org.apache.hadoop.mapred.JobClient;
import org.apache.hadoop.mapred.JobConf;
import org.apache.hadoop.mapred.Mapper;
import org.apache.hadoop.mapred.OutputCollector;
import org.apache.hadoop.mapred.Reporter;
import org.apache.hadoop.mapred.SequenceFileInputFormat;
import org.apache.hadoop.util.ReflectionUtils;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

public class TestDFSIO
implements Tool {
    private static final Log LOG = LogFactory.getLog(TestDFSIO.class);
    private static final int DEFAULT_BUFFER_SIZE = 1000000;
    private static final String BASE_FILE_NAME = "test_io_";
    private static final String DEFAULT_RES_FILE_NAME = "TestDFSIO_results.log";
    private static final long MEGA = ByteMultiple.MB.value();
    private static final int DEFAULT_NR_BYTES = 1;
    private static final int DEFAULT_NR_FILES = 4;
    private static final String USAGE = "Usage: " + TestDFSIO.class.getSimpleName() + " [genericOptions]" + " -read [-random | -backward | -skip [-skipSize Size]] |" + " -write | -append | -clean" + " [-compression codecClassName]" + " [-nrFiles N]" + " [-size Size[B|KB|MB|GB|TB]]" + " [-resFile resultFileName] [-bufferSize Bytes]" + " [-rootDir]";
    private Configuration config = new Configuration();
    private static MiniDFSCluster cluster;
    private static TestDFSIO bench;

    private static String getBaseDir(Configuration conf) {
        return conf.get("test.build.data", "/benchmarks/TestDFSIO");
    }

    private static Path getControlDir(Configuration conf) {
        return new Path(TestDFSIO.getBaseDir(conf), "io_control");
    }

    private static Path getWriteDir(Configuration conf) {
        return new Path(TestDFSIO.getBaseDir(conf), "io_write");
    }

    private static Path getReadDir(Configuration conf) {
        return new Path(TestDFSIO.getBaseDir(conf), "io_read");
    }

    private static Path getAppendDir(Configuration conf) {
        return new Path(TestDFSIO.getBaseDir(conf), "io_append");
    }

    private static Path getRandomReadDir(Configuration conf) {
        return new Path(TestDFSIO.getBaseDir(conf), "io_random_read");
    }

    private static Path getDataDir(Configuration conf) {
        return new Path(TestDFSIO.getBaseDir(conf), "io_data");
    }

    @BeforeClass
    public static void beforeClass() throws Exception {
        bench = new TestDFSIO();
        bench.getConf().setBoolean("dfs.support.append", true);
        cluster = new MiniDFSCluster.Builder(bench.getConf()).numDataNodes(2).format(true).build();
        DistributedFileSystem fs = cluster.getFileSystem();
        bench.createControlFile((FileSystem)fs, 1L, 4);
    }

    @AfterClass
    public static void afterClass() throws Exception {
        if (cluster == null) {
            return;
        }
        DistributedFileSystem fs = cluster.getFileSystem();
        bench.cleanup((FileSystem)fs);
        cluster.shutdown();
    }

    @Test
    public void testWrite() throws Exception {
        DistributedFileSystem fs = cluster.getFileSystem();
        long tStart = System.currentTimeMillis();
        bench.writeTest((FileSystem)fs);
        long execTime = System.currentTimeMillis() - tStart;
        bench.analyzeResult((FileSystem)fs, TestType.TEST_TYPE_WRITE, execTime);
    }

    @Test
    public void testRead() throws Exception {
        DistributedFileSystem fs = cluster.getFileSystem();
        long tStart = System.currentTimeMillis();
        bench.readTest((FileSystem)fs);
        long execTime = System.currentTimeMillis() - tStart;
        bench.analyzeResult((FileSystem)fs, TestType.TEST_TYPE_READ, execTime);
    }

    @Test
    public void testReadRandom() throws Exception {
        DistributedFileSystem fs = cluster.getFileSystem();
        long tStart = System.currentTimeMillis();
        bench.getConf().setLong("test.io.skip.size", 0L);
        bench.randomReadTest((FileSystem)fs);
        long execTime = System.currentTimeMillis() - tStart;
        bench.analyzeResult((FileSystem)fs, TestType.TEST_TYPE_READ_RANDOM, execTime);
    }

    @Test
    public void testReadBackward() throws Exception {
        DistributedFileSystem fs = cluster.getFileSystem();
        long tStart = System.currentTimeMillis();
        bench.getConf().setLong("test.io.skip.size", -1000000L);
        bench.randomReadTest((FileSystem)fs);
        long execTime = System.currentTimeMillis() - tStart;
        bench.analyzeResult((FileSystem)fs, TestType.TEST_TYPE_READ_BACKWARD, execTime);
    }

    @Test
    public void testReadSkip() throws Exception {
        DistributedFileSystem fs = cluster.getFileSystem();
        long tStart = System.currentTimeMillis();
        bench.getConf().setLong("test.io.skip.size", 1L);
        bench.randomReadTest((FileSystem)fs);
        long execTime = System.currentTimeMillis() - tStart;
        bench.analyzeResult((FileSystem)fs, TestType.TEST_TYPE_READ_SKIP, execTime);
    }

    @Test
    public void testAppend() throws Exception {
        DistributedFileSystem fs = cluster.getFileSystem();
        long tStart = System.currentTimeMillis();
        bench.appendTest((FileSystem)fs);
        long execTime = System.currentTimeMillis() - tStart;
        bench.analyzeResult((FileSystem)fs, TestType.TEST_TYPE_APPEND, execTime);
    }

    private void createControlFile(FileSystem fs, long nrBytes, int nrFiles) throws IOException {
        LOG.info((Object)("creating control file: " + nrBytes + " bytes, " + nrFiles + " files"));
        Path controlDir = TestDFSIO.getControlDir(this.config);
        fs.delete(controlDir, true);
        for (int i = 0; i < nrFiles; ++i) {
            String name = TestDFSIO.getFileName(i);
            Path controlFile = new Path(controlDir, "in_file_" + name);
            SequenceFile.Writer writer = null;
            try {
                writer = SequenceFile.createWriter((FileSystem)fs, (Configuration)this.config, (Path)controlFile, Text.class, LongWritable.class, (SequenceFile.CompressionType)SequenceFile.CompressionType.NONE);
                writer.append((Writable)new Text(name), (Writable)new LongWritable(nrBytes));
                continue;
            }
            catch (Exception e) {
                throw new IOException(e.getLocalizedMessage());
            }
            finally {
                if (writer != null) {
                    writer.close();
                }
                writer = null;
            }
        }
        LOG.info((Object)("created control files for: " + nrFiles + " files"));
    }

    private static String getFileName(int fIdx) {
        return BASE_FILE_NAME + Integer.toString(fIdx);
    }

    private void writeTest(FileSystem fs) throws IOException {
        Path writeDir = TestDFSIO.getWriteDir(this.config);
        fs.delete(TestDFSIO.getDataDir(this.config), true);
        fs.delete(writeDir, true);
        this.runIOTest(WriteMapper.class, writeDir);
    }

    private void runIOTest(Class<? extends Mapper<Text, LongWritable, Text, Text>> mapperClass, Path outputDir) throws IOException {
        JobConf job = new JobConf(this.config, TestDFSIO.class);
        FileInputFormat.setInputPaths((JobConf)job, (Path[])new Path[]{TestDFSIO.getControlDir(this.config)});
        job.setInputFormat(SequenceFileInputFormat.class);
        job.setMapperClass(mapperClass);
        job.setReducerClass(AccumulatingReducer.class);
        FileOutputFormat.setOutputPath((JobConf)job, (Path)outputDir);
        job.setOutputKeyClass(Text.class);
        job.setOutputValueClass(Text.class);
        job.setNumReduceTasks(1);
        JobClient.runJob((JobConf)job);
    }

    private void appendTest(FileSystem fs) throws IOException {
        Path appendDir = TestDFSIO.getAppendDir(this.config);
        fs.delete(appendDir, true);
        this.runIOTest(AppendMapper.class, appendDir);
    }

    private void readTest(FileSystem fs) throws IOException {
        Path readDir = TestDFSIO.getReadDir(this.config);
        fs.delete(readDir, true);
        this.runIOTest(ReadMapper.class, readDir);
    }

    private void randomReadTest(FileSystem fs) throws IOException {
        Path readDir = TestDFSIO.getRandomReadDir(this.config);
        fs.delete(readDir, true);
        this.runIOTest(RandomReadMapper.class, readDir);
    }

    private void sequentialTest(FileSystem fs, TestType testType, long fileSize, int nrFiles) throws IOException {
        IOStatMapper ioer = null;
        switch (testType) {
            case TEST_TYPE_READ: {
                ioer = new ReadMapper();
                break;
            }
            case TEST_TYPE_WRITE: {
                ioer = new WriteMapper();
                break;
            }
            case TEST_TYPE_APPEND: {
                ioer = new AppendMapper();
                break;
            }
            case TEST_TYPE_READ_RANDOM: 
            case TEST_TYPE_READ_BACKWARD: 
            case TEST_TYPE_READ_SKIP: {
                ioer = new RandomReadMapper();
                break;
            }
            default: {
                return;
            }
        }
        for (int i = 0; i < nrFiles; ++i) {
            ioer.doIO(Reporter.NULL, BASE_FILE_NAME + Integer.toString(i), fileSize);
        }
    }

    public static void main(String[] args) {
        TestDFSIO bench = new TestDFSIO();
        int res = -1;
        try {
            res = ToolRunner.run((Tool)bench, (String[])args);
        }
        catch (Exception e) {
            System.err.print(StringUtils.stringifyException((Throwable)e));
            res = -2;
        }
        if (res == -1) {
            System.err.print(USAGE);
        }
        System.exit(res);
    }

    public int run(String[] args) throws IOException {
        TestType testType = null;
        int bufferSize = 1000000;
        long nrBytes = 1L * MEGA;
        int nrFiles = 1;
        long skipSize = 0L;
        String resFileName = DEFAULT_RES_FILE_NAME;
        String compressionClass = null;
        boolean isSequential = false;
        String version = TestDFSIO.class.getSimpleName() + ".1.7";
        LOG.info((Object)version);
        if (args.length == 0) {
            System.err.println("Missing arguments.");
            return -1;
        }
        for (int i = 0; i < args.length; ++i) {
            if (args[i].startsWith("-read")) {
                testType = TestType.TEST_TYPE_READ;
                continue;
            }
            if (args[i].equals("-write")) {
                testType = TestType.TEST_TYPE_WRITE;
                continue;
            }
            if (args[i].equals("-append")) {
                testType = TestType.TEST_TYPE_APPEND;
                continue;
            }
            if (args[i].equals("-random")) {
                if (testType != TestType.TEST_TYPE_READ) {
                    return -1;
                }
                testType = TestType.TEST_TYPE_READ_RANDOM;
                continue;
            }
            if (args[i].equals("-backward")) {
                if (testType != TestType.TEST_TYPE_READ) {
                    return -1;
                }
                testType = TestType.TEST_TYPE_READ_BACKWARD;
                continue;
            }
            if (args[i].equals("-skip")) {
                if (testType != TestType.TEST_TYPE_READ) {
                    return -1;
                }
                testType = TestType.TEST_TYPE_READ_SKIP;
                continue;
            }
            if (args[i].equals("-clean")) {
                testType = TestType.TEST_TYPE_CLEANUP;
                continue;
            }
            if (args[i].startsWith("-seq")) {
                isSequential = true;
                continue;
            }
            if (args[i].startsWith("-compression")) {
                compressionClass = args[++i];
                continue;
            }
            if (args[i].equals("-nrFiles")) {
                nrFiles = Integer.parseInt(args[++i]);
                continue;
            }
            if (args[i].equals("-fileSize") || args[i].equals("-size")) {
                nrBytes = TestDFSIO.parseSize(args[++i]);
                continue;
            }
            if (args[i].equals("-skipSize")) {
                skipSize = TestDFSIO.parseSize(args[++i]);
                continue;
            }
            if (args[i].equals("-bufferSize")) {
                bufferSize = Integer.parseInt(args[++i]);
                continue;
            }
            if (args[i].equals("-resFile")) {
                resFileName = args[++i];
                continue;
            }
            System.err.println("Illegal argument: " + args[i]);
            return -1;
        }
        if (testType == null) {
            return -1;
        }
        if (testType == TestType.TEST_TYPE_READ_BACKWARD) {
            skipSize = -bufferSize;
        } else if (testType == TestType.TEST_TYPE_READ_SKIP && skipSize == 0L) {
            skipSize = bufferSize;
        }
        LOG.info((Object)("nrFiles = " + nrFiles));
        LOG.info((Object)("nrBytes (MB) = " + TestDFSIO.toMB(nrBytes)));
        LOG.info((Object)("bufferSize = " + bufferSize));
        if (skipSize > 0L) {
            LOG.info((Object)("skipSize = " + skipSize));
        }
        LOG.info((Object)("baseDir = " + TestDFSIO.getBaseDir(this.config)));
        if (compressionClass != null) {
            this.config.set("test.io.compression.class", compressionClass);
            LOG.info((Object)("compressionClass = " + compressionClass));
        }
        this.config.setInt("test.io.file.buffer.size", bufferSize);
        this.config.setLong("test.io.skip.size", skipSize);
        this.config.setBoolean("dfs.support.append", true);
        FileSystem fs = FileSystem.get((Configuration)this.config);
        if (isSequential) {
            long tStart = System.currentTimeMillis();
            this.sequentialTest(fs, testType, nrBytes, nrFiles);
            long execTime = System.currentTimeMillis() - tStart;
            String resultLine = "Seq Test exec time sec: " + (float)execTime / 1000.0f;
            LOG.info((Object)resultLine);
            return 0;
        }
        if (testType == TestType.TEST_TYPE_CLEANUP) {
            this.cleanup(fs);
            return 0;
        }
        this.createControlFile(fs, nrBytes, nrFiles);
        long tStart = System.currentTimeMillis();
        switch (testType) {
            case TEST_TYPE_WRITE: {
                this.writeTest(fs);
                break;
            }
            case TEST_TYPE_READ: {
                this.readTest(fs);
                break;
            }
            case TEST_TYPE_APPEND: {
                this.appendTest(fs);
                break;
            }
            case TEST_TYPE_READ_RANDOM: 
            case TEST_TYPE_READ_BACKWARD: 
            case TEST_TYPE_READ_SKIP: {
                this.randomReadTest(fs);
            }
        }
        long execTime = System.currentTimeMillis() - tStart;
        this.analyzeResult(fs, testType, execTime, resFileName);
        return 0;
    }

    public Configuration getConf() {
        return this.config;
    }

    public void setConf(Configuration conf) {
        this.config = conf;
    }

    static long parseSize(String arg) {
        String[] args = arg.split("\\D", 2);
        assert (args.length <= 2);
        long nrBytes = Long.parseLong(args[0]);
        String bytesMult = arg.substring(args[0].length());
        return nrBytes * ByteMultiple.parseString(bytesMult).value();
    }

    static float toMB(long bytes) {
        return (float)bytes / (float)MEGA;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void analyzeResult(FileSystem fs, TestType testType, long execTime, String resFileName) throws IOException {
        Path reduceFile = this.getReduceFilePath(testType);
        long tasks = 0L;
        long size = 0L;
        long time = 0L;
        float rate = 0.0f;
        float sqrate = 0.0f;
        FilterInputStream in = null;
        BufferedReader lines = null;
        try {
            String line;
            in = new DataInputStream((InputStream)fs.open(reduceFile));
            lines = new BufferedReader(new InputStreamReader(in));
            while ((line = lines.readLine()) != null) {
                StringTokenizer tokens = new StringTokenizer(line, " \t\n\r\f%");
                String attr = tokens.nextToken();
                if (attr.endsWith(":tasks")) {
                    tasks = Long.parseLong(tokens.nextToken());
                    continue;
                }
                if (attr.endsWith(":size")) {
                    size = Long.parseLong(tokens.nextToken());
                    continue;
                }
                if (attr.endsWith(":time")) {
                    time = Long.parseLong(tokens.nextToken());
                    continue;
                }
                if (attr.endsWith(":rate")) {
                    rate = Float.parseFloat(tokens.nextToken());
                    continue;
                }
                if (!attr.endsWith(":sqrate")) continue;
                sqrate = Float.parseFloat(tokens.nextToken());
            }
        }
        finally {
            if (in != null) {
                in.close();
            }
            if (lines != null) {
                lines.close();
            }
        }
        double med = rate / 1000.0f / (float)tasks;
        double stdDev = Math.sqrt(Math.abs((double)(sqrate / 1000.0f / (float)tasks) - med * med));
        String[] resultLines = new String[]{"----- TestDFSIO ----- : " + (Object)((Object)testType), "           Date & time: " + new Date(System.currentTimeMillis()), "       Number of files: " + tasks, "Total MBytes processed: " + TestDFSIO.toMB(size), "     Throughput mb/sec: " + (double)size * 1000.0 / (double)(time * MEGA), "Average IO rate mb/sec: " + med, " IO rate std deviation: " + stdDev, "    Test exec time sec: " + (float)execTime / 1000.0f, ""};
        PrintStream res = null;
        try {
            res = new PrintStream(new FileOutputStream(new File(resFileName), true));
            for (int i = 0; i < resultLines.length; ++i) {
                LOG.info((Object)resultLines[i]);
                res.println(resultLines[i]);
            }
        }
        finally {
            if (res != null) {
                res.close();
            }
        }
    }

    private Path getReduceFilePath(TestType testType) {
        switch (testType) {
            case TEST_TYPE_WRITE: {
                return new Path(TestDFSIO.getWriteDir(this.config), "part-00000");
            }
            case TEST_TYPE_APPEND: {
                return new Path(TestDFSIO.getAppendDir(this.config), "part-00000");
            }
            case TEST_TYPE_READ: {
                return new Path(TestDFSIO.getReadDir(this.config), "part-00000");
            }
            case TEST_TYPE_READ_RANDOM: 
            case TEST_TYPE_READ_BACKWARD: 
            case TEST_TYPE_READ_SKIP: {
                return new Path(TestDFSIO.getRandomReadDir(this.config), "part-00000");
            }
        }
        return null;
    }

    private void analyzeResult(FileSystem fs, TestType testType, long execTime) throws IOException {
        this.analyzeResult(fs, testType, execTime, DEFAULT_RES_FILE_NAME);
    }

    private void cleanup(FileSystem fs) throws IOException {
        LOG.info((Object)"Cleaning up test files");
        fs.delete(new Path(TestDFSIO.getBaseDir(this.config)), true);
    }

    static {
        Configuration.addDefaultResource((String)"hdfs-default.xml");
        Configuration.addDefaultResource((String)"hdfs-site.xml");
        Configuration.addDefaultResource((String)"mapred-default.xml");
        Configuration.addDefaultResource((String)"mapred-site.xml");
    }

    public static class RandomReadMapper
    extends IOStatMapper {
        private Random rnd = new Random();
        private long fileSize;
        private long skipSize;

        @Override
        public void configure(JobConf conf) {
            super.configure(conf);
            this.skipSize = conf.getLong("test.io.skip.size", 0L);
        }

        @Override
        public Closeable getIOStream(String name) throws IOException {
            Path filePath = new Path(TestDFSIO.getDataDir(this.getConf()), name);
            this.fileSize = this.fs.getFileStatus(filePath).getLen();
            FSDataInputStream in = this.fs.open(filePath);
            if (this.compressionCodec != null) {
                in = new FSDataInputStream((InputStream)this.compressionCodec.createInputStream((InputStream)in));
            }
            LOG.info((Object)("in = " + in.getClass().getName()));
            LOG.info((Object)("skipSize = " + this.skipSize));
            return in;
        }

        @Override
        public Long doIO(Reporter reporter, String name, long totalSize) throws IOException {
            int curSize;
            PositionedReadable in = (PositionedReadable)this.stream;
            long actualSize = 0L;
            long pos = this.nextOffset(-1L);
            while (actualSize < totalSize && (curSize = in.read(pos, this.buffer, 0, this.bufferSize)) >= 0) {
                reporter.setStatus("reading " + name + "@" + (actualSize += (long)curSize) + "/" + totalSize + " ::host = " + this.hostName);
                pos = this.nextOffset(pos);
            }
            return actualSize;
        }

        private long nextOffset(long current) {
            if (this.skipSize == 0L) {
                return this.rnd.nextInt((int)this.fileSize);
            }
            if (this.skipSize > 0L) {
                return current < 0L ? 0L : current + (long)this.bufferSize + this.skipSize;
            }
            return current < 0L ? Math.max(0L, this.fileSize - (long)this.bufferSize) : Math.max(0L, current + this.skipSize);
        }
    }

    public static class ReadMapper
    extends IOStatMapper {
        @Override
        public Closeable getIOStream(String name) throws IOException {
            FSDataInputStream in = this.fs.open(new Path(TestDFSIO.getDataDir(this.getConf()), name));
            if (this.compressionCodec != null) {
                in = this.compressionCodec.createInputStream((InputStream)in);
            }
            LOG.info((Object)("in = " + in.getClass().getName()));
            return in;
        }

        @Override
        public Long doIO(Reporter reporter, String name, long totalSize) throws IOException {
            int curSize;
            InputStream in = (InputStream)this.stream;
            long actualSize = 0L;
            while (actualSize < totalSize && (curSize = in.read(this.buffer, 0, this.bufferSize)) >= 0) {
                reporter.setStatus("reading " + name + "@" + (actualSize += (long)curSize) + "/" + totalSize + " ::host = " + this.hostName);
            }
            return actualSize;
        }
    }

    public static class AppendMapper
    extends IOStatMapper {
        public AppendMapper() {
            for (int i = 0; i < this.bufferSize; ++i) {
                this.buffer[i] = (byte)(48 + i % 50);
            }
        }

        @Override
        public Closeable getIOStream(String name) throws IOException {
            FSDataOutputStream out = this.fs.append(new Path(TestDFSIO.getDataDir(this.getConf()), name), this.bufferSize);
            if (this.compressionCodec != null) {
                out = this.compressionCodec.createOutputStream((OutputStream)out);
            }
            LOG.info((Object)("out = " + out.getClass().getName()));
            return out;
        }

        @Override
        public Long doIO(Reporter reporter, String name, long totalSize) throws IOException {
            OutputStream out = (OutputStream)this.stream;
            for (long nrRemaining = totalSize; nrRemaining > 0L; nrRemaining -= (long)this.bufferSize) {
                int curSize = (long)this.bufferSize < nrRemaining ? this.bufferSize : (int)nrRemaining;
                out.write(this.buffer, 0, curSize);
                reporter.setStatus("writing " + name + "@" + (totalSize - nrRemaining) + "/" + totalSize + " ::host = " + this.hostName);
            }
            return totalSize;
        }
    }

    public static class WriteMapper
    extends IOStatMapper {
        public WriteMapper() {
            for (int i = 0; i < this.bufferSize; ++i) {
                this.buffer[i] = (byte)(48 + i % 50);
            }
        }

        @Override
        public Closeable getIOStream(String name) throws IOException {
            FSDataOutputStream out = this.fs.create(new Path(TestDFSIO.getDataDir(this.getConf()), name), true, this.bufferSize);
            if (this.compressionCodec != null) {
                out = this.compressionCodec.createOutputStream((OutputStream)out);
            }
            LOG.info((Object)("out = " + out.getClass().getName()));
            return out;
        }

        @Override
        public Long doIO(Reporter reporter, String name, long totalSize) throws IOException {
            OutputStream out = (OutputStream)this.stream;
            for (long nrRemaining = totalSize; nrRemaining > 0L; nrRemaining -= (long)this.bufferSize) {
                int curSize = (long)this.bufferSize < nrRemaining ? this.bufferSize : (int)nrRemaining;
                out.write(this.buffer, 0, curSize);
                reporter.setStatus("writing " + name + "@" + (totalSize - nrRemaining) + "/" + totalSize + " ::host = " + this.hostName);
            }
            return totalSize;
        }
    }

    private static abstract class IOStatMapper
    extends IOMapperBase<Long> {
        protected CompressionCodec compressionCodec;

        IOStatMapper() {
        }

        @Override
        public void configure(JobConf conf) {
            Class<CompressionCodec> codec;
            super.configure(conf);
            String compression = this.getConf().get("test.io.compression.class", null);
            try {
                codec = compression == null ? null : Class.forName(compression).asSubclass(CompressionCodec.class);
            }
            catch (Exception e) {
                throw new RuntimeException("Compression codec not found: ", e);
            }
            if (codec != null) {
                this.compressionCodec = (CompressionCodec)ReflectionUtils.newInstance(codec, (Configuration)this.getConf());
            }
        }

        @Override
        void collectStats(OutputCollector<Text, Text> output, String name, long execTime, Long objSize) throws IOException {
            long totalSize = objSize;
            float ioRateMbSec = (float)totalSize * 1000.0f / (float)(execTime * MEGA);
            LOG.info((Object)("Number of bytes processed = " + totalSize));
            LOG.info((Object)("Exec time = " + execTime));
            LOG.info((Object)("IO rate = " + ioRateMbSec));
            output.collect((Object)new Text("l:tasks"), (Object)new Text(String.valueOf(1)));
            output.collect((Object)new Text("l:size"), (Object)new Text(String.valueOf(totalSize)));
            output.collect((Object)new Text("l:time"), (Object)new Text(String.valueOf(execTime)));
            output.collect((Object)new Text("f:rate"), (Object)new Text(String.valueOf(ioRateMbSec * 1000.0f)));
            output.collect((Object)new Text("f:sqrate"), (Object)new Text(String.valueOf(ioRateMbSec * ioRateMbSec * 1000.0f)));
        }
    }

    static enum ByteMultiple {
        B(1L),
        KB(1024L),
        MB(0x100000L),
        GB(0x40000000L),
        TB(0x10000000000L);

        private long multiplier;

        private ByteMultiple(long mult) {
            this.multiplier = mult;
        }

        long value() {
            return this.multiplier;
        }

        static ByteMultiple parseString(String sMultiple) {
            if (sMultiple == null || sMultiple.isEmpty()) {
                return MB;
            }
            String sMU = sMultiple.toUpperCase();
            if (B.name().toUpperCase().endsWith(sMU)) {
                return B;
            }
            if (KB.name().toUpperCase().endsWith(sMU)) {
                return KB;
            }
            if (MB.name().toUpperCase().endsWith(sMU)) {
                return MB;
            }
            if (GB.name().toUpperCase().endsWith(sMU)) {
                return GB;
            }
            if (TB.name().toUpperCase().endsWith(sMU)) {
                return TB;
            }
            throw new IllegalArgumentException("Unsupported ByteMultiple " + sMultiple);
        }
    }

    private static enum TestType {
        TEST_TYPE_READ("read"),
        TEST_TYPE_WRITE("write"),
        TEST_TYPE_CLEANUP("cleanup"),
        TEST_TYPE_APPEND("append"),
        TEST_TYPE_READ_RANDOM("random read"),
        TEST_TYPE_READ_BACKWARD("backward read"),
        TEST_TYPE_READ_SKIP("skip read");

        private String type;

        private TestType(String t) {
            this.type = t;
        }

        public String toString() {
            return this.type;
        }
    }
}

