/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.authenticator;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.security.Principal;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.Map;
import javax.security.auth.Subject;
import javax.security.auth.login.Configuration;
import javax.security.sasl.Sasl;
import javax.security.sasl.SaslException;
import javax.security.sasl.SaslServer;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.network.Authenticator;
import org.apache.kafka.common.network.NetworkReceive;
import org.apache.kafka.common.network.NetworkSend;
import org.apache.kafka.common.network.TransportLayer;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.apache.kafka.common.security.auth.PrincipalBuilder;
import org.apache.kafka.common.security.authenticator.SaslServerCallbackHandler;
import org.apache.kafka.common.security.kerberos.KerberosName;
import org.apache.kafka.common.security.kerberos.KerberosShortNamer;
import org.ietf.jgss.GSSCredential;
import org.ietf.jgss.GSSException;
import org.ietf.jgss.GSSManager;
import org.ietf.jgss.GSSName;
import org.ietf.jgss.Oid;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SaslServerAuthenticator
implements Authenticator {
    private static final Logger LOG = LoggerFactory.getLogger(SaslServerAuthenticator.class);
    private final SaslServer saslServer;
    private final Subject subject;
    private final String node;
    private final KerberosShortNamer kerberosNamer;
    private TransportLayer transportLayer;
    private NetworkReceive netInBuffer;
    private NetworkSend netOutBuffer;

    public SaslServerAuthenticator(String node, Subject subject, KerberosShortNamer kerberosNameParser) throws IOException {
        if (subject == null) {
            throw new IllegalArgumentException("subject cannot be null");
        }
        if (subject.getPrincipals().isEmpty()) {
            throw new IllegalArgumentException("subject must have at least one principal");
        }
        this.node = node;
        this.subject = subject;
        this.kerberosNamer = kerberosNameParser;
        this.saslServer = this.createSaslServer();
    }

    @Override
    public void configure(TransportLayer transportLayer, PrincipalBuilder principalBuilder, Map<String, ?> configs) {
        this.transportLayer = transportLayer;
    }

    private SaslServer createSaslServer() throws IOException {
        KerberosName kerberosName;
        final SaslServerCallbackHandler saslServerCallbackHandler = new SaslServerCallbackHandler(Configuration.getConfiguration(), this.kerberosNamer);
        Principal servicePrincipal = this.subject.getPrincipals().iterator().next();
        try {
            kerberosName = KerberosName.parse(servicePrincipal.getName());
        }
        catch (IllegalArgumentException e) {
            throw new KafkaException("Principal has name with unexpected format " + servicePrincipal);
        }
        final String servicePrincipalName = kerberosName.serviceName();
        final String serviceHostname = kerberosName.hostName();
        String mech = "GSSAPI";
        LOG.debug("Creating SaslServer for {} with mechanism {}", kerberosName, (Object)"GSSAPI");
        boolean usingNativeJgss = Boolean.getBoolean("sun.security.jgss.native");
        if (usingNativeJgss) {
            try {
                GSSManager manager = GSSManager.getInstance();
                Oid krb5Mechanism = new Oid("1.2.840.113554.1.2.2");
                GSSName gssName = manager.createName(servicePrincipalName + "@" + serviceHostname, GSSName.NT_HOSTBASED_SERVICE);
                GSSCredential cred = manager.createCredential(gssName, Integer.MAX_VALUE, krb5Mechanism, 2);
                this.subject.getPrivateCredentials().add(cred);
            }
            catch (GSSException ex) {
                LOG.warn("Cannot add private credential to subject; clients authentication may fail", ex);
            }
        }
        try {
            return Subject.doAs(this.subject, new PrivilegedExceptionAction<SaslServer>(){

                @Override
                public SaslServer run() throws SaslException {
                    return Sasl.createSaslServer("GSSAPI", servicePrincipalName, serviceHostname, null, saslServerCallbackHandler);
                }
            });
        }
        catch (PrivilegedActionException e) {
            throw new SaslException("Kafka Server failed to create a SaslServer to interact with a client during session authentication", e.getCause());
        }
    }

    @Override
    public void authenticate() throws IOException {
        if (this.netOutBuffer != null && !this.flushNetOutBufferAndUpdateInterestOps()) {
            return;
        }
        if (this.saslServer.isComplete()) {
            this.transportLayer.removeInterestOps(4);
            return;
        }
        if (this.netInBuffer == null) {
            this.netInBuffer = new NetworkReceive(this.node);
        }
        this.netInBuffer.readFrom(this.transportLayer);
        if (this.netInBuffer.complete()) {
            this.netInBuffer.payload().rewind();
            byte[] clientToken = new byte[this.netInBuffer.payload().remaining()];
            this.netInBuffer.payload().get(clientToken, 0, clientToken.length);
            this.netInBuffer = null;
            try {
                byte[] response = this.saslServer.evaluateResponse(clientToken);
                if (response != null) {
                    this.netOutBuffer = new NetworkSend(this.node, ByteBuffer.wrap(response));
                    this.flushNetOutBufferAndUpdateInterestOps();
                }
            }
            catch (Exception e) {
                throw new IOException(e);
            }
        }
    }

    @Override
    public Principal principal() {
        return new KafkaPrincipal("User", this.saslServer.getAuthorizationID());
    }

    @Override
    public boolean complete() {
        return this.saslServer.isComplete();
    }

    @Override
    public void close() throws IOException {
        this.saslServer.dispose();
    }

    private boolean flushNetOutBufferAndUpdateInterestOps() throws IOException {
        boolean flushedCompletely = this.flushNetOutBuffer();
        if (flushedCompletely) {
            this.transportLayer.removeInterestOps(4);
        } else {
            this.transportLayer.addInterestOps(4);
        }
        return flushedCompletely;
    }

    private boolean flushNetOutBuffer() throws IOException {
        if (!this.netOutBuffer.completed()) {
            this.netOutBuffer.writeTo(this.transportLayer);
        }
        return this.netOutBuffer.completed();
    }
}

