/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.network;

import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.CancelledKeyException;
import java.nio.channels.FileChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;
import java.security.Principal;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;
import org.apache.kafka.common.network.TransportLayer;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SslTransportLayer
implements TransportLayer {
    private static final Logger log = LoggerFactory.getLogger(SslTransportLayer.class);
    private final String channelId;
    private final SSLEngine sslEngine;
    private final SelectionKey key;
    private final SocketChannel socketChannel;
    private final boolean enableRenegotiation;
    private SSLEngineResult.HandshakeStatus handshakeStatus;
    private SSLEngineResult handshakeResult;
    private boolean handshakeComplete = false;
    private boolean closing = false;
    private ByteBuffer netReadBuffer;
    private ByteBuffer netWriteBuffer;
    private ByteBuffer appReadBuffer;
    private ByteBuffer emptyBuf = ByteBuffer.allocate(0);

    public static SslTransportLayer create(String channelId, SelectionKey key, SSLEngine sslEngine) throws IOException {
        SslTransportLayer transportLayer = new SslTransportLayer(channelId, key, sslEngine, false);
        transportLayer.startHandshake();
        return transportLayer;
    }

    SslTransportLayer(String channelId, SelectionKey key, SSLEngine sslEngine, boolean enableRenegotiation) throws IOException {
        this.channelId = channelId;
        this.key = key;
        this.socketChannel = (SocketChannel)key.channel();
        this.sslEngine = sslEngine;
        this.enableRenegotiation = enableRenegotiation;
    }

    protected void startHandshake() throws IOException {
        this.netReadBuffer = ByteBuffer.allocate(this.netReadBufferSize());
        this.netWriteBuffer = ByteBuffer.allocate(this.netWriteBufferSize());
        this.appReadBuffer = ByteBuffer.allocate(this.applicationBufferSize());
        this.netWriteBuffer.position(0);
        this.netWriteBuffer.limit(0);
        this.netReadBuffer.position(0);
        this.netReadBuffer.limit(0);
        this.handshakeComplete = false;
        this.closing = false;
        this.sslEngine.beginHandshake();
        this.handshakeStatus = this.sslEngine.getHandshakeStatus();
    }

    @Override
    public boolean ready() {
        return this.handshakeComplete;
    }

    @Override
    public void finishConnect() throws IOException {
        this.socketChannel.finishConnect();
        this.key.interestOps(this.key.interestOps() & 0xFFFFFFF7 | 1);
    }

    @Override
    public void disconnect() {
        this.key.cancel();
    }

    @Override
    public SocketChannel socketChannel() {
        return this.socketChannel;
    }

    @Override
    public boolean isOpen() {
        return this.socketChannel.isOpen();
    }

    @Override
    public boolean isConnected() {
        return this.socketChannel.isConnected();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        if (this.closing) {
            return;
        }
        this.closing = true;
        this.sslEngine.closeOutbound();
        try {
            if (this.isConnected()) {
                if (!this.flush(this.netWriteBuffer)) {
                    throw new IOException("Remaining data in the network buffer, can't send SSL close message.");
                }
                this.netWriteBuffer.clear();
                SSLEngineResult wrapResult = this.sslEngine.wrap(this.emptyBuf, this.netWriteBuffer);
                if (wrapResult.getStatus() != SSLEngineResult.Status.CLOSED) {
                    throw new IOException("Unexpected status returned by SSLEngine.wrap, expected CLOSED, received " + (Object)((Object)wrapResult.getStatus()) + ". Will not send close message to peer.");
                }
                this.netWriteBuffer.flip();
                this.flush(this.netWriteBuffer);
            }
        }
        catch (IOException ie) {
            log.warn("Failed to send SSL Close message ", ie);
        }
        finally {
            try {
                this.socketChannel.socket().close();
                this.socketChannel.close();
            }
            catch (IOException e) {
                log.warn("Failed to close SSL socket channel: " + e);
            }
        }
        this.key.attach(null);
        this.key.cancel();
    }

    @Override
    public boolean hasPendingWrites() {
        return this.netWriteBuffer.hasRemaining();
    }

    private boolean flush(ByteBuffer buf) throws IOException {
        int remaining = buf.remaining();
        if (remaining > 0) {
            int written = this.socketChannel.write(buf);
            return written >= remaining;
        }
        return true;
    }

    @Override
    public void handshake() throws IOException {
        boolean read2 = this.key.isReadable();
        boolean write2 = this.key.isWritable();
        this.handshakeComplete = false;
        this.handshakeStatus = this.sslEngine.getHandshakeStatus();
        if (!this.flush(this.netWriteBuffer)) {
            this.key.interestOps(this.key.interestOps() | 4);
            return;
        }
        try {
            switch (this.handshakeStatus) {
                case NEED_TASK: {
                    log.trace("SSLHandshake NEED_TASK channelId {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {}", new Object[]{this.channelId, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
                    this.handshakeStatus = this.runDelegatedTasks();
                    break;
                }
                case NEED_WRAP: {
                    log.trace("SSLHandshake NEED_WRAP channelId {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {}", new Object[]{this.channelId, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
                    this.handshakeResult = this.handshakeWrap(write2);
                    if (this.handshakeResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                        int currentNetWriteBufferSize = this.netWriteBufferSize();
                        this.netWriteBuffer.compact();
                        this.netWriteBuffer = Utils.ensureCapacity(this.netWriteBuffer, currentNetWriteBufferSize);
                        this.netWriteBuffer.flip();
                        if (this.netWriteBuffer.limit() >= currentNetWriteBufferSize) {
                            throw new IllegalStateException("Buffer overflow when available data size (" + this.netWriteBuffer.limit() + ") >= network buffer size (" + currentNetWriteBufferSize + ")");
                        }
                    } else {
                        if (this.handshakeResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                            throw new IllegalStateException("Should not have received BUFFER_UNDERFLOW during handshake WRAP.");
                        }
                        if (this.handshakeResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                            throw new EOFException();
                        }
                    }
                    log.trace("SSLHandshake NEED_WRAP channelId {}, handshakeResult {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {}", new Object[]{this.channelId, this.handshakeResult, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
                    if (this.handshakeStatus != SSLEngineResult.HandshakeStatus.NEED_UNWRAP || !this.flush(this.netWriteBuffer)) {
                        this.key.interestOps(this.key.interestOps() | 4);
                        break;
                    }
                }
                case NEED_UNWRAP: {
                    log.trace("SSLHandshake NEED_UNWRAP channelId {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {}", new Object[]{this.channelId, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
                    do {
                        this.handshakeResult = this.handshakeUnwrap(read2);
                        if (this.handshakeResult.getStatus() != SSLEngineResult.Status.BUFFER_OVERFLOW) continue;
                        int currentAppBufferSize = this.applicationBufferSize();
                        this.appReadBuffer = Utils.ensureCapacity(this.appReadBuffer, currentAppBufferSize);
                        if (this.appReadBuffer.position() <= currentAppBufferSize) continue;
                        throw new IllegalStateException("Buffer underflow when available data size (" + this.appReadBuffer.position() + ") > packet buffer size (" + currentAppBufferSize + ")");
                    } while (this.handshakeResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW);
                    if (this.handshakeResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                        int currentNetReadBufferSize = this.netReadBufferSize();
                        this.netReadBuffer = Utils.ensureCapacity(this.netReadBuffer, currentNetReadBufferSize);
                        if (this.netReadBuffer.position() >= currentNetReadBufferSize) {
                            throw new IllegalStateException("Buffer underflow when there is available data");
                        }
                    } else if (this.handshakeResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                        throw new EOFException("SSL handshake status CLOSED during handshake UNWRAP");
                    }
                    log.trace("SSLHandshake NEED_UNWRAP channelId {}, handshakeResult {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {}", new Object[]{this.channelId, this.handshakeResult, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
                    if (this.handshakeStatus != SSLEngineResult.HandshakeStatus.FINISHED) {
                        if (this.handshakeStatus == SSLEngineResult.HandshakeStatus.NEED_WRAP) {
                            this.key.interestOps(this.key.interestOps() | 4);
                            break;
                        }
                        if (this.handshakeStatus == SSLEngineResult.HandshakeStatus.NEED_UNWRAP) {
                            this.key.interestOps(this.key.interestOps() & 0xFFFFFFFB);
                        }
                        break;
                    }
                }
                case FINISHED: {
                    this.handshakeFinished();
                    break;
                }
                case NOT_HANDSHAKING: {
                    this.handshakeFinished();
                    break;
                }
                default: {
                    throw new IllegalStateException(String.format("Unexpected status [%s]", new Object[]{this.handshakeStatus}));
                }
            }
        }
        catch (SSLException e) {
            this.handshakeFailure();
            throw e;
        }
    }

    private void renegotiate() throws IOException {
        if (!this.enableRenegotiation) {
            throw new SSLHandshakeException("Renegotiation is not supported");
        }
        this.handshake();
    }

    private SSLEngineResult.HandshakeStatus runDelegatedTasks() {
        Runnable task;
        while ((task = this.delegatedTask()) != null) {
            task.run();
        }
        return this.sslEngine.getHandshakeStatus();
    }

    private void handshakeFinished() throws IOException {
        if (this.handshakeResult.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.FINISHED) {
            boolean bl = this.handshakeComplete = !this.netWriteBuffer.hasRemaining();
            if (!this.handshakeComplete) {
                this.key.interestOps(this.key.interestOps() | 4);
            } else {
                this.key.interestOps(this.key.interestOps() & 0xFFFFFFFB);
            }
        } else {
            throw new IOException("NOT_HANDSHAKING during handshake");
        }
        log.trace("SSLHandshake FINISHED channelId {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {} ", new Object[]{this.channelId, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
    }

    private SSLEngineResult handshakeWrap(boolean doWrite) throws IOException {
        log.trace("SSLHandshake handshakeWrap {}", (Object)this.channelId);
        if (this.netWriteBuffer.hasRemaining()) {
            throw new IllegalStateException("handshakeWrap called with netWriteBuffer not empty");
        }
        this.netWriteBuffer.clear();
        SSLEngineResult result2 = this.sslEngine.wrap(this.emptyBuf, this.netWriteBuffer);
        this.netWriteBuffer.flip();
        this.handshakeStatus = result2.getHandshakeStatus();
        if (result2.getStatus() == SSLEngineResult.Status.OK && result2.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_TASK) {
            this.handshakeStatus = this.runDelegatedTasks();
        }
        if (doWrite) {
            this.flush(this.netWriteBuffer);
        }
        return result2;
    }

    private SSLEngineResult handshakeUnwrap(boolean doRead) throws IOException {
        SSLEngineResult result2;
        log.trace("SSLHandshake handshakeUnwrap {}", (Object)this.channelId);
        boolean cont = false;
        int read2 = 0;
        if (doRead && (read2 = this.socketChannel.read(this.netReadBuffer)) == -1) {
            throw new EOFException("EOF during handshake.");
        }
        do {
            this.netReadBuffer.flip();
            result2 = this.sslEngine.unwrap(this.netReadBuffer, this.appReadBuffer);
            this.netReadBuffer.compact();
            this.handshakeStatus = result2.getHandshakeStatus();
            if (result2.getStatus() == SSLEngineResult.Status.OK && result2.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NEED_TASK) {
                this.handshakeStatus = this.runDelegatedTasks();
            }
            cont = result2.getStatus() == SSLEngineResult.Status.OK && this.handshakeStatus == SSLEngineResult.HandshakeStatus.NEED_UNWRAP;
            log.trace("SSLHandshake handshakeUnwrap: handshakeStatus {} status {}", (Object)this.handshakeStatus, (Object)result2.getStatus());
        } while (this.netReadBuffer.position() != 0 && cont);
        return result2;
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        if (this.closing) {
            return -1;
        }
        int read2 = 0;
        if (!this.handshakeComplete) {
            return read2;
        }
        if (this.appReadBuffer.position() > 0) {
            read2 = this.readFromAppBuffer(dst);
        }
        if (dst.remaining() > 0) {
            this.netReadBuffer = Utils.ensureCapacity(this.netReadBuffer, this.netReadBufferSize());
            if (this.netReadBuffer.remaining() > 0) {
                int netread = this.socketChannel.read(this.netReadBuffer);
                if (netread == 0 && this.netReadBuffer.position() == 0) {
                    return netread;
                }
                if (netread < 0) {
                    throw new EOFException("EOF during read");
                }
            }
            do {
                this.netReadBuffer.flip();
                SSLEngineResult unwrapResult = this.sslEngine.unwrap(this.netReadBuffer, this.appReadBuffer);
                this.netReadBuffer.compact();
                if (unwrapResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING && unwrapResult.getStatus() == SSLEngineResult.Status.OK) {
                    log.trace("SSLChannel Read begin renegotiation channelId {}, appReadBuffer pos {}, netReadBuffer pos {}, netWriteBuffer pos {}", new Object[]{this.channelId, this.appReadBuffer.position(), this.netReadBuffer.position(), this.netWriteBuffer.position()});
                    this.renegotiate();
                    break;
                }
                if (unwrapResult.getStatus() == SSLEngineResult.Status.OK) {
                    read2 += this.readFromAppBuffer(dst);
                    continue;
                }
                if (unwrapResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
                    int currentApplicationBufferSize = this.applicationBufferSize();
                    this.appReadBuffer = Utils.ensureCapacity(this.appReadBuffer, currentApplicationBufferSize);
                    if (this.appReadBuffer.position() >= currentApplicationBufferSize) {
                        throw new IllegalStateException("Buffer overflow when available data size (" + this.appReadBuffer.position() + ") >= application buffer size (" + currentApplicationBufferSize + ")");
                    }
                    if (!dst.hasRemaining()) break;
                    read2 += this.readFromAppBuffer(dst);
                    continue;
                }
                if (unwrapResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                    int currentNetReadBufferSize = this.netReadBufferSize();
                    this.netReadBuffer = Utils.ensureCapacity(this.netReadBuffer, currentNetReadBufferSize);
                    if (this.netReadBuffer.position() < currentNetReadBufferSize) break;
                    throw new IllegalStateException("Buffer underflow when available data size (" + this.netReadBuffer.position() + ") > packet buffer size (" + currentNetReadBufferSize + ")");
                }
                if (unwrapResult.getStatus() != SSLEngineResult.Status.CLOSED) continue;
                throw new EOFException();
            } while (this.netReadBuffer.position() != 0);
        }
        return read2;
    }

    @Override
    public long read(ByteBuffer[] dsts) throws IOException {
        return this.read(dsts, 0, dsts.length);
    }

    @Override
    public long read(ByteBuffer[] dsts, int offset, int length) throws IOException {
        if (offset < 0 || length < 0 || offset > dsts.length - length) {
            throw new IndexOutOfBoundsException();
        }
        int totalRead = 0;
        int i = offset;
        while (i < length) {
            if (dsts[i].hasRemaining()) {
                int read2 = this.read(dsts[i]);
                if (read2 <= 0) break;
                totalRead += read2;
            }
            if (dsts[i].hasRemaining()) continue;
            ++i;
        }
        return totalRead;
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        int written = 0;
        if (this.closing) {
            throw new IllegalStateException("Channel is in closing state");
        }
        if (!this.handshakeComplete) {
            return written;
        }
        if (!this.flush(this.netWriteBuffer)) {
            return written;
        }
        this.netWriteBuffer.clear();
        SSLEngineResult wrapResult = this.sslEngine.wrap(src, this.netWriteBuffer);
        this.netWriteBuffer.flip();
        if (wrapResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING && wrapResult.getStatus() == SSLEngineResult.Status.OK) {
            this.renegotiate();
            return written;
        }
        if (wrapResult.getStatus() == SSLEngineResult.Status.OK) {
            written = wrapResult.bytesConsumed();
            this.flush(this.netWriteBuffer);
        } else if (wrapResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW) {
            int currentNetWriteBufferSize = this.netWriteBufferSize();
            this.netWriteBuffer.compact();
            this.netWriteBuffer = Utils.ensureCapacity(this.netWriteBuffer, currentNetWriteBufferSize);
            this.netWriteBuffer.flip();
            if (this.netWriteBuffer.limit() >= currentNetWriteBufferSize) {
                throw new IllegalStateException("SSL BUFFER_OVERFLOW when available data size (" + this.netWriteBuffer.limit() + ") >= network buffer size (" + currentNetWriteBufferSize + ")");
            }
        } else {
            if (wrapResult.getStatus() == SSLEngineResult.Status.BUFFER_UNDERFLOW) {
                throw new IllegalStateException("SSL BUFFER_UNDERFLOW during write");
            }
            if (wrapResult.getStatus() == SSLEngineResult.Status.CLOSED) {
                throw new EOFException();
            }
        }
        return written;
    }

    @Override
    public long write(ByteBuffer[] srcs, int offset, int length) throws IOException {
        if (offset < 0 || length < 0 || offset > srcs.length - length) {
            throw new IndexOutOfBoundsException();
        }
        int totalWritten = 0;
        for (int i = offset; i < length; ++i) {
            int written;
            if ((srcs[i].hasRemaining() || this.hasPendingWrites()) && (written = this.write(srcs[i])) > 0) {
                totalWritten += written;
            }
            if (srcs[i].hasRemaining() || this.hasPendingWrites()) break;
        }
        return totalWritten;
    }

    @Override
    public long write(ByteBuffer[] srcs) throws IOException {
        return this.write(srcs, 0, srcs.length);
    }

    @Override
    public Principal peerPrincipal() throws IOException {
        try {
            return this.sslEngine.getSession().getPeerPrincipal();
        }
        catch (SSLPeerUnverifiedException se) {
            log.warn("SSL peer is not authenticated, returning ANONYMOUS instead");
            return KafkaPrincipal.ANONYMOUS;
        }
    }

    public SSLSession sslSession() throws IllegalStateException {
        return this.sslEngine.getSession();
    }

    @Override
    public void addInterestOps(int ops2) {
        if (!this.key.isValid()) {
            throw new CancelledKeyException();
        }
        if (!this.handshakeComplete) {
            throw new IllegalStateException("handshake is not completed");
        }
        this.key.interestOps(this.key.interestOps() | ops2);
    }

    @Override
    public void removeInterestOps(int ops2) {
        if (!this.key.isValid()) {
            throw new CancelledKeyException();
        }
        if (!this.handshakeComplete) {
            throw new IllegalStateException("handshake is not completed");
        }
        this.key.interestOps(this.key.interestOps() & ~ops2);
    }

    protected Runnable delegatedTask() {
        return this.sslEngine.getDelegatedTask();
    }

    private int readFromAppBuffer(ByteBuffer dst) {
        this.appReadBuffer.flip();
        int remaining = Math.min(this.appReadBuffer.remaining(), dst.remaining());
        if (remaining > 0) {
            int limit = this.appReadBuffer.limit();
            this.appReadBuffer.limit(this.appReadBuffer.position() + remaining);
            dst.put(this.appReadBuffer);
            this.appReadBuffer.limit(limit);
        }
        this.appReadBuffer.compact();
        return remaining;
    }

    protected int netReadBufferSize() {
        return this.sslEngine.getSession().getPacketBufferSize();
    }

    protected int netWriteBufferSize() {
        return this.sslEngine.getSession().getPacketBufferSize();
    }

    protected int applicationBufferSize() {
        return this.sslEngine.getSession().getApplicationBufferSize();
    }

    protected ByteBuffer netReadBuffer() {
        return this.netReadBuffer;
    }

    private void handshakeFailure() {
        this.sslEngine.closeOutbound();
        try {
            this.sslEngine.closeInbound();
        }
        catch (SSLException e) {
            log.debug("SSLEngine.closeInBound() raised an exception.", e);
        }
    }

    @Override
    public boolean isMute() {
        return this.key.isValid() && (this.key.interestOps() & 1) == 0;
    }

    @Override
    public long transferFrom(FileChannel fileChannel, long position, long count2) throws IOException {
        return fileChannel.transferTo(position, count2, this);
    }
}

