/*
 * Copyright (C) 2013  The Async HBase Authors.  All rights reserved.
 * This file is part of Async HBase.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 *   - Neither the name of the StumbleUpon nor the names of its contributors
 *     may be used to endorse or promote products derived from this software
 *     without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package com.google.protobuf;  // This is a lie.

import java.lang.reflect.Field;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Helper class to extract byte arrays from {@link ByteString} without copy.
 * <p>
 * Without this protobufs would force us to copy every single byte array out
 * of the objects de-serialized from the wire (which already do one copy, on
 * top of the copies the JVM does to go from kernel buffer to C buffer and
 * from C buffer to JVM buffer).
 * <p>
 * <strong>This class isn't part of the public API of AsyncHBase.</strong>
 * @since 1.5
 */
public final class ZeroCopyLiteralByteString extends LiteralByteStringAsyncHBase {
  private static final long serialVersionUID = 1L;
  
  private static final Logger LOG = LoggerFactory.getLogger(ZeroCopyLiteralByteString.class);
  
  private static final Class<?> _clsLiteralByteString;
  private static final Field _fieldBytes;
  private static final boolean _canUseEfficientCopy;
  static {
    Class<?> clsLiteralByteString = null;
    Field fieldBytes = null;
    boolean canUseEfficientCopy = false;
    try {
      // setup.
      clsLiteralByteString = Class.forName(
          "com.google.protobuf.ByteString$LiteralByteString");
      fieldBytes = clsLiteralByteString.getDeclaredField("bytes");
      fieldBytes.setAccessible(true);

      // test?
      final ByteString testByteString = ByteString.copyFromUtf8("");
      @SuppressWarnings("unused")
      final byte[] wrappedBytes = (byte[]) fieldBytes.get(testByteString);

      // yay!
      canUseEfficientCopy = true;
    } catch (ClassNotFoundException | NoSuchFieldException | ClassCastException
        | IllegalArgumentException | IllegalAccessException e) {
      LOG.warn("AsyncHBase: Could not load/access "
          + "'com.google.protobuf.ByteString$LiteralByteString' class."
          + "Make sure Google Protobuf version library > 3.5 is in the classpath", e);
      LOG.warn("Protobuf zero-copy feature will be unavailable.");
    }
    LOG.info("Protobuf zero-copy feature initialized.");
    _clsLiteralByteString = clsLiteralByteString;
    _fieldBytes = fieldBytes;
    _canUseEfficientCopy = canUseEfficientCopy;
  }

  /** Private constructor so this class cannot be instantiated. */
  private ZeroCopyLiteralByteString() {
    super(null);
    throw new UnsupportedOperationException("Should never be here.");
  }

  /**
   * Wraps a byte array in a {@link ByteString} without copying it.
   */
  public static ByteString wrap(final byte[] array) {
    return new LiteralByteStringAsyncHBase(array);
  }

  /**
   * Extracts the byte array from the given {@link ByteString} without copy.
   * @param buf A buffer from which to extract the array.  This buffer must be
   * actually an instance of a {@code LiteralByteString}.
   */
  public static byte[] zeroCopyGetBytes(final ByteString buf) {
    if (buf instanceof LiteralByteStringAsyncHBase) {
      return ((LiteralByteStringAsyncHBase) buf).bytes;
    } else if (_clsLiteralByteString.isAssignableFrom(buf.getClass())) {
      if (_canUseEfficientCopy) {
        try {
          return (byte[]) _fieldBytes.get(buf);
        } catch (Throwable e) {
          // should never happen, we tested this above
          throw new RuntimeException("Unexpected Exception while"
              + " accessing internal field `bytes`.", e);
        }
      } else {
        return buf.toByteArray();
      }
    }
    throw new UnsupportedOperationException("Need a LiteralByteString or"
        + " LiteralByteStringAsyncHBase, got a " + buf.getClass().getName());
  }

}
