/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode.snapshot;

import com.google.common.base.Preconditions;
import com.google.common.primitives.SignedBytes;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.hdfs.protocol.SnapshotDiffReport;
import org.apache.hadoop.hdfs.protocol.SnapshotException;
import org.apache.hadoop.hdfs.server.namenode.Content;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeDirectory;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.hadoop.hdfs.server.namenode.INodeMap;
import org.apache.hadoop.hdfs.server.namenode.Quota;
import org.apache.hadoop.hdfs.server.namenode.snapshot.FileWithSnapshot;
import org.apache.hadoop.hdfs.server.namenode.snapshot.INodeDirectoryWithSnapshot;
import org.apache.hadoop.hdfs.server.namenode.snapshot.Snapshot;
import org.apache.hadoop.hdfs.util.Diff;
import org.apache.hadoop.hdfs.util.ReadOnlyList;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
public class INodeDirectorySnapshottable
extends INodeDirectoryWithSnapshot {
    static final int SNAPSHOT_LIMIT = 65536;
    private final List<Snapshot> snapshotsByNames = new ArrayList<Snapshot>();
    private int snapshotQuota = 65536;

    public static INodeDirectorySnapshottable valueOf(INode inode, String src) throws IOException {
        INodeDirectory dir = INodeDirectory.valueOf(inode, src);
        if (!dir.isSnapshottable()) {
            throw new SnapshotException("Directory is not a snapshottable directory: " + src);
        }
        return (INodeDirectorySnapshottable)dir;
    }

    ReadOnlyList<Snapshot> getSnapshotsByNames() {
        return ReadOnlyList.Util.asReadOnlyList(this.snapshotsByNames);
    }

    public INodeDirectorySnapshottable(INodeDirectory dir) {
        super(dir, true, dir instanceof INodeDirectoryWithSnapshot ? ((INodeDirectoryWithSnapshot)dir).getDiffs() : null);
    }

    public int getNumSnapshots() {
        return this.snapshotsByNames.size();
    }

    private int searchSnapshot(byte[] snapshotName) {
        return Collections.binarySearch(this.snapshotsByNames, snapshotName);
    }

    public Snapshot getSnapshot(byte[] snapshotName) {
        int i = this.searchSnapshot(snapshotName);
        return i < 0 ? null : this.snapshotsByNames.get(i);
    }

    public ReadOnlyList<Snapshot> getSnapshotList() {
        return ReadOnlyList.Util.asReadOnlyList(this.snapshotsByNames);
    }

    public void renameSnapshot(String path, String oldName, String newName) throws SnapshotException {
        if (newName.equals(oldName)) {
            return;
        }
        int indexOfOld = this.searchSnapshot(DFSUtil.string2Bytes(oldName));
        if (indexOfOld < 0) {
            throw new SnapshotException("The snapshot " + oldName + " does not exist for directory " + path);
        }
        byte[] newNameBytes = DFSUtil.string2Bytes(newName);
        int indexOfNew = this.searchSnapshot(newNameBytes);
        if (indexOfNew > 0) {
            throw new SnapshotException("The snapshot " + newName + " already exists for directory " + path);
        }
        Snapshot snapshot = this.snapshotsByNames.remove(indexOfOld);
        Snapshot.Root ssRoot = snapshot.getRoot();
        ssRoot.setLocalName(newNameBytes);
        indexOfNew = -indexOfNew - 1;
        if (indexOfNew <= indexOfOld) {
            this.snapshotsByNames.add(indexOfNew, snapshot);
        } else {
            this.snapshotsByNames.add(indexOfNew - 1, snapshot);
        }
    }

    public int getSnapshotQuota() {
        return this.snapshotQuota;
    }

    public void setSnapshotQuota(int snapshotQuota) {
        if (snapshotQuota < 0) {
            throw new HadoopIllegalArgumentException("Cannot set snapshot quota to " + snapshotQuota + " < 0");
        }
        this.snapshotQuota = snapshotQuota;
    }

    @Override
    public boolean isSnapshottable() {
        return true;
    }

    void addSnapshot(Snapshot snapshot) {
        this.snapshotsByNames.add(snapshot);
    }

    Snapshot addSnapshot(int id, String name) throws SnapshotException, QuotaExceededException {
        int n = this.getNumSnapshots();
        if (n + 1 > this.snapshotQuota) {
            throw new SnapshotException("Failed to add snapshot: there are already " + n + " snapshot(s) and the snapshot quota is " + this.snapshotQuota);
        }
        Snapshot s2 = new Snapshot(id, name, this);
        byte[] nameBytes = s2.getRoot().getLocalNameBytes();
        int i = this.searchSnapshot(nameBytes);
        if (i >= 0) {
            throw new SnapshotException("Failed to add snapshot: there is already a snapshot with the same name \"" + Snapshot.getSnapshotName(s2) + "\".");
        }
        INodeDirectoryWithSnapshot.DirectoryDiff d = (INodeDirectoryWithSnapshot.DirectoryDiff)this.getDiffs().addDiff(s2, this);
        d.snapshotINode = s2.getRoot();
        this.snapshotsByNames.add(-i - 1, s2);
        this.updateModificationTime(Time.now(), null, null);
        s2.getRoot().setModificationTime(this.getModificationTime(), null, null);
        return s2;
    }

    Snapshot removeSnapshot(String snapshotName, INode.BlocksMapUpdateInfo collectedBlocks, List<INode> removedINodes) throws SnapshotException {
        int i = this.searchSnapshot(DFSUtil.string2Bytes(snapshotName));
        if (i < 0) {
            throw new SnapshotException("Cannot delete snapshot " + snapshotName + " from path " + this.getFullPathName() + ": the snapshot does not exist.");
        }
        Snapshot snapshot = this.snapshotsByNames.get(i);
        Snapshot prior = Snapshot.findLatestSnapshot(this, snapshot);
        try {
            Quota.Counts counts = this.cleanSubtree(snapshot, prior, collectedBlocks, removedINodes, true);
            INodeDirectory parent = this.getParent();
            if (parent != null) {
                parent.addSpaceConsumed(-counts.get(Quota.NAMESPACE), -counts.get(Quota.DISKSPACE), true);
            }
        }
        catch (QuotaExceededException e) {
            LOG.error("BUG: removeSnapshot increases namespace usage.", e);
        }
        this.snapshotsByNames.remove(i);
        return snapshot;
    }

    @Override
    public Content.Counts computeContentSummary(Content.Counts counts) {
        super.computeContentSummary(counts);
        counts.add(Content.SNAPSHOT, this.snapshotsByNames.size());
        counts.add(Content.SNAPSHOTTABLE_DIRECTORY, 1L);
        return counts;
    }

    SnapshotDiffInfo computeDiff(String from2, String to2) throws SnapshotException {
        Snapshot fromSnapshot = this.getSnapshotByName(from2);
        Snapshot toSnapshot = this.getSnapshotByName(to2);
        if (from2.equals(to2)) {
            return null;
        }
        SnapshotDiffInfo diffs = new SnapshotDiffInfo(this, fromSnapshot, toSnapshot);
        this.computeDiffRecursively(this, new ArrayList<byte[]>(), diffs);
        return diffs;
    }

    private Snapshot getSnapshotByName(String snapshotName) throws SnapshotException {
        Snapshot s2 = null;
        if (snapshotName != null && !snapshotName.isEmpty()) {
            int index2 = this.searchSnapshot(DFSUtil.string2Bytes(snapshotName));
            if (index2 < 0) {
                throw new SnapshotException("Cannot find the snapshot of directory " + this.getFullPathName() + " with name " + snapshotName);
            }
            s2 = this.snapshotsByNames.get(index2);
        }
        return s2;
    }

    private void computeDiffRecursively(INode node, List<byte[]> parentPath, SnapshotDiffInfo diffReport) {
        INodeDirectoryWithSnapshot.ChildrenDiff diff2 = new INodeDirectoryWithSnapshot.ChildrenDiff();
        byte[][] relativePath = (byte[][])parentPath.toArray((T[])new byte[parentPath.size()][]);
        if (node.isDirectory()) {
            INodeDirectoryWithSnapshot sdir;
            boolean change;
            INodeDirectory dir = node.asDirectory();
            if (dir instanceof INodeDirectoryWithSnapshot && (change = (sdir = (INodeDirectoryWithSnapshot)dir).computeDiffBetweenSnapshots(diffReport.from, diffReport.to, diff2))) {
                diffReport.addDirDiff(sdir, relativePath, diff2);
            }
            ReadOnlyList<INode> children = dir.getChildrenList(diffReport.isFromEarlier() ? diffReport.to : diffReport.from);
            for (INode child : children) {
                byte[] name = child.getLocalNameBytes();
                if (diff2.searchIndex(Diff.ListType.CREATED, name) >= 0 || diff2.searchIndex(Diff.ListType.DELETED, name) >= 0) continue;
                parentPath.add(name);
                this.computeDiffRecursively(child, parentPath, diffReport);
                parentPath.remove(parentPath.size() - 1);
            }
        } else if (node.isFile() && node.asFile() instanceof FileWithSnapshot) {
            FileWithSnapshot file = (FileWithSnapshot)((Object)node.asFile());
            Snapshot earlierSnapshot = diffReport.isFromEarlier() ? diffReport.from : diffReport.to;
            Snapshot laterSnapshot = diffReport.isFromEarlier() ? diffReport.to : diffReport.from;
            boolean change = file.getDiffs().changedBetweenSnapshots(earlierSnapshot, laterSnapshot);
            if (change) {
                diffReport.addFileDiff(file.asINodeFile(), relativePath);
            }
        }
    }

    INodeDirectory replaceSelf(Snapshot latest, INodeMap inodeMap) throws QuotaExceededException {
        if (latest == null) {
            Preconditions.checkState(this.getLastSnapshot() == null, "latest == null but getLastSnapshot() != null, this=%s", this);
            return this.replaceSelf4INodeDirectory(inodeMap);
        }
        return this.replaceSelf4INodeDirectoryWithSnapshot(inodeMap).recordModification(latest, null);
    }

    @Override
    public String toDetailString() {
        return super.toDetailString() + ", snapshotsByNames=" + this.snapshotsByNames;
    }

    @Override
    public void dumpTreeRecursively(PrintWriter out, StringBuilder prefix, Snapshot snapshot) {
        super.dumpTreeRecursively(out, prefix, snapshot);
        if (snapshot == null) {
            out.println();
            out.print(prefix);
            out.print("Snapshot of ");
            String name = this.getLocalName();
            out.print(name.isEmpty() ? "/" : name);
            out.print(": quota=");
            out.print(this.getSnapshotQuota());
            int n = 0;
            for (INodeDirectoryWithSnapshot.DirectoryDiff diff2 : this.getDiffs()) {
                if (!diff2.isSnapshotRoot()) continue;
                ++n;
            }
            Preconditions.checkState(n == this.snapshotsByNames.size());
            out.print(", #snapshot=");
            out.println(n);
            INodeDirectorySnapshottable.dumpTreeRecursively(out, prefix, new Iterable<INodeDirectory.SnapshotAndINode>(){

                @Override
                public Iterator<INodeDirectory.SnapshotAndINode> iterator() {
                    return new Iterator<INodeDirectory.SnapshotAndINode>(){
                        final Iterator<INodeDirectoryWithSnapshot.DirectoryDiff> i;
                        private INodeDirectoryWithSnapshot.DirectoryDiff next;
                        {
                            this.i = INodeDirectorySnapshottable.this.getDiffs().iterator();
                            this.next = this.findNext();
                        }

                        private INodeDirectoryWithSnapshot.DirectoryDiff findNext() {
                            while (this.i.hasNext()) {
                                INodeDirectoryWithSnapshot.DirectoryDiff diff2 = this.i.next();
                                if (!diff2.isSnapshotRoot()) continue;
                                return diff2;
                            }
                            return null;
                        }

                        @Override
                        public boolean hasNext() {
                            return this.next != null;
                        }

                        @Override
                        public INodeDirectory.SnapshotAndINode next() {
                            Snapshot s2 = this.next.snapshot;
                            INodeDirectory.SnapshotAndINode pair2 = new INodeDirectory.SnapshotAndINode(s2);
                            this.next = this.findNext();
                            return pair2;
                        }

                        @Override
                        public void remove() {
                            throw new UnsupportedOperationException();
                        }
                    };
                }
            });
        }
    }

    public static class SnapshotDiffInfo {
        public static final Comparator<INode> INODE_COMPARATOR = new Comparator<INode>(){

            @Override
            public int compare(INode left, INode right) {
                if (left == null) {
                    return right == null ? 0 : -1;
                }
                if (right == null) {
                    return 1;
                }
                int cmp = this.compare(left.getParent(), right.getParent());
                return cmp == 0 ? SignedBytes.lexicographicalComparator().compare(left.getLocalNameBytes(), right.getLocalNameBytes()) : cmp;
            }
        };
        private final INodeDirectorySnapshottable snapshotRoot;
        private final Snapshot from;
        private final Snapshot to;
        private final SortedMap<INode, byte[][]> diffMap = new TreeMap<INode, byte[][]>(INODE_COMPARATOR);
        private final Map<INodeDirectoryWithSnapshot, INodeDirectoryWithSnapshot.ChildrenDiff> dirDiffMap = new HashMap<INodeDirectoryWithSnapshot, INodeDirectoryWithSnapshot.ChildrenDiff>();

        SnapshotDiffInfo(INodeDirectorySnapshottable snapshotRoot, Snapshot start, Snapshot end) {
            this.snapshotRoot = snapshotRoot;
            this.from = start;
            this.to = end;
        }

        private void addDirDiff(INodeDirectoryWithSnapshot dir, byte[][] relativePath, INodeDirectoryWithSnapshot.ChildrenDiff diff2) {
            this.dirDiffMap.put(dir, diff2);
            this.diffMap.put(dir, relativePath);
        }

        private void addFileDiff(INodeFile file, byte[][] relativePath) {
            this.diffMap.put(file, relativePath);
        }

        private boolean isFromEarlier() {
            return Snapshot.ID_COMPARATOR.compare(this.from, this.to) < 0;
        }

        public SnapshotDiffReport generateReport() {
            ArrayList<SnapshotDiffReport.DiffReportEntry> diffReportList = new ArrayList<SnapshotDiffReport.DiffReportEntry>();
            for (INode node : this.diffMap.keySet()) {
                diffReportList.add(new SnapshotDiffReport.DiffReportEntry(SnapshotDiffReport.DiffType.MODIFY, (byte[][])this.diffMap.get(node)));
                if (!node.isDirectory()) continue;
                INodeDirectoryWithSnapshot.ChildrenDiff dirDiff = this.dirDiffMap.get(node);
                List<SnapshotDiffReport.DiffReportEntry> subList = dirDiff.generateReport((byte[][])this.diffMap.get(node), (INodeDirectoryWithSnapshot)node, this.isFromEarlier());
                diffReportList.addAll(subList);
            }
            return new SnapshotDiffReport(this.snapshotRoot.getFullPathName(), Snapshot.getSnapshotName(this.from), Snapshot.getSnapshotName(this.to), diffReportList);
        }
    }
}

