/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import java.io.ByteArrayOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.io.HeapSize;
import org.apache.hadoop.hbase.io.encoding.DataBlockEncoding;
import org.apache.hadoop.hbase.io.hfile.BlockCacheKey;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.BlockWithScanInfo;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileBlock;
import org.apache.hadoop.hbase.io.hfile.InlineBlockWriter;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ClassSize;
import org.apache.hadoop.io.WritableUtils;
import org.apache.hadoop.util.StringUtils;

@InterfaceAudience.Private
public class HFileBlockIndex {
    private static final Log LOG = LogFactory.getLog(HFileBlockIndex.class);
    static final int DEFAULT_MAX_CHUNK_SIZE = 131072;
    public static final String MAX_CHUNK_SIZE_KEY = "hfile.index.block.max.size";
    static final int SECONDARY_INDEX_ENTRY_OVERHEAD = 12;
    private static final String INLINE_BLOCKS_NOT_ALLOWED = "Inline blocks are not allowed in the single-level-only mode";
    private static final int MID_KEY_METADATA_SIZE = 16;

    public static int getMaxChunkSize(Configuration conf) {
        return conf.getInt(MAX_CHUNK_SIZE_KEY, 131072);
    }

    static class BlockIndexChunk {
        private final List<byte[]> blockKeys = new ArrayList<byte[]>();
        private final List<Long> blockOffsets = new ArrayList<Long>();
        private final List<Integer> onDiskDataSizes = new ArrayList<Integer>();
        private final List<Long> numSubEntriesAt = new ArrayList<Long>();
        private int curTotalNonRootEntrySize = 0;
        private int curTotalRootSize = 0;
        private final List<Integer> secondaryIndexOffsetMarks = new ArrayList<Integer>();

        BlockIndexChunk() {
        }

        void add(byte[] firstKey, long blockOffset, int onDiskDataSize, long curTotalNumSubEntries) {
            this.secondaryIndexOffsetMarks.add(this.curTotalNonRootEntrySize);
            this.curTotalNonRootEntrySize += 12 + firstKey.length;
            this.curTotalRootSize += 12 + WritableUtils.getVIntSize(firstKey.length) + firstKey.length;
            this.blockKeys.add(firstKey);
            this.blockOffsets.add(blockOffset);
            this.onDiskDataSizes.add(onDiskDataSize);
            if (curTotalNumSubEntries != -1L) {
                this.numSubEntriesAt.add(curTotalNumSubEntries);
                if (this.numSubEntriesAt.size() != this.blockKeys.size()) {
                    throw new IllegalStateException("Only have key/value count stats for " + this.numSubEntriesAt.size() + " block index " + "entries out of " + this.blockKeys.size());
                }
            }
        }

        public void add(byte[] firstKey, long blockOffset, int onDiskDataSize) {
            this.add(firstKey, blockOffset, onDiskDataSize, -1L);
        }

        public void clear() {
            this.blockKeys.clear();
            this.blockOffsets.clear();
            this.onDiskDataSizes.clear();
            this.secondaryIndexOffsetMarks.clear();
            this.numSubEntriesAt.clear();
            this.curTotalNonRootEntrySize = 0;
            this.curTotalRootSize = 0;
        }

        public int getEntryBySubEntry(long k) {
            int i = Collections.binarySearch(this.numSubEntriesAt, k);
            if (i >= 0) {
                return i + 1;
            }
            return -i - 1;
        }

        public byte[] getMidKeyMetadata() throws IOException {
            ByteArrayOutputStream baos = new ByteArrayOutputStream(16);
            DataOutputStream baosDos = new DataOutputStream(baos);
            long totalNumSubEntries = this.numSubEntriesAt.get(this.blockKeys.size() - 1);
            if (totalNumSubEntries == 0L) {
                throw new IOException("No leaf-level entries, mid-key unavailable");
            }
            long midKeySubEntry = (totalNumSubEntries - 1L) / 2L;
            int midKeyEntry = this.getEntryBySubEntry(midKeySubEntry);
            baosDos.writeLong(this.blockOffsets.get(midKeyEntry));
            baosDos.writeInt(this.onDiskDataSizes.get(midKeyEntry));
            long numSubEntriesBefore = midKeyEntry > 0 ? this.numSubEntriesAt.get(midKeyEntry - 1) : 0L;
            long subEntryWithinEntry = midKeySubEntry - numSubEntriesBefore;
            if (subEntryWithinEntry < 0L || subEntryWithinEntry > Integer.MAX_VALUE) {
                throw new IOException("Could not identify mid-key index within the leaf-level block containing mid-key: out of range (" + subEntryWithinEntry + ", numSubEntriesBefore=" + numSubEntriesBefore + ", midKeySubEntry=" + midKeySubEntry + ")");
            }
            baosDos.writeInt((int)subEntryWithinEntry);
            if (baosDos.size() != 16) {
                throw new IOException("Could not write mid-key metadata: size=" + baosDos.size() + ", correct size: " + 16);
            }
            baos.close();
            return baos.toByteArray();
        }

        void writeNonRoot(DataOutput out) throws IOException {
            out.writeInt(this.blockKeys.size());
            if (this.secondaryIndexOffsetMarks.size() != this.blockKeys.size()) {
                throw new IOException("Corrupted block index chunk writer: " + this.blockKeys.size() + " entries but " + this.secondaryIndexOffsetMarks.size() + " secondary index items");
            }
            for (int currentSecondaryIndex : this.secondaryIndexOffsetMarks) {
                out.writeInt(currentSecondaryIndex);
            }
            out.writeInt(this.curTotalNonRootEntrySize);
            for (int i = 0; i < this.blockKeys.size(); ++i) {
                out.writeLong(this.blockOffsets.get(i));
                out.writeInt(this.onDiskDataSizes.get(i));
                out.write(this.blockKeys.get(i));
            }
        }

        int getNonRootSize() {
            return 4 + 4 * (this.blockKeys.size() + 1) + this.curTotalNonRootEntrySize;
        }

        void writeRoot(DataOutput out) throws IOException {
            for (int i = 0; i < this.blockKeys.size(); ++i) {
                out.writeLong(this.blockOffsets.get(i));
                out.writeInt(this.onDiskDataSizes.get(i));
                Bytes.writeByteArray(out, this.blockKeys.get(i));
            }
        }

        int getRootSize() {
            return this.curTotalRootSize;
        }

        public int getNumEntries() {
            return this.blockKeys.size();
        }

        public byte[] getBlockKey(int i) {
            return this.blockKeys.get(i);
        }

        public long getBlockOffset(int i) {
            return this.blockOffsets.get(i);
        }

        public int getOnDiskDataSize(int i) {
            return this.onDiskDataSizes.get(i);
        }

        public long getCumulativeNumKV(int i) {
            if (i < 0) {
                return 0L;
            }
            return this.numSubEntriesAt.get(i);
        }
    }

    public static class BlockIndexWriter
    implements InlineBlockWriter {
        private BlockIndexChunk rootChunk = new BlockIndexChunk();
        private BlockIndexChunk curInlineChunk = new BlockIndexChunk();
        private int numLevels = 1;
        private HFileBlock.Writer blockWriter;
        private byte[] firstKey = null;
        private long totalNumEntries;
        private long totalBlockOnDiskSize;
        private long totalBlockUncompressedSize;
        private int maxChunkSize;
        private boolean singleLevelOnly;
        private CacheConfig cacheConf;
        private String nameForCaching;

        public BlockIndexWriter() {
            this(null, null, null);
            this.singleLevelOnly = true;
        }

        public BlockIndexWriter(HFileBlock.Writer blockWriter, CacheConfig cacheConf, String nameForCaching) {
            if (cacheConf == null != (nameForCaching == null)) {
                throw new IllegalArgumentException("Block cache and file name for caching must be both specified or both null");
            }
            this.blockWriter = blockWriter;
            this.cacheConf = cacheConf;
            this.nameForCaching = nameForCaching;
            this.maxChunkSize = 131072;
        }

        public void setMaxChunkSize(int maxChunkSize) {
            if (maxChunkSize <= 0) {
                throw new IllegalArgumentException("Invald maximum index block size");
            }
            this.maxChunkSize = maxChunkSize;
        }

        public long writeIndexBlocks(FSDataOutputStream out) throws IOException {
            byte[] midKeyMetadata;
            if (this.curInlineChunk != null && this.curInlineChunk.getNumEntries() != 0) {
                throw new IOException("Trying to write a multi-level block index, but are " + this.curInlineChunk.getNumEntries() + " entries in the " + "last inline chunk.");
            }
            byte[] byArray = midKeyMetadata = this.numLevels > 1 ? this.rootChunk.getMidKeyMetadata() : null;
            if (this.curInlineChunk != null) {
                while (this.rootChunk.getRootSize() > this.maxChunkSize) {
                    this.rootChunk = this.writeIntermediateLevel(out, this.rootChunk);
                    ++this.numLevels;
                }
            }
            long rootLevelIndexPos = out.getPos();
            DataOutputStream blockStream = this.blockWriter.startWriting(BlockType.ROOT_INDEX);
            this.rootChunk.writeRoot(blockStream);
            if (midKeyMetadata != null) {
                blockStream.write(midKeyMetadata);
            }
            this.blockWriter.writeHeaderAndData(out);
            this.totalBlockOnDiskSize += (long)this.blockWriter.getOnDiskSizeWithoutHeader();
            this.totalBlockUncompressedSize += (long)this.blockWriter.getUncompressedSizeWithoutHeader();
            if (LOG.isTraceEnabled()) {
                LOG.trace("Wrote a " + this.numLevels + "-level index with root level at pos " + rootLevelIndexPos + ", " + this.rootChunk.getNumEntries() + " root-level entries, " + this.totalNumEntries + " total entries, " + StringUtils.humanReadableInt(this.totalBlockOnDiskSize) + " on-disk size, " + StringUtils.humanReadableInt(this.totalBlockUncompressedSize) + " total uncompressed size.");
            }
            return rootLevelIndexPos;
        }

        public void writeSingleLevelIndex(DataOutput out, String description) throws IOException {
            this.expectNumLevels(1);
            if (!this.singleLevelOnly) {
                throw new IOException("Single-level mode is turned off");
            }
            if (this.rootChunk.getNumEntries() > 0) {
                throw new IOException("Root-level entries already added in single-level mode");
            }
            this.rootChunk = this.curInlineChunk;
            this.curInlineChunk = new BlockIndexChunk();
            if (LOG.isTraceEnabled()) {
                LOG.trace("Wrote a single-level " + description + " index with " + this.rootChunk.getNumEntries() + " entries, " + this.rootChunk.getRootSize() + " bytes");
            }
            this.rootChunk.writeRoot(out);
        }

        private BlockIndexChunk writeIntermediateLevel(FSDataOutputStream out, BlockIndexChunk currentLevel) throws IOException {
            BlockIndexChunk parent = new BlockIndexChunk();
            BlockIndexChunk curChunk = new BlockIndexChunk();
            for (int i = 0; i < currentLevel.getNumEntries(); ++i) {
                curChunk.add(currentLevel.getBlockKey(i), currentLevel.getBlockOffset(i), currentLevel.getOnDiskDataSize(i));
                if (curChunk.getRootSize() < this.maxChunkSize) continue;
                this.writeIntermediateBlock(out, parent, curChunk);
            }
            if (curChunk.getNumEntries() > 0) {
                this.writeIntermediateBlock(out, parent, curChunk);
            }
            return parent;
        }

        private void writeIntermediateBlock(FSDataOutputStream out, BlockIndexChunk parent, BlockIndexChunk curChunk) throws IOException {
            long beginOffset = out.getPos();
            DataOutputStream dos = this.blockWriter.startWriting(BlockType.INTERMEDIATE_INDEX);
            curChunk.writeNonRoot(dos);
            byte[] curFirstKey = curChunk.getBlockKey(0);
            this.blockWriter.writeHeaderAndData(out);
            if (this.cacheConf != null) {
                HFileBlock blockForCaching = this.blockWriter.getBlockForCaching(this.cacheConf);
                this.cacheConf.getBlockCache().cacheBlock(new BlockCacheKey(this.nameForCaching, beginOffset, DataBlockEncoding.NONE, blockForCaching.getBlockType()), blockForCaching);
            }
            this.totalBlockOnDiskSize += (long)this.blockWriter.getOnDiskSizeWithoutHeader();
            this.totalBlockUncompressedSize += (long)this.blockWriter.getUncompressedSizeWithoutHeader();
            parent.add(curFirstKey, beginOffset, this.blockWriter.getOnDiskSizeWithHeader());
            curChunk.clear();
            curFirstKey = null;
        }

        public final int getNumRootEntries() {
            return this.rootChunk.getNumEntries();
        }

        public int getNumLevels() {
            return this.numLevels;
        }

        private void expectNumLevels(int expectedNumLevels) {
            if (this.numLevels != expectedNumLevels) {
                throw new IllegalStateException("Number of block index levels is " + this.numLevels + "but is expected to be " + expectedNumLevels);
            }
        }

        @Override
        public boolean shouldWriteBlock(boolean closing) {
            if (this.singleLevelOnly) {
                throw new UnsupportedOperationException(HFileBlockIndex.INLINE_BLOCKS_NOT_ALLOWED);
            }
            if (this.curInlineChunk == null) {
                throw new IllegalStateException("curInlineChunk is null; has shouldWriteBlock been called with closing=true and then called again?");
            }
            if (this.curInlineChunk.getNumEntries() == 0) {
                return false;
            }
            if (closing) {
                if (this.rootChunk.getNumEntries() == 0) {
                    this.expectNumLevels(1);
                    this.rootChunk = this.curInlineChunk;
                    this.curInlineChunk = null;
                    return false;
                }
                return true;
            }
            return this.curInlineChunk.getNonRootSize() >= this.maxChunkSize;
        }

        @Override
        public void writeInlineBlock(DataOutput out) throws IOException {
            if (this.singleLevelOnly) {
                throw new UnsupportedOperationException(HFileBlockIndex.INLINE_BLOCKS_NOT_ALLOWED);
            }
            this.curInlineChunk.writeNonRoot(out);
            this.firstKey = this.curInlineChunk.getBlockKey(0);
            this.curInlineChunk.clear();
        }

        @Override
        public void blockWritten(long offset, int onDiskSize, int uncompressedSize) {
            this.totalBlockOnDiskSize += (long)onDiskSize;
            this.totalBlockUncompressedSize += (long)uncompressedSize;
            if (this.singleLevelOnly) {
                throw new UnsupportedOperationException(HFileBlockIndex.INLINE_BLOCKS_NOT_ALLOWED);
            }
            if (this.firstKey == null) {
                throw new IllegalStateException("Trying to add second-level index entry with offset=" + offset + " and onDiskSize=" + onDiskSize + "but the first key was not set in writeInlineBlock");
            }
            if (this.rootChunk.getNumEntries() == 0) {
                this.expectNumLevels(1);
                this.numLevels = 2;
            }
            this.rootChunk.add(this.firstKey, offset, onDiskSize, this.totalNumEntries);
            this.firstKey = null;
        }

        @Override
        public BlockType getInlineBlockType() {
            return BlockType.LEAF_INDEX;
        }

        public void addEntry(byte[] firstKey, long blockOffset, int blockDataSize) {
            this.curInlineChunk.add(firstKey, blockOffset, blockDataSize);
            ++this.totalNumEntries;
        }

        public void ensureSingleLevel() throws IOException {
            if (this.numLevels > 1) {
                throw new IOException("Wrote a " + this.numLevels + "-level index with " + this.rootChunk.getNumEntries() + " root-level entries, but " + "this is expected to be a single-level block index.");
            }
        }

        @Override
        public boolean getCacheOnWrite() {
            return this.cacheConf != null && this.cacheConf.shouldCacheIndexesOnWrite();
        }

        public long getTotalUncompressedSize() {
            return this.totalBlockUncompressedSize;
        }
    }

    public static class BlockIndexReader
    implements HeapSize {
        private final KeyValue.KVComparator comparator;
        private byte[][] blockKeys;
        private long[] blockOffsets;
        private int[] blockDataSizes;
        private int rootCount = 0;
        private long midLeafBlockOffset = -1L;
        private int midLeafBlockOnDiskSize = -1;
        private int midKeyEntry = -1;
        private AtomicReference<byte[]> midKey = new AtomicReference();
        private int searchTreeLevel;
        private HFile.CachingBlockReader cachingBlockReader;

        public BlockIndexReader(KeyValue.KVComparator c, int treeLevel, HFile.CachingBlockReader cachingBlockReader) {
            this(c, treeLevel);
            this.cachingBlockReader = cachingBlockReader;
        }

        public BlockIndexReader(KeyValue.KVComparator c, int treeLevel) {
            this.comparator = c;
            this.searchTreeLevel = treeLevel;
        }

        public boolean isEmpty() {
            return this.blockKeys.length == 0;
        }

        public void ensureNonEmpty() {
            if (this.blockKeys.length == 0) {
                throw new IllegalStateException("Block index is empty or not loaded");
            }
        }

        public HFileBlock seekToDataBlock(byte[] key, int keyOffset, int keyLength, HFileBlock currentBlock, boolean cacheBlocks, boolean pread, boolean isCompaction) throws IOException {
            BlockWithScanInfo blockWithScanInfo = this.loadDataBlockWithScanInfo(key, keyOffset, keyLength, currentBlock, cacheBlocks, pread, isCompaction);
            if (blockWithScanInfo == null) {
                return null;
            }
            return blockWithScanInfo.getHFileBlock();
        }

        public BlockWithScanInfo loadDataBlockWithScanInfo(byte[] key, int keyOffset, int keyLength, HFileBlock currentBlock, boolean cacheBlocks, boolean pread, boolean isCompaction) throws IOException {
            HFileBlock block;
            int rootLevelIndex = this.rootBlockContainingKey(key, keyOffset, keyLength);
            if (rootLevelIndex < 0 || rootLevelIndex >= this.blockOffsets.length) {
                return null;
            }
            byte[] nextIndexedKey = null;
            long currentOffset = this.blockOffsets[rootLevelIndex];
            int currentOnDiskSize = this.blockDataSizes[rootLevelIndex];
            nextIndexedKey = rootLevelIndex < this.blockKeys.length - 1 ? this.blockKeys[rootLevelIndex + 1] : HConstants.NO_NEXT_INDEXED_KEY;
            int lookupLevel = 1;
            int index2 = -1;
            while (true) {
                if (currentBlock != null && currentBlock.getOffset() == currentOffset) {
                    block = currentBlock;
                } else {
                    boolean shouldCache;
                    boolean bl = shouldCache = cacheBlocks || lookupLevel < this.searchTreeLevel;
                    BlockType expectedBlockType = lookupLevel < this.searchTreeLevel - 1 ? BlockType.INTERMEDIATE_INDEX : (lookupLevel == this.searchTreeLevel - 1 ? BlockType.LEAF_INDEX : BlockType.DATA);
                    block = this.cachingBlockReader.readBlock(currentOffset, currentOnDiskSize, shouldCache, pread, isCompaction, true, expectedBlockType);
                }
                if (block == null) {
                    throw new IOException("Failed to read block at offset " + currentOffset + ", onDiskSize=" + currentOnDiskSize);
                }
                if (block.getBlockType().isData()) break;
                if (++lookupLevel > this.searchTreeLevel) {
                    throw new IOException("Search Tree Level overflow: lookupLevel=" + lookupLevel + ", searchTreeLevel=" + this.searchTreeLevel);
                }
                ByteBuffer buffer = block.getBufferWithoutHeader();
                index2 = BlockIndexReader.locateNonRootIndexEntry(buffer, key, keyOffset, keyLength, this.comparator);
                if (index2 == -1) {
                    throw new IOException("The key " + Bytes.toStringBinary(key, keyOffset, keyLength) + " is before the" + " first key of the non-root index block " + block);
                }
                currentOffset = buffer.getLong();
                currentOnDiskSize = buffer.getInt();
                byte[] tmpNextIndexedKey = this.getNonRootIndexedKey(buffer, index2 + 1);
                if (tmpNextIndexedKey == null) continue;
                nextIndexedKey = tmpNextIndexedKey;
            }
            if (lookupLevel != this.searchTreeLevel) {
                throw new IOException("Reached a data block at level " + lookupLevel + " but the number of levels is " + this.searchTreeLevel);
            }
            BlockWithScanInfo blockWithScanInfo = new BlockWithScanInfo(block, nextIndexedKey);
            return blockWithScanInfo;
        }

        public byte[] midkey() throws IOException {
            if (this.rootCount == 0) {
                throw new IOException("HFile empty");
            }
            byte[] targetMidKey = this.midKey.get();
            if (targetMidKey != null) {
                return targetMidKey;
            }
            if (this.midLeafBlockOffset >= 0L) {
                if (this.cachingBlockReader == null) {
                    throw new IOException("Have to read the middle leaf block but no block reader available");
                }
                HFileBlock midLeafBlock = this.cachingBlockReader.readBlock(this.midLeafBlockOffset, this.midLeafBlockOnDiskSize, true, true, false, true, BlockType.LEAF_INDEX);
                ByteBuffer b = midLeafBlock.getBufferWithoutHeader();
                int numDataBlocks = b.getInt();
                int keyRelOffset = b.getInt(4 * (this.midKeyEntry + 1));
                int keyLen = b.getInt(4 * (this.midKeyEntry + 2)) - keyRelOffset;
                int keyOffset = b.arrayOffset() + 4 * (numDataBlocks + 2) + keyRelOffset + 12;
                targetMidKey = Arrays.copyOfRange(b.array(), keyOffset, keyOffset + keyLen);
            } else {
                targetMidKey = this.blockKeys[this.rootCount / 2];
            }
            this.midKey.set(targetMidKey);
            return targetMidKey;
        }

        public byte[] getRootBlockKey(int i) {
            return this.blockKeys[i];
        }

        public long getRootBlockOffset(int i) {
            return this.blockOffsets[i];
        }

        public int getRootBlockDataSize(int i) {
            return this.blockDataSizes[i];
        }

        public int getRootBlockCount() {
            return this.rootCount;
        }

        public int rootBlockContainingKey(byte[] key, int offset, int length) {
            int pos = Bytes.binarySearch(this.blockKeys, key, offset, length, this.comparator);
            if (pos >= 0) {
                assert (pos < this.blockKeys.length);
                return pos;
            }
            int i = -pos - 1;
            assert (0 <= i && i <= this.blockKeys.length);
            return i - 1;
        }

        private void add(byte[] key, long offset, int dataSize) {
            this.blockOffsets[this.rootCount] = offset;
            this.blockKeys[this.rootCount] = key;
            this.blockDataSizes[this.rootCount] = dataSize;
            ++this.rootCount;
        }

        private byte[] getNonRootIndexedKey(ByteBuffer nonRootIndex, int i) {
            int numEntries = nonRootIndex.getInt(0);
            if (i < 0 || i >= numEntries) {
                return null;
            }
            int entriesOffset = 4 * (numEntries + 2);
            int targetKeyRelOffset = nonRootIndex.getInt(4 * (i + 1));
            int targetKeyOffset = entriesOffset + targetKeyRelOffset + 12;
            int targetKeyLength = nonRootIndex.getInt(4 * (i + 2)) - targetKeyRelOffset - 12;
            int from2 = nonRootIndex.arrayOffset() + targetKeyOffset;
            int to2 = from2 + targetKeyLength;
            return Arrays.copyOfRange(nonRootIndex.array(), from2, to2);
        }

        static int binarySearchNonRootIndex(byte[] key, int keyOffset, int keyLength, ByteBuffer nonRootIndex, KeyValue.KVComparator comparator) {
            int numEntries = nonRootIndex.getInt(0);
            int low = 0;
            int high = numEntries - 1;
            int mid = 0;
            int entriesOffset = 4 * (numEntries + 2);
            while (low <= high) {
                mid = low + high >>> 1;
                int midKeyRelOffset = nonRootIndex.getInt(4 * (mid + 1));
                int midKeyOffset = entriesOffset + midKeyRelOffset + 12;
                int midLength = nonRootIndex.getInt(4 * (mid + 2)) - midKeyRelOffset - 12;
                int cmp = comparator.compareFlatKey(key, keyOffset, keyLength, nonRootIndex.array(), nonRootIndex.arrayOffset() + midKeyOffset, midLength);
                if (cmp > 0) {
                    low = mid + 1;
                    continue;
                }
                if (cmp < 0) {
                    high = mid - 1;
                    continue;
                }
                return mid;
            }
            if (low != high + 1) {
                throw new IllegalStateException("Binary search broken: low=" + low + " " + "instead of " + (high + 1));
            }
            int i = low - 1;
            if (i < -1 || i >= numEntries) {
                throw new IllegalStateException("Binary search broken: result is " + i + " but expected to be between -1 and (numEntries - 1) = " + (numEntries - 1));
            }
            return i;
        }

        static int locateNonRootIndexEntry(ByteBuffer nonRootBlock, byte[] key, int keyOffset, int keyLength, KeyValue.KVComparator comparator) {
            int entryIndex = BlockIndexReader.binarySearchNonRootIndex(key, keyOffset, keyLength, nonRootBlock, comparator);
            if (entryIndex != -1) {
                int numEntries = nonRootBlock.getInt(0);
                int entriesOffset = 4 * (numEntries + 2);
                int entryRelOffset = nonRootBlock.getInt(4 * (1 + entryIndex));
                nonRootBlock.position(entriesOffset + entryRelOffset);
            }
            return entryIndex;
        }

        public void readRootIndex(DataInput in, int numEntries) throws IOException {
            this.blockOffsets = new long[numEntries];
            this.blockKeys = new byte[numEntries][];
            this.blockDataSizes = new int[numEntries];
            if (numEntries > 0) {
                for (int i = 0; i < numEntries; ++i) {
                    long offset = in.readLong();
                    int dataSize = in.readInt();
                    byte[] key = Bytes.readByteArray(in);
                    this.add(key, offset, dataSize);
                }
            }
        }

        public DataInputStream readRootIndex(HFileBlock blk, int numEntries) throws IOException {
            DataInputStream in = blk.getByteStream();
            this.readRootIndex(in, numEntries);
            return in;
        }

        public void readMultiLevelIndexRoot(HFileBlock blk, int numEntries) throws IOException {
            DataInputStream in = this.readRootIndex(blk, numEntries);
            int checkSumBytes = blk.totalChecksumBytes();
            if (in.available() - checkSumBytes < 16) {
                return;
            }
            this.midLeafBlockOffset = in.readLong();
            this.midLeafBlockOnDiskSize = in.readInt();
            this.midKeyEntry = in.readInt();
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("size=" + this.rootCount).append("\n");
            for (int i = 0; i < this.rootCount; ++i) {
                sb.append("key=").append(KeyValue.keyToString(this.blockKeys[i])).append("\n  offset=").append(this.blockOffsets[i]).append(", dataSize=" + this.blockDataSizes[i]).append("\n");
            }
            return sb.toString();
        }

        @Override
        public long heapSize() {
            long heapSize = ClassSize.align(6 * ClassSize.REFERENCE + 8 + ClassSize.OBJECT);
            heapSize += 16L;
            if (this.blockKeys != null) {
                heapSize += (long)ClassSize.align(ClassSize.ARRAY + this.blockKeys.length * ClassSize.REFERENCE);
                for (byte[] key : this.blockKeys) {
                    heapSize += (long)ClassSize.align(ClassSize.ARRAY + key.length);
                }
            }
            if (this.blockOffsets != null) {
                heapSize += (long)ClassSize.align(ClassSize.ARRAY + this.blockOffsets.length * 8);
            }
            if (this.blockDataSizes != null) {
                heapSize += (long)ClassSize.align(ClassSize.ARRAY + this.blockDataSizes.length * 4);
            }
            return ClassSize.align(heapSize);
        }
    }
}

