/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sentry.provider.db.service.persistent;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import org.apache.curator.RetryPolicy;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.api.ACLProvider;
import org.apache.curator.framework.api.BackgroundPathable;
import org.apache.curator.framework.imps.CuratorFrameworkState;
import org.apache.curator.framework.imps.DefaultACLProvider;
import org.apache.curator.retry.RetryNTimes;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.sentry.service.thrift.JaasConfiguration;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Id;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HAContext {
    private static final Logger LOGGER = LoggerFactory.getLogger(HAContext.class);
    private static HAContext serverHAContext = null;
    private static boolean aclChecked = false;
    public static final String SENTRY_SERVICE_REGISTER_NAMESPACE = "sentry-service";
    public static final String SENTRY_ZK_JAAS_NAME = "SentryClient";
    private final String zookeeperQuorum;
    private final int retriesMaxCount;
    private final int sleepMsBetweenRetries;
    private final String namespace;
    private final boolean zkSecure;
    private List<ACL> saslACL;
    private final CuratorFramework curatorFramework;
    private final RetryPolicy retryPolicy;

    protected HAContext(Configuration conf) throws Exception {
        Object aclProvider;
        this.zookeeperQuorum = conf.get("sentry.ha.zookeeper.quorum", "localhost:2181");
        this.retriesMaxCount = conf.getInt("sentry.ha.zookeeper.session.retries.max.count", 3);
        this.sleepMsBetweenRetries = conf.getInt("sentry.ha.zookeeper.session.sleep.between.retries.ms", 100);
        this.namespace = conf.get("sentry.ha.zookeeper.namespace", "sentry");
        this.zkSecure = conf.getBoolean("sentry.ha.zookeeper.security", false);
        this.validateConf();
        if (this.zkSecure) {
            LOGGER.info("Connecting to ZooKeeper with SASL/Kerberos and using 'sasl' ACLs");
            this.setJaasConfiguration(conf);
            System.setProperty("zookeeper.sasl.clientconfig", SENTRY_ZK_JAAS_NAME);
            this.saslACL = Lists.newArrayList();
            this.saslACL.add(new ACL(31, new Id("sasl", this.getServicePrincipal(conf, "sentry.service.server.principal"))));
            this.saslACL.add(new ACL(31, new Id("sasl", this.getServicePrincipal(conf, "sentry.zookeeper.client.principal"))));
            aclProvider = new SASLOwnerACLProvider();
            String allowConnect = conf.get("sentry.service.allow.connect");
            if (!Strings.isNullOrEmpty((String)allowConnect)) {
                for (String principal : Arrays.asList(allowConnect.split("\\s*,\\s*"))) {
                    LOGGER.info("Adding acls for " + principal);
                    this.saslACL.add(new ACL(31, new Id("sasl", principal)));
                }
            }
        } else {
            LOGGER.info("Connecting to ZooKeeper without authentication");
            aclProvider = new DefaultACLProvider();
        }
        this.retryPolicy = new RetryNTimes(this.retriesMaxCount, this.sleepMsBetweenRetries);
        this.curatorFramework = CuratorFrameworkFactory.builder().namespace(this.namespace).connectString(this.zookeeperQuorum).retryPolicy(this.retryPolicy).aclProvider((ACLProvider)aclProvider).build();
        this.startCuratorFramework();
    }

    public static HAContext getHAContext(Configuration conf) throws Exception {
        if (serverHAContext == null) {
            serverHAContext = new HAContext(conf);
            Runtime.getRuntime().addShutdownHook(new Thread(){

                @Override
                public void run() {
                    LOGGER.info("ShutdownHook closing curator framework");
                    try {
                        HAContext.clearServerContext();
                    }
                    catch (Throwable t) {
                        LOGGER.error("Error stopping SentryService", t);
                    }
                }
            });
        }
        return serverHAContext;
    }

    public static HAContext getHAServerContext(Configuration conf) throws Exception {
        HAContext serverContext = HAContext.getHAContext(conf);
        serverContext.checkAndSetACLs();
        return serverContext;
    }

    @VisibleForTesting
    public static synchronized void clearServerContext() {
        if (serverHAContext != null) {
            serverHAContext.getCuratorFramework().close();
            serverHAContext = null;
        }
    }

    public void startCuratorFramework() {
        if (this.curatorFramework.getState() != CuratorFrameworkState.STARTED) {
            this.curatorFramework.start();
        }
    }

    public CuratorFramework getCuratorFramework() {
        return this.curatorFramework;
    }

    public String getZookeeperQuorum() {
        return this.zookeeperQuorum;
    }

    public static boolean isHaEnabled(Configuration conf) {
        return conf.getBoolean("sentry.ha.enabled", false);
    }

    public String getNamespace() {
        return this.namespace;
    }

    public RetryPolicy getRetryPolicy() {
        return this.retryPolicy;
    }

    private void validateConf() {
        Preconditions.checkNotNull((Object)this.zookeeperQuorum, (Object)"Zookeeper Quorum should not be null.");
        Preconditions.checkNotNull((Object)this.namespace, (Object)"Zookeeper namespace should not be null.");
    }

    protected String getServicePrincipal(Configuration conf, String confProperty) throws IOException {
        String principal = conf.get(confProperty);
        Preconditions.checkNotNull((Object)principal);
        Preconditions.checkArgument((principal.length() != 0 ? 1 : 0) != 0, (Object)"Server principal is not right.");
        return principal.split("[/@]")[0];
    }

    private void checkAndSetACLs() throws Exception {
        if (this.zkSecure && !aclChecked) {
            List acls;
            this.startCuratorFramework();
            String namespace = "/" + this.curatorFramework.getNamespace();
            if (this.curatorFramework.getZookeeperClient().getZooKeeper().exists(namespace, null) != null && ((acls = this.curatorFramework.getZookeeperClient().getZooKeeper().getACL(namespace, new Stat())).isEmpty() || !((ACL)acls.get(0)).getId().getScheme().equals("sasl"))) {
                LOGGER.info("'sasl' ACLs not set; setting...");
                List children = this.curatorFramework.getZookeeperClient().getZooKeeper().getChildren(namespace, null);
                for (String child : children) {
                    this.checkAndSetACLs("/" + child);
                }
                this.curatorFramework.getZookeeperClient().getZooKeeper().setACL(namespace, this.saslACL, -1);
            }
            aclChecked = true;
        }
    }

    private void checkAndSetACLs(String path) throws Exception {
        LOGGER.info("Setting acls on " + path);
        List children = (List)this.curatorFramework.getChildren().forPath(path);
        for (String child : children) {
            this.checkAndSetACLs(path + "/" + child);
        }
        ((BackgroundPathable)this.curatorFramework.setACL().withACL(this.saslACL)).forPath(path);
    }

    private void setJaasConfiguration(Configuration conf) throws IOException {
        if ("false".equalsIgnoreCase(conf.get("sentry.zookeeper.client.ticketcache", "false"))) {
            String keytabFile = conf.get("sentry.zookeeper.client.keytab");
            Preconditions.checkArgument((keytabFile.length() != 0 ? 1 : 0) != 0, (Object)"Keytab File is not right.");
            String principal = conf.get("sentry.zookeeper.client.principal");
            principal = SecurityUtil.getServerPrincipal((String)principal, (String)conf.get("sentry.service.server.rpc-address", "0.0.0.0"));
            Preconditions.checkArgument((principal.length() != 0 ? 1 : 0) != 0, (Object)"Kerberos principal is not right.");
            JaasConfiguration.addEntryForKeytab(SENTRY_ZK_JAAS_NAME, principal, keytabFile);
        } else {
            JaasConfiguration.addEntryForTicketCache(SENTRY_ZK_JAAS_NAME);
        }
        javax.security.auth.login.Configuration.setConfiguration(JaasConfiguration.getInstance());
    }

    public class SASLOwnerACLProvider
    implements ACLProvider {
        public List<ACL> getDefaultAcl() {
            return HAContext.this.saslACL;
        }

        public List<ACL> getAclForPath(String path) {
            return HAContext.this.saslACL;
        }
    }
}

