/*
 * Decompiled with CFR 0.152.
 */
package org.apache.curator.x.discovery.details;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.Closeable;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.api.ACLBackgroundPathAndBytesable;
import org.apache.curator.framework.api.BackgroundPathable;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.curator.framework.state.ConnectionStateListener;
import org.apache.curator.utils.CloseableUtils;
import org.apache.curator.utils.ThreadUtils;
import org.apache.curator.utils.ZKPaths;
import org.apache.curator.x.discovery.ServiceCache;
import org.apache.curator.x.discovery.ServiceCacheBuilder;
import org.apache.curator.x.discovery.ServiceDiscovery;
import org.apache.curator.x.discovery.ServiceInstance;
import org.apache.curator.x.discovery.ServiceProvider;
import org.apache.curator.x.discovery.ServiceProviderBuilder;
import org.apache.curator.x.discovery.ServiceType;
import org.apache.curator.x.discovery.details.InstanceSerializer;
import org.apache.curator.x.discovery.details.ServiceCacheBuilderImpl;
import org.apache.curator.x.discovery.details.ServiceProviderBuilderImpl;
import org.apache.curator.x.discovery.strategies.RoundRobinStrategy;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServiceDiscoveryImpl<T>
implements ServiceDiscovery<T> {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final CuratorFramework client;
    private final String basePath;
    private final InstanceSerializer<T> serializer;
    private final Map<String, ServiceInstance<T>> services = Maps.newConcurrentMap();
    private final Collection<ServiceCache<T>> caches = Sets.newSetFromMap((Map)Maps.newConcurrentMap());
    private final Collection<ServiceProvider<T>> providers = Sets.newSetFromMap((Map)Maps.newConcurrentMap());
    private final ConnectionStateListener connectionStateListener = new ConnectionStateListener(){

        public void stateChanged(CuratorFramework client, ConnectionState newState) {
            if (newState == ConnectionState.RECONNECTED) {
                try {
                    ServiceDiscoveryImpl.this.log.debug("Re-registering due to reconnection");
                    ServiceDiscoveryImpl.this.reRegisterServices();
                }
                catch (Exception e) {
                    ServiceDiscoveryImpl.this.log.error("Could not re-register instances after reconnection", (Throwable)e);
                }
            }
        }
    };

    public ServiceDiscoveryImpl(CuratorFramework client, String basePath, InstanceSerializer<T> serializer, ServiceInstance<T> thisInstance) {
        this.client = (CuratorFramework)Preconditions.checkNotNull((Object)client, (Object)"client cannot be null");
        this.basePath = (String)Preconditions.checkNotNull((Object)basePath, (Object)"basePath cannot be null");
        this.serializer = (InstanceSerializer)Preconditions.checkNotNull(serializer, (Object)"serializer cannot be null");
        if (thisInstance != null) {
            this.services.put(thisInstance.getId(), thisInstance);
        }
    }

    @Override
    public void start() throws Exception {
        this.client.getConnectionStateListenable().addListener((Object)this.connectionStateListener);
        this.reRegisterServices();
    }

    @Override
    public void close() throws IOException {
        for (ServiceCache cache : Lists.newArrayList(this.caches)) {
            CloseableUtils.closeQuietly((Closeable)cache);
        }
        for (ServiceProvider provider : Lists.newArrayList(this.providers)) {
            CloseableUtils.closeQuietly((Closeable)provider);
        }
        Iterator<ServiceInstance<T>> it = this.services.values().iterator();
        while (it.hasNext()) {
            ServiceInstance<T> service = it.next();
            String path = this.pathForInstance(service.getName(), service.getId());
            boolean doRemove = true;
            try {
                this.client.delete().forPath(path);
            }
            catch (KeeperException.NoNodeException ignore) {
            }
            catch (Exception e) {
                doRemove = false;
                this.log.error("Could not unregister instance: " + service.getName(), (Throwable)e);
            }
            if (!doRemove) continue;
            it.remove();
        }
        this.client.getConnectionStateListenable().removeListener((Object)this.connectionStateListener);
    }

    @Override
    public void registerService(ServiceInstance<T> service) throws Exception {
        this.services.put(service.getId(), service);
        this.internalRegisterService(service);
    }

    @Override
    public void updateService(ServiceInstance<T> service) throws Exception {
        Preconditions.checkArgument((boolean)this.services.containsKey(service.getId()), (Object)("Service is not registered: " + service));
        byte[] bytes = this.serializer.serialize(service);
        String path = this.pathForInstance(service.getName(), service.getId());
        this.client.setData().forPath(path, bytes);
    }

    @VisibleForTesting
    protected void internalRegisterService(ServiceInstance<T> service) throws Exception {
        byte[] bytes = this.serializer.serialize(service);
        String path = this.pathForInstance(service.getName(), service.getId());
        int MAX_TRIES = 2;
        boolean isDone = false;
        for (int i = 0; !isDone && i < 2; ++i) {
            try {
                CreateMode mode = service.getServiceType() == ServiceType.DYNAMIC ? CreateMode.EPHEMERAL : CreateMode.PERSISTENT;
                ((ACLBackgroundPathAndBytesable)this.client.create().creatingParentsIfNeeded().withMode(mode)).forPath(path, bytes);
                isDone = true;
                continue;
            }
            catch (KeeperException.NodeExistsException e) {
                this.client.delete().forPath(path);
            }
        }
    }

    @Override
    public void unregisterService(ServiceInstance<T> service) throws Exception {
        String path = this.pathForInstance(service.getName(), service.getId());
        try {
            this.client.delete().forPath(path);
        }
        catch (KeeperException.NoNodeException noNodeException) {
            // empty catch block
        }
        this.services.remove(service.getId());
    }

    @Override
    public ServiceProviderBuilder<T> serviceProviderBuilder() {
        return new ServiceProviderBuilderImpl(this).providerStrategy(new RoundRobinStrategy()).threadFactory(ThreadUtils.newThreadFactory((String)"ServiceProvider"));
    }

    @Override
    public ServiceCacheBuilder<T> serviceCacheBuilder() {
        return new ServiceCacheBuilderImpl(this).threadFactory(ThreadUtils.newThreadFactory((String)"ServiceCache"));
    }

    @Override
    public Collection<String> queryForNames() throws Exception {
        List names = (List)this.client.getChildren().forPath(this.basePath);
        return ImmutableList.copyOf((Collection)names);
    }

    @Override
    public Collection<ServiceInstance<T>> queryForInstances(String name) throws Exception {
        return this.queryForInstances(name, null);
    }

    @Override
    public ServiceInstance<T> queryForInstance(String name, String id) throws Exception {
        String path = this.pathForInstance(name, id);
        try {
            byte[] bytes = (byte[])this.client.getData().forPath(path);
            return this.serializer.deserialize(bytes);
        }
        catch (KeeperException.NoNodeException ignore) {
            return null;
        }
    }

    void cacheOpened(ServiceCache<T> cache) {
        this.caches.add(cache);
    }

    void cacheClosed(ServiceCache<T> cache) {
        this.caches.remove(cache);
    }

    void providerOpened(ServiceProvider<T> provider) {
        this.providers.add(provider);
    }

    void providerClosed(ServiceProvider<T> cache) {
        this.providers.remove(cache);
    }

    CuratorFramework getClient() {
        return this.client;
    }

    String pathForName(String name) {
        return ZKPaths.makePath((String)this.basePath, (String)name);
    }

    InstanceSerializer<T> getSerializer() {
        return this.serializer;
    }

    List<ServiceInstance<T>> queryForInstances(String name, Watcher watcher) throws Exception {
        List<Object> instanceIds;
        ImmutableList.Builder builder = ImmutableList.builder();
        String path = this.pathForName(name);
        if (watcher != null) {
            instanceIds = this.getChildrenWatched(path, watcher, true);
        } else {
            try {
                instanceIds = (List)this.client.getChildren().forPath(path);
            }
            catch (KeeperException.NoNodeException e) {
                instanceIds = Lists.newArrayList();
            }
        }
        for (String string : instanceIds) {
            ServiceInstance<T> instance = this.queryForInstance(name, string);
            if (instance == null) continue;
            builder.add(instance);
        }
        return builder.build();
    }

    private List<String> getChildrenWatched(String path, Watcher watcher, boolean recurse) throws Exception {
        List<String> instanceIds;
        try {
            instanceIds = (List<String>)((BackgroundPathable)this.client.getChildren().usingWatcher(watcher)).forPath(path);
        }
        catch (KeeperException.NoNodeException e) {
            if (recurse) {
                try {
                    this.client.create().creatingParentsIfNeeded().forPath(path);
                }
                catch (KeeperException.NodeExistsException ignore) {
                    // empty catch block
                }
                instanceIds = this.getChildrenWatched(path, watcher, false);
            }
            throw e;
        }
        return instanceIds;
    }

    private String pathForInstance(String name, String id) throws UnsupportedEncodingException {
        return ZKPaths.makePath((String)this.pathForName(name), (String)id);
    }

    private void reRegisterServices() throws Exception {
        for (ServiceInstance<T> service : this.services.values()) {
            this.internalRegisterService(service);
        }
    }
}

