/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.producer.internals;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.apache.kafka.clients.ClientRequest;
import org.apache.kafka.clients.ClientResponse;
import org.apache.kafka.clients.KafkaClient;
import org.apache.kafka.clients.Metadata;
import org.apache.kafka.clients.RequestCompletionHandler;
import org.apache.kafka.clients.producer.internals.RecordAccumulator;
import org.apache.kafka.clients.producer.internals.RecordBatch;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.InvalidMetadataException;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.metrics.Measurable;
import org.apache.kafka.common.metrics.MetricConfig;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Avg;
import org.apache.kafka.common.metrics.stats.Max;
import org.apache.kafka.common.metrics.stats.Rate;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.ProduceRequest;
import org.apache.kafka.common.requests.ProduceResponse;
import org.apache.kafka.common.requests.RequestSend;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Sender
implements Runnable {
    private static final Logger log = LoggerFactory.getLogger(Sender.class);
    private final KafkaClient client;
    private final RecordAccumulator accumulator;
    private final Metadata metadata;
    private final int maxRequestSize;
    private final short acks;
    private final int requestTimeout;
    private final int retries;
    private final Time time;
    private volatile boolean running;
    private volatile boolean forceClose;
    private final SenderMetrics sensors;
    private String clientId;

    public Sender(KafkaClient client, Metadata metadata, RecordAccumulator accumulator, int maxRequestSize, short acks, int retries, int requestTimeout, Metrics metrics, Time time, String clientId) {
        this.client = client;
        this.accumulator = accumulator;
        this.metadata = metadata;
        this.maxRequestSize = maxRequestSize;
        this.running = true;
        this.requestTimeout = requestTimeout;
        this.acks = acks;
        this.retries = retries;
        this.time = time;
        this.clientId = clientId;
        this.sensors = new SenderMetrics(metrics);
    }

    @Override
    public void run() {
        log.debug("Starting Kafka producer I/O thread.");
        while (this.running) {
            try {
                this.run(this.time.milliseconds());
            }
            catch (Exception e) {
                log.error("Uncaught error in kafka producer I/O thread: ", (Throwable)e);
            }
        }
        log.debug("Beginning shutdown of Kafka producer I/O thread, sending remaining records.");
        while (!this.forceClose && (this.accumulator.hasUnsent() || this.client.inFlightRequestCount() > 0)) {
            try {
                this.run(this.time.milliseconds());
            }
            catch (Exception e) {
                log.error("Uncaught error in kafka producer I/O thread: ", (Throwable)e);
            }
        }
        if (this.forceClose) {
            this.accumulator.abortIncompleteBatches();
        }
        try {
            this.client.close();
        }
        catch (Exception e) {
            log.error("Failed to close network client", (Throwable)e);
        }
        log.debug("Shutdown of Kafka producer I/O thread has completed.");
    }

    public void run(long now) {
        Cluster cluster = this.metadata.fetch();
        RecordAccumulator.ReadyCheckResult result = this.accumulator.ready(cluster, now);
        if (result.unknownLeadersExist) {
            this.metadata.requestUpdate();
        }
        Iterator<Node> iter = result.readyNodes.iterator();
        long notReadyTimeout = Long.MAX_VALUE;
        while (iter.hasNext()) {
            Node node = iter.next();
            if (this.client.ready(node, now)) continue;
            iter.remove();
            notReadyTimeout = Math.min(notReadyTimeout, this.client.connectionDelay(node, now));
        }
        Map<Integer, List<RecordBatch>> batches = this.accumulator.drain(cluster, result.readyNodes, this.maxRequestSize, now);
        this.sensors.updateProduceRequestMetrics(batches);
        List<ClientRequest> requests = this.createProduceRequests(batches, now);
        long pollTimeout = Math.min(result.nextReadyCheckDelayMs, notReadyTimeout);
        if (result.readyNodes.size() > 0) {
            log.trace("Nodes with data ready to send: {}", result.readyNodes);
            log.trace("Created {} produce requests: {}", (Object)requests.size(), requests);
            pollTimeout = 0L;
        }
        for (ClientRequest request : requests) {
            this.client.send(request);
        }
        this.client.poll(pollTimeout, now);
    }

    public void initiateClose() {
        this.running = false;
        this.accumulator.close();
        this.wakeup();
    }

    public void forceClose() {
        this.forceClose = true;
        this.initiateClose();
    }

    private void handleProduceResponse(ClientResponse response, Map<TopicPartition, RecordBatch> batches, long now) {
        int correlationId = response.request().request().header().correlationId();
        if (response.wasDisconnected()) {
            log.trace("Cancelled request {} due to node {} being disconnected", (Object)response, (Object)response.request().request().destination());
            for (RecordBatch batch : batches.values()) {
                this.completeBatch(batch, Errors.NETWORK_EXCEPTION, -1L, correlationId, now);
            }
        } else {
            log.trace("Received produce response from node {} with correlation id {}", (Object)response.request().request().destination(), (Object)correlationId);
            if (response.hasResponse()) {
                ProduceResponse produceResponse = new ProduceResponse(response.responseBody());
                for (Map.Entry<TopicPartition, ProduceResponse.PartitionResponse> entry : produceResponse.responses().entrySet()) {
                    TopicPartition tp = entry.getKey();
                    ProduceResponse.PartitionResponse partResp = entry.getValue();
                    Errors error = Errors.forCode(partResp.errorCode);
                    RecordBatch batch = batches.get(tp);
                    this.completeBatch(batch, error, partResp.baseOffset, correlationId, now);
                }
                this.sensors.recordLatency(response.request().request().destination(), response.requestLatencyMs());
            } else {
                for (RecordBatch batch : batches.values()) {
                    this.completeBatch(batch, Errors.NONE, -1L, correlationId, now);
                }
            }
        }
    }

    private void completeBatch(RecordBatch batch, Errors error, long baseOffset, long correlationId, long now) {
        if (error != Errors.NONE && this.canRetry(batch, error)) {
            log.warn("Got error produce response with correlation id {} on topic-partition {}, retrying ({} attempts left). Error: {}", new Object[]{correlationId, batch.topicPartition, this.retries - batch.attempts - 1, error});
            this.accumulator.reenqueue(batch, now);
            this.sensors.recordRetries(batch.topicPartition.topic(), batch.recordCount);
        } else {
            batch.done(baseOffset, error.exception());
            this.accumulator.deallocate(batch);
            if (error != Errors.NONE) {
                this.sensors.recordErrors(batch.topicPartition.topic(), batch.recordCount);
            }
        }
        if (error.exception() instanceof InvalidMetadataException) {
            this.metadata.requestUpdate();
        }
    }

    private boolean canRetry(RecordBatch batch, Errors error) {
        return batch.attempts < this.retries && error.exception() instanceof RetriableException;
    }

    private List<ClientRequest> createProduceRequests(Map<Integer, List<RecordBatch>> collated, long now) {
        ArrayList<ClientRequest> requests = new ArrayList<ClientRequest>(collated.size());
        for (Map.Entry<Integer, List<RecordBatch>> entry : collated.entrySet()) {
            requests.add(this.produceRequest(now, entry.getKey(), this.acks, this.requestTimeout, entry.getValue()));
        }
        return requests;
    }

    private ClientRequest produceRequest(long now, int destination, short acks, int timeout, List<RecordBatch> batches) {
        HashMap<TopicPartition, ByteBuffer> produceRecordsByPartition = new HashMap<TopicPartition, ByteBuffer>(batches.size());
        final HashMap<TopicPartition, RecordBatch> recordsByPartition = new HashMap<TopicPartition, RecordBatch>(batches.size());
        for (RecordBatch batch : batches) {
            TopicPartition tp = batch.topicPartition;
            produceRecordsByPartition.put(tp, (ByteBuffer)batch.records.buffer().flip());
            recordsByPartition.put(tp, batch);
        }
        ProduceRequest request = new ProduceRequest(acks, timeout, produceRecordsByPartition);
        RequestSend send = new RequestSend(Integer.toString(destination), this.client.nextRequestHeader(ApiKeys.PRODUCE), request.toStruct());
        RequestCompletionHandler callback = new RequestCompletionHandler(){

            @Override
            public void onComplete(ClientResponse response) {
                Sender.this.handleProduceResponse(response, recordsByPartition, Sender.this.time.milliseconds());
            }
        };
        return new ClientRequest(now, acks != 0, send, callback);
    }

    public void wakeup() {
        this.client.wakeup();
    }

    private class SenderMetrics {
        private final Metrics metrics;
        public final Sensor retrySensor;
        public final Sensor errorSensor;
        public final Sensor queueTimeSensor;
        public final Sensor requestTimeSensor;
        public final Sensor recordsPerRequestSensor;
        public final Sensor batchSizeSensor;
        public final Sensor compressionRateSensor;
        public final Sensor maxRecordSizeSensor;

        public SenderMetrics(Metrics metrics) {
            this.metrics = metrics;
            LinkedHashMap<String, String> metricTags = new LinkedHashMap<String, String>();
            metricTags.put("client-id", Sender.this.clientId);
            String metricGrpName = "producer-metrics";
            this.batchSizeSensor = metrics.sensor("batch-size");
            MetricName m = new MetricName("batch-size-avg", metricGrpName, "The average number of bytes sent per partition per-request.", metricTags);
            this.batchSizeSensor.add(m, new Avg());
            m = new MetricName("batch-size-max", metricGrpName, "The max number of bytes sent per partition per-request.", metricTags);
            this.batchSizeSensor.add(m, new Max());
            this.compressionRateSensor = metrics.sensor("compression-rate");
            m = new MetricName("compression-rate-avg", metricGrpName, "The average compression rate of record batches.", metricTags);
            this.compressionRateSensor.add(m, new Avg());
            this.queueTimeSensor = metrics.sensor("queue-time");
            m = new MetricName("record-queue-time-avg", metricGrpName, "The average time in ms record batches spent in the record accumulator.", metricTags);
            this.queueTimeSensor.add(m, new Avg());
            m = new MetricName("record-queue-time-max", metricGrpName, "The maximum time in ms record batches spent in the record accumulator.", metricTags);
            this.queueTimeSensor.add(m, new Max());
            this.requestTimeSensor = metrics.sensor("request-time");
            m = new MetricName("request-latency-avg", metricGrpName, "The average request latency in ms", metricTags);
            this.requestTimeSensor.add(m, new Avg());
            m = new MetricName("request-latency-max", metricGrpName, "The maximum request latency in ms", metricTags);
            this.requestTimeSensor.add(m, new Max());
            this.recordsPerRequestSensor = metrics.sensor("records-per-request");
            m = new MetricName("record-send-rate", metricGrpName, "The average number of records sent per second.", metricTags);
            this.recordsPerRequestSensor.add(m, new Rate());
            m = new MetricName("records-per-request-avg", metricGrpName, "The average number of records per request.", metricTags);
            this.recordsPerRequestSensor.add(m, new Avg());
            this.retrySensor = metrics.sensor("record-retries");
            m = new MetricName("record-retry-rate", metricGrpName, "The average per-second number of retried record sends", metricTags);
            this.retrySensor.add(m, new Rate());
            this.errorSensor = metrics.sensor("errors");
            m = new MetricName("record-error-rate", metricGrpName, "The average per-second number of record sends that resulted in errors", metricTags);
            this.errorSensor.add(m, new Rate());
            this.maxRecordSizeSensor = metrics.sensor("record-size-max");
            m = new MetricName("record-size-max", metricGrpName, "The maximum record size", metricTags);
            this.maxRecordSizeSensor.add(m, new Max());
            m = new MetricName("record-size-avg", metricGrpName, "The average record size", metricTags);
            this.maxRecordSizeSensor.add(m, new Avg());
            m = new MetricName("requests-in-flight", metricGrpName, "The current number of in-flight requests awaiting a response.", metricTags);
            this.metrics.addMetric(m, new Measurable(){

                @Override
                public double measure(MetricConfig config, long now) {
                    return Sender.this.client.inFlightRequestCount();
                }
            });
            m = new MetricName("metadata-age", metricGrpName, "The age in seconds of the current producer metadata being used.", metricTags);
            metrics.addMetric(m, new Measurable(){

                @Override
                public double measure(MetricConfig config, long now) {
                    return (double)(now - Sender.this.metadata.lastSuccessfulUpdate()) / 1000.0;
                }
            });
        }

        public void maybeRegisterTopicMetrics(String topic) {
            String topicRecordsCountName = "topic." + topic + ".records-per-batch";
            Sensor topicRecordCount = this.metrics.getSensor(topicRecordsCountName);
            if (topicRecordCount == null) {
                LinkedHashMap<String, String> metricTags = new LinkedHashMap<String, String>();
                metricTags.put("client-id", Sender.this.clientId);
                metricTags.put("topic", topic);
                String metricGrpName = "producer-topic-metrics";
                topicRecordCount = this.metrics.sensor(topicRecordsCountName);
                MetricName m = new MetricName("record-send-rate", metricGrpName, metricTags);
                topicRecordCount.add(m, new Rate());
                String topicByteRateName = "topic." + topic + ".bytes";
                Sensor topicByteRate = this.metrics.sensor(topicByteRateName);
                m = new MetricName("byte-rate", metricGrpName, metricTags);
                topicByteRate.add(m, new Rate());
                String topicCompressionRateName = "topic." + topic + ".compression-rate";
                Sensor topicCompressionRate = this.metrics.sensor(topicCompressionRateName);
                m = new MetricName("compression-rate", metricGrpName, metricTags);
                topicCompressionRate.add(m, new Avg());
                String topicRetryName = "topic." + topic + ".record-retries";
                Sensor topicRetrySensor = this.metrics.sensor(topicRetryName);
                m = new MetricName("record-retry-rate", metricGrpName, metricTags);
                topicRetrySensor.add(m, new Rate());
                String topicErrorName = "topic." + topic + ".record-errors";
                Sensor topicErrorSensor = this.metrics.sensor(topicErrorName);
                m = new MetricName("record-error-rate", metricGrpName, metricTags);
                topicErrorSensor.add(m, new Rate());
            }
        }

        public void updateProduceRequestMetrics(Map<Integer, List<RecordBatch>> batches) {
            long now = Sender.this.time.milliseconds();
            for (List<RecordBatch> nodeBatch : batches.values()) {
                int records = 0;
                for (RecordBatch batch : nodeBatch) {
                    String topic = batch.topicPartition.topic();
                    this.maybeRegisterTopicMetrics(topic);
                    String topicRecordsCountName = "topic." + topic + ".records-per-batch";
                    Sensor topicRecordCount = Utils.notNull(this.metrics.getSensor(topicRecordsCountName));
                    topicRecordCount.record(batch.recordCount);
                    String topicByteRateName = "topic." + topic + ".bytes";
                    Sensor topicByteRate = Utils.notNull(this.metrics.getSensor(topicByteRateName));
                    topicByteRate.record(batch.records.sizeInBytes());
                    String topicCompressionRateName = "topic." + topic + ".compression-rate";
                    Sensor topicCompressionRate = Utils.notNull(this.metrics.getSensor(topicCompressionRateName));
                    topicCompressionRate.record(batch.records.compressionRate());
                    this.batchSizeSensor.record(batch.records.sizeInBytes(), now);
                    this.queueTimeSensor.record(batch.drainedMs - batch.createdMs, now);
                    this.compressionRateSensor.record(batch.records.compressionRate());
                    this.maxRecordSizeSensor.record(batch.maxRecordSize, now);
                    records += batch.recordCount;
                }
                this.recordsPerRequestSensor.record(records, now);
            }
        }

        public void recordRetries(String topic, int count) {
            long now = Sender.this.time.milliseconds();
            this.retrySensor.record(count, now);
            String topicRetryName = "topic." + topic + ".record-retries";
            Sensor topicRetrySensor = this.metrics.getSensor(topicRetryName);
            if (topicRetrySensor != null) {
                topicRetrySensor.record(count, now);
            }
        }

        public void recordErrors(String topic, int count) {
            long now = Sender.this.time.milliseconds();
            this.errorSensor.record(count, now);
            String topicErrorName = "topic." + topic + ".record-errors";
            Sensor topicErrorSensor = this.metrics.getSensor(topicErrorName);
            if (topicErrorSensor != null) {
                topicErrorSensor.record(count, now);
            }
        }

        public void recordLatency(String node, long latency) {
            String nodeTimeName;
            Sensor nodeRequestTime;
            long now = Sender.this.time.milliseconds();
            this.requestTimeSensor.record(latency, now);
            if (!node.isEmpty() && (nodeRequestTime = this.metrics.getSensor(nodeTimeName = "node-" + node + ".latency")) != null) {
                nodeRequestTime.record(latency, now);
            }
        }
    }
}

