/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.Set;
import org.apache.kafka.clients.ClientRequest;
import org.apache.kafka.clients.ClientResponse;
import org.apache.kafka.clients.ClusterConnectionStates;
import org.apache.kafka.clients.ConnectionState;
import org.apache.kafka.clients.InFlightRequests;
import org.apache.kafka.clients.KafkaClient;
import org.apache.kafka.clients.Metadata;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.network.NetworkReceive;
import org.apache.kafka.common.network.Selectable;
import org.apache.kafka.common.network.Send;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.ProtoUtils;
import org.apache.kafka.common.protocol.types.Struct;
import org.apache.kafka.common.requests.MetadataRequest;
import org.apache.kafka.common.requests.MetadataResponse;
import org.apache.kafka.common.requests.RequestHeader;
import org.apache.kafka.common.requests.RequestSend;
import org.apache.kafka.common.requests.ResponseHeader;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NetworkClient
implements KafkaClient {
    private static final Logger log = LoggerFactory.getLogger(NetworkClient.class);
    private final Selectable selector;
    private final Metadata metadata;
    private final ClusterConnectionStates connectionStates;
    private final InFlightRequests inFlightRequests;
    private final int socketSendBuffer;
    private final int socketReceiveBuffer;
    private final String clientId;
    private final int nodeIndexOffset;
    private int correlation;
    private boolean metadataFetchInProgress;
    private long lastNoNodeAvailableMs;

    public NetworkClient(Selectable selector, Metadata metadata, String clientId, int maxInFlightRequestsPerConnection, long reconnectBackoffMs, int socketSendBuffer, int socketReceiveBuffer) {
        this.selector = selector;
        this.metadata = metadata;
        this.clientId = clientId;
        this.inFlightRequests = new InFlightRequests(maxInFlightRequestsPerConnection);
        this.connectionStates = new ClusterConnectionStates(reconnectBackoffMs);
        this.socketSendBuffer = socketSendBuffer;
        this.socketReceiveBuffer = socketReceiveBuffer;
        this.correlation = 0;
        this.nodeIndexOffset = new Random().nextInt(Integer.MAX_VALUE);
        this.metadataFetchInProgress = false;
        this.lastNoNodeAvailableMs = 0L;
    }

    @Override
    public boolean ready(Node node, long now) {
        if (this.isReady(node, now)) {
            return true;
        }
        if (this.connectionStates.canConnect(node.idString(), now)) {
            this.initiateConnect(node, now);
        }
        return false;
    }

    @Override
    public long connectionDelay(Node node, long now) {
        return this.connectionStates.connectionDelay(node.idString(), now);
    }

    @Override
    public boolean connectionFailed(Node node) {
        return this.connectionStates.connectionState(node.idString()).equals((Object)ConnectionState.DISCONNECTED);
    }

    @Override
    public boolean isReady(Node node, long now) {
        String nodeId = node.idString();
        if (!this.metadataFetchInProgress && this.metadata.timeToNextUpdate(now) == 0L) {
            return false;
        }
        return this.isSendable(nodeId);
    }

    private boolean isSendable(String node) {
        return this.connectionStates.isConnected(node) && this.inFlightRequests.canSendMore(node);
    }

    public ConnectionState connectionState(String node) {
        return this.connectionStates.connectionState(node);
    }

    @Override
    public void send(ClientRequest request) {
        String nodeId = request.request().destination();
        if (!this.isSendable(nodeId)) {
            throw new IllegalStateException("Attempt to send a request to node " + nodeId + " which is not ready.");
        }
        this.inFlightRequests.add(request);
        this.selector.send(request.request());
    }

    @Override
    public List<ClientResponse> poll(long timeout, long now) {
        long timeToNextMetadataUpdate = this.metadata.timeToNextUpdate(now);
        long timeToNextReconnectAttempt = Math.max(this.lastNoNodeAvailableMs + this.metadata.refreshBackoff() - now, 0L);
        long waitForMetadataFetch = this.metadataFetchInProgress ? Integer.MAX_VALUE : 0L;
        long metadataTimeout = Math.max(Math.max(timeToNextMetadataUpdate, timeToNextReconnectAttempt), waitForMetadataFetch);
        if (metadataTimeout == 0L) {
            this.maybeUpdateMetadata(now);
        }
        try {
            this.selector.poll(Math.min(timeout, metadataTimeout));
        }
        catch (IOException e) {
            log.error("Unexpected error during I/O in producer network thread", (Throwable)e);
        }
        ArrayList<ClientResponse> responses = new ArrayList<ClientResponse>();
        this.handleCompletedSends(responses, now);
        this.handleCompletedReceives(responses, now);
        this.handleDisconnections(responses, now);
        this.handleConnections();
        for (ClientResponse response : responses) {
            if (!response.request().hasCallback()) continue;
            try {
                response.request().callback().onComplete(response);
            }
            catch (Exception e) {
                log.error("Uncaught error in request completion:", (Throwable)e);
            }
        }
        return responses;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<ClientResponse> completeAll(String node, long now) {
        try {
            this.selector.muteAll();
            this.selector.unmute(node);
            ArrayList<ClientResponse> responses = new ArrayList<ClientResponse>();
            while (this.inFlightRequestCount(node) > 0) {
                responses.addAll(this.poll(Integer.MAX_VALUE, now));
            }
            ArrayList<ClientResponse> arrayList = responses;
            return arrayList;
        }
        finally {
            this.selector.unmuteAll();
        }
    }

    @Override
    public List<ClientResponse> completeAll(long now) {
        ArrayList<ClientResponse> responses = new ArrayList<ClientResponse>();
        while (this.inFlightRequestCount() > 0) {
            responses.addAll(this.poll(Integer.MAX_VALUE, now));
        }
        return responses;
    }

    @Override
    public int inFlightRequestCount() {
        return this.inFlightRequests.inFlightRequestCount();
    }

    @Override
    public int inFlightRequestCount(String node) {
        return this.inFlightRequests.inFlightRequestCount(node);
    }

    @Override
    public RequestHeader nextRequestHeader(ApiKeys key) {
        return new RequestHeader(key.id, this.clientId, this.correlation++);
    }

    @Override
    public void wakeup() {
        this.selector.wakeup();
    }

    @Override
    public void close() {
        this.selector.close();
    }

    @Override
    public Node leastLoadedNode(long now) {
        List<Node> nodes = this.metadata.fetch().nodes();
        int inflight = Integer.MAX_VALUE;
        Node found = null;
        for (int i = 0; i < nodes.size(); ++i) {
            int idx = Utils.abs((this.nodeIndexOffset + i) % nodes.size());
            Node node = nodes.get(idx);
            int currInflight = this.inFlightRequests.inFlightRequestCount(node.idString());
            if (currInflight == 0 && this.connectionStates.isConnected(node.idString())) {
                return node;
            }
            if (this.connectionStates.isBlackedOut(node.idString(), now) || currInflight >= inflight) continue;
            inflight = currInflight;
            found = node;
        }
        return found;
    }

    private void handleCompletedSends(List<ClientResponse> responses, long now) {
        for (Send send : this.selector.completedSends()) {
            ClientRequest request = this.inFlightRequests.lastSent(send.destination());
            if (request.expectResponse()) continue;
            this.inFlightRequests.completeLastSent(send.destination());
            responses.add(new ClientResponse(request, now, false, null));
        }
    }

    private void handleCompletedReceives(List<ClientResponse> responses, long now) {
        for (NetworkReceive receive : this.selector.completedReceives()) {
            String source = receive.source();
            ClientRequest req = this.inFlightRequests.completeNext(source);
            ResponseHeader header = ResponseHeader.parse(receive.payload());
            short apiKey = req.request().header().apiKey();
            Struct body = (Struct)ProtoUtils.currentResponseSchema(apiKey).read(receive.payload());
            this.correlate(req.request().header(), header);
            if (apiKey == ApiKeys.METADATA.id) {
                this.handleMetadataResponse(req.request().header(), body, now);
                continue;
            }
            responses.add(new ClientResponse(req, now, false, body));
        }
    }

    private void handleMetadataResponse(RequestHeader header, Struct body, long now) {
        this.metadataFetchInProgress = false;
        MetadataResponse response = new MetadataResponse(body);
        Cluster cluster = response.cluster();
        if (response.errors().size() > 0) {
            log.warn("Error while fetching metadata with correlation id {} : {}", (Object)header.correlationId(), response.errors());
        }
        if (cluster.nodes().size() > 0) {
            this.metadata.update(cluster, now);
        } else {
            log.trace("Ignoring empty metadata response with correlation id {}.", (Object)header.correlationId());
            this.metadata.failedUpdate(now);
        }
    }

    private void handleDisconnections(List<ClientResponse> responses, long now) {
        for (String node : this.selector.disconnected()) {
            this.connectionStates.disconnected(node);
            log.debug("Node {} disconnected.", (Object)node);
            for (ClientRequest request : this.inFlightRequests.clearAll(node)) {
                log.trace("Cancelled request {} due to node {} being disconnected", (Object)request, (Object)node);
                ApiKeys requestKey = ApiKeys.forId(request.request().header().apiKey());
                if (requestKey == ApiKeys.METADATA) {
                    this.metadataFetchInProgress = false;
                    continue;
                }
                responses.add(new ClientResponse(request, now, true, null));
            }
        }
        if (this.selector.disconnected().size() > 0) {
            this.metadata.requestUpdate();
        }
    }

    private void handleConnections() {
        for (String node : this.selector.connected()) {
            log.debug("Completed connection to node {}", (Object)node);
            this.connectionStates.connected(node);
        }
    }

    private void correlate(RequestHeader requestHeader, ResponseHeader responseHeader) {
        if (requestHeader.correlationId() != responseHeader.correlationId()) {
            throw new IllegalStateException("Correlation id for response (" + responseHeader.correlationId() + ") does not match request (" + requestHeader.correlationId() + ")");
        }
    }

    private ClientRequest metadataRequest(long now, String node, Set<String> topics) {
        MetadataRequest metadata = new MetadataRequest(new ArrayList<String>(topics));
        RequestSend send = new RequestSend(node, this.nextRequestHeader(ApiKeys.METADATA), metadata.toStruct());
        return new ClientRequest(now, true, send, null);
    }

    private void maybeUpdateMetadata(long now) {
        Node node = this.leastLoadedNode(now);
        if (node == null) {
            log.debug("Give up sending metadata request since no node is available");
            this.lastNoNodeAvailableMs = now;
            return;
        }
        String nodeConnectionId = node.idString();
        if (this.connectionStates.isConnected(nodeConnectionId) && this.inFlightRequests.canSendMore(nodeConnectionId)) {
            Set<String> topics = this.metadata.topics();
            this.metadataFetchInProgress = true;
            ClientRequest metadataRequest = this.metadataRequest(now, nodeConnectionId, topics);
            log.debug("Sending metadata request {} to node {}", (Object)metadataRequest, (Object)node.id());
            this.selector.send(metadataRequest.request());
            this.inFlightRequests.add(metadataRequest);
        } else if (this.connectionStates.canConnect(nodeConnectionId, now)) {
            log.debug("Initialize connection to node {} for sending metadata request", (Object)node.id());
            this.initiateConnect(node, now);
        } else {
            this.lastNoNodeAvailableMs = now;
        }
    }

    private void initiateConnect(Node node, long now) {
        String nodeConnectionId = node.idString();
        try {
            log.debug("Initiating connection to node {} at {}:{}.", new Object[]{node.id(), node.host(), node.port()});
            this.connectionStates.connecting(nodeConnectionId, now);
            this.selector.connect(nodeConnectionId, new InetSocketAddress(node.host(), node.port()), this.socketSendBuffer, this.socketReceiveBuffer);
        }
        catch (IOException e) {
            this.connectionStates.disconnected(nodeConnectionId);
            this.metadata.requestUpdate();
            log.debug("Error connecting to node {} at {}:{}:", new Object[]{node.id(), node.host(), node.port(), e});
        }
    }
}

