/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hive.ql.hooks;

import junit.framework.Assert;

import org.apache.hadoop.hive.ql.metadata.Partition;
import org.apache.hadoop.hive.ql.metadata.Table;

// This hook verifies that the location of every partition in the inputs and outputs does not
// start with the location of the table.  It is a very simple check to make sure the location is
// not a subdirectory.
public class VerifyPartitionIsNotSubdirectoryOfTableHook implements ExecuteWithHookContext {

  @Override
  public void run(HookContext hookContext) {
    for (WriteEntity output : hookContext.getOutputs()) {
      if (output.getType() == WriteEntity.Type.PARTITION) {
        verify (output.getPartition(), output.getTable());
      }
    }

    for (ReadEntity input : hookContext.getInputs()) {
      if (input.getType() == ReadEntity.Type.PARTITION) {
        verify (input.getPartition(), input.getTable());
      }
    }
  }

  private void verify(Partition partition, Table table) {
    Assert.assertFalse("The location of the partition: " + partition.getName() + " was a " +
        "subdirectory of the location of the table: " + table.getTableName(),
        partition.getDataLocation().toString().startsWith(table.getPath().toString()));
  }
}
