/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.net.InetSocketAddress;
import java.util.Iterator;
import java.util.Random;
import junit.framework.TestCase;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.TestDatanodeBlockScanner;
import org.apache.hadoop.hdfs.protocol.ClientProtocol;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.FSConstants;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;

public class TestReplication
extends TestCase {
    private static final long seed = 3735928559L;
    private static final int blockSize = 8192;
    private static final int fileSize = 16384;
    private static final String[] racks = new String[]{"/d1/r1", "/d1/r1", "/d1/r2", "/d1/r2", "/d1/r2", "/d2/r3", "/d2/r3"};
    private static final int numDatanodes = racks.length;
    private static final Log LOG = LogFactory.getLog((String)"org.apache.hadoop.hdfs.TestReplication");

    private void writeFile(FileSystem fileSys, Path name, int repl) throws IOException {
        FSDataOutputStream stm = fileSys.create(name, true, fileSys.getConf().getInt("io.file.buffer.size", 4096), (short)repl, 8192L);
        byte[] buffer = new byte[16384];
        Random rand = new Random(3735928559L);
        rand.nextBytes(buffer);
        stm.write(buffer);
        stm.close();
    }

    private void checkFile(FileSystem fileSys, Path name, int repl) throws IOException {
        LocatedBlock blk;
        DatanodeInfo[] datanodes;
        Configuration conf = fileSys.getConf();
        ClientProtocol namenode = DFSClient.createNamenode((Configuration)conf);
        this.waitForBlockReplication(name.toString(), namenode, Math.min(numDatanodes, repl), -1L);
        LocatedBlocks locations = namenode.getBlockLocations(name.toString(), 0L, Long.MAX_VALUE);
        FileStatus stat = fileSys.getFileStatus(name);
        BlockLocation[] blockLocations = fileSys.getFileBlockLocations(stat, 0L, Long.MAX_VALUE);
        TestReplication.assertTrue((blockLocations.length == locations.locatedBlockCount() ? 1 : 0) != 0);
        for (int i = 0; i < blockLocations.length; ++i) {
            LocatedBlock blk2 = locations.get(i);
            DatanodeInfo[] datanodes2 = blk2.getLocations();
            String[] topologyPaths = blockLocations[i].getTopologyPaths();
            TestReplication.assertTrue((topologyPaths.length == datanodes2.length ? 1 : 0) != 0);
            for (int j = 0; j < topologyPaths.length; ++j) {
                boolean found = false;
                for (int k = 0; k < racks.length; ++k) {
                    if (!topologyPaths[j].startsWith(racks[k])) continue;
                    found = true;
                    break;
                }
                TestReplication.assertTrue((boolean)found);
            }
        }
        boolean isOnSameRack = true;
        boolean isNotOnSameRack = true;
        Iterator i$ = locations.getLocatedBlocks().iterator();
        while (i$.hasNext() && (datanodes = (blk = (LocatedBlock)i$.next()).getLocations()).length > 1) {
            if (datanodes.length == 2) {
                isNotOnSameRack = !datanodes[0].getNetworkLocation().equals(datanodes[1].getNetworkLocation());
                break;
            }
            isOnSameRack = false;
            isNotOnSameRack = false;
            for (int i = 0; i < datanodes.length - 1; ++i) {
                LOG.info((Object)("datanode " + i + ": " + datanodes[i].getName()));
                boolean onRack = false;
                for (int j = i + 1; j < datanodes.length; ++j) {
                    if (!datanodes[i].getNetworkLocation().equals(datanodes[j].getNetworkLocation())) continue;
                    onRack = true;
                }
                if (onRack) {
                    isOnSameRack = true;
                }
                if (!onRack) {
                    isNotOnSameRack = true;
                }
                if (isOnSameRack && isNotOnSameRack) break;
            }
            if (isOnSameRack && isNotOnSameRack) continue;
            break;
        }
        TestReplication.assertTrue((boolean)isOnSameRack);
        TestReplication.assertTrue((boolean)isNotOnSameRack);
    }

    private void cleanupFile(FileSystem fileSys, Path name) throws IOException {
        TestReplication.assertTrue((boolean)fileSys.exists(name));
        fileSys.delete(name, true);
        TestReplication.assertTrue((!fileSys.exists(name) ? 1 : 0) != 0);
    }

    public void testBadBlockReportOnTransfer() throws Exception {
        Configuration conf = new Configuration();
        FileSystem fs = null;
        DFSClient dfsClient = null;
        LocatedBlocks blocks = null;
        int replicaCount = 0;
        MiniDFSCluster cluster = new MiniDFSCluster(conf, 2, true, null);
        cluster.waitActive();
        fs = cluster.getFileSystem();
        dfsClient = new DFSClient(new InetSocketAddress("localhost", cluster.getNameNodePort()), conf);
        Path file1 = new Path("/tmp/testBadBlockReportOnTransfer/file1");
        DFSTestUtil.createFile(fs, file1, 1024L, (short)1, 0L);
        DFSTestUtil.waitReplication(fs, file1, (short)1);
        String block = DFSTestUtil.getFirstBlock(fs, file1).getBlockName();
        cluster.corruptBlockOnDataNodes(block);
        fs.setReplication(file1, (short)2);
        blocks = dfsClient.namenode.getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
        while (!blocks.get(0).isCorrupt()) {
            try {
                LOG.info((Object)"Waiting until block is marked as corrupt...");
                Thread.sleep(1000L);
            }
            catch (InterruptedException ie) {
                // empty catch block
            }
            blocks = dfsClient.namenode.getBlockLocations(file1.toString(), 0L, Long.MAX_VALUE);
        }
        replicaCount = blocks.get(0).getLocations().length;
        TestReplication.assertTrue((replicaCount == 1 ? 1 : 0) != 0);
        cluster.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void runReplication(boolean simulated) throws IOException {
        Configuration conf = new Configuration();
        conf.setBoolean("dfs.replication.considerLoad", false);
        if (simulated) {
            conf.setBoolean("dfs.datanode.simulateddatastorage", true);
        }
        MiniDFSCluster cluster = new MiniDFSCluster(conf, numDatanodes, true, racks);
        cluster.waitActive();
        InetSocketAddress addr = new InetSocketAddress("localhost", cluster.getNameNodePort());
        DFSClient client = new DFSClient(addr, conf);
        DatanodeInfo[] info = client.datanodeReport(FSConstants.DatanodeReportType.LIVE);
        TestReplication.assertEquals((String)"Number of Datanodes ", (int)numDatanodes, (int)info.length);
        FileSystem fileSys = cluster.getFileSystem();
        try {
            Path file1 = new Path("/smallblocktest.dat");
            this.writeFile(fileSys, file1, 3);
            this.checkFile(fileSys, file1, 3);
            this.cleanupFile(fileSys, file1);
            this.writeFile(fileSys, file1, 10);
            this.checkFile(fileSys, file1, 10);
            this.cleanupFile(fileSys, file1);
            this.writeFile(fileSys, file1, 4);
            this.checkFile(fileSys, file1, 4);
            this.cleanupFile(fileSys, file1);
            this.writeFile(fileSys, file1, 1);
            this.checkFile(fileSys, file1, 1);
            this.cleanupFile(fileSys, file1);
            this.writeFile(fileSys, file1, 2);
            this.checkFile(fileSys, file1, 2);
            this.cleanupFile(fileSys, file1);
        }
        finally {
            fileSys.close();
            cluster.shutdown();
        }
    }

    public void testReplicationSimulatedStorag() throws IOException {
        this.runReplication(true);
    }

    public void testReplication() throws IOException {
        this.runReplication(false);
    }

    private void waitForBlockReplication(String filename, ClientProtocol namenode, int expected, long maxWaitSec) throws IOException {
        long start = System.currentTimeMillis();
        LOG.info((Object)("Checking for block replication for " + filename));
        int iters = 0;
        while (true) {
            boolean replOk = true;
            LocatedBlocks blocks = namenode.getBlockLocations(filename, 0L, Long.MAX_VALUE);
            for (LocatedBlock block : blocks.getLocatedBlocks()) {
                int actual = block.getLocations().length;
                if (actual >= expected) continue;
                LOG.info((Object)("Not enough replicas for " + block.getBlock() + " yet. Expecting " + expected + ", got " + actual + "."));
                replOk = false;
                break;
            }
            if (replOk) {
                return;
            }
            ++iters;
            if (maxWaitSec > 0L && System.currentTimeMillis() - start > maxWaitSec * 1000L) {
                throw new IOException("Timedout while waiting for all blocks to  be replicated for " + filename);
            }
            try {
                Thread.sleep(500L);
            }
            catch (InterruptedException ignored) {
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testPendingReplicationRetry() throws IOException {
        MiniDFSCluster cluster = null;
        int numDataNodes = 4;
        String testFile = "/replication-test-file";
        Path testPath = new Path(testFile);
        byte[] buffer = new byte[1024];
        for (int i = 0; i < buffer.length; ++i) {
            buffer[i] = 49;
        }
        try {
            Configuration conf = new Configuration();
            conf.set("dfs.replication", Integer.toString(numDataNodes));
            cluster = new MiniDFSCluster(0, conf, numDataNodes, true, true, null, null);
            cluster.waitActive();
            DFSClient dfsClient = new DFSClient(new InetSocketAddress("localhost", cluster.getNameNodePort()), conf);
            FSDataOutputStream out = cluster.getFileSystem().create(testPath);
            out.write(buffer);
            out.close();
            this.waitForBlockReplication(testFile, dfsClient.namenode, numDataNodes, -1L);
            String block = dfsClient.namenode.getBlockLocations(testFile, 0L, Long.MAX_VALUE).get(0).getBlock().getBlockName();
            cluster.shutdown();
            cluster = null;
            File baseDir = new File(System.getProperty("test.build.data"), "dfs/data");
            for (int i = 0; i < 25; ++i) {
                buffer[i] = 48;
            }
            int fileCount = 0;
            for (int i = 0; i < 6; ++i) {
                File blockFile = new File(baseDir, "data" + (i + 1) + "/current/" + block);
                LOG.info((Object)("Checking for file " + blockFile));
                if (!blockFile.exists()) continue;
                if (fileCount == 0) {
                    LOG.info((Object)("Deleting file " + blockFile));
                    TestReplication.assertTrue((boolean)blockFile.delete());
                } else {
                    LOG.info((Object)("Corrupting file " + blockFile));
                    long len = blockFile.length();
                    TestReplication.assertTrue((len > 50L ? 1 : 0) != 0);
                    RandomAccessFile blockOut = new RandomAccessFile(blockFile, "rw");
                    try {
                        blockOut.seek(len / 3L);
                        blockOut.write(buffer, 0, 25);
                    }
                    finally {
                        blockOut.close();
                    }
                }
                ++fileCount;
            }
            TestReplication.assertEquals((int)3, (int)fileCount);
            LOG.info((Object)"Restarting minicluster after deleting a replica and corrupting 2 crcs");
            conf = new Configuration();
            conf.set("dfs.replication", Integer.toString(numDataNodes));
            conf.set("dfs.replication.pending.timeout.sec", Integer.toString(2));
            conf.set("dfs.datanode.block.write.timeout.sec", Integer.toString(5));
            conf.set("dfs.safemode.threshold.pct", "0.75f");
            cluster = new MiniDFSCluster(0, conf, numDataNodes * 2, false, true, null, null);
            cluster.waitActive();
            dfsClient = new DFSClient(new InetSocketAddress("localhost", cluster.getNameNodePort()), conf);
            this.waitForBlockReplication(testFile, dfsClient.namenode, numDataNodes, -1L);
        }
        finally {
            if (cluster != null) {
                cluster.shutdown();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testReplicateLenMismatchedBlock() throws Exception {
        MiniDFSCluster cluster = new MiniDFSCluster(new Configuration(), 2, true, null);
        try {
            cluster.waitActive();
            this.changeBlockLen(cluster, -1);
            this.changeBlockLen(cluster, 1);
        }
        finally {
            cluster.shutdown();
        }
    }

    private void changeBlockLen(MiniDFSCluster cluster, int lenDelta) throws IOException, InterruptedException {
        Path fileName = new Path("/file1");
        boolean REPLICATION_FACTOR = true;
        FileSystem fs = cluster.getFileSystem();
        int fileLen = fs.getConf().getInt("io.bytes.per.checksum", 512);
        DFSTestUtil.createFile(fs, fileName, fileLen, (short)1, 0L);
        DFSTestUtil.waitReplication(fs, fileName, (short)1);
        String block = DFSTestUtil.getFirstBlock(fs, fileName).getBlockName();
        for (int i = 0; i < cluster.getDataNodes().size() && !TestDatanodeBlockScanner.changeReplicaLength(block, i, lenDelta); ++i) {
        }
        fs.setReplication(fileName, (short)2);
        DFSClient dfsClient = new DFSClient(new InetSocketAddress("localhost", cluster.getNameNodePort()), fs.getConf());
        LocatedBlocks blocks = dfsClient.namenode.getBlockLocations(fileName.toString(), 0L, (long)fileLen);
        if (lenDelta < 0) {
            while (!blocks.get(0).isCorrupt() || 1 != blocks.get(0).getLocations().length) {
                Thread.sleep(100L);
                blocks = dfsClient.namenode.getBlockLocations(fileName.toString(), 0L, (long)fileLen);
            }
        } else {
            while (2 != blocks.get(0).getLocations().length) {
                Thread.sleep(100L);
                blocks = dfsClient.namenode.getBlockLocations(fileName.toString(), 0L, (long)fileLen);
            }
        }
        fs.delete(fileName, true);
    }
}

