/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.nfs.nfs3;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class IdUserGroup {
    static final Log LOG = LogFactory.getLog(IdUserGroup.class);
    private static final String OS = System.getProperty("os.name");
    static final String LINUX_GET_ALL_USERS_CMD = "getent passwd | cut -d: -f1,3";
    static final String LINUX_GET_ALL_GROUPS_CMD = "getent group | cut -d: -f1,3";
    static final String MAC_GET_ALL_USERS_CMD = "dscl . -list /Users UniqueID";
    static final String MAC_GET_ALL_GROUPS_CMD = "dscl . -list /Groups PrimaryGroupID";
    static final long TIMEOUT = 900000L;
    private BiMap<Integer, String> uidNameMap = HashBiMap.create();
    private BiMap<Integer, String> gidNameMap = HashBiMap.create();
    private long lastUpdateTime = 0L;
    private static final String DUPLICATE_NAME_ID_DEBUG_INFO = "NFS gateway can't start with duplicate name or id on the host system.\nThis is because HDFS (non-kerberos cluster) uses name as the only way to identify a user or group.\nThe host system with duplicated user/group name or id might work fine most of the time by itself.\nHowever when NFS gateway talks to HDFS, HDFS accepts only user and group name.\nTherefore, same name means the same user or same group. To find the duplicated names/ids, one can do:\n<getent passwd | cut -d: -f1,3> and <getent group | cut -d: -f1,3> on Linux systms,\n<dscl . -list /Users UniqueID> and <dscl . -list /Groups PrimaryGroupID> on MacOS.";

    public IdUserGroup() throws IOException {
        this.updateMaps();
    }

    private boolean isExpired() {
        return this.lastUpdateTime - System.currentTimeMillis() > 900000L;
    }

    private void checkAndUpdateMaps() {
        if (this.isExpired()) {
            LOG.info((Object)"Update cache now");
            try {
                this.updateMaps();
            }
            catch (IOException e) {
                LOG.error((Object)"Can't update the maps. Will use the old ones, which can potentially cause problem.", (Throwable)e);
            }
        }
    }

    @VisibleForTesting
    public static void updateMapInternal(BiMap<Integer, String> map, String mapName, String command, String regex) throws IOException {
        BufferedReader br = null;
        try {
            Process process = Runtime.getRuntime().exec(new String[]{"bash", "-c", command});
            br = new BufferedReader(new InputStreamReader(process.getInputStream()));
            String line = null;
            while ((line = br.readLine()) != null) {
                String[] nameId = line.split(regex);
                if (nameId == null || nameId.length != 2) {
                    throw new IOException("Can't parse " + mapName + " list entry:" + line);
                }
                LOG.debug((Object)("add to " + mapName + "map:" + nameId[0] + " id:" + nameId[1]));
                Integer key = Integer.valueOf(nameId[1]);
                String value = nameId[0];
                if (map.containsKey((Object)key)) {
                    LOG.error((Object)String.format("Got duplicate id:(%d, %s), existing entry: (%d, %s).\n%s", key, value, key, map.get((Object)key), DUPLICATE_NAME_ID_DEBUG_INFO));
                    throw new DuplicateNameOrIdException("Got duplicate id.");
                }
                if (map.containsValue((Object)nameId[0])) {
                    LOG.error((Object)String.format("Got duplicate name:(%d, %s), existing entry: (%d, %s) \n%s", key, value, map.inverse().get((Object)value), value, DUPLICATE_NAME_ID_DEBUG_INFO));
                    throw new DuplicateNameOrIdException("Got duplicate name");
                }
                map.put((Object)Integer.valueOf(nameId[1]), (Object)nameId[0]);
            }
            LOG.info((Object)("Updated " + mapName + " map size:" + map.size()));
        }
        catch (IOException e) {
            LOG.error((Object)("Can't update " + mapName + " map"));
            throw e;
        }
        finally {
            if (br != null) {
                try {
                    br.close();
                }
                catch (IOException e1) {
                    LOG.error((Object)"Can't close BufferedReader of command result");
                    e1.printStackTrace();
                }
            }
        }
    }

    public synchronized void updateMaps() throws IOException {
        HashBiMap uMap = HashBiMap.create();
        HashBiMap gMap = HashBiMap.create();
        if (!OS.startsWith("Linux") && !OS.startsWith("Mac")) {
            LOG.error((Object)("Platform is not supported:" + OS + ". Can't update user map and group map and" + " 'nobody' will be used for any user and group."));
            return;
        }
        if (OS.startsWith("Linux")) {
            IdUserGroup.updateMapInternal((BiMap<Integer, String>)uMap, "user", LINUX_GET_ALL_USERS_CMD, ":");
            IdUserGroup.updateMapInternal((BiMap<Integer, String>)gMap, "group", LINUX_GET_ALL_GROUPS_CMD, ":");
        } else {
            IdUserGroup.updateMapInternal((BiMap<Integer, String>)uMap, "user", MAC_GET_ALL_USERS_CMD, "\\s+");
            IdUserGroup.updateMapInternal((BiMap<Integer, String>)gMap, "group", MAC_GET_ALL_GROUPS_CMD, "\\s+");
        }
        this.uidNameMap = uMap;
        this.gidNameMap = gMap;
        this.lastUpdateTime = System.currentTimeMillis();
    }

    public synchronized int getUid(String user) throws IOException {
        this.checkAndUpdateMaps();
        Integer id = (Integer)this.uidNameMap.inverse().get((Object)user);
        if (id == null) {
            throw new IOException("User just deleted?:" + user);
        }
        return id;
    }

    public synchronized int getGid(String group) throws IOException {
        this.checkAndUpdateMaps();
        Integer id = (Integer)this.gidNameMap.inverse().get((Object)group);
        if (id == null) {
            throw new IOException("No such group:" + group);
        }
        return id;
    }

    public synchronized String getUserName(int uid, String unknown) {
        this.checkAndUpdateMaps();
        String uname = (String)this.uidNameMap.get((Object)uid);
        if (uname == null) {
            LOG.warn((Object)("Can't find user name for uid " + uid + ". Use default user name " + unknown));
            uname = unknown;
        }
        return uname;
    }

    public synchronized String getGroupName(int gid, String unknown) {
        this.checkAndUpdateMaps();
        String gname = (String)this.gidNameMap.get((Object)gid);
        if (gname == null) {
            LOG.warn((Object)("Can't find group name for gid " + gid + ". Use default group name " + unknown));
            gname = unknown;
        }
        return gname;
    }

    public int getUidAllowingUnknown(String user) {
        int uid;
        this.checkAndUpdateMaps();
        try {
            uid = this.getUid(user);
        }
        catch (IOException e) {
            uid = user.hashCode();
            LOG.info((Object)("Can't map user " + user + ". Use its string hashcode:" + uid));
        }
        return uid;
    }

    public int getGidAllowingUnknown(String group) {
        int gid;
        this.checkAndUpdateMaps();
        try {
            gid = this.getGid(group);
        }
        catch (IOException e) {
            gid = group.hashCode();
            LOG.debug((Object)("Can't map group " + group + ". Use its string hashcode:" + gid));
        }
        return gid;
    }

    public static class DuplicateNameOrIdException
    extends IOException {
        private static final long serialVersionUID = 1L;

        public DuplicateNameOrIdException(String msg) {
            super(msg);
        }
    }
}

