/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.balancer;

import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.URI;
import java.util.EnumSet;
import org.apache.commons.logging.Log;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.NameNodeProxies;
import org.apache.hadoop.hdfs.protocol.AlreadyBeingCreatedException;
import org.apache.hadoop.hdfs.protocol.ClientProtocol;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenIdentifier;
import org.apache.hadoop.hdfs.security.token.block.BlockTokenSecretManager;
import org.apache.hadoop.hdfs.security.token.block.DataEncryptionKey;
import org.apache.hadoop.hdfs.security.token.block.ExportedBlockKeys;
import org.apache.hadoop.hdfs.server.balancer.Balancer;
import org.apache.hadoop.hdfs.server.protocol.NamenodeProtocol;
import org.apache.hadoop.hdfs.server.protocol.NamespaceInfo;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.util.Daemon;

@InterfaceAudience.Private
class NameNodeConnector {
    private static final Log LOG = Balancer.LOG;
    private static final Path BALANCER_ID_PATH = new Path("/system/balancer.id");
    private static final int MAX_NOT_CHANGED_ITERATIONS = 5;
    final URI nameNodeUri;
    final String blockpoolID;
    final NamenodeProtocol namenode;
    final ClientProtocol client;
    final FileSystem fs;
    final OutputStream out;
    private final boolean isBlockTokenEnabled;
    private final boolean encryptDataTransfer;
    private boolean shouldRun;
    private long keyUpdaterInterval;
    private int notChangedIterations = 0;
    private BlockTokenSecretManager blockTokenSecretManager;
    private Daemon keyupdaterthread;
    private DataEncryptionKey encryptionKey;

    NameNodeConnector(URI nameNodeUri, Configuration conf) throws IOException {
        this.nameNodeUri = nameNodeUri;
        this.namenode = NameNodeProxies.createProxy(conf, nameNodeUri, NamenodeProtocol.class).getProxy();
        this.client = NameNodeProxies.createProxy(conf, nameNodeUri, ClientProtocol.class).getProxy();
        this.fs = FileSystem.get((URI)nameNodeUri, (Configuration)conf);
        NamespaceInfo namespaceinfo = this.namenode.versionRequest();
        this.blockpoolID = namespaceinfo.getBlockPoolID();
        ExportedBlockKeys keys = this.namenode.getBlockKeys();
        this.isBlockTokenEnabled = keys.isBlockTokenEnabled();
        if (this.isBlockTokenEnabled) {
            long blockKeyUpdateInterval = keys.getKeyUpdateInterval();
            long blockTokenLifetime = keys.getTokenLifetime();
            LOG.info((Object)("Block token params received from NN: keyUpdateInterval=" + blockKeyUpdateInterval / 60000L + " min(s), tokenLifetime=" + blockTokenLifetime / 60000L + " min(s)"));
            String encryptionAlgorithm = conf.get("dfs.encrypt.data.transfer.algorithm");
            this.blockTokenSecretManager = new BlockTokenSecretManager(blockKeyUpdateInterval, blockTokenLifetime, this.blockpoolID, encryptionAlgorithm);
            this.blockTokenSecretManager.addKeys(keys);
            this.keyUpdaterInterval = blockKeyUpdateInterval / 4L;
            LOG.info((Object)("Balancer will update its block keys every " + this.keyUpdaterInterval / 60000L + " minute(s)"));
            this.keyupdaterthread = new Daemon((Runnable)new BlockKeyUpdater());
            this.shouldRun = true;
            this.keyupdaterthread.start();
        }
        this.encryptDataTransfer = this.fs.getServerDefaults(new Path("/")).getEncryptDataTransfer();
        this.out = this.checkAndMarkRunningBalancer();
        if (this.out == null) {
            throw new IOException("Another balancer is running");
        }
    }

    boolean shouldContinue(long dispatchBlockMoveBytes) {
        if (dispatchBlockMoveBytes > 0L) {
            this.notChangedIterations = 0;
        } else {
            ++this.notChangedIterations;
            if (this.notChangedIterations >= 5) {
                System.out.println("No block has been moved for " + this.notChangedIterations + " iterations. Exiting...");
                return false;
            }
        }
        return true;
    }

    Token<BlockTokenIdentifier> getAccessToken(ExtendedBlock eb) throws IOException {
        if (!this.isBlockTokenEnabled) {
            return BlockTokenSecretManager.DUMMY_TOKEN;
        }
        if (!this.shouldRun) {
            throw new IOException("Can not get access token. BlockKeyUpdater is not running");
        }
        return this.blockTokenSecretManager.generateToken(null, eb, EnumSet.of(BlockTokenSecretManager.AccessMode.REPLACE, BlockTokenSecretManager.AccessMode.COPY));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    DataEncryptionKey getDataEncryptionKey() throws IOException {
        if (this.encryptDataTransfer) {
            NameNodeConnector nameNodeConnector = this;
            synchronized (nameNodeConnector) {
                if (this.encryptionKey == null) {
                    this.encryptionKey = this.blockTokenSecretManager.generateDataEncryptionKey();
                }
                return this.encryptionKey;
            }
        }
        return null;
    }

    private OutputStream checkAndMarkRunningBalancer() throws IOException {
        try {
            FSDataOutputStream out = this.fs.create(BALANCER_ID_PATH);
            out.writeBytes(InetAddress.getLocalHost().getHostName());
            out.flush();
            return out;
        }
        catch (RemoteException e) {
            if (AlreadyBeingCreatedException.class.getName().equals(e.getClassName())) {
                return null;
            }
            throw e;
        }
    }

    void close() {
        this.shouldRun = false;
        try {
            if (this.keyupdaterthread != null) {
                this.keyupdaterthread.interrupt();
            }
        }
        catch (Exception e) {
            LOG.warn((Object)"Exception shutting down access key updater thread", (Throwable)e);
        }
        IOUtils.closeStream((Closeable)this.out);
        if (this.fs != null) {
            try {
                this.fs.delete(BALANCER_ID_PATH, true);
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Failed to delete " + BALANCER_ID_PATH), (Throwable)ioe);
            }
        }
    }

    public String toString() {
        return this.getClass().getSimpleName() + "[namenodeUri=" + this.nameNodeUri + ", id=" + this.blockpoolID + "]";
    }

    class BlockKeyUpdater
    implements Runnable {
        BlockKeyUpdater() {
        }

        @Override
        public void run() {
            try {
                while (NameNodeConnector.this.shouldRun) {
                    try {
                        NameNodeConnector.this.blockTokenSecretManager.addKeys(NameNodeConnector.this.namenode.getBlockKeys());
                    }
                    catch (IOException e) {
                        LOG.error((Object)"Failed to set keys", (Throwable)e);
                    }
                    Thread.sleep(NameNodeConnector.this.keyUpdaterInterval);
                }
            }
            catch (InterruptedException e) {
                LOG.debug((Object)"InterruptedException in block key updater thread", (Throwable)e);
            }
            catch (Throwable e) {
                LOG.error((Object)"Exception in block key updater thread", e);
                NameNodeConnector.this.shouldRun = false;
            }
        }
    }
}

