/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.NavigableMap;
import java.util.TreeMap;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.CipherSuite;
import org.apache.hadoop.crypto.CryptoProtocolVersion;
import org.apache.hadoop.fs.BatchedRemoteIterator;
import org.apache.hadoop.fs.UnresolvedLinkException;
import org.apache.hadoop.fs.XAttr;
import org.apache.hadoop.fs.XAttrSetFlag;
import org.apache.hadoop.hdfs.XAttrHelper;
import org.apache.hadoop.hdfs.protocol.EncryptionZone;
import org.apache.hadoop.hdfs.protocol.SnapshotAccessControlException;
import org.apache.hadoop.hdfs.protocol.proto.HdfsProtos;
import org.apache.hadoop.hdfs.protocolPB.PBHelper;
import org.apache.hadoop.hdfs.server.namenode.FSDirXAttrOp;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodesInPath;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EncryptionZoneManager {
    public static Logger LOG = LoggerFactory.getLogger(EncryptionZoneManager.class);
    private final TreeMap<Long, EncryptionZoneInt> encryptionZones;
    private final FSDirectory dir;
    private final int maxListEncryptionZonesResponses;

    public EncryptionZoneManager(FSDirectory dir, Configuration conf) {
        this.dir = dir;
        this.encryptionZones = new TreeMap();
        this.maxListEncryptionZonesResponses = conf.getInt("dfs.namenode.list.encryption.zones.num.responses", 100);
        Preconditions.checkArgument((this.maxListEncryptionZonesResponses >= 0 ? 1 : 0) != 0, (Object)"dfs.namenode.list.encryption.zones.num.responses must be a positive integer.");
    }

    void addEncryptionZone(Long inodeId, CipherSuite suite, CryptoProtocolVersion version, String keyName) {
        assert (this.dir.hasWriteLock());
        this.unprotectedAddEncryptionZone(inodeId, suite, version, keyName);
    }

    void unprotectedAddEncryptionZone(Long inodeId, CipherSuite suite, CryptoProtocolVersion version, String keyName) {
        EncryptionZoneInt ez = new EncryptionZoneInt(inodeId, suite, version, keyName);
        this.encryptionZones.put(inodeId, ez);
    }

    void removeEncryptionZone(Long inodeId) {
        assert (this.dir.hasWriteLock());
        this.encryptionZones.remove(inodeId);
    }

    boolean isInAnEZ(INodesInPath iip) throws UnresolvedLinkException, SnapshotAccessControlException {
        assert (this.dir.hasReadLock());
        return this.getEncryptionZoneForPath(iip) != null;
    }

    private String getFullPathName(EncryptionZoneInt ezi) {
        assert (this.dir.hasReadLock());
        return this.dir.getInode(ezi.getINodeId()).getFullPathName();
    }

    String getKeyName(INodesInPath iip) {
        assert (this.dir.hasReadLock());
        EncryptionZoneInt ezi = this.getEncryptionZoneForPath(iip);
        if (ezi == null) {
            return null;
        }
        return ezi.getKeyName();
    }

    private EncryptionZoneInt getEncryptionZoneForPath(INodesInPath iip) {
        assert (this.dir.hasReadLock());
        Preconditions.checkNotNull((Object)iip);
        List<INode> inodes = iip.getReadOnlyINodes();
        for (int i = inodes.size() - 1; i >= 0; --i) {
            EncryptionZoneInt ezi;
            INode inode = inodes.get(i);
            if (inode == null || (ezi = this.encryptionZones.get(inode.getId())) == null) continue;
            return ezi;
        }
        return null;
    }

    EncryptionZone getEZINodeForPath(INodesInPath iip) {
        EncryptionZoneInt ezi = this.getEncryptionZoneForPath(iip);
        if (ezi == null) {
            return null;
        }
        return new EncryptionZone(ezi.getINodeId(), this.getFullPathName(ezi), ezi.getSuite(), ezi.getVersion(), ezi.getKeyName());
    }

    void checkMoveValidity(INodesInPath srcIIP, INodesInPath dstIIP, String src) throws IOException {
        boolean dstInEZ;
        assert (this.dir.hasReadLock());
        EncryptionZoneInt srcEZI = this.getEncryptionZoneForPath(srcIIP);
        EncryptionZoneInt dstEZI = this.getEncryptionZoneForPath(dstIIP);
        boolean srcInEZ = srcEZI != null;
        boolean bl = dstInEZ = dstEZI != null;
        if (srcInEZ) {
            if (!dstInEZ) {
                if (srcEZI.getINodeId() == srcIIP.getLastINode().getId()) {
                    return;
                }
                throw new IOException(src + " can't be moved from an encryption zone.");
            }
        } else if (dstInEZ) {
            throw new IOException(src + " can't be moved into an encryption zone.");
        }
        if (srcInEZ && srcEZI != dstEZI) {
            String srcEZPath = this.getFullPathName(srcEZI);
            String dstEZPath = this.getFullPathName(dstEZI);
            StringBuilder sb = new StringBuilder(src);
            sb.append(" can't be moved from encryption zone ");
            sb.append(srcEZPath);
            sb.append(" to encryption zone ");
            sb.append(dstEZPath);
            sb.append(".");
            throw new IOException(sb.toString());
        }
    }

    XAttr createEncryptionZone(String src, CipherSuite suite, CryptoProtocolVersion version, String keyName) throws IOException {
        assert (this.dir.hasWriteLock());
        INodesInPath srcIIP = this.dir.getINodesInPath4Write(src, false);
        if (this.dir.isNonEmptyDirectory(srcIIP)) {
            throw new IOException("Attempt to create an encryption zone for a non-empty directory.");
        }
        if (srcIIP != null && srcIIP.getLastINode() != null && !srcIIP.getLastINode().isDirectory()) {
            throw new IOException("Attempt to create an encryption zone for a file.");
        }
        EncryptionZoneInt ezi = this.getEncryptionZoneForPath(srcIIP);
        if (ezi != null) {
            throw new IOException("Directory " + src + " is already in an " + "encryption zone. (" + this.getFullPathName(ezi) + ")");
        }
        HdfsProtos.ZoneEncryptionInfoProto proto = PBHelper.convert(suite, version, keyName);
        XAttr ezXAttr = XAttrHelper.buildXAttr("raw.hdfs.crypto.encryption.zone", proto.toByteArray());
        ArrayList xattrs = Lists.newArrayListWithCapacity((int)1);
        xattrs.add(ezXAttr);
        FSDirXAttrOp.unprotectedSetXAttrs(this.dir, src, xattrs, EnumSet.of(XAttrSetFlag.CREATE));
        return ezXAttr;
    }

    BatchedRemoteIterator.BatchedListEntries<EncryptionZone> listEncryptionZones(long prevId) throws IOException {
        assert (this.dir.hasReadLock());
        NavigableMap<Long, EncryptionZoneInt> tailMap = this.encryptionZones.tailMap(prevId, false);
        int numResponses = Math.min(this.maxListEncryptionZonesResponses, tailMap.size());
        ArrayList zones = Lists.newArrayListWithExpectedSize((int)numResponses);
        int count = 0;
        for (EncryptionZoneInt ezi : tailMap.values()) {
            String pathName = this.getFullPathName(ezi);
            INodesInPath iip = this.dir.getINodesInPath(pathName, false);
            INode lastINode = iip.getLastINode();
            if (lastINode == null || lastINode.getId() != ezi.getINodeId()) continue;
            zones.add(new EncryptionZone(ezi.getINodeId(), pathName, ezi.getSuite(), ezi.getVersion(), ezi.getKeyName()));
            if (++count < numResponses) continue;
            break;
        }
        boolean hasMore = numResponses < tailMap.size();
        return new BatchedRemoteIterator.BatchedListEntries((List)zones, hasMore);
    }

    private static class EncryptionZoneInt {
        private final long inodeId;
        private final CipherSuite suite;
        private final CryptoProtocolVersion version;
        private final String keyName;

        EncryptionZoneInt(long inodeId, CipherSuite suite, CryptoProtocolVersion version, String keyName) {
            Preconditions.checkArgument((suite != CipherSuite.UNKNOWN ? 1 : 0) != 0);
            Preconditions.checkArgument((version != CryptoProtocolVersion.UNKNOWN ? 1 : 0) != 0);
            this.inodeId = inodeId;
            this.suite = suite;
            this.version = version;
            this.keyName = keyName;
        }

        long getINodeId() {
            return this.inodeId;
        }

        CipherSuite getSuite() {
            return this.suite;
        }

        CryptoProtocolVersion getVersion() {
            return this.version;
        }

        String getKeyName() {
            return this.keyName;
        }
    }
}

