/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.crypto;

import java.io.DataOutput;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.EnumSet;
import java.util.Random;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.crypto.CryptoCodec;
import org.apache.hadoop.fs.ByteBufferReadable;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.HasEnhancedByteBufferAccess;
import org.apache.hadoop.fs.PositionedReadable;
import org.apache.hadoop.fs.ReadOption;
import org.apache.hadoop.fs.Seekable;
import org.apache.hadoop.fs.Syncable;
import org.apache.hadoop.io.ByteBufferPool;
import org.apache.hadoop.io.DataOutputBuffer;
import org.apache.hadoop.io.RandomDatum;
import org.apache.hadoop.test.GenericTestUtils;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public abstract class CryptoStreamsTestBase {
    protected static final Log LOG = LogFactory.getLog(CryptoStreamsTestBase.class);
    protected static CryptoCodec codec;
    private static final byte[] key;
    private static final byte[] iv;
    protected static final int count = 10000;
    protected static int defaultBufferSize;
    protected static int smallBufferSize;
    private byte[] data;
    private int dataLen;

    @Before
    public void setUp() throws IOException {
        int seed = new Random().nextInt();
        DataOutputBuffer dataBuf = new DataOutputBuffer();
        RandomDatum.Generator generator = new RandomDatum.Generator(seed);
        for (int i = 0; i < 10000; ++i) {
            generator.next();
            RandomDatum key = generator.getKey();
            RandomDatum value = generator.getValue();
            key.write((DataOutput)dataBuf);
            value.write((DataOutput)dataBuf);
        }
        LOG.info((Object)"Generated 10000 records");
        this.data = dataBuf.getData();
        this.dataLen = dataBuf.getLength();
    }

    protected void writeData(OutputStream out) throws Exception {
        out.write(this.data, 0, this.dataLen);
        out.close();
    }

    protected int getDataLen() {
        return this.dataLen;
    }

    private int readAll(InputStream in, byte[] b, int off, int len) throws IOException {
        int n = 0;
        int total = 0;
        while (n != -1 && (total += n) < len) {
            n = in.read(b, off + total, len - total);
        }
        return total;
    }

    protected OutputStream getOutputStream(int bufferSize) throws IOException {
        return this.getOutputStream(bufferSize, key, iv);
    }

    protected abstract OutputStream getOutputStream(int var1, byte[] var2, byte[] var3) throws IOException;

    protected InputStream getInputStream(int bufferSize) throws IOException {
        return this.getInputStream(bufferSize, key, iv);
    }

    protected abstract InputStream getInputStream(int var1, byte[] var2, byte[] var3) throws IOException;

    @Test(timeout=120000L)
    public void testRead() throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize);
        this.writeData(out);
        InputStream in = this.getInputStream(defaultBufferSize);
        this.readCheck(in);
        in.close();
        in = this.getInputStream(smallBufferSize);
        this.readCheck(in);
        in.close();
    }

    private void readCheck(InputStream in) throws Exception {
        byte[] result = new byte[this.dataLen];
        int n = this.readAll(in, result, 0, this.dataLen);
        Assert.assertEquals((long)this.dataLen, (long)n);
        byte[] expectedData = new byte[n];
        System.arraycopy(this.data, 0, expectedData, 0, n);
        Assert.assertArrayEquals((byte[])result, (byte[])expectedData);
        n = in.read(result, 0, this.dataLen);
        Assert.assertEquals((long)n, (long)-1L);
        in.close();
    }

    @Test(timeout=120000L)
    public void testWrite() throws Exception {
        this.writeCheck(defaultBufferSize);
        this.writeCheck(smallBufferSize);
    }

    private void writeCheck(int bufferSize) throws Exception {
        OutputStream out = this.getOutputStream(bufferSize);
        this.writeData(out);
        if (out instanceof FSDataOutputStream) {
            Assert.assertEquals((long)((FSDataOutputStream)out).getPos(), (long)this.getDataLen());
        }
    }

    @Test(timeout=120000L)
    public void testCryptoIV() throws Exception {
        byte[] iv1 = (byte[])iv.clone();
        this.setCounterBaseForIV(iv1, Long.MAX_VALUE);
        this.cryptoCheck(iv1);
        this.setCounterBaseForIV(iv1, 0x7FFFFFFFFFFFFFFEL);
        this.cryptoCheck(iv1);
        this.setCounterBaseForIV(iv1, Integer.MAX_VALUE);
        this.cryptoCheck(iv1);
        this.setCounterBaseForIV(iv1, 0L);
        this.cryptoCheck(iv1);
        this.setCounterBaseForIV(iv1, -1L);
        this.cryptoCheck(iv1);
    }

    private void cryptoCheck(byte[] iv) throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize, key, iv);
        this.writeData(out);
        InputStream in = this.getInputStream(defaultBufferSize, key, iv);
        this.readCheck(in);
        in.close();
    }

    private void setCounterBaseForIV(byte[] iv, long counterBase) {
        ByteBuffer buf = ByteBuffer.wrap(iv);
        buf.order(ByteOrder.BIG_ENDIAN);
        buf.putLong(iv.length - 8, counterBase);
    }

    @Test(timeout=120000L)
    public void testSyncable() throws IOException {
        this.syncableCheck();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void syncableCheck() throws IOException {
        try (OutputStream out = this.getOutputStream(smallBufferSize);){
            int bytesWritten = this.dataLen / 3;
            out.write(this.data, 0, bytesWritten);
            ((Syncable)out).hflush();
            InputStream in = this.getInputStream(defaultBufferSize);
            this.verify(in, bytesWritten, this.data);
            in.close();
            out.write(this.data, bytesWritten, this.dataLen - bytesWritten);
            ((Syncable)out).hsync();
            in = this.getInputStream(defaultBufferSize);
            this.verify(in, this.dataLen, this.data);
            in.close();
        }
    }

    private void verify(InputStream in, int bytesToVerify, byte[] expectedBytes) throws IOException {
        byte[] readBuf = new byte[bytesToVerify];
        this.readAll(in, readBuf, 0, bytesToVerify);
        for (int i = 0; i < bytesToVerify; ++i) {
            Assert.assertEquals((long)expectedBytes[i], (long)readBuf[i]);
        }
    }

    private int readAll(InputStream in, long pos, byte[] b, int off, int len) throws IOException {
        int n = 0;
        int total = 0;
        while (n != -1 && (total += n) < len) {
            n = ((PositionedReadable)in).read(pos + (long)total, b, off + total, len - total);
        }
        return total;
    }

    @Test(timeout=120000L)
    public void testPositionedRead() throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize);
        this.writeData(out);
        InputStream in = this.getInputStream(defaultBufferSize);
        this.positionedReadCheck(in, this.dataLen / 3);
        this.positionedReadCheck(in, this.dataLen / 2);
        in.close();
    }

    private void positionedReadCheck(InputStream in, int pos) throws Exception {
        byte[] result = new byte[this.dataLen];
        int n = this.readAll(in, pos, result, 0, this.dataLen);
        Assert.assertEquals((long)this.dataLen, (long)(n + pos));
        byte[] readData = new byte[n];
        System.arraycopy(result, 0, readData, 0, n);
        byte[] expectedData = new byte[n];
        System.arraycopy(this.data, pos, expectedData, 0, n);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
    }

    @Test(timeout=120000L)
    public void testReadFully() throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize);
        this.writeData(out);
        InputStream in = this.getInputStream(defaultBufferSize);
        int len1 = this.dataLen / 4;
        byte[] readData = new byte[len1];
        this.readAll(in, readData, 0, len1);
        byte[] expectedData = new byte[len1];
        System.arraycopy(this.data, 0, expectedData, 0, len1);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
        this.readFullyCheck(in, this.dataLen / 3);
        readData = new byte[len1];
        this.readAll(in, readData, 0, len1);
        expectedData = new byte[len1];
        System.arraycopy(this.data, len1, expectedData, 0, len1);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
        this.readFullyCheck(in, this.dataLen / 2);
        readData = new byte[len1];
        this.readAll(in, readData, 0, len1);
        expectedData = new byte[len1];
        System.arraycopy(this.data, 2 * len1, expectedData, 0, len1);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
        in.close();
    }

    private void readFullyCheck(InputStream in, int pos) throws Exception {
        byte[] result = new byte[this.dataLen - pos];
        ((PositionedReadable)in).readFully((long)pos, result);
        byte[] expectedData = new byte[this.dataLen - pos];
        System.arraycopy(this.data, pos, expectedData, 0, this.dataLen - pos);
        Assert.assertArrayEquals((byte[])result, (byte[])expectedData);
        result = new byte[this.dataLen];
        try {
            ((PositionedReadable)in).readFully((long)pos, result);
            Assert.fail((String)"Read fully exceeds maximum length should fail.");
        }
        catch (IOException e) {
            // empty catch block
        }
    }

    @Test(timeout=120000L)
    public void testSeek() throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize);
        this.writeData(out);
        InputStream in = this.getInputStream(defaultBufferSize);
        this.seekCheck(in, this.dataLen / 3);
        this.seekCheck(in, 0);
        this.seekCheck(in, this.dataLen / 2);
        long pos = ((Seekable)in).getPos();
        try {
            this.seekCheck(in, -3);
            Assert.fail((String)"Seek to negative offset should fail.");
        }
        catch (IllegalArgumentException e) {
            GenericTestUtils.assertExceptionContains("Cannot seek to negative offset", e);
        }
        Assert.assertEquals((long)pos, (long)((Seekable)in).getPos());
        try {
            this.seekCheck(in, this.dataLen + 3);
            Assert.fail((String)"Seek after EOF should fail.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("Cannot seek after EOF", e);
        }
        Assert.assertEquals((long)pos, (long)((Seekable)in).getPos());
        in.close();
    }

    private void seekCheck(InputStream in, int pos) throws Exception {
        byte[] result = new byte[this.dataLen];
        ((Seekable)in).seek((long)pos);
        int n = this.readAll(in, result, 0, this.dataLen);
        Assert.assertEquals((long)this.dataLen, (long)(n + pos));
        byte[] readData = new byte[n];
        System.arraycopy(result, 0, readData, 0, n);
        byte[] expectedData = new byte[n];
        System.arraycopy(this.data, pos, expectedData, 0, n);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
    }

    @Test(timeout=120000L)
    public void testGetPos() throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize);
        this.writeData(out);
        InputStream in = this.getInputStream(defaultBufferSize);
        byte[] result = new byte[this.dataLen];
        int n1 = this.readAll(in, result, 0, this.dataLen / 3);
        Assert.assertEquals((long)n1, (long)((Seekable)in).getPos());
        int n2 = this.readAll(in, result, n1, this.dataLen - n1);
        Assert.assertEquals((long)(n1 + n2), (long)((Seekable)in).getPos());
        in.close();
    }

    @Test(timeout=120000L)
    public void testAvailable() throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize);
        this.writeData(out);
        InputStream in = this.getInputStream(defaultBufferSize);
        byte[] result = new byte[this.dataLen];
        int n1 = this.readAll(in, result, 0, this.dataLen / 3);
        Assert.assertEquals((long)in.available(), (long)(this.dataLen - n1));
        int n2 = this.readAll(in, result, n1, this.dataLen - n1);
        Assert.assertEquals((long)in.available(), (long)(this.dataLen - n1 - n2));
        in.close();
    }

    @Test(timeout=120000L)
    public void testSkip() throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize);
        this.writeData(out);
        InputStream in = this.getInputStream(defaultBufferSize);
        byte[] result = new byte[this.dataLen];
        int n1 = this.readAll(in, result, 0, this.dataLen / 3);
        Assert.assertEquals((long)n1, (long)((Seekable)in).getPos());
        long skipped = in.skip(this.dataLen / 3);
        int n2 = this.readAll(in, result, 0, this.dataLen);
        Assert.assertEquals((long)this.dataLen, (long)((long)n1 + skipped + (long)n2));
        byte[] readData = new byte[n2];
        System.arraycopy(result, 0, readData, 0, n2);
        byte[] expectedData = new byte[n2];
        System.arraycopy(this.data, this.dataLen - n2, expectedData, 0, n2);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
        try {
            skipped = in.skip(-3L);
            Assert.fail((String)"Skip Negative length should fail.");
        }
        catch (IllegalArgumentException e) {
            GenericTestUtils.assertExceptionContains("Negative skip length", e);
        }
        skipped = in.skip(3L);
        Assert.assertEquals((long)skipped, (long)0L);
        in.close();
    }

    private void byteBufferReadCheck(InputStream in, ByteBuffer buf, int bufPos) throws Exception {
        buf.position(bufPos);
        int n = ((ByteBufferReadable)in).read(buf);
        Assert.assertEquals((long)(bufPos + n), (long)buf.position());
        byte[] readData = new byte[n];
        buf.rewind();
        buf.position(bufPos);
        buf.get(readData);
        byte[] expectedData = new byte[n];
        System.arraycopy(this.data, 0, expectedData, 0, n);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
    }

    @Test(timeout=120000L)
    public void testByteBufferRead() throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize);
        this.writeData(out);
        InputStream in = this.getInputStream(defaultBufferSize);
        ByteBuffer buf = ByteBuffer.allocate(this.dataLen + 100);
        this.byteBufferReadCheck(in, buf, 0);
        in.close();
        in = this.getInputStream(defaultBufferSize);
        buf.clear();
        this.byteBufferReadCheck(in, buf, 11);
        in.close();
        in = this.getInputStream(smallBufferSize);
        buf.clear();
        this.byteBufferReadCheck(in, buf, 0);
        in.close();
        in = this.getInputStream(smallBufferSize);
        buf.clear();
        this.byteBufferReadCheck(in, buf, 11);
        in.close();
        in = this.getInputStream(defaultBufferSize);
        buf = ByteBuffer.allocateDirect(this.dataLen + 100);
        this.byteBufferReadCheck(in, buf, 0);
        in.close();
        in = this.getInputStream(defaultBufferSize);
        buf.clear();
        this.byteBufferReadCheck(in, buf, 11);
        in.close();
        in = this.getInputStream(smallBufferSize);
        buf.clear();
        this.byteBufferReadCheck(in, buf, 0);
        in.close();
        in = this.getInputStream(smallBufferSize);
        buf.clear();
        this.byteBufferReadCheck(in, buf, 11);
        in.close();
    }

    @Test(timeout=120000L)
    public void testCombinedOp() throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize);
        this.writeData(out);
        int len1 = this.dataLen / 8;
        int len2 = this.dataLen / 10;
        InputStream in = this.getInputStream(defaultBufferSize);
        byte[] readData = new byte[len1];
        this.readAll(in, readData, 0, len1);
        byte[] expectedData = new byte[len1];
        System.arraycopy(this.data, 0, expectedData, 0, len1);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
        long pos = ((Seekable)in).getPos();
        Assert.assertEquals((long)len1, (long)pos);
        ((Seekable)in).seek(pos + (long)len2);
        long n = in.skip(len2);
        Assert.assertEquals((long)len2, (long)n);
        this.positionedReadCheck(in, this.dataLen / 4);
        pos = ((Seekable)in).getPos();
        Assert.assertEquals((long)(len1 + len2 + len2), (long)pos);
        ByteBuffer buf = ByteBuffer.allocate(len1);
        int nRead = ((ByteBufferReadable)in).read(buf);
        Assert.assertEquals((long)nRead, (long)buf.position());
        readData = new byte[nRead];
        buf.rewind();
        buf.get(readData);
        expectedData = new byte[nRead];
        System.arraycopy(this.data, (int)pos, expectedData, 0, nRead);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
        long lastPos = pos;
        pos = ((Seekable)in).getPos();
        Assert.assertEquals((long)(lastPos + (long)nRead), (long)pos);
        this.positionedReadCheck(in, this.dataLen / 3);
        readData = new byte[len1];
        this.readAll(in, readData, 0, len1);
        expectedData = new byte[len1];
        System.arraycopy(this.data, (int)pos, expectedData, 0, len1);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
        lastPos = pos;
        pos = ((Seekable)in).getPos();
        Assert.assertEquals((long)(lastPos + (long)len1), (long)pos);
        buf = ByteBuffer.allocate(len1);
        nRead = ((ByteBufferReadable)in).read(buf);
        Assert.assertEquals((long)nRead, (long)buf.position());
        readData = new byte[nRead];
        buf.rewind();
        buf.get(readData);
        expectedData = new byte[nRead];
        System.arraycopy(this.data, (int)pos, expectedData, 0, nRead);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
        lastPos = pos;
        pos = ((Seekable)in).getPos();
        Assert.assertEquals((long)(lastPos + (long)nRead), (long)pos);
        ((Seekable)in).seek((long)this.dataLen);
        buf.clear();
        n = ((ByteBufferReadable)in).read(buf);
        Assert.assertEquals((long)n, (long)-1L);
        in.close();
    }

    @Test(timeout=120000L)
    public void testSeekToNewSource() throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize);
        this.writeData(out);
        InputStream in = this.getInputStream(defaultBufferSize);
        int len1 = this.dataLen / 8;
        byte[] readData = new byte[len1];
        this.readAll(in, readData, 0, len1);
        this.seekToNewSourceCheck(in, this.dataLen / 3);
        this.seekToNewSourceCheck(in, 0);
        this.seekToNewSourceCheck(in, this.dataLen / 2);
        try {
            this.seekToNewSourceCheck(in, -3);
            Assert.fail((String)"Seek to negative offset should fail.");
        }
        catch (IllegalArgumentException e) {
            GenericTestUtils.assertExceptionContains("Cannot seek to negative offset", e);
        }
        try {
            this.seekToNewSourceCheck(in, this.dataLen + 3);
            Assert.fail((String)"Seek after EOF should fail.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("Attempted to read past end of file", e);
        }
        in.close();
    }

    private void seekToNewSourceCheck(InputStream in, int targetPos) throws Exception {
        byte[] result = new byte[this.dataLen];
        ((Seekable)in).seekToNewSource((long)targetPos);
        int n = this.readAll(in, result, 0, this.dataLen);
        Assert.assertEquals((long)this.dataLen, (long)(n + targetPos));
        byte[] readData = new byte[n];
        System.arraycopy(result, 0, readData, 0, n);
        byte[] expectedData = new byte[n];
        System.arraycopy(this.data, targetPos, expectedData, 0, n);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
    }

    private ByteBufferPool getBufferPool() {
        return new ByteBufferPool(){

            public ByteBuffer getBuffer(boolean direct, int length) {
                return ByteBuffer.allocateDirect(length);
            }

            public void putBuffer(ByteBuffer buffer) {
            }
        };
    }

    @Test(timeout=120000L)
    public void testHasEnhancedByteBufferAccess() throws Exception {
        OutputStream out = this.getOutputStream(defaultBufferSize);
        this.writeData(out);
        InputStream in = this.getInputStream(defaultBufferSize);
        int len1 = this.dataLen / 8;
        ByteBuffer buffer = ((HasEnhancedByteBufferAccess)in).read(this.getBufferPool(), len1, EnumSet.of(ReadOption.SKIP_CHECKSUMS));
        int n1 = buffer.remaining();
        byte[] readData = new byte[n1];
        buffer.get(readData);
        byte[] expectedData = new byte[n1];
        System.arraycopy(this.data, 0, expectedData, 0, n1);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
        ((HasEnhancedByteBufferAccess)in).releaseBuffer(buffer);
        readData = new byte[len1];
        this.readAll(in, readData, 0, len1);
        expectedData = new byte[len1];
        System.arraycopy(this.data, n1, expectedData, 0, len1);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
        buffer = ((HasEnhancedByteBufferAccess)in).read(this.getBufferPool(), len1, EnumSet.of(ReadOption.SKIP_CHECKSUMS));
        int n2 = buffer.remaining();
        readData = new byte[n2];
        buffer.get(readData);
        expectedData = new byte[n2];
        System.arraycopy(this.data, n1 + len1, expectedData, 0, n2);
        Assert.assertArrayEquals((byte[])readData, (byte[])expectedData);
        ((HasEnhancedByteBufferAccess)in).releaseBuffer(buffer);
        in.close();
    }

    static {
        key = new byte[]{1, 2, 3, 4, 5, 6, 7, 8, 9, 16, 17, 18, 19, 20, 21, 22};
        iv = new byte[]{1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 7, 8};
        defaultBufferSize = 8192;
        smallBufferSize = 1024;
    }
}

