/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.io.DataOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.server.datanode.BlockSender;
import org.apache.hadoop.hdfs.server.datanode.CachingStrategy;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.FinalizedReplica;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.RollingLogs;
import org.apache.hadoop.hdfs.util.DataTransferThrottler;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.GSet;
import org.apache.hadoop.util.LightWeightGSet;
import org.apache.hadoop.util.Time;

class BlockPoolSliceScanner {
    public static final Log LOG = LogFactory.getLog(BlockPoolSliceScanner.class);
    private static final String DATA_FORMAT = "yyyy-MM-dd HH:mm:ss,SSS";
    private static final int MAX_SCAN_RATE = 0x800000;
    private static final int MIN_SCAN_RATE = 0x100000;
    private static final long DEFAULT_SCAN_PERIOD_HOURS = 504L;
    private static final String VERIFICATION_PREFIX = "dncp_block_verification.log";
    private final String blockPoolId;
    private final long scanPeriod;
    private final AtomicLong lastScanTime = new AtomicLong();
    private final DataNode datanode;
    private final FsDatasetSpi<? extends FsVolumeSpi> dataset;
    private final SortedSet<BlockScanInfo> blockInfoSet = new TreeSet<BlockScanInfo>(BlockScanInfo.LAST_SCAN_TIME_COMPARATOR);
    private final GSet<Block, BlockScanInfo> blockMap = new LightWeightGSet(LightWeightGSet.computeCapacity((double)0.5, (String)"BlockMap"));
    private volatile HashMap<Long, Integer> processedBlocks;
    private long totalScans = 0L;
    private long totalScanErrors = 0L;
    private long totalTransientErrors = 0L;
    private final AtomicInteger totalBlocksScannedInLastRun = new AtomicInteger();
    private long currentPeriodStart = Time.monotonicNow();
    private long bytesLeft = 0L;
    private long totalBytesToScan = 0L;
    private boolean isNewPeriod = true;
    private final LogFileHandler verificationLog;
    private final DataTransferThrottler throttler = new DataTransferThrottler(200L, 0x800000L);

    BlockPoolSliceScanner(String bpid, DataNode datanode, FsDatasetSpi<? extends FsVolumeSpi> dataset, Configuration conf) {
        this.datanode = datanode;
        this.dataset = dataset;
        this.blockPoolId = bpid;
        long hours = conf.getInt("dfs.datanode.scan.period.hours", 0);
        if (hours <= 0L) {
            hours = 504L;
        }
        this.scanPeriod = hours * 3600L * 1000L;
        LOG.info((Object)("Periodic Block Verification Scanner initialized with interval " + hours + " hours for block pool " + bpid));
        List<FinalizedReplica> arr = dataset.getFinalizedBlocks(this.blockPoolId);
        Collections.shuffle(arr);
        long scanTime = -1L;
        for (FinalizedReplica block : arr) {
            BlockScanInfo info = new BlockScanInfo(block);
            --scanTime;
            info.lastScanTime = info.lastScanTime;
            this.addBlockInfo(info);
        }
        RollingLogs rollingLogs = null;
        try {
            rollingLogs = dataset.createRollingLogs(this.blockPoolId, VERIFICATION_PREFIX);
        }
        catch (IOException e) {
            LOG.warn((Object)"Could not open verfication log. Verification times are not stored.");
        }
        this.verificationLog = rollingLogs == null ? null : new LogFileHandler(rollingLogs);
    }

    String getBlockPoolId() {
        return this.blockPoolId;
    }

    private void updateBytesToScan(long len, long lastScanTime) {
        this.totalBytesToScan += len;
        if (lastScanTime < this.currentPeriodStart) {
            this.bytesLeft += len;
        }
    }

    private synchronized void addBlockInfo(BlockScanInfo info) {
        boolean added = this.blockInfoSet.add(info);
        this.blockMap.put((Object)info);
        if (added) {
            this.updateBytesToScan(info.getNumBytes(), info.lastScanTime);
        }
    }

    private synchronized void delBlockInfo(BlockScanInfo info) {
        boolean exists = this.blockInfoSet.remove(info);
        this.blockMap.remove((Object)info);
        if (exists) {
            this.updateBytesToScan(-info.getNumBytes(), info.lastScanTime);
        }
    }

    private synchronized void updateBlockInfo(LogEntry e) {
        BlockScanInfo info = (BlockScanInfo)this.blockMap.get((Object)new Block(e.blockId, 0L, e.genStamp));
        if (info != null && e.verificationTime > 0L && info.lastScanTime < e.verificationTime) {
            this.delBlockInfo(info);
            info.lastScanTime = e.verificationTime;
            info.lastScanType = ScanType.VERIFICATION_SCAN;
            this.addBlockInfo(info);
        }
    }

    private synchronized long getNewBlockScanTime() {
        long period = Math.min(this.scanPeriod, (long)(Math.max(this.blockMap.size(), 1) * 600) * 1000L);
        int periodInt = Math.abs((int)period);
        return Time.monotonicNow() - this.scanPeriod + (long)DFSUtil.getRandom().nextInt(periodInt);
    }

    synchronized void addBlock(ExtendedBlock block) {
        BlockScanInfo info = (BlockScanInfo)this.blockMap.get((Object)block.getLocalBlock());
        if (info != null) {
            LOG.warn((Object)("Adding an already existing block " + block));
            this.delBlockInfo(info);
        }
        info = new BlockScanInfo(block.getLocalBlock());
        info.lastScanTime = this.getNewBlockScanTime();
        this.addBlockInfo(info);
        this.adjustThrottler();
    }

    synchronized void deleteBlock(Block block) {
        BlockScanInfo info = (BlockScanInfo)this.blockMap.get((Object)block);
        if (info != null) {
            this.delBlockInfo(info);
        }
    }

    @VisibleForTesting
    long getTotalScans() {
        return this.totalScans;
    }

    long getLastScanTime() {
        return this.lastScanTime.get();
    }

    synchronized long getLastScanTime(Block block) {
        BlockScanInfo info = (BlockScanInfo)this.blockMap.get((Object)block);
        return info == null ? 0L : info.lastScanTime;
    }

    void deleteBlocks(Block[] blocks) {
        for (Block b : blocks) {
            this.deleteBlock(b);
        }
    }

    private synchronized void updateScanStatus(Block block, ScanType type, boolean scanOk) {
        BlockScanInfo info = (BlockScanInfo)this.blockMap.get((Object)block);
        if (info != null) {
            this.delBlockInfo(info);
        } else {
            info = new BlockScanInfo(block);
        }
        long now = Time.monotonicNow();
        info.lastScanType = type;
        info.lastScanTime = now;
        info.lastScanOk = scanOk;
        this.addBlockInfo(info);
        if (!scanOk) {
            return;
        }
        if (this.verificationLog != null) {
            this.verificationLog.append(now, block.getGenerationStamp(), block.getBlockId());
        }
    }

    private void handleScanFailure(ExtendedBlock block) {
        LOG.info((Object)("Reporting bad " + block));
        try {
            this.datanode.reportBadBlocks(block);
        }
        catch (IOException ie) {
            LOG.warn((Object)("Cannot report bad " + block.getBlockId()));
        }
    }

    private synchronized void adjustThrottler() {
        long timeLeft = Math.max(1L, this.currentPeriodStart + this.scanPeriod - Time.monotonicNow());
        long bw = Math.max(this.bytesLeft * 1000L / timeLeft, 0x100000L);
        this.throttler.setBandwidth(Math.min(bw, 0x800000L));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void verifyBlock(ExtendedBlock block) {
        BlockSender blockSender = null;
        for (int i = 0; i < 2; ++i) {
            boolean second = i > 0;
            try {
                this.adjustThrottler();
                blockSender = new BlockSender(block, 0L, -1L, false, true, true, this.datanode, null, CachingStrategy.newDropBehind());
                DataOutputStream out = new DataOutputStream((OutputStream)new IOUtils.NullOutputStream());
                blockSender.sendBlock(out, null, this.throttler);
                LOG.info((Object)((second ? "Second " : "") + "Verification succeeded for " + block));
                if (second) {
                    ++this.totalTransientErrors;
                }
                this.updateScanStatus(block.getLocalBlock(), ScanType.VERIFICATION_SCAN, true);
            }
            catch (IOException e) {
                block10: {
                    block9: {
                        block8: {
                            try {
                                this.updateScanStatus(block.getLocalBlock(), ScanType.VERIFICATION_SCAN, false);
                                if (this.dataset.contains(block)) break block8;
                                LOG.info((Object)(block + " is no longer in the dataset"));
                                this.deleteBlock(block.getLocalBlock());
                            }
                            catch (Throwable throwable) {
                                IOUtils.closeStream(blockSender);
                                this.datanode.getMetrics().incrBlocksVerified();
                                ++this.totalScans;
                                throw throwable;
                            }
                            IOUtils.closeStream((Closeable)blockSender);
                            this.datanode.getMetrics().incrBlocksVerified();
                            ++this.totalScans;
                            return;
                        }
                        if (!(e instanceof FileNotFoundException)) break block9;
                        LOG.info((Object)("Verification failed for " + block + " - may be due to race with write"));
                        this.deleteBlock(block.getLocalBlock());
                        IOUtils.closeStream((Closeable)blockSender);
                        this.datanode.getMetrics().incrBlocksVerified();
                        ++this.totalScans;
                        return;
                    }
                    LOG.warn((Object)((second ? "Second " : "First ") + "Verification failed for " + block), (Throwable)e);
                    if (!second) break block10;
                    ++this.totalScanErrors;
                    this.datanode.getMetrics().incrBlockVerificationFailures();
                    this.handleScanFailure(block);
                    IOUtils.closeStream((Closeable)blockSender);
                    this.datanode.getMetrics().incrBlocksVerified();
                    ++this.totalScans;
                    return;
                }
                IOUtils.closeStream((Closeable)blockSender);
                this.datanode.getMetrics().incrBlocksVerified();
                ++this.totalScans;
                continue;
            }
            IOUtils.closeStream((Closeable)blockSender);
            this.datanode.getMetrics().incrBlocksVerified();
            ++this.totalScans;
            return;
        }
    }

    private synchronized long getEarliestScanTime() {
        if (!this.blockInfoSet.isEmpty()) {
            return this.blockInfoSet.first().lastScanTime;
        }
        return Long.MAX_VALUE;
    }

    private synchronized boolean isFirstBlockProcessed() {
        long blockId;
        return !this.blockInfoSet.isEmpty() && this.processedBlocks.get(blockId = this.blockInfoSet.first().getBlockId()) != null && this.processedBlocks.get(blockId) == 1;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyFirstBlock() {
        Block block = null;
        BlockPoolSliceScanner blockPoolSliceScanner = this;
        synchronized (blockPoolSliceScanner) {
            if (!this.blockInfoSet.isEmpty()) {
                block = this.blockInfoSet.first();
            }
        }
        if (block != null) {
            this.verifyBlock(new ExtendedBlock(this.blockPoolId, block));
            this.processedBlocks.put(block.getBlockId(), 1);
        }
    }

    int getBlocksScannedInLastRun() {
        return this.totalBlocksScannedInLastRun.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean assignInitialVerificationTimes() {
        boolean bl;
        RollingLogs.LineIterator logIterator;
        block12: {
            block13: {
                if (this.verificationLog == null) break block13;
                long now = Time.monotonicNow();
                logIterator = null;
                try {
                    logIterator = this.verificationLog.logs.iterator(false);
                    while (logIterator.hasNext()) {
                        BlockScanInfo info;
                        if (!this.datanode.shouldRun || this.datanode.blockScanner.blockScannerThread.isInterrupted()) {
                            bl = false;
                            break block12;
                        }
                        LogEntry entry = LogEntry.parseEntry((String)logIterator.next());
                        if (entry == null) continue;
                        this.updateBlockInfo(entry);
                        if (now - entry.verificationTime >= this.scanPeriod || (info = (BlockScanInfo)this.blockMap.get((Object)new Block(entry.blockId, 0L, entry.genStamp))) == null) continue;
                        if (this.processedBlocks.get(entry.blockId) == null) {
                            if (this.isNewPeriod) {
                                this.updateBytesLeft(-info.getNumBytes());
                            }
                            this.processedBlocks.put(entry.blockId, 1);
                        }
                        if (!logIterator.isLastReadFromPrevious()) continue;
                        this.verificationLog.append(entry.verificationTime, entry.genStamp, entry.blockId);
                    }
                }
                catch (IOException e) {
                    try {
                        LOG.warn((Object)"Failed to read previous verification times.", (Throwable)e);
                    }
                    catch (Throwable throwable) {
                        IOUtils.closeStream(logIterator);
                        throw throwable;
                    }
                    IOUtils.closeStream((Closeable)logIterator);
                }
                IOUtils.closeStream((Closeable)logIterator);
                this.isNewPeriod = false;
            }
            BlockPoolSliceScanner blockPoolSliceScanner = this;
            synchronized (blockPoolSliceScanner) {
                int numBlocks = Math.max(this.blockMap.size(), 1);
                long verifyInterval = Math.min(this.scanPeriod / (2L * (long)numBlocks), 600000L);
                long lastScanTime = Time.monotonicNow() - this.scanPeriod;
                if (this.blockInfoSet.isEmpty()) return true;
                while (true) {
                    BlockScanInfo info = this.blockInfoSet.first();
                    if (info.lastScanTime >= 0L) return true;
                    this.delBlockInfo(info);
                    info.lastScanTime = lastScanTime;
                    lastScanTime += verifyInterval;
                    this.addBlockInfo(info);
                }
            }
        }
        IOUtils.closeStream((Closeable)logIterator);
        return bl;
    }

    private synchronized void updateBytesLeft(long len) {
        this.bytesLeft += len;
    }

    private synchronized void startNewPeriod() {
        LOG.info((Object)("Starting a new period : work left in prev period : " + String.format("%.2f%%", this.totalBytesToScan == 0L ? 0.0 : (double)this.bytesLeft * 100.0 / (double)this.totalBytesToScan)));
        this.bytesLeft = this.totalBytesToScan;
        this.currentPeriodStart = Time.monotonicNow();
        this.isNewPeriod = true;
    }

    private synchronized boolean workRemainingInCurrentPeriod() {
        if (this.bytesLeft <= 0L && Time.monotonicNow() < this.currentPeriodStart + this.scanPeriod) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Skipping scan since bytesLeft=" + this.bytesLeft + ", Start=" + this.currentPeriodStart + ", period=" + this.scanPeriod + ", now=" + Time.monotonicNow() + " " + this.blockPoolId));
            }
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void scanBlockPoolSlice() {
        if (!this.workRemainingInCurrentPeriod()) {
            return;
        }
        this.processedBlocks = new HashMap();
        if (!this.assignInitialVerificationTimes()) {
            return;
        }
        try {
            this.scan();
        }
        finally {
            this.totalBlocksScannedInLastRun.set(this.processedBlocks.size());
            this.lastScanTime.set(Time.monotonicNow());
        }
    }

    void shutdown() {
        if (this.verificationLog != null) {
            this.verificationLog.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void scan() {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Starting to scan blockpool: " + this.blockPoolId));
        }
        try {
            this.adjustThrottler();
            while (this.datanode.shouldRun && !this.datanode.blockScanner.blockScannerThread.isInterrupted() && this.datanode.isBPServiceAlive(this.blockPoolId)) {
                long now = Time.monotonicNow();
                BlockPoolSliceScanner blockPoolSliceScanner = this;
                synchronized (blockPoolSliceScanner) {
                    if (now >= this.currentPeriodStart + this.scanPeriod) {
                        this.startNewPeriod();
                    }
                }
                if (now - this.getEarliestScanTime() >= this.scanPeriod || !this.blockInfoSet.isEmpty() && !this.isFirstBlockProcessed()) {
                    this.verifyFirstBlock();
                    continue;
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)"All remaining blocks were processed recently, so this run is complete");
                }
                break;
            }
        }
        catch (RuntimeException e) {
            LOG.warn((Object)"RuntimeException during BlockPoolScanner.scan()", (Throwable)e);
            throw e;
        }
        finally {
            this.rollVerificationLogs();
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Done scanning block pool: " + this.blockPoolId));
            }
        }
    }

    private synchronized void rollVerificationLogs() {
        if (this.verificationLog != null) {
            try {
                this.verificationLog.logs.roll();
            }
            catch (IOException ex) {
                LOG.warn((Object)"Received exception: ", (Throwable)ex);
                this.verificationLog.close();
            }
        }
    }

    synchronized void printBlockReport(StringBuilder buffer, boolean summaryOnly) {
        long oneHour = 3600000L;
        long oneDay = 24L * oneHour;
        long oneWeek = 7L * oneDay;
        long fourWeeks = 4L * oneWeek;
        int inOneHour = 0;
        int inOneDay = 0;
        int inOneWeek = 0;
        int inFourWeeks = 0;
        int inScanPeriod = 0;
        int neverScanned = 0;
        SimpleDateFormat dateFormat = new SimpleDateFormat(DATA_FORMAT);
        int total = this.blockInfoSet.size();
        long now = Time.monotonicNow();
        Date date = new Date();
        for (BlockScanInfo info : this.blockInfoSet) {
            long scanTime = info.getLastScanTime();
            long diff = now - scanTime;
            if (diff <= oneHour) {
                ++inOneHour;
            }
            if (diff <= oneDay) {
                ++inOneDay;
            }
            if (diff <= oneWeek) {
                ++inOneWeek;
            }
            if (diff <= fourWeeks) {
                ++inFourWeeks;
            }
            if (diff <= this.scanPeriod) {
                ++inScanPeriod;
            }
            if (scanTime <= 0L) {
                ++neverScanned;
            }
            if (summaryOnly) continue;
            date.setTime(scanTime);
            String scanType = info.lastScanType == ScanType.VERIFICATION_SCAN ? "local" : "none";
            buffer.append(String.format("%-26s : status : %-6s type : %-6s scan time : %-15d %s%n", info, info.lastScanOk ? "ok" : "failed", scanType, scanTime, scanTime <= 0L ? "not yet verified" : dateFormat.format(date)));
        }
        double pctPeriodLeft = (double)(this.scanPeriod + this.currentPeriodStart - now) * 100.0 / (double)this.scanPeriod;
        double pctProgress = this.totalBytesToScan == 0L ? 100.0 : (double)(this.totalBytesToScan - this.bytesLeft) * 100.0 / (double)this.totalBytesToScan;
        buffer.append(String.format("%nTotal Blocks                 : %6d%nVerified in last hour        : %6d%nVerified in last day         : %6d%nVerified in last week        : %6d%nVerified in last four weeks  : %6d%nVerified in SCAN_PERIOD      : %6d%nNot yet verified             : %6d%nVerified since restart       : %6d%nScans since restart          : %6d%nScan errors since restart    : %6d%nTransient scan errors        : %6d%nCurrent scan rate limit KBps : %6d%nProgress this period         : %6.0f%%%nTime left in cur period      : %6.2f%%%n", total, inOneHour, inOneDay, inOneWeek, inFourWeeks, inScanPeriod, neverScanned, this.totalScans, this.totalScans, this.totalScanErrors, this.totalTransientErrors, Math.round((double)this.throttler.getBandwidth() / 1024.0), pctProgress, pctPeriodLeft));
    }

    private static class LogFileHandler {
        private final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss,SSS");
        private final RollingLogs logs;

        private LogFileHandler(RollingLogs logs) {
            this.logs = logs;
        }

        void append(long verificationTime, long genStamp, long blockId) {
            String m = LogEntry.toString(verificationTime, genStamp, blockId, this.dateFormat);
            try {
                this.logs.appender().append(m);
            }
            catch (IOException e) {
                LOG.warn((Object)("Failed to append to " + this.logs + ", m=" + m), (Throwable)e);
            }
        }

        void close() {
            try {
                this.logs.appender().close();
            }
            catch (IOException e) {
                LOG.warn((Object)("Failed to close the appender of " + this.logs), (Throwable)e);
            }
        }
    }

    private static class LogEntry {
        long blockId = -1L;
        long verificationTime = -1L;
        long genStamp = 0L;
        private static final Pattern entryPattern = Pattern.compile("\\G\\s*([^=\\p{Space}]+)=\"(.*?)\"\\s*");

        private LogEntry() {
        }

        static String toString(long verificationTime, long genStamp, long blockId, DateFormat dateFormat) {
            return "\ndate=\"" + dateFormat.format(new Date(verificationTime)) + "\"\t time=\"" + verificationTime + "\"\t genstamp=\"" + genStamp + "\"\t id=\"" + blockId + "\"";
        }

        static LogEntry parseEntry(String line) {
            LogEntry entry = new LogEntry();
            Matcher matcher = entryPattern.matcher(line);
            while (matcher.find()) {
                String name = matcher.group(1);
                String value = matcher.group(2);
                try {
                    if (name.equals("id")) {
                        entry.blockId = Long.parseLong(value);
                        continue;
                    }
                    if (name.equals("time")) {
                        entry.verificationTime = Long.parseLong(value);
                        continue;
                    }
                    if (!name.equals("genstamp")) continue;
                    entry.genStamp = Long.parseLong(value);
                }
                catch (NumberFormatException nfe) {
                    LOG.warn((Object)("Cannot parse line: " + line), (Throwable)nfe);
                    return null;
                }
            }
            return entry;
        }
    }

    static class BlockScanInfo
    extends Block
    implements LightWeightGSet.LinkedElement {
        static final Comparator<BlockScanInfo> LAST_SCAN_TIME_COMPARATOR = new Comparator<BlockScanInfo>(){

            @Override
            public int compare(BlockScanInfo left, BlockScanInfo right) {
                long l = left.lastScanTime;
                long r = right.lastScanTime;
                return l < r ? -1 : (l > r ? 1 : left.compareTo(right));
            }
        };
        long lastScanTime = 0L;
        ScanType lastScanType = ScanType.NONE;
        boolean lastScanOk = true;
        private LightWeightGSet.LinkedElement next;

        BlockScanInfo(Block block) {
            super(block);
        }

        @Override
        public int hashCode() {
            return super.hashCode();
        }

        @Override
        public boolean equals(Object that) {
            if (this == that) {
                return true;
            }
            return super.equals(that);
        }

        long getLastScanTime() {
            return this.lastScanType == ScanType.NONE ? 0L : this.lastScanTime;
        }

        public void setNext(LightWeightGSet.LinkedElement next) {
            this.next = next;
        }

        public LightWeightGSet.LinkedElement getNext() {
            return this.next;
        }
    }

    private static enum ScanType {
        VERIFICATION_SCAN,
        NONE;

    }
}

