/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.contract;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.util.Random;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.contract.AbstractFSContractTestBase;
import org.apache.hadoop.fs.contract.ContractTestUtils;
import org.apache.hadoop.io.IOUtils;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractContractSeekTest
extends AbstractFSContractTestBase {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractContractSeekTest.class);
    public static final int DEFAULT_RANDOM_SEEK_COUNT = 100;
    private Path testPath;
    private Path smallSeekFile;
    private Path zeroByteFile;
    private FSDataInputStream instream;

    @Override
    public void setup() throws Exception {
        super.setup();
        this.skipIfUnsupported("supports-seek");
        this.testPath = this.getContract().getTestPath();
        this.smallSeekFile = this.path("seekfile.txt");
        this.zeroByteFile = this.path("zero.txt");
        byte[] block = ContractTestUtils.dataset(1024, 0, 255);
        ContractTestUtils.createFile(this.getFileSystem(), this.smallSeekFile, false, block);
        ContractTestUtils.touch(this.getFileSystem(), this.zeroByteFile);
    }

    @Override
    protected Configuration createConfiguration() {
        Configuration conf = super.createConfiguration();
        conf.setInt("io.file.buffer.size", 4096);
        return conf;
    }

    @Override
    public void teardown() throws Exception {
        IOUtils.closeStream((Closeable)this.instream);
        this.instream = null;
        super.teardown();
    }

    @Test
    public void testSeekZeroByteFile() throws Throwable {
        this.describe("seek and read a 0 byte file");
        this.instream = this.getFileSystem().open(this.zeroByteFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        int result = this.instream.read();
        this.assertMinusOne("initial byte read", result);
        byte[] buffer = new byte[1];
        this.instream.seek(0L);
        result = this.instream.read();
        this.assertMinusOne("post-seek byte read", result);
        result = this.instream.read(buffer, 0, 1);
        this.assertMinusOne("post-seek buffer read", result);
    }

    @Test
    public void testBlockReadZeroByteFile() throws Throwable {
        this.describe("do a block read on a 0 byte file");
        this.instream = this.getFileSystem().open(this.zeroByteFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        byte[] buffer = new byte[1];
        int result = this.instream.read(buffer, 0, 1);
        this.assertMinusOne("block read zero byte file", result);
    }

    @Test
    public void testSeekReadClosedFile() throws Throwable {
        boolean supportsSeekOnClosedFiles = this.isSupported("supports-seek-on-closed-file");
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        AbstractContractSeekTest.getLog().debug("Stream is of type " + this.instream.getClass().getCanonicalName());
        this.instream.close();
        try {
            this.instream.seek(0L);
            if (!supportsSeekOnClosedFiles) {
                AbstractContractSeekTest.fail((String)"seek succeeded on a closed stream");
            }
        }
        catch (IOException e) {
            // empty catch block
        }
        try {
            int data = this.instream.available();
            AbstractContractSeekTest.fail((String)("read() succeeded on a closed stream, got " + data));
        }
        catch (IOException e) {
            // empty catch block
        }
        try {
            int data = this.instream.read();
            AbstractContractSeekTest.fail((String)("read() succeeded on a closed stream, got " + data));
        }
        catch (IOException e) {
            // empty catch block
        }
        try {
            byte[] buffer = new byte[1];
            int result = this.instream.read(buffer, 0, 1);
            AbstractContractSeekTest.fail((String)("read(buffer, 0, 1) succeeded on a closed stream, got " + result));
        }
        catch (IOException e) {
            // empty catch block
        }
        try {
            long offset = this.instream.getPos();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.instream.close();
    }

    @Test
    public void testNegativeSeek() throws Throwable {
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        try {
            this.instream.seek(-1L);
            long p = this.instream.getPos();
            LOG.warn("Seek to -1 returned a position of " + p);
            int result = this.instream.read();
            AbstractContractSeekTest.fail((String)("expected an exception, got data " + result + " at a position of " + p));
        }
        catch (EOFException e) {
            this.handleExpectedException(e);
        }
        catch (IOException e) {
            this.handleRelaxedException("a negative seek", "EOFException", e);
        }
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
    }

    @Test
    public void testSeekFile() throws Throwable {
        this.describe("basic seek operations");
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        this.instream.seek(0L);
        int result = this.instream.read();
        AbstractContractSeekTest.assertEquals((long)0L, (long)result);
        AbstractContractSeekTest.assertEquals((long)1L, (long)this.instream.read());
        AbstractContractSeekTest.assertEquals((long)2L, (long)this.instream.getPos());
        AbstractContractSeekTest.assertEquals((long)2L, (long)this.instream.read());
        AbstractContractSeekTest.assertEquals((long)3L, (long)this.instream.getPos());
        this.instream.seek(128L);
        AbstractContractSeekTest.assertEquals((long)128L, (long)this.instream.getPos());
        AbstractContractSeekTest.assertEquals((long)128L, (long)this.instream.read());
        this.instream.seek(63L);
        AbstractContractSeekTest.assertEquals((long)63L, (long)this.instream.read());
    }

    @Test
    public void testSeekAndReadPastEndOfFile() throws Throwable {
        this.describe("verify that reading past the last bytes in the file returns -1");
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        this.instream.seek(1022L);
        AbstractContractSeekTest.assertTrue((String)"Premature EOF", (this.instream.read() != -1 ? 1 : 0) != 0);
        AbstractContractSeekTest.assertTrue((String)"Premature EOF", (this.instream.read() != -1 ? 1 : 0) != 0);
        this.assertMinusOne("read past end of file", this.instream.read());
    }

    @Test
    public void testSeekPastEndOfFileThenReseekAndRead() throws Throwable {
        this.describe("do a seek past the EOF, then verify the stream recovers");
        this.instream = this.getFileSystem().open(this.smallSeekFile);
        boolean canSeekPastEOF = !this.getContract().isSupported("rejects-seek-past-eof", true);
        try {
            this.instream.seek(1025L);
            this.assertMinusOne("read after seeking past EOF", this.instream.read());
        }
        catch (EOFException e) {
            if (canSeekPastEOF) {
                throw e;
            }
            this.handleExpectedException(e);
        }
        catch (IOException e) {
            if (canSeekPastEOF) {
                throw e;
            }
            this.handleRelaxedException("a seek past the end of the file", "EOFException", e);
        }
        this.instream.seek(1L);
        AbstractContractSeekTest.assertTrue((String)"Premature EOF", (this.instream.read() != -1 ? 1 : 0) != 0);
    }

    @Test
    public void testSeekBigFile() throws Throwable {
        this.describe("Seek round a large file and verify the bytes are what is expected");
        Path testSeekFile = this.path("bigseekfile.txt");
        byte[] block = ContractTestUtils.dataset(65536, 0, 255);
        ContractTestUtils.createFile(this.getFileSystem(), testSeekFile, false, block);
        this.instream = this.getFileSystem().open(testSeekFile);
        AbstractContractSeekTest.assertEquals((long)0L, (long)this.instream.getPos());
        this.instream.seek(0L);
        int result = this.instream.read();
        AbstractContractSeekTest.assertEquals((long)0L, (long)result);
        AbstractContractSeekTest.assertEquals((long)1L, (long)this.instream.read());
        AbstractContractSeekTest.assertEquals((long)2L, (long)this.instream.read());
        this.instream.seek(32768L);
        AbstractContractSeekTest.assertEquals((String)"@32768", (long)block[32768], (long)((byte)this.instream.read()));
        this.instream.seek(40000L);
        AbstractContractSeekTest.assertEquals((String)"@40000", (long)block[40000], (long)((byte)this.instream.read()));
        this.instream.seek(8191L);
        AbstractContractSeekTest.assertEquals((String)"@8191", (long)block[8191], (long)((byte)this.instream.read()));
        this.instream.seek(0L);
        AbstractContractSeekTest.assertEquals((String)"@0", (long)0L, (long)((byte)this.instream.read()));
    }

    @Test
    public void testPositionedBulkReadDoesntChangePosition() throws Throwable {
        this.describe("verify that a positioned read does not change the getPos() value");
        Path testSeekFile = this.path("bigseekfile.txt");
        byte[] block = ContractTestUtils.dataset(65536, 0, 255);
        ContractTestUtils.createFile(this.getFileSystem(), testSeekFile, false, block);
        this.instream = this.getFileSystem().open(testSeekFile);
        this.instream.seek(39999L);
        AbstractContractSeekTest.assertTrue((-1 != this.instream.read() ? 1 : 0) != 0);
        AbstractContractSeekTest.assertEquals((long)40000L, (long)this.instream.getPos());
        byte[] readBuffer = new byte[256];
        this.instream.read(128L, readBuffer, 0, readBuffer.length);
        AbstractContractSeekTest.assertEquals((long)40000L, (long)this.instream.getPos());
        AbstractContractSeekTest.assertEquals((String)"@40000", (long)block[40000], (long)((byte)this.instream.read()));
        for (int i = 0; i < 256; ++i) {
            AbstractContractSeekTest.assertEquals((String)("@" + i), (long)block[i + 128], (long)readBuffer[i]);
        }
    }

    @Test
    public void testRandomSeeks() throws Throwable {
        int limit = this.getContract().getLimit("test.random-seek-count", 100);
        this.describe("Testing " + limit + " random seeks");
        int filesize = 10240;
        byte[] buf = ContractTestUtils.dataset(filesize, 0, 255);
        Path randomSeekFile = this.path("testrandomseeks.bin");
        ContractTestUtils.createFile(this.getFileSystem(), randomSeekFile, false, buf);
        Random r = new Random();
        FSDataInputStream stm = this.getFileSystem().open(randomSeekFile);
        int[] seeks = new int[10];
        int[] reads = new int[10];
        try {
            for (int i = 0; i < limit; ++i) {
                int seekOff = r.nextInt(buf.length);
                int toRead = r.nextInt(Math.min(buf.length - seekOff, 32000));
                seeks[i % seeks.length] = seekOff;
                reads[i % reads.length] = toRead;
                ContractTestUtils.verifyRead(stm, buf, seekOff, toRead);
            }
        }
        catch (AssertionError afe) {
            StringBuilder sb = new StringBuilder();
            sb.append("Sequence of actions:\n");
            for (int j = 0; j < seeks.length; ++j) {
                sb.append("seek @ ").append(seeks[j]).append("  ").append("read ").append(reads[j]).append("\n");
            }
            LOG.error(sb.toString());
            throw afe;
        }
        finally {
            stm.close();
        }
    }
}

