/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.sink.hive;

import java.io.IOException;
import java.util.ArrayList;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.flume.Event;
import org.apache.flume.instrumentation.SinkCounter;
import org.apache.flume.sink.hive.HiveEventSerializer;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hive.hcatalog.streaming.HiveEndPoint;
import org.apache.hive.hcatalog.streaming.RecordWriter;
import org.apache.hive.hcatalog.streaming.SerializationError;
import org.apache.hive.hcatalog.streaming.StreamingConnection;
import org.apache.hive.hcatalog.streaming.StreamingException;
import org.apache.hive.hcatalog.streaming.StreamingIOFailure;
import org.apache.hive.hcatalog.streaming.TransactionBatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class HiveWriter {
    private static final Logger LOG = LoggerFactory.getLogger(HiveWriter.class);
    private final HiveEndPoint endPoint;
    private HiveEventSerializer serializer;
    private final StreamingConnection connection;
    private final int txnsPerBatch;
    private final RecordWriter recordWriter;
    private TransactionBatch txnBatch;
    private final ExecutorService callTimeoutPool;
    private final long callTimeout;
    private long lastUsed;
    private SinkCounter sinkCounter;
    private int batchCounter;
    private long eventCounter;
    private long processSize;
    protected boolean closed;
    private boolean autoCreatePartitions;
    private boolean hearbeatNeeded = false;
    private UserGroupInformation ugi;
    private final int writeBatchSz = 1000;
    private ArrayList<Event> batch = new ArrayList(1000);

    HiveWriter(HiveEndPoint endPoint, int txnsPerBatch, boolean autoCreatePartitions, long callTimeout, ExecutorService callTimeoutPool, UserGroupInformation ugi, HiveEventSerializer serializer, SinkCounter sinkCounter) throws ConnectException, InterruptedException {
        try {
            this.autoCreatePartitions = autoCreatePartitions;
            this.sinkCounter = sinkCounter;
            this.callTimeout = callTimeout;
            this.callTimeoutPool = callTimeoutPool;
            this.endPoint = endPoint;
            this.ugi = ugi;
            this.connection = this.newConnection(ugi);
            this.txnsPerBatch = txnsPerBatch;
            this.serializer = serializer;
            this.recordWriter = serializer.createRecordWriter(endPoint);
            this.txnBatch = this.nextTxnBatch(this.recordWriter);
            this.closed = false;
            this.lastUsed = System.currentTimeMillis();
        }
        catch (InterruptedException e) {
            throw e;
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new ConnectException(endPoint, (Throwable)e);
        }
    }

    public String toString() {
        return this.endPoint.toString();
    }

    private void resetCounters() {
        this.eventCounter = 0L;
        this.processSize = 0L;
        this.batchCounter = 0;
    }

    void setHearbeatNeeded() {
        this.hearbeatNeeded = true;
    }

    public synchronized void write(Event event) throws WriteException, InterruptedException {
        if (this.closed) {
            throw new IllegalStateException("Writer closed. Cannot write to : " + this.endPoint);
        }
        this.batch.add(event);
        if (this.batch.size() == 1000) {
            this.writeEventBatchToSerializer();
        }
        this.processSize += (long)event.getBody().length;
        ++this.eventCounter;
    }

    private void writeEventBatchToSerializer() throws InterruptedException, WriteException {
        try {
            this.timedCall(new CallRunner1<Void>(){

                @Override
                public Void call() throws InterruptedException, StreamingException {
                    try {
                        for (Event event : HiveWriter.this.batch) {
                            try {
                                HiveWriter.this.serializer.write(HiveWriter.this.txnBatch, event);
                            }
                            catch (SerializationError err) {
                                LOG.info("Parse failed : {}  : {}", (Object)err.getMessage(), (Object)new String(event.getBody()));
                            }
                        }
                        return null;
                    }
                    catch (IOException e) {
                        throw new StreamingIOFailure(e.getMessage(), (Exception)e);
                    }
                }
            });
            this.batch.clear();
        }
        catch (StreamingException e) {
            throw new WriteException(this.endPoint, this.txnBatch.getCurrentTxnId(), e);
        }
        catch (TimeoutException e) {
            throw new WriteException(this.endPoint, this.txnBatch.getCurrentTxnId(), e);
        }
    }

    public void flush(boolean rollToNext) throws CommitException, TxnBatchException, TxnFailure, InterruptedException, WriteException {
        if (!this.batch.isEmpty()) {
            this.writeEventBatchToSerializer();
            this.batch.clear();
        }
        if (this.hearbeatNeeded) {
            this.hearbeatNeeded = false;
            this.heartBeat();
        }
        this.lastUsed = System.currentTimeMillis();
        try {
            this.commitTxn();
            if (this.txnBatch.remainingTransactions() == 0) {
                this.closeTxnBatch();
                this.txnBatch = null;
                if (rollToNext) {
                    this.txnBatch = this.nextTxnBatch(this.recordWriter);
                }
            }
            if (rollToNext) {
                LOG.debug("Switching to next Txn for {}", (Object)this.endPoint);
                this.txnBatch.beginNextTransaction();
            }
        }
        catch (StreamingException e) {
            throw new TxnFailure(this.txnBatch, e);
        }
    }

    public void abort() throws InterruptedException {
        this.batch.clear();
        this.abortTxn();
    }

    public void heartBeat() throws InterruptedException {
        try {
            this.timedCall(new CallRunner1<Void>(){

                @Override
                public Void call() throws StreamingException {
                    LOG.info("Sending heartbeat on batch " + HiveWriter.this.txnBatch);
                    HiveWriter.this.txnBatch.heartbeat();
                    return null;
                }
            });
        }
        catch (InterruptedException e) {
            throw e;
        }
        catch (Exception e) {
            LOG.warn("Unable to send heartbeat on Txn Batch " + this.txnBatch, (Throwable)e);
        }
    }

    public void close() throws InterruptedException {
        this.batch.clear();
        this.closeTxnBatch();
        this.closeConnection();
        this.closed = true;
    }

    public void closeConnection() throws InterruptedException {
        LOG.info("Closing connection to EndPoint : {}", (Object)this.endPoint);
        try {
            this.timedCall(new CallRunner1<Void>(){

                @Override
                public Void call() {
                    HiveWriter.this.connection.close();
                    return null;
                }
            });
            this.sinkCounter.incrementConnectionClosedCount();
        }
        catch (Exception e) {
            LOG.warn("Error closing connection to EndPoint : " + this.endPoint, (Throwable)e);
        }
    }

    private void commitTxn() throws CommitException, InterruptedException {
        if (LOG.isInfoEnabled()) {
            LOG.info("Committing Txn " + this.txnBatch.getCurrentTxnId() + " on EndPoint: " + this.endPoint);
        }
        try {
            this.timedCall(new CallRunner1<Void>(){

                @Override
                public Void call() throws StreamingException, InterruptedException {
                    HiveWriter.this.txnBatch.commit();
                    return null;
                }
            });
        }
        catch (Exception e) {
            throw new CommitException(this.endPoint, this.txnBatch.getCurrentTxnId(), e);
        }
    }

    private void abortTxn() throws InterruptedException {
        LOG.info("Aborting Txn id {} on End Point {}", (Object)this.txnBatch.getCurrentTxnId(), (Object)this.endPoint);
        try {
            this.timedCall(new CallRunner1<Void>(){

                @Override
                public Void call() throws StreamingException, InterruptedException {
                    HiveWriter.this.txnBatch.abort();
                    return null;
                }
            });
        }
        catch (InterruptedException e) {
            throw e;
        }
        catch (TimeoutException e) {
            LOG.warn("Timeout while aborting Txn " + this.txnBatch.getCurrentTxnId() + " on EndPoint: " + this.endPoint, (Throwable)e);
        }
        catch (Exception e) {
            LOG.warn("Error aborting Txn " + this.txnBatch.getCurrentTxnId() + " on EndPoint: " + this.endPoint, (Throwable)e);
        }
    }

    private StreamingConnection newConnection(final UserGroupInformation ugi) throws InterruptedException, ConnectException {
        try {
            return this.timedCall(new CallRunner1<StreamingConnection>(){

                @Override
                public StreamingConnection call() throws InterruptedException, StreamingException {
                    return HiveWriter.this.endPoint.newConnection(HiveWriter.this.autoCreatePartitions, null, ugi);
                }
            });
        }
        catch (Exception e) {
            throw new ConnectException(this.endPoint, (Throwable)e);
        }
    }

    private TransactionBatch nextTxnBatch(final RecordWriter recordWriter) throws InterruptedException, TxnBatchException {
        LOG.debug("Fetching new Txn Batch for {}", (Object)this.endPoint);
        TransactionBatch batch = null;
        try {
            batch = this.timedCall(new CallRunner1<TransactionBatch>(){

                @Override
                public TransactionBatch call() throws InterruptedException, StreamingException {
                    return HiveWriter.this.connection.fetchTransactionBatch(HiveWriter.this.txnsPerBatch, recordWriter);
                }
            });
            LOG.info("Acquired Txn Batch {}. Switching to first txn", (Object)batch);
            batch.beginNextTransaction();
        }
        catch (Exception e) {
            throw new TxnBatchException(this.endPoint, (Throwable)e);
        }
        return batch;
    }

    private void closeTxnBatch() throws InterruptedException {
        try {
            LOG.debug("Closing Txn Batch {}", (Object)this.txnBatch);
            this.timedCall(new CallRunner1<Void>(){

                @Override
                public Void call() throws InterruptedException, StreamingException {
                    HiveWriter.this.txnBatch.close();
                    return null;
                }
            });
        }
        catch (InterruptedException e) {
            throw e;
        }
        catch (Exception e) {
            LOG.warn("Error closing Txn Batch " + this.txnBatch, (Throwable)e);
        }
    }

    private <T> T timedCall(final CallRunner1<T> callRunner) throws TimeoutException, InterruptedException, StreamingException {
        Future future = this.callTimeoutPool.submit(new Callable<T>(){

            @Override
            public T call() throws StreamingException, InterruptedException, Failure {
                return callRunner.call();
            }
        });
        try {
            if (this.callTimeout > 0L) {
                return future.get(this.callTimeout, TimeUnit.MILLISECONDS);
            }
            return future.get();
        }
        catch (TimeoutException eT) {
            future.cancel(true);
            this.sinkCounter.incrementConnectionFailedCount();
            throw eT;
        }
        catch (ExecutionException e1) {
            this.sinkCounter.incrementConnectionFailedCount();
            Throwable cause = e1.getCause();
            if (cause instanceof IOException) {
                throw new StreamingException("I/O Failure", (Exception)((IOException)cause));
            }
            if (cause instanceof StreamingException) {
                throw (StreamingException)cause;
            }
            if (cause instanceof TimeoutException) {
                throw new StreamingException("Operation Timed Out.", (Exception)((TimeoutException)cause));
            }
            if (cause instanceof RuntimeException) {
                throw (RuntimeException)cause;
            }
            if (cause instanceof InterruptedException) {
                throw (InterruptedException)cause;
            }
            throw new StreamingException(e1.getMessage(), (Exception)e1);
        }
    }

    long getLastUsed() {
        return this.lastUsed;
    }

    private class TxnFailure
    extends Failure {
        public TxnFailure(TransactionBatch txnBatch, Throwable cause) {
            super("Failed switching to next Txn in TxnBatch " + txnBatch, cause);
        }
    }

    public static class TxnBatchException
    extends Failure {
        public TxnBatchException(HiveEndPoint ep, Throwable cause) {
            super("Failed acquiring Transaction Batch from EndPoint: " + ep, cause);
        }
    }

    public static class ConnectException
    extends Failure {
        public ConnectException(HiveEndPoint ep, Throwable cause) {
            super("Failed connecting to EndPoint " + ep, cause);
        }
    }

    public static class CommitException
    extends Failure {
        public CommitException(HiveEndPoint endPoint, Long txnID, Throwable cause) {
            super("Commit of Txn " + txnID + " failed on EndPoint: " + endPoint, cause);
        }
    }

    public static class WriteException
    extends Failure {
        public WriteException(HiveEndPoint endPoint, Long currentTxnId, Throwable cause) {
            super("Failed writing to : " + endPoint + ". TxnID : " + currentTxnId, cause);
        }
    }

    public static class Failure
    extends Exception {
        public Failure(String msg, Throwable cause) {
            super(msg, cause);
        }
    }

    private static interface CallRunner1<T> {
        public T call() throws StreamingException, InterruptedException, Failure;
    }

    private static interface CallRunner<T> {
        public T call() throws Exception;
    }
}

