/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.client.avro;

import com.google.common.base.Charsets;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.io.Files;
import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Pattern;
import org.apache.flume.Context;
import org.apache.flume.Event;
import org.apache.flume.FlumeException;
import org.apache.flume.annotations.InterfaceAudience;
import org.apache.flume.annotations.InterfaceStability;
import org.apache.flume.client.avro.ReliableEventReader;
import org.apache.flume.serialization.DurablePositionTracker;
import org.apache.flume.serialization.EventDeserializer;
import org.apache.flume.serialization.EventDeserializerFactory;
import org.apache.flume.serialization.ResettableFileInputStream;
import org.apache.flume.tools.PlatformDetect;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class ReliableSpoolingFileEventReader
implements ReliableEventReader {
    private static final Logger logger = LoggerFactory.getLogger(ReliableSpoolingFileEventReader.class);
    static final String metaFileName = ".flumespool-main.meta";
    private final File spoolDirectory;
    private final String completedSuffix;
    private final String deserializerType;
    private final Context deserializerContext;
    private final Pattern ignorePattern;
    private final File metaFile;
    private final boolean annotateFileName;
    private final String fileNameHeader;
    private final String deletePolicy;
    private final Charset inputCharset;
    private Optional<FileInfo> currentFile = Optional.absent();
    private Optional<FileInfo> lastFileRead = Optional.absent();
    private boolean committed = true;

    private ReliableSpoolingFileEventReader(File spoolDirectory, String completedSuffix, String ignorePattern, String trackerDirPath, boolean annotateFileName, String fileNameHeader, String deserializerType, Context deserializerContext, String deletePolicy, String inputCharset) throws IOException {
        Preconditions.checkNotNull((Object)spoolDirectory);
        Preconditions.checkNotNull((Object)completedSuffix);
        Preconditions.checkNotNull((Object)ignorePattern);
        Preconditions.checkNotNull((Object)trackerDirPath);
        Preconditions.checkNotNull((Object)deserializerType);
        Preconditions.checkNotNull((Object)deserializerContext);
        Preconditions.checkNotNull((Object)deletePolicy);
        Preconditions.checkNotNull((Object)inputCharset);
        if (!deletePolicy.equalsIgnoreCase(DeletePolicy.NEVER.name()) && !deletePolicy.equalsIgnoreCase(DeletePolicy.IMMEDIATE.name())) {
            throw new IllegalArgumentException("Delete policies other than NEVER and IMMEDIATE are not yet supported");
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Initializing {} with directory={}, metaDir={}, deserializer={}", new Object[]{ReliableSpoolingFileEventReader.class.getSimpleName(), spoolDirectory, trackerDirPath, deserializerType});
        }
        Preconditions.checkState((boolean)spoolDirectory.exists(), (Object)("Directory does not exist: " + spoolDirectory.getAbsolutePath()));
        Preconditions.checkState((boolean)spoolDirectory.isDirectory(), (Object)("Path is not a directory: " + spoolDirectory.getAbsolutePath()));
        try {
            File f1 = File.createTempFile("flume", "test", spoolDirectory);
            Files.write((CharSequence)"testing flume file permissions\n", (File)f1, (Charset)Charsets.UTF_8);
            Files.readLines((File)f1, (Charset)Charsets.UTF_8);
            if (!f1.delete()) {
                throw new IOException("Unable to delete canary file " + f1);
            }
        }
        catch (IOException e) {
            throw new FlumeException("Unable to read and modify files in the spooling directory: " + spoolDirectory, (Throwable)e);
        }
        this.spoolDirectory = spoolDirectory;
        this.completedSuffix = completedSuffix;
        this.deserializerType = deserializerType;
        this.deserializerContext = deserializerContext;
        this.annotateFileName = annotateFileName;
        this.fileNameHeader = fileNameHeader;
        this.ignorePattern = Pattern.compile(ignorePattern);
        this.deletePolicy = deletePolicy;
        this.inputCharset = Charset.forName(inputCharset);
        File trackerDirectory = new File(trackerDirPath);
        if (!trackerDirectory.isAbsolute()) {
            trackerDirectory = new File(spoolDirectory, trackerDirPath);
        }
        if (!trackerDirectory.exists() && !trackerDirectory.mkdir()) {
            throw new IOException("Unable to mkdir nonexistent meta directory " + trackerDirectory);
        }
        if (!trackerDirectory.isDirectory()) {
            throw new IOException("Specified meta directory is not a directory" + trackerDirectory);
        }
        this.metaFile = new File(trackerDirectory, metaFileName);
    }

    public String getLastFileRead() {
        if (!this.lastFileRead.isPresent()) {
            return null;
        }
        return ((FileInfo)this.lastFileRead.get()).getFile().getAbsolutePath();
    }

    @Override
    public Event readEvent() throws IOException {
        List<Event> events = this.readEvents(1);
        if (!events.isEmpty()) {
            return events.get(0);
        }
        return null;
    }

    @Override
    public List<Event> readEvents(int numEvents) throws IOException {
        if (!this.committed) {
            if (!this.currentFile.isPresent()) {
                throw new IllegalStateException("File should not roll when commit is outstanding.");
            }
            logger.info("Last read was never committed - resetting mark position.");
            ((FileInfo)this.currentFile.get()).getDeserializer().reset();
        } else {
            if (!this.currentFile.isPresent()) {
                this.currentFile = this.getNextFile();
            }
            if (!this.currentFile.isPresent()) {
                return Collections.emptyList();
            }
        }
        EventDeserializer des = ((FileInfo)this.currentFile.get()).getDeserializer();
        List<Event> events = des.readEvents(numEvents);
        if (events.isEmpty()) {
            this.retireCurrentFile();
            this.currentFile = this.getNextFile();
            if (!this.currentFile.isPresent()) {
                return Collections.emptyList();
            }
            events = ((FileInfo)this.currentFile.get()).getDeserializer().readEvents(numEvents);
        }
        if (this.annotateFileName) {
            String filename = ((FileInfo)this.currentFile.get()).getFile().getAbsolutePath();
            for (Event event : events) {
                event.getHeaders().put(this.fileNameHeader, filename);
            }
        }
        this.committed = false;
        this.lastFileRead = this.currentFile;
        return events;
    }

    @Override
    public void close() throws IOException {
        if (this.currentFile.isPresent()) {
            ((FileInfo)this.currentFile.get()).getDeserializer().close();
            this.currentFile = Optional.absent();
        }
    }

    @Override
    public void commit() throws IOException {
        if (!this.committed && this.currentFile.isPresent()) {
            ((FileInfo)this.currentFile.get()).getDeserializer().mark();
            this.committed = true;
        }
    }

    private void retireCurrentFile() throws IOException {
        Preconditions.checkState((boolean)this.currentFile.isPresent());
        File fileToRoll = new File(((FileInfo)this.currentFile.get()).getFile().getAbsolutePath());
        ((FileInfo)this.currentFile.get()).getDeserializer().close();
        if (fileToRoll.lastModified() != ((FileInfo)this.currentFile.get()).getLastModified()) {
            String message = "File has been modified since being read: " + fileToRoll;
            throw new IllegalStateException(message);
        }
        if (fileToRoll.length() != ((FileInfo)this.currentFile.get()).getLength()) {
            String message = "File has changed size since being read: " + fileToRoll;
            throw new IllegalStateException(message);
        }
        if (this.deletePolicy.equalsIgnoreCase(DeletePolicy.NEVER.name())) {
            this.rollCurrentFile(fileToRoll);
        } else if (this.deletePolicy.equalsIgnoreCase(DeletePolicy.IMMEDIATE.name())) {
            this.deleteCurrentFile(fileToRoll);
        } else {
            throw new IllegalArgumentException("Unsupported delete policy: " + this.deletePolicy);
        }
    }

    /*
     * Enabled aggressive block sorting
     */
    private void rollCurrentFile(File fileToRoll) throws IOException {
        File dest = new File(fileToRoll.getPath() + this.completedSuffix);
        logger.info("Preparing to move file {} to {}", (Object)fileToRoll, (Object)dest);
        if (dest.exists() && PlatformDetect.isWindows()) {
            if (Files.equal((File)((FileInfo)this.currentFile.get()).getFile(), (File)dest)) {
                logger.warn("Completed file " + dest + " already exists, but files match, so continuing.");
                boolean deleted = fileToRoll.delete();
                if (deleted) return;
                logger.error("Unable to delete file " + fileToRoll.getAbsolutePath() + ". It will likely be ingested another time.");
                return;
            }
            String message = "File name has been re-used with different files. Spooling assumptions violated for " + dest;
            throw new IllegalStateException(message);
        }
        if (dest.exists()) {
            String message = "File name has been re-used with different files. Spooling assumptions violated for " + dest;
            throw new IllegalStateException(message);
        }
        boolean renamed = fileToRoll.renameTo(dest);
        if (renamed) {
            logger.debug("Successfully rolled file {} to {}", (Object)fileToRoll, (Object)dest);
            this.deleteMetaFile();
            return;
        }
        String message = "Unable to move " + fileToRoll + " to " + dest + ". This will likely cause duplicate events. Please verify that " + "flume has sufficient permissions to perform these operations.";
        throw new FlumeException(message);
    }

    private void deleteCurrentFile(File fileToDelete) throws IOException {
        logger.info("Preparing to delete file {}", (Object)fileToDelete);
        if (!fileToDelete.exists()) {
            logger.warn("Unable to delete nonexistent file: {}", (Object)fileToDelete);
            return;
        }
        if (!fileToDelete.delete()) {
            throw new IOException("Unable to delete spool file: " + fileToDelete);
        }
        this.deleteMetaFile();
    }

    private Optional<FileInfo> getNextFile() {
        FileFilter filter = new FileFilter(){

            @Override
            public boolean accept(File candidate) {
                String fileName = candidate.getName();
                return !candidate.isDirectory() && !fileName.endsWith(ReliableSpoolingFileEventReader.this.completedSuffix) && !fileName.startsWith(".") && !ReliableSpoolingFileEventReader.this.ignorePattern.matcher(fileName).matches();
            }
        };
        List<File> candidateFiles = Arrays.asList(this.spoolDirectory.listFiles(filter));
        if (candidateFiles.isEmpty()) {
            return Optional.absent();
        }
        Collections.sort(candidateFiles, new Comparator<File>(){

            @Override
            public int compare(File a, File b) {
                int timeComparison = new Long(a.lastModified()).compareTo(new Long(b.lastModified()));
                if (timeComparison != 0) {
                    return timeComparison;
                }
                return a.getName().compareTo(b.getName());
            }
        });
        File nextFile = candidateFiles.get(0);
        try {
            String nextPath = nextFile.getPath();
            DurablePositionTracker tracker = DurablePositionTracker.getInstance(this.metaFile, nextPath);
            if (!tracker.getTarget().equals(nextPath)) {
                tracker.close();
                this.deleteMetaFile();
                tracker = DurablePositionTracker.getInstance(this.metaFile, nextPath);
            }
            Preconditions.checkState((boolean)tracker.getTarget().equals(nextPath), (String)"Tracker target %s does not equal expected filename %s", (Object[])new Object[]{tracker.getTarget(), nextPath});
            ResettableFileInputStream in = new ResettableFileInputStream(nextFile, tracker, 16384, this.inputCharset);
            EventDeserializer deserializer = EventDeserializerFactory.getInstance(this.deserializerType, this.deserializerContext, in);
            return Optional.of((Object)new FileInfo(nextFile, deserializer));
        }
        catch (FileNotFoundException e) {
            logger.warn("Could not find file: " + nextFile, (Throwable)e);
            return Optional.absent();
        }
        catch (IOException e) {
            logger.error("Exception opening file: " + nextFile, (Throwable)e);
            return Optional.absent();
        }
    }

    private void deleteMetaFile() throws IOException {
        if (this.metaFile.exists() && !this.metaFile.delete()) {
            throw new IOException("Unable to delete old meta file " + this.metaFile);
        }
    }

    public static class Builder {
        private File spoolDirectory;
        private String completedSuffix = "fileSuffix";
        private String ignorePattern = "^$";
        private String trackerDirPath = ".flumespool";
        private Boolean annotateFileName = false;
        private String fileNameHeader = "file";
        private String deserializerType = "LINE";
        private Context deserializerContext = new Context();
        private String deletePolicy = "never";
        private String inputCharset = "UTF-8";

        public Builder spoolDirectory(File directory) {
            this.spoolDirectory = directory;
            return this;
        }

        public Builder completedSuffix(String completedSuffix) {
            this.completedSuffix = completedSuffix;
            return this;
        }

        public Builder ignorePattern(String ignorePattern) {
            this.ignorePattern = ignorePattern;
            return this;
        }

        public Builder trackerDirPath(String trackerDirPath) {
            this.trackerDirPath = trackerDirPath;
            return this;
        }

        public Builder annotateFileName(Boolean annotateFileName) {
            this.annotateFileName = annotateFileName;
            return this;
        }

        public Builder fileNameHeader(String fileNameHeader) {
            this.fileNameHeader = fileNameHeader;
            return this;
        }

        public Builder deserializerType(String deserializerType) {
            this.deserializerType = deserializerType;
            return this;
        }

        public Builder deserializerContext(Context deserializerContext) {
            this.deserializerContext = deserializerContext;
            return this;
        }

        public Builder deletePolicy(String deletePolicy) {
            this.deletePolicy = deletePolicy;
            return this;
        }

        public Builder inputCharset(String inputCharset) {
            this.inputCharset = inputCharset;
            return this;
        }

        public ReliableSpoolingFileEventReader build() throws IOException {
            return new ReliableSpoolingFileEventReader(this.spoolDirectory, this.completedSuffix, this.ignorePattern, this.trackerDirPath, this.annotateFileName, this.fileNameHeader, this.deserializerType, this.deserializerContext, this.deletePolicy, this.inputCharset);
        }
    }

    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    static enum DeletePolicy {
        NEVER,
        IMMEDIATE,
        DELAY;

    }

    private static class FileInfo {
        private final File file;
        private final long length;
        private final long lastModified;
        private final EventDeserializer deserializer;

        public FileInfo(File file, EventDeserializer deserializer) {
            this.file = file;
            this.length = file.length();
            this.lastModified = file.lastModified();
            this.deserializer = deserializer;
        }

        public long getLength() {
            return this.length;
        }

        public long getLastModified() {
            return this.lastModified;
        }

        public EventDeserializer getDeserializer() {
            return this.deserializer;
        }

        public File getFile() {
            return this.file;
        }
    }
}

