/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.channel.file;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nullable;
import org.apache.flume.channel.file.FlumeEvent;
import org.apache.flume.channel.file.FlumeEventPointer;
import org.apache.flume.channel.file.LogFileFactory;
import org.apache.flume.channel.file.LogFileRetryableIOException;
import org.apache.flume.channel.file.LogRecord;
import org.apache.flume.channel.file.Pair;
import org.apache.flume.channel.file.Put;
import org.apache.flume.channel.file.Take;
import org.apache.flume.channel.file.TransactionEventRecord;
import org.apache.flume.channel.file.encryption.CipherProvider;
import org.apache.flume.channel.file.encryption.KeyProvider;
import org.apache.flume.tools.DirectMemoryUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class LogFile {
    private static final Logger LOG = LoggerFactory.getLogger(LogFile.class);
    private static final ByteBuffer FILL = DirectMemoryUtils.allocate((int)0x100000);
    protected static final byte OP_RECORD = 127;
    protected static final byte OP_EOF = -128;

    LogFile() {
    }

    protected static void writeDelimitedBuffer(ByteBuffer output, ByteBuffer buffer) throws IOException {
        output.putInt(buffer.limit());
        output.put(buffer);
    }

    protected static byte[] readDelimitedBuffer(RandomAccessFile fileHandle) throws IOException {
        int length = fileHandle.readInt();
        Preconditions.checkState((length >= 0 ? 1 : 0) != 0, (Object)Integer.toHexString(length));
        byte[] buffer = new byte[length];
        fileHandle.readFully(buffer);
        return buffer;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] args) throws EOFException, IOException {
        File file = new File(args[0]);
        SequentialReader reader = null;
        try {
            LogRecord entry;
            reader = LogFileFactory.getSequentialReader(file, null);
            int fileId = reader.getLogFileID();
            int count = 0;
            int readCount = 0;
            int putCount = 0;
            int takeCount = 0;
            int rollbackCount = 0;
            int commitCount = 0;
            while ((entry = reader.next()) != null) {
                int offset = entry.getOffset();
                TransactionEventRecord record = entry.getEvent();
                short type = record.getRecordType();
                long trans = record.getTransactionID();
                long ts = record.getLogWriteOrderID();
                ++readCount;
                FlumeEventPointer ptr = null;
                if (type == TransactionEventRecord.Type.PUT.get()) {
                    ++putCount;
                    ptr = new FlumeEventPointer(fileId, offset);
                } else if (type == TransactionEventRecord.Type.TAKE.get()) {
                    ++takeCount;
                    Take take = (Take)record;
                    ptr = new FlumeEventPointer(take.getFileID(), take.getOffset());
                } else if (type == TransactionEventRecord.Type.ROLLBACK.get()) {
                    ++rollbackCount;
                } else if (type == TransactionEventRecord.Type.COMMIT.get()) {
                    ++commitCount;
                } else {
                    Preconditions.checkArgument((boolean)false, (Object)("Unknown record type: " + Integer.toHexString(type)));
                }
                System.out.println(Joiner.on((String)", ").skipNulls().join((Object)trans, (Object)ts, new Object[]{fileId, offset, TransactionEventRecord.getName(type), ptr}));
            }
            System.out.println("Replayed " + count + " from " + file + " read: " + readCount + ", put: " + putCount + ", take: " + takeCount + ", rollback: " + rollbackCount + ", commit: " + commitCount);
        }
        catch (EOFException e) {
            System.out.println("Hit EOF on " + file);
        }
        finally {
            if (reader != null) {
                reader.close();
            }
        }
    }

    static {
        for (int i = 0; i < FILL.capacity(); ++i) {
            FILL.put((byte)-128);
        }
    }

    static abstract class SequentialReader {
        private final RandomAccessFile fileHandle;
        private final FileChannel fileChannel;
        private final File file;
        private final KeyProvider encryptionKeyProvider;
        private int logFileID;
        private long lastCheckpointPosition;
        private long lastCheckpointWriteOrderID;

        SequentialReader(File file, @Nullable KeyProvider encryptionKeyProvider) throws IOException, EOFException {
            this.file = file;
            this.encryptionKeyProvider = encryptionKeyProvider;
            this.fileHandle = new RandomAccessFile(file, "r");
            this.fileChannel = this.fileHandle.getChannel();
        }

        abstract LogRecord doNext(int var1) throws IOException;

        abstract int getVersion();

        protected void setLastCheckpointPosition(long lastCheckpointPosition) {
            this.lastCheckpointPosition = lastCheckpointPosition;
        }

        protected void setLastCheckpointWriteOrderID(long lastCheckpointWriteOrderID) {
            this.lastCheckpointWriteOrderID = lastCheckpointWriteOrderID;
        }

        protected void setLogFileID(int logFileID) {
            this.logFileID = logFileID;
            Preconditions.checkArgument((logFileID >= 0 ? 1 : 0) != 0, (Object)("LogFileID is not positive: " + Integer.toHexString(logFileID)));
        }

        protected KeyProvider getKeyProvider() {
            return this.encryptionKeyProvider;
        }

        protected RandomAccessFile getFileHandle() {
            return this.fileHandle;
        }

        int getLogFileID() {
            return this.logFileID;
        }

        void skipToLastCheckpointPosition(long checkpointWriteOrderID) throws IOException {
            if (this.lastCheckpointPosition > 0L && this.lastCheckpointWriteOrderID <= checkpointWriteOrderID) {
                LOG.info("fast-forward to checkpoint position: " + this.lastCheckpointPosition);
                this.fileChannel.position(this.lastCheckpointPosition);
            } else {
                LOG.warn("Checkpoint for file(" + this.file.getAbsolutePath() + ") " + "is: " + this.lastCheckpointWriteOrderID + ", which is beyond the " + "requested checkpoint time: " + checkpointWriteOrderID + " and position " + this.lastCheckpointPosition);
            }
        }

        LogRecord next() throws IOException {
            int offset = -1;
            try {
                long position = this.fileChannel.position();
                if (position > 1623195647L) {
                    LOG.info("File position exceeds the threshold: 1623195647, position: " + position);
                }
                Preconditions.checkState(((offset = (int)position) >= 0 ? 1 : 0) != 0);
                byte operation = this.fileHandle.readByte();
                if (operation != 127) {
                    if (operation == -128) {
                        LOG.info("Encountered EOF at " + offset + " in " + this.file);
                    } else {
                        LOG.error("Encountered non op-record at " + offset + " " + Integer.toHexString(operation) + " in " + this.file);
                    }
                    return null;
                }
                return this.doNext(offset);
            }
            catch (EOFException e) {
                return null;
            }
            catch (IOException e) {
                throw new IOException("Unable to read next Transaction from log file " + this.file.getCanonicalPath() + " at offset " + offset, e);
            }
        }

        void close() {
            if (this.fileHandle != null) {
                try {
                    this.fileHandle.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
        }
    }

    static abstract class RandomReader {
        private final File file;
        private final BlockingQueue<RandomAccessFile> readFileHandles = new ArrayBlockingQueue<RandomAccessFile>(50, true);
        private final KeyProvider encryptionKeyProvider;
        private volatile boolean open;

        public RandomReader(File file, @Nullable KeyProvider encryptionKeyProvider) throws IOException {
            this.file = file;
            this.encryptionKeyProvider = encryptionKeyProvider;
            this.readFileHandles.add(this.open());
            this.open = true;
        }

        protected abstract TransactionEventRecord doGet(RandomAccessFile var1) throws IOException;

        abstract int getVersion();

        File getFile() {
            return this.file;
        }

        protected KeyProvider getKeyProvider() {
            return this.encryptionKeyProvider;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        FlumeEvent get(int offset) throws IOException, InterruptedException {
            Preconditions.checkState((boolean)this.open, (Object)"File closed");
            RandomAccessFile fileHandle = this.checkOut();
            boolean error = true;
            try {
                fileHandle.seek(offset);
                byte operation = fileHandle.readByte();
                Preconditions.checkState((operation == 127 ? 1 : 0) != 0, (Object)Integer.toHexString(operation));
                TransactionEventRecord record = this.doGet(fileHandle);
                if (!(record instanceof Put)) {
                    Preconditions.checkState((boolean)false, (Object)("Record is " + record.getClass().getSimpleName()));
                }
                error = false;
                FlumeEvent flumeEvent = ((Put)record).getEvent();
                return flumeEvent;
            }
            finally {
                if (error) {
                    RandomReader.close(fileHandle, this.file);
                } else {
                    this.checkIn(fileHandle);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        synchronized void close() {
            if (this.open) {
                this.open = false;
                LOG.info("Closing RandomReader " + this.file);
                ArrayList fileHandles = Lists.newArrayList();
                while (this.readFileHandles.drainTo(fileHandles) > 0) {
                    Iterator i$ = fileHandles.iterator();
                    while (i$.hasNext()) {
                        RandomAccessFile fileHandle;
                        RandomAccessFile randomAccessFile = fileHandle = (RandomAccessFile)i$.next();
                        synchronized (randomAccessFile) {
                            try {
                                fileHandle.close();
                            }
                            catch (IOException e) {
                                LOG.warn("Unable to close fileHandle for " + this.file, (Throwable)e);
                            }
                        }
                    }
                    fileHandles.clear();
                    try {
                        Thread.sleep(5L);
                    }
                    catch (InterruptedException interruptedException) {}
                }
            }
        }

        private RandomAccessFile open() throws IOException {
            return new RandomAccessFile(this.file, "r");
        }

        private void checkIn(RandomAccessFile fileHandle) {
            if (!this.readFileHandles.offer(fileHandle)) {
                RandomReader.close(fileHandle, this.file);
            }
        }

        private RandomAccessFile checkOut() throws IOException, InterruptedException {
            RandomAccessFile fileHandle = (RandomAccessFile)this.readFileHandles.poll();
            if (fileHandle != null) {
                return fileHandle;
            }
            int remaining = this.readFileHandles.remainingCapacity();
            if (remaining > 0) {
                LOG.info("Opening " + this.file + " for read, remaining capacity is " + remaining);
                return this.open();
            }
            return this.readFileHandles.take();
        }

        private static void close(RandomAccessFile fileHandle, File file) {
            if (fileHandle != null) {
                try {
                    fileHandle.close();
                }
                catch (IOException e) {
                    LOG.warn("Unable to close " + file, (Throwable)e);
                }
            }
        }
    }

    static abstract class Writer {
        private final int logFileID;
        private final File file;
        private final long maxFileSize;
        private final RandomAccessFile writeFileHandle;
        private final FileChannel writeFileChannel;
        private final CipherProvider.Encryptor encryptor;
        private final CachedFSUsableSpace usableSpace;
        private volatile boolean open;

        Writer(File file, int logFileID, long maxFileSize, CipherProvider.Encryptor encryptor, long usableSpaceRefreshInterval) throws IOException {
            this.file = file;
            this.logFileID = logFileID;
            this.maxFileSize = Math.min(maxFileSize, 1623195647L);
            this.encryptor = encryptor;
            this.writeFileHandle = new RandomAccessFile(file, "rw");
            this.writeFileChannel = this.writeFileHandle.getChannel();
            this.usableSpace = new CachedFSUsableSpace(file, usableSpaceRefreshInterval);
            LOG.info("Opened " + file);
            this.open = true;
        }

        abstract int getVersion();

        protected CipherProvider.Encryptor getEncryptor() {
            return this.encryptor;
        }

        int getLogFileID() {
            return this.logFileID;
        }

        File getFile() {
            return this.file;
        }

        String getParent() {
            return this.file.getParent();
        }

        long getUsableSpace() {
            return this.usableSpace.getUsableSpace();
        }

        long getMaxSize() {
            return this.maxFileSize;
        }

        synchronized long position() throws IOException {
            return this.getFileChannel().position();
        }

        synchronized FlumeEventPointer put(ByteBuffer buffer) throws IOException {
            if (this.encryptor != null) {
                buffer = ByteBuffer.wrap(this.encryptor.encrypt(buffer.array()));
            }
            Pair<Integer, Integer> pair = this.write(buffer);
            return new FlumeEventPointer(pair.getLeft(), pair.getRight());
        }

        synchronized void take(ByteBuffer buffer) throws IOException {
            if (this.encryptor != null) {
                buffer = ByteBuffer.wrap(this.encryptor.encrypt(buffer.array()));
            }
            this.write(buffer);
        }

        synchronized void rollback(ByteBuffer buffer) throws IOException {
            if (this.encryptor != null) {
                buffer = ByteBuffer.wrap(this.encryptor.encrypt(buffer.array()));
            }
            this.write(buffer);
        }

        synchronized void commit(ByteBuffer buffer) throws IOException {
            if (this.encryptor != null) {
                buffer = ByteBuffer.wrap(this.encryptor.encrypt(buffer.array()));
            }
            this.write(buffer);
            this.sync();
        }

        private Pair<Integer, Integer> write(ByteBuffer buffer) throws IOException {
            if (!this.isOpen()) {
                throw new LogFileRetryableIOException("File closed " + this.file);
            }
            long length = this.position();
            long expectedLength = length + (long)buffer.limit();
            if (expectedLength > this.maxFileSize) {
                throw new LogFileRetryableIOException(expectedLength + " > " + this.maxFileSize);
            }
            int offset = (int)length;
            Preconditions.checkState((offset >= 0 ? 1 : 0) != 0, (Object)String.valueOf(offset));
            int recordLength = 5 + buffer.limit();
            this.usableSpace.decrement(recordLength);
            this.preallocate(recordLength);
            ByteBuffer toWrite = ByteBuffer.allocate(recordLength);
            toWrite.put((byte)127);
            LogFile.writeDelimitedBuffer(toWrite, buffer);
            toWrite.position(0);
            int wrote = this.getFileChannel().write(toWrite);
            Preconditions.checkState((wrote == toWrite.limit() ? 1 : 0) != 0);
            return Pair.of(this.getLogFileID(), offset);
        }

        synchronized boolean isRollRequired(ByteBuffer buffer) throws IOException {
            return this.isOpen() && this.position() + (long)buffer.limit() > this.getMaxSize();
        }

        private void sync() throws IOException {
            if (!this.isOpen()) {
                throw new LogFileRetryableIOException("File closed " + this.file);
            }
            this.getFileChannel().force(false);
        }

        protected boolean isOpen() {
            return this.open;
        }

        protected RandomAccessFile getFileHandle() {
            return this.writeFileHandle;
        }

        protected FileChannel getFileChannel() {
            return this.writeFileChannel;
        }

        synchronized void close() {
            if (this.open) {
                this.open = false;
                if (this.writeFileChannel.isOpen()) {
                    LOG.info("Closing " + this.file);
                    try {
                        this.writeFileChannel.force(true);
                    }
                    catch (IOException e) {
                        LOG.warn("Unable to flush to disk " + this.file, (Throwable)e);
                    }
                    try {
                        this.writeFileHandle.close();
                    }
                    catch (IOException e) {
                        LOG.warn("Unable to close " + this.file, (Throwable)e);
                    }
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void preallocate(int size) throws IOException {
            long position = this.position();
            if (position + (long)size > this.getFileChannel().size()) {
                LOG.debug("Preallocating at position " + position);
                ByteBuffer byteBuffer = FILL;
                synchronized (byteBuffer) {
                    FILL.position(0);
                    this.getFileChannel().write(FILL, position);
                }
            }
        }
    }

    @VisibleForTesting
    static class CachedFSUsableSpace {
        private final File fs;
        private final long interval;
        private final AtomicLong lastRefresh;
        private final AtomicLong value;

        CachedFSUsableSpace(File fs, long interval) {
            this.fs = fs;
            this.interval = interval;
            this.value = new AtomicLong(fs.getUsableSpace());
            this.lastRefresh = new AtomicLong(System.currentTimeMillis());
        }

        void decrement(long numBytes) {
            Preconditions.checkArgument((numBytes >= 0L ? 1 : 0) != 0, (Object)"numBytes less than zero");
            this.value.addAndGet(-numBytes);
        }

        long getUsableSpace() {
            long now = System.currentTimeMillis();
            if (now - this.interval > this.lastRefresh.get()) {
                this.value.set(this.fs.getUsableSpace());
                this.lastRefresh.set(now);
            }
            return Math.max(this.value.get(), 0L);
        }
    }

    static abstract class MetaDataWriter {
        private final File file;
        private final int logFileID;
        private final RandomAccessFile writeFileHandle;
        private long lastCheckpointOffset;
        private long lastCheckpointWriteOrderID;

        protected MetaDataWriter(File file, int logFileID) throws IOException {
            this.file = file;
            this.logFileID = logFileID;
            this.writeFileHandle = new RandomAccessFile(file, "rw");
        }

        protected RandomAccessFile getFileHandle() {
            return this.writeFileHandle;
        }

        protected void setLastCheckpointOffset(long lastCheckpointOffset) {
            this.lastCheckpointOffset = lastCheckpointOffset;
        }

        protected void setLastCheckpointWriteOrderID(long lastCheckpointWriteOrderID) {
            this.lastCheckpointWriteOrderID = lastCheckpointWriteOrderID;
        }

        protected long getLastCheckpointOffset() {
            return this.lastCheckpointOffset;
        }

        protected long getLastCheckpointWriteOrderID() {
            return this.lastCheckpointWriteOrderID;
        }

        protected File getFile() {
            return this.file;
        }

        protected int getLogFileID() {
            return this.logFileID;
        }

        void markCheckpoint(long logWriteOrderID) throws IOException {
            this.markCheckpoint(this.lastCheckpointOffset, logWriteOrderID);
        }

        abstract void markCheckpoint(long var1, long var3) throws IOException;

        abstract int getVersion();

        void close() {
            try {
                this.writeFileHandle.close();
            }
            catch (IOException e) {
                LOG.warn("Unable to close " + this.file, (Throwable)e);
            }
        }
    }
}

