/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.source;

import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.collect.Maps;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.regex.Pattern;
import org.apache.flume.Event;
import org.apache.flume.event.EventBuilder;
import org.joda.time.DateTime;
import org.joda.time.ReadableInstant;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SyslogParser {
    private static final Logger logger = LoggerFactory.getLogger(SyslogParser.class);
    private static final int TS_CACHE_MAX = 1000;
    private static final Pattern TWO_SPACES = Pattern.compile("  ");
    private static final DateTimeFormatter rfc3164Format = DateTimeFormat.forPattern((String)"MMM d HH:mm:ss").withZoneUTC();
    private static final String timePat = "yyyy-MM-dd'T'HH:mm:ss";
    private static final int RFC3164_LEN = 15;
    private static final int RFC5424_PREFIX_LEN = 19;
    private final DateTimeFormatter timeParser = DateTimeFormat.forPattern((String)"yyyy-MM-dd'T'HH:mm:ss").withZoneUTC();
    private Cache<String, Long> timestampCache = CacheBuilder.newBuilder().maximumSize(1000).build((CacheLoader)new CacheLoader<String, Long>(){

        public Long load(String key) throws Exception {
            return SyslogParser.this.timeParser.parseMillis(key);
        }
    });

    public Event parseMessage(String msg, Charset charset) {
        int nextSpace;
        long ts;
        HashMap headers = Maps.newHashMap();
        int msgLen = msg.length();
        int curPos = 0;
        Preconditions.checkArgument((msg.charAt(curPos) == '<' ? 1 : 0) != 0, (String)"Bad format: invalid priority: cannot find open bracket '<' (%s)", (Object[])new Object[]{msg});
        int endBracketPos = msg.indexOf(62);
        Preconditions.checkArgument((endBracketPos > 0 && endBracketPos <= 6 ? 1 : 0) != 0, (String)"Bad format: invalid priority: cannot find end bracket '>' (%s)", (Object[])new Object[]{msg});
        String priority = msg.substring(1, endBracketPos);
        int pri = Integer.parseInt(priority);
        int facility = pri / 8;
        int severity = pri % 8;
        headers.put("Facility", String.valueOf(facility));
        headers.put("Severity", String.valueOf(severity));
        Preconditions.checkArgument((msgLen > endBracketPos + 1 ? 1 : 0) != 0, (String)"Bad format: no data except priority (%s)", (Object[])new Object[]{msg});
        curPos = endBracketPos + 1;
        if (msgLen > curPos + 2 && "1 ".equals(msg.substring(curPos, curPos + 2))) {
            curPos += 2;
        }
        char dateStartChar = msg.charAt(curPos);
        try {
            if (dateStartChar == '-') {
                ts = System.currentTimeMillis();
                if (msgLen <= curPos + 2) {
                    throw new IllegalArgumentException("bad syslog format (missing hostname)");
                }
                curPos += 2;
            } else if (dateStartChar >= 'A' && dateStartChar <= 'Z') {
                if (msgLen <= curPos + 15) {
                    throw new IllegalArgumentException("bad timestamp format");
                }
                ts = this.parseRfc3164Time(msg.substring(curPos, curPos + 15));
                curPos += 16;
            } else {
                nextSpace = msg.indexOf(32, curPos);
                if (nextSpace == -1) {
                    throw new IllegalArgumentException("bad timestamp format");
                }
                ts = this.parseRfc5424Date(msg.substring(curPos, nextSpace));
                curPos = nextSpace + 1;
            }
        }
        catch (IllegalArgumentException ex) {
            throw new IllegalArgumentException("Unable to parse message: " + msg, ex);
        }
        headers.put("timestamp", String.valueOf(ts));
        nextSpace = msg.indexOf(32, curPos);
        if (nextSpace == -1) {
            throw new IllegalArgumentException("bad syslog format (missing hostname)");
        }
        String hostname = new String(msg.substring(curPos, nextSpace));
        headers.put("host", hostname);
        String data = "";
        if (msgLen > nextSpace + 1) {
            curPos = nextSpace + 1;
            data = msg.substring(curPos);
        }
        Event event = EventBuilder.withBody((String)data, (Charset)charset, (Map)headers);
        return event;
    }

    protected long parseRfc5424Date(String msg) {
        char tzFirst;
        Long ts = null;
        int curPos = 0;
        int msgLen = msg.length();
        Preconditions.checkArgument((msgLen > 19 ? 1 : 0) != 0, (String)"Bad format: Not a valid RFC5424 timestamp: %s", (Object[])new Object[]{msg});
        String timestampPrefix = msg.substring(curPos, 19);
        try {
            ts = (Long)this.timestampCache.get((Object)timestampPrefix);
        }
        catch (ExecutionException ex) {
            throw new IllegalArgumentException("bad timestamp format", ex);
        }
        Preconditions.checkArgument((ts != null ? 1 : 0) != 0, (Object)"Parsing error: timestamp is null");
        if (msg.charAt(curPos += 19) == '.') {
            boolean foundEnd = false;
            int endMillisPos = curPos + 1;
            if (msgLen <= endMillisPos) {
                throw new IllegalArgumentException("bad timestamp format (no TZ)");
            }
            while (!foundEnd) {
                char curDigit = msg.charAt(endMillisPos);
                if (curDigit >= '0' && curDigit <= '9') {
                    ++endMillisPos;
                    continue;
                }
                foundEnd = true;
            }
            if (endMillisPos - (curPos + 1) <= 0) {
                throw new IllegalArgumentException("Bad format: Invalid timestamp (fractional portion): " + msg);
            }
            float frac = Float.parseFloat(msg.substring(curPos, endMillisPos));
            long milliseconds = (long)(frac * 1000.0f);
            ts = ts + milliseconds;
            curPos = endMillisPos;
        }
        if ((tzFirst = msg.charAt(curPos)) != 'Z' && (tzFirst == '+' || tzFirst == '-')) {
            Preconditions.checkArgument((msgLen > curPos + 5 ? 1 : 0) != 0, (String)"Bad format: Invalid timezone (%s)", (Object[])new Object[]{msg});
            int polarity = tzFirst == '+' ? 1 : -1;
            char[] h = new char[5];
            for (int i = 0; i < 5; ++i) {
                h[i] = msg.charAt(curPos + 1 + i);
            }
            if (h[0] >= '0' && h[0] <= '9' && h[1] >= '0' && h[1] <= '9' && h[2] == ':' && h[3] >= '0' && h[3] <= '9' && h[4] >= '0' && h[4] <= '9') {
                int hourOffset = Integer.parseInt(msg.substring(curPos + 1, curPos + 3));
                int minOffset = Integer.parseInt(msg.substring(curPos + 4, curPos + 6));
                ts = ts - (long)(polarity * (hourOffset * 60 + minOffset) * 60000);
            } else {
                throw new IllegalArgumentException("Bad format: Invalid timezone: " + msg);
            }
        }
        return ts;
    }

    protected long parseRfc3164Time(String ts) {
        DateTime date;
        DateTime now = DateTime.now();
        int year = now.getYear();
        ts = TWO_SPACES.matcher(ts).replaceFirst(" ");
        try {
            date = rfc3164Format.parseDateTime(ts);
        }
        catch (IllegalArgumentException e) {
            logger.debug("rfc3164 date parse failed on (" + ts + "): invalid format", (Throwable)e);
            return 0L;
        }
        if (date != null) {
            DateTime fixed = date.withYear(year);
            if (fixed.isAfter((ReadableInstant)now) && fixed.minusMonths(1).isAfter((ReadableInstant)now)) {
                fixed = date.withYear(year - 1);
            } else if (fixed.isBefore((ReadableInstant)now) && fixed.plusMonths(1).isBefore((ReadableInstant)now)) {
                fixed = date.withYear(year + 1);
            }
            date = fixed;
        }
        if (date == null) {
            return 0L;
        }
        return date.getMillis();
    }
}

