/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.web.security.x509.ocsp;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import jakarta.ws.rs.ProcessingException;
import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.Entity;
import jakarta.ws.rs.client.WebTarget;
import jakarta.ws.rs.core.Response;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.net.URI;
import java.security.KeyStore;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;
import javax.security.auth.x500.X500Principal;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.security.util.KeyStoreUtils;
import org.apache.nifi.security.util.SslContextFactory;
import org.apache.nifi.security.util.StandardTlsConfiguration;
import org.apache.nifi.security.util.TlsConfiguration;
import org.apache.nifi.util.FormatUtils;
import org.apache.nifi.util.NiFiProperties;
import org.apache.nifi.web.security.x509.ocsp.CertificateStatusException;
import org.apache.nifi.web.security.x509.ocsp.OcspRequest;
import org.apache.nifi.web.security.x509.ocsp.OcspStatus;
import org.apache.nifi.web.util.WebUtils;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.Extensions;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cert.ocsp.CertificateID;
import org.bouncycastle.cert.ocsp.CertificateStatus;
import org.bouncycastle.cert.ocsp.OCSPException;
import org.bouncycastle.cert.ocsp.OCSPReq;
import org.bouncycastle.cert.ocsp.OCSPReqBuilder;
import org.bouncycastle.cert.ocsp.OCSPResp;
import org.bouncycastle.cert.ocsp.RevokedStatus;
import org.bouncycastle.cert.ocsp.SingleResp;
import org.bouncycastle.operator.DigestCalculatorProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;
import org.bouncycastle.operator.jcajce.JcaDigestCalculatorProviderBuilder;
import org.glassfish.jersey.client.ClientConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OcspCertificateValidator {
    private static final Logger logger = LoggerFactory.getLogger(OcspCertificateValidator.class);
    private static final String HTTPS = "https";
    private static final String OCSP_REQUEST_CONTENT_TYPE = "application/ocsp-request";
    private static final int CONNECT_TIMEOUT = 10000;
    private static final int READ_TIMEOUT = 10000;
    private URI validationAuthorityURI;
    private Client client;
    private Map<String, X509Certificate> trustedCAs;
    private LoadingCache<OcspRequest, OcspStatus> ocspCache;

    public OcspCertificateValidator(NiFiProperties properties) {
        String rawValidationAuthorityUrl = properties.getProperty("nifi.security.ocsp.responder.url");
        if (StringUtils.isNotBlank((CharSequence)rawValidationAuthorityUrl)) {
            try {
                this.validationAuthorityURI = URI.create(rawValidationAuthorityUrl);
                ClientConfig clientConfig = new ClientConfig();
                clientConfig.property("jersey.config.client.readTimeout", (Object)10000);
                clientConfig.property("jersey.config.client.connectTimeout", (Object)10000);
                if (HTTPS.equalsIgnoreCase(this.validationAuthorityURI.getScheme())) {
                    TlsConfiguration tlsConfiguration = StandardTlsConfiguration.fromNiFiProperties((NiFiProperties)properties);
                    this.client = WebUtils.createClient((ClientConfig)clientConfig, (SSLContext)SslContextFactory.createSslContext((TlsConfiguration)tlsConfiguration));
                } else {
                    this.client = WebUtils.createClient((ClientConfig)clientConfig);
                }
                this.trustedCAs = this.getTrustedCAs(properties);
                X509Certificate ocspCertificate = this.getOcspCertificate(properties);
                if (ocspCertificate != null) {
                    this.trustedCAs.put(ocspCertificate.getSubjectX500Principal().getName(), ocspCertificate);
                }
                long cacheDurationMillis = FormatUtils.getTimeDuration((String)"12 hours", (TimeUnit)TimeUnit.MILLISECONDS);
                this.ocspCache = Caffeine.newBuilder().expireAfterWrite(cacheDurationMillis, TimeUnit.MILLISECONDS).build(ocspRequest -> {
                    String subjectDn = ocspRequest.getSubjectCertificate().getSubjectX500Principal().getName();
                    logger.info(String.format("Validating client certificate via OCSP: <%s>", subjectDn));
                    OcspStatus ocspStatus = this.getOcspStatus((OcspRequest)ocspRequest);
                    logger.info(String.format("Client certificate status for <%s>: %s", subjectDn, ocspStatus.toString()));
                    return ocspStatus;
                });
            }
            catch (Exception e) {
                logger.error("Disabling OCSP certificate validation. Unable to load OCSP configuration: " + e, (Throwable)e);
                this.client = null;
            }
        }
    }

    private X509Certificate getOcspCertificate(NiFiProperties properties) {
        X509Certificate validationAuthorityCertificate = null;
        String validationAuthorityCertificatePath = properties.getProperty("nifi.security.ocsp.responder.certificate");
        if (StringUtils.isNotBlank((CharSequence)validationAuthorityCertificatePath)) {
            try (FileInputStream fis = new FileInputStream(validationAuthorityCertificatePath);){
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                validationAuthorityCertificate = (X509Certificate)cf.generateCertificate(fis);
            }
            catch (Exception e) {
                throw new IllegalStateException("Unable to load the validation authority certificate: " + e);
            }
        }
        return validationAuthorityCertificate;
    }

    private Map<String, X509Certificate> getTrustedCAs(NiFiProperties properties) {
        HashMap<String, X509Certificate> certificateAuthorities = new HashMap<String, X509Certificate>();
        String truststorePath = properties.getProperty("nifi.security.truststore");
        if (truststorePath == null) {
            throw new IllegalArgumentException("The truststore path is required.");
        }
        String rawTruststorePassword = properties.getProperty("nifi.security.truststorePasswd");
        char[] truststorePassword = rawTruststorePassword == null ? new char[]{} : rawTruststorePassword.toCharArray();
        try (FileInputStream fis = new FileInputStream(truststorePath);){
            KeyStore truststore = KeyStoreUtils.getKeyStore((String)KeyStore.getDefaultType());
            truststore.load(fis, truststorePassword);
            TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init(truststore);
            for (TrustManager trustManager : trustManagerFactory.getTrustManagers()) {
                if (!(trustManager instanceof X509TrustManager)) continue;
                for (X509Certificate ca : ((X509TrustManager)trustManager).getAcceptedIssuers()) {
                    certificateAuthorities.put(ca.getSubjectX500Principal().getName(), ca);
                }
            }
        }
        catch (Exception e) {
            throw new IllegalStateException("Unable to load the configured truststore: " + e);
        }
        return certificateAuthorities;
    }

    public void validate(X509Certificate[] certificates) throws CertificateStatusException {
        if (this.client != null && certificates != null && certificates.length > 0) {
            X509Certificate subjectCertificate = this.getSubjectCertificate(certificates);
            X509Certificate issuerCertificate = this.getIssuerCertificate(certificates);
            if (issuerCertificate == null) {
                throw new IllegalArgumentException(String.format("Unable to obtain certificate of issuer <%s> for the specified subject certificate <%s>.", subjectCertificate.getIssuerX500Principal().getName(), subjectCertificate.getSubjectX500Principal().getName()));
            }
            OcspRequest ocspRequest = new OcspRequest(subjectCertificate, issuerCertificate);
            OcspStatus ocspStatus = (OcspStatus)this.ocspCache.get((Object)ocspRequest);
            if (OcspStatus.VerificationStatus.Verified.equals((Object)ocspStatus.getVerificationStatus()) && OcspStatus.ValidationStatus.Revoked.equals((Object)ocspStatus.getValidationStatus())) {
                throw new CertificateStatusException(String.format("Client certificate for <%s> is revoked according to the certificate authority.", subjectCertificate.getSubjectX500Principal().getName()));
            }
        }
    }

    private X509Certificate getSubjectCertificate(X509Certificate[] certificates) {
        return certificates[0];
    }

    private X509Certificate getIssuerCertificate(X509Certificate[] certificates) {
        if (certificates.length > 1) {
            return certificates[1];
        }
        if (certificates.length == 1) {
            X509Certificate subjectCertificate = this.getSubjectCertificate(certificates);
            X500Principal issuerPrincipal = subjectCertificate.getIssuerX500Principal();
            return this.trustedCAs.get(issuerPrincipal.getName());
        }
        return null;
    }

    private OcspStatus getOcspStatus(OcspRequest ocspStatusKey) {
        X509Certificate subjectCertificate = ocspStatusKey.getSubjectCertificate();
        X509Certificate issuerCertificate = ocspStatusKey.getIssuerCertificate();
        OcspStatus ocspStatus = new OcspStatus();
        ocspStatus.setVerificationStatus(OcspStatus.VerificationStatus.Unknown);
        ocspStatus.setValidationStatus(OcspStatus.ValidationStatus.Unknown);
        try {
            SingleResp[] responses;
            BigInteger subjectSerialNumber = subjectCertificate.getSerialNumber();
            DigestCalculatorProvider calculatorProviderBuilder = new JcaDigestCalculatorProviderBuilder().setProvider("BC").build();
            CertificateID certificateId = new CertificateID(calculatorProviderBuilder.get(CertificateID.HASH_SHA1), new X509CertificateHolder(issuerCertificate.getEncoded()), subjectSerialNumber);
            OCSPReqBuilder requestGenerator = new OCSPReqBuilder();
            requestGenerator.addRequest(certificateId);
            BigInteger nonce = BigInteger.valueOf(System.currentTimeMillis());
            Extension ext = new Extension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce, true, (ASN1OctetString)new DEROctetString(nonce.toByteArray()));
            requestGenerator.setRequestExtensions(new Extensions(new Extension[]{ext}));
            OCSPReq ocspRequest = requestGenerator.build();
            Response response = this.getClientResponse(ocspRequest);
            if (Response.Status.OK.getStatusCode() != response.getStatusInfo().getStatusCode()) {
                logger.warn(String.format("OCSP request was unsuccessful (%s).", response.getStatus()));
                return ocspStatus;
            }
            OCSPResp ocspResponse = new OCSPResp((InputStream)response.readEntity(InputStream.class));
            switch (ocspResponse.getStatus()) {
                case 0: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.Successful);
                    break;
                }
                case 2: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.InternalError);
                    break;
                }
                case 1: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.MalformedRequest);
                    break;
                }
                case 5: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.SignatureRequired);
                    break;
                }
                case 3: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.TryLater);
                    break;
                }
                case 6: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.Unauthorized);
                    break;
                }
                default: {
                    ocspStatus.setResponseStatus(OcspStatus.ResponseStatus.Unknown);
                }
            }
            if (ocspResponse.getStatus() != 0) {
                logger.warn(String.format("OCSP request was unsuccessful (%s).", ocspStatus.getResponseStatus().toString()));
                return ocspStatus;
            }
            Object ocspResponseObject = ocspResponse.getResponseObject();
            if (!(ocspResponseObject instanceof BasicOCSPResp)) {
                logger.warn(String.format("Unexpected OCSP response object: %s", ocspResponseObject));
                return ocspStatus;
            }
            BasicOCSPResp basicOcspResponse = (BasicOCSPResp)ocspResponse.getResponseObject();
            X509CertificateHolder[] responderCertificates = basicOcspResponse.getCerts();
            if (responderCertificates.length != 1) {
                logger.warn(String.format("Unexpected number of OCSP responder certificates: %s", responderCertificates.length));
                return ocspStatus;
            }
            X509Certificate trustedResponderCertificate = this.getTrustedResponderCertificate(responderCertificates[0], issuerCertificate);
            if (trustedResponderCertificate != null) {
                if (basicOcspResponse.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider("BC").build(trustedResponderCertificate.getPublicKey()))) {
                    ocspStatus.setVerificationStatus(OcspStatus.VerificationStatus.Verified);
                } else {
                    ocspStatus.setVerificationStatus(OcspStatus.VerificationStatus.Unverified);
                }
            } else {
                ocspStatus.setVerificationStatus(OcspStatus.VerificationStatus.Unverified);
            }
            for (SingleResp singleResponse : responses = basicOcspResponse.getResponses()) {
                CertificateID responseCertificateId = singleResponse.getCertID();
                BigInteger responseSerialNumber = responseCertificateId.getSerialNumber();
                if (!responseSerialNumber.equals(subjectSerialNumber)) continue;
                CertificateStatus certStatus = singleResponse.getCertStatus();
                if (CertificateStatus.GOOD == certStatus) {
                    ocspStatus.setValidationStatus(OcspStatus.ValidationStatus.Good);
                    continue;
                }
                if (certStatus instanceof RevokedStatus) {
                    ocspStatus.setValidationStatus(OcspStatus.ValidationStatus.Revoked);
                    continue;
                }
                ocspStatus.setValidationStatus(OcspStatus.ValidationStatus.Unknown);
            }
        }
        catch (ProcessingException | IOException | OCSPException | OperatorCreationException e) {
            logger.error(e.getMessage(), e);
        }
        catch (CertificateException e) {
            e.printStackTrace();
        }
        return ocspStatus;
    }

    private Response getClientResponse(OCSPReq ocspRequest) throws IOException {
        WebTarget webTarget = this.client.target(this.validationAuthorityURI);
        return webTarget.request().post(Entity.entity((Object)ocspRequest.getEncoded(), (String)OCSP_REQUEST_CONTENT_TYPE));
    }

    private X509Certificate getTrustedResponderCertificate(X509CertificateHolder responderCertificateHolder, X509Certificate issuerCertificate) throws CertificateException {
        X509Certificate responderCertificate = new JcaX509CertificateConverter().setProvider("BC").getCertificate(responderCertificateHolder);
        String trustedCAName = responderCertificate.getSubjectX500Principal().getName();
        if (this.trustedCAs.containsKey(trustedCAName)) {
            return this.trustedCAs.get(trustedCAName);
        }
        X500Principal issuerCA = issuerCertificate.getSubjectX500Principal();
        if (responderCertificate.getIssuerX500Principal().equals(issuerCA)) {
            return null;
        }
        return null;
    }
}

