/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.controller.reporting;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.nifi.controller.Counter;
import org.apache.nifi.controller.ProcessorNode;
import org.apache.nifi.controller.flow.FlowManager;
import org.apache.nifi.controller.repository.CounterRepository;
import org.apache.nifi.controller.repository.FlowFileEvent;
import org.apache.nifi.controller.repository.FlowFileEventRepository;
import org.apache.nifi.groups.ProcessGroup;
import org.apache.nifi.util.FormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogComponentStatuses
implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(LogComponentStatuses.class);
    private static final int METRIC_CACHE_SECONDS = 300;
    private static final String PROCESSOR_LINE_FORMAT = "| %1$-30.30s | %2$-36.36s | %3$-30.30s | %4$28.28s | %5$30.30s | %6$14.14s | %7$14.14s | %8$28.28s |\n";
    private static final String COUNTER_LINE_FORMAT = "| %1$-36.36s | %2$-36.36s | %3$28.28s | %4$28.28s |\n";
    private final FlowFileEventRepository flowFileEventRepository;
    private final CounterRepository counterRepository;
    private final FlowManager flowManager;
    private final String processorHeader;
    private final String processorBorderLine;
    private final String counterHeader;
    private final String counterBorderLine;
    private final Map<String, Long> previousCounterValues = new ConcurrentHashMap<String, Long>();
    private volatile long lastTriggerTime = System.currentTimeMillis();

    public LogComponentStatuses(FlowFileEventRepository flowFileEventRepository, CounterRepository counterRepository, FlowManager flowManager) {
        this.flowFileEventRepository = flowFileEventRepository;
        this.counterRepository = counterRepository;
        this.flowManager = flowManager;
        this.processorHeader = String.format(PROCESSOR_LINE_FORMAT, "Processor Name", "Processor ID", "Processor Type", "Bytes Read/sec", "Bytes Written/sec", "Tasks/sec", "Nanos/Task", "Percent of Processing Time");
        this.processorBorderLine = this.createLine(this.processorHeader);
        this.counterHeader = String.format(COUNTER_LINE_FORMAT, "Counter Context", "Counter Name", "Counter Value", "Increase/sec");
        this.counterBorderLine = this.createLine(this.counterHeader);
    }

    private String createLine(String valueToUnderscore) {
        StringBuilder processorBorderBuilder = new StringBuilder(valueToUnderscore.length());
        for (int i = 0; i < valueToUnderscore.length(); ++i) {
            processorBorderBuilder.append('-');
        }
        return processorBorderBuilder.toString();
    }

    @Override
    public void run() {
        try {
            if (!logger.isInfoEnabled()) {
                return;
            }
            this.logFlowFileEvents();
            this.logCounters();
        }
        catch (Exception e) {
            logger.error("Failed to log component statuses", (Throwable)e);
        }
    }

    private void logFlowFileEvents() {
        long timestamp = System.currentTimeMillis();
        ProcessGroup rootGroup = this.flowManager.getRootGroup();
        List allProcessors = rootGroup.findAllProcessors();
        long totalNanos = 0L;
        ArrayList<ProcessorAndEvent> processorsAndEvents = new ArrayList<ProcessorAndEvent>();
        for (ProcessorNode processorNode : allProcessors) {
            FlowFileEvent flowFileEvent = this.flowFileEventRepository.reportTransferEvents(processorNode.getIdentifier(), timestamp);
            if (flowFileEvent == null) continue;
            processorsAndEvents.add(new ProcessorAndEvent(processorNode, flowFileEvent));
            totalNanos += flowFileEvent.getProcessingNanoseconds();
        }
        Comparator<ProcessorAndEvent> comparator = Comparator.comparing(procAndEvent -> procAndEvent.getEvent().getProcessingNanoseconds());
        processorsAndEvents.sort(comparator.reversed());
        StringBuilder builder = new StringBuilder();
        builder.append("Processor Statuses:\n");
        builder.append(this.processorBorderLine);
        builder.append("\n");
        builder.append(this.processorHeader);
        builder.append(this.processorBorderLine);
        builder.append("\n");
        for (ProcessorAndEvent processorAndEvent : processorsAndEvents) {
            this.addStatus(processorAndEvent, builder, 300, totalNanos);
        }
        builder.append(this.processorBorderLine);
        logger.info(builder.toString());
    }

    private void addStatus(ProcessorAndEvent processorAndEvent, StringBuilder builder, int secondsInEvent, long totalNanos) {
        ProcessorNode processorNode = processorAndEvent.getProcessorNode();
        FlowFileEvent flowFileEvent = processorAndEvent.getEvent();
        long bytesReadPerSecond = flowFileEvent.getBytesRead() / (long)secondsInEvent;
        long bytesWrittenPerSecond = flowFileEvent.getBytesWritten() / (long)secondsInEvent;
        double invocations = (double)flowFileEvent.getInvocations() / (double)secondsInEvent;
        long nanos = flowFileEvent.getProcessingNanoseconds();
        double nanosPer = (double)nanos / invocations;
        double nanosRatio = (double)nanos / (double)totalNanos;
        double processingPercent = nanosRatio * 100.0;
        String processingPercentTwoDecimals = String.format("%.2f %%", processingPercent);
        String bytesRead = FormatUtils.formatDataSize((double)bytesReadPerSecond);
        String bytesWritten = FormatUtils.formatDataSize((double)bytesWrittenPerSecond);
        String invocationsPerSec = String.format("%.2f", invocations);
        String nanosPerInvocation = String.format("%.2f", nanosPer);
        builder.append(String.format(PROCESSOR_LINE_FORMAT, processorNode.getName(), processorNode.getIdentifier(), processorNode.getComponentType(), bytesRead, bytesWritten, invocationsPerSec, nanosPerInvocation, processingPercentTwoDecimals));
    }

    private void logCounters() {
        StringBuilder builder = new StringBuilder();
        builder.append("Counters:\n");
        builder.append(this.counterBorderLine);
        builder.append("\n");
        builder.append(this.counterHeader);
        builder.append(this.counterBorderLine);
        builder.append("\n");
        long now = System.currentTimeMillis();
        long millisSinceLastTrigger = now - this.lastTriggerTime;
        double secondsSinceLastTrigger = (double)millisSinceLastTrigger / 1000.0;
        this.lastTriggerTime = now;
        List counters = this.counterRepository.getCounters();
        counters.sort(Comparator.comparing(Counter::getContext).thenComparing(Counter::getName));
        for (Counter counter : counters) {
            String counterId = counter.getIdentifier();
            long lastValue = this.previousCounterValues.getOrDefault(counterId, 0L);
            this.previousCounterValues.put(counterId, counter.getValue());
            long increaseSinceLast = counter.getValue() - lastValue;
            double increasePerSecond = (double)increaseSinceLast / secondsSinceLastTrigger;
            String increase = String.format("%.2f", increasePerSecond);
            builder.append(String.format(COUNTER_LINE_FORMAT, counter.getContext(), counter.getName(), counter.getValue(), increase));
        }
        builder.append(this.counterBorderLine);
        logger.info(builder.toString());
    }

    private static class ProcessorAndEvent {
        private final ProcessorNode processorNode;
        private final FlowFileEvent event;

        public ProcessorAndEvent(ProcessorNode processorNode, FlowFileEvent event) {
            this.processorNode = processorNode;
            this.event = event;
        }

        public ProcessorNode getProcessorNode() {
            return this.processorNode;
        }

        public FlowFileEvent getEvent() {
            return this.event;
        }
    }
}

